/*
 * Z80SIM  -  a Z80-CPU simulator
 *
 * Copyright (C) 1987-92 by Udo Munk
 *
 * This modul contains a complex I/O-simulation for the Z80-CPU
 * simulation. Because this is an example, what you can do with
 * the CPU-emulation, you may change this modul for your needs,
 * and use it, or parts of it, in your own I/O-simulations.
 *
 * History:
 * 28-SEP-87 Development on TARGON/35 with AT&T Unix System V.3
 * 19-MAY-89 Additions for CP/M 3.0 und MP/M
 * 23-DEC-90 Ported to COHERENT 3.0
 * 10-JUN-92 Some optimization done
 */

/*
 *      Dieses Modul enthaelt die I/O-Handler zur Simulation
 *      der I/O-Umgebung eines CP/M-Rechners unter UNIX.
 *
 *      Belegungs-Tabelle der I/O-Ports:
 *
 *       0 - Konsole Status
 *       1 - Konsole Daten
 *
 *       2 - Printer Status
 *       3 - Printer Daten
 *
 *       4 - Auxilary Status
 *       5 - Auxilary Daten
 *
 *      10 - FDC Drive
 *      11 - FDC Track
 *      12 - FDC Sector
 *      13 - FDC Command
 *      14 - FDC Status
 *
 *      15 - DMA destination address low
 *      16 - DMA destination address high
 *
 *      20 - MMU Initialisierung
 *      21 - MMU Bank Select
 *
 *      25 - Clock Command
 *      26 - Clock Data
 *
 */

#include <stdio.h>
#include <signal.h>
#ifdef COHERENT
#include <sys/fcntl.h>
#else
#include <fcntl.h>
#include <malloc.h>
#include <memory.h>
#endif
#include <time.h>
#include "sim.h"
#include "simglb.h"

/*
 *      Struktur zur Beschreibung eines Laufwerks:
 *              Pointer auf Dateinamen
 *              Pointer auf den fd
 *              Anzahl Spuren
 *              Anzahl Sektoren
 */
struct dskdef {
	char *fn;
	int *fd;
	unsigned int tracks;
	unsigned int sectors;
};

static BYTE drive;      /* aktuelles Laufwerk A..P (0..15) */
static BYTE track;      /* aktuelle Spur (0..255) */
static BYTE sector;     /* aktueller Sektor (0..255) */
static BYTE status;     /* Status der letzten I/O-Operation */
static BYTE dmadl;      /* aktuelle DMA-Adresse destination low */
static BYTE dmadh;      /* aktuelle DMA-Adresse destination high */
static BYTE clkcmd;     /* Clock command */
static int drivea;      /* fd fuer Datei "drivea.cpm" */
static int driveb;      /* fd fuer Datei "driveb.cpm" */
static int drivec;      /* fd fuer Datei "drivec.cpm" */
static int drived;      /* fd fuer Datei "drived.cpm" */
static int drivee;      /* fd fuer Datei "drivee.cpm" */
static int drivef;      /* fd fuer Datei "drivef.cpm" */
static int driveg;      /* fd fuer Datei "driveg.cpm" */
static int driveh;      /* fd fuer Datei "driveh.cpm" */
static int drivei;      /* fd fuer Datei "drivei.cpm" */
static int drivej;      /* fd fuer Datei "drivej.cpm" */
static int drivek;      /* fd fuer Datei "drivek.cpm" */
static int drivel;      /* fd fuer Datei "drivel.cpm" */
static int drivem;      /* fd fuer Datei "drivem.cpm" */
static int driven;      /* fd fuer Datei "driven.cpm" */
static int driveo;      /* fd fuer Datei "driveo.cpm" */
static int drivep;      /* fd fuer Datei "drivep.cpm" */
static int printer;     /* fd fuer Datei "printer.cpm" */
static int auxin;       /* fd fuer pipe "auxin" */
static int auxout;      /* fd fuer pipe "auxout" */
static int aux_in_eof;  /* Status der pipe "auxin" (<>0 ist EOF) */
static int pid_rec;     /* PID des Emfangsprozesses fuer auxiliary */
static char last_char;  /* Puffer fuer 1 Zeichen (Console Status) */

static struct dskdef disks[16] = {
	{ "disks/drivea.cpm", &drivea, 77, 26 },
	{ "disks/driveb.cpm", &driveb, 77, 26 },
	{ "disks/drivec.cpm", &drivec, 77, 26 },
	{ "disks/drived.cpm", &drived, 77, 26 },
	{ "disks/drivee.cpm", &drivee, -1, -1 },
	{ "disks/drivef.cpm", &drivef, -1, -1 },
	{ "disks/driveg.cpm", &driveg, -1, -1 },
	{ "disks/driveh.cpm", &driveh, -1, -1 },
	{ "disks/drivei.cpm", &drivei, -1, -1 },
	{ "disks/drivej.cpm", &drivej, -1, -1 },
	{ "disks/drivek.cpm", &drivek, -1, -1 },
	{ "disks/drivel.cpm", &drivel, -1, -1 },
	{ "disks/drivem.cpm", &drivem, -1, -1 },
	{ "disks/driven.cpm", &driven, -1, -1 },
	{ "disks/driveo.cpm", &driveo, -1, -1 },
	{ "disks/drivep.cpm", &drivep, -1, -1 }
};

/*
 *      MMU:
 *      ===
 *
 *      +--------+
 * 16KB | Common |
 *      +--------+
 *      +--------+  +--------+  ..........  +--------+
 *      |        |  |        |              |        |
 * 48KB |        |  |        |  ..........  |        |
 *      | Bank 0 |  | Bank 1 |              | Bank n |
 *      +--------+  +--------+  ..........  +--------+
 */
#define MAXSEG 16               /* maximale Anzahl Speicherbaenke */
#define SEGSIZ 49152            /* Groesse einer Speicherbank = 48KBytes */
static char *mmu[MAXSEG];       /* MMU mit Pointern auf die Speicherbaenke */
static int selbnk;              /* momentan selektierte Bank */
static int maxbnk;              /* Anzahl initialisierter Speicherbaenke */

/*
 *      Vorwaerts-Deklaration der I/O-Unterprogramme
 *      fuer alle Portadressen
 */
BYTE io_trap();
BYTE cond_in(), cond_out(), cons_in(), cons_out();
BYTE prtd_in(), prtd_out(), prts_in(), prts_out();
BYTE auxd_in(), auxd_out(), auxs_in(), auxs_out();
BYTE fdcd_in(), fdcd_out();
BYTE fdct_in(), fdct_out();
BYTE fdcs_in(), fdcs_out();
BYTE fdco_in(), fdco_out();
BYTE fdcx_in(), fdcx_out();
BYTE dmal_in(), dmal_out();
BYTE dmah_in(), dmah_out();
BYTE mmui_in(), mmui_out(), mmus_in(), mmus_out();
BYTE clkc_in(), clkc_out(), clkd_in(), clkd_out();

/*
 *      Das folgende Array enthaelt fuer jede Port-Adresse
 *      jeweils eine Adresse fuer eine Funktion, die den
 *      Input und den Output durchfuehrt.
 */
static BYTE (*port[256][2]) () = {
	{ cons_in, cons_out },          /* port 0 */
	{ cond_in, cond_out },          /* port 1 */
	{ prts_in, prts_out },          /* port 2 */
	{ prtd_in, prtd_out },          /* port 3 */
	{ auxs_in, auxs_out },          /* port 4 */
	{ auxd_in, auxd_out },          /* port 5 */
	{ io_trap, io_trap  },          /* port 6 */
	{ io_trap, io_trap  },          /* port 7 */
	{ io_trap, io_trap  },          /* port 8 */
	{ io_trap, io_trap  },          /* port 9 */
	{ fdcd_in, fdcd_out },          /* port 10 */
	{ fdct_in, fdct_out },          /* port 11 */
	{ fdcs_in, fdcs_out },          /* port 12 */
	{ fdco_in, fdco_out },          /* port 13 */
	{ fdcx_in, fdcx_out },          /* port 14 */
	{ dmal_in, dmal_out },          /* port 15 */
	{ dmah_in, dmah_out },          /* port 16 */
	{ io_trap, io_trap  },          /* port 17 */
	{ io_trap, io_trap  },          /* port 18 */
	{ io_trap, io_trap  },          /* port 19 */
	{ mmui_in, mmui_out },          /* port 20 */
	{ mmus_in, mmus_out },          /* port 21 */
	{ io_trap, io_trap  },          /* port 22 */
	{ io_trap, io_trap  },          /* port 23 */
	{ io_trap, io_trap  },          /* port 24 */
	{ clkc_in, clkc_out },          /* port 25 */
	{ clkd_in, clkd_out }           /* port 26 */
};

/*
 *      Diese Funktion initialisiert die I/O-Handler:
 *      1. alle nicht verwendeten Port-Adressen mit dem
 *         I/O-Trap-Handler belegen.
 *      2. Die MMU wird mit NULL-Pointern initialisiert
 *      3. die Dateien, die die Plattenlaufwerke simulieren,
 *         werden geoeffnet. Die Datei fuer Drive A muss sich
 *         oeffnen lassen, sonst kann nicht gebootet werden !
 *         Fehlerhafte Dateieroeffnungen fuer die restlichen
 *         15 Laufwerke werden durch Eintragen des fd-Pointers
 *         NULL in die Struktur dskdef des jeweiligen Laufwerks
 *         gekennzeichnet.
 *      4. Die Datei "printer.cpm" zur Simulation des Druckers
 *         wird angelegt".
 *      5. Der Emfangsprozess fuer die serielle Schnittstelle
 *         wird gestartet.
 *      6. Die named pipes "auxin" und "auxout" zur Simulation einer
 *         seriellen Schnittstelle werden geoeffnet.
 */
void init_io()
{
	void exit(), perror();
	register int i;

	for (i = 27; i <= 255; i++) {
		port[i][0] = io_trap;
		port[i][1] = io_trap;
	}
	for (i = 0; i < MAXSEG; i++)
		mmu[i] = NULL;
	if ((*disks[0].fd = open(disks[0].fn, O_RDWR)) == -1) {
		perror("file disks/drivea.cpm");
		exit(1);
	}
	for (i = 1; i <= 15; i++)
		if ((*disks[i].fd = open(disks[i].fn, O_RDWR)) == -1)
			disks[i].fd = NULL;
	if ((printer = creat("printer.cpm", 0644)) == -1) {
		perror("file printer.cpm");
		exit(1);
	}
	pid_rec = fork();
	switch (pid_rec) {
	case -1:
		puts("can't fork");
		exit(1);
	case 0:
		execlp("receive", "receive", "auxiliary.cpm", 0);
		puts("can't exec receive process");
		exit(1);
	}
	if ((auxin = open("auxin", O_RDONLY | O_NDELAY)) == -1) {
		perror("pipe auxin");
		exit(1);
	}
	if ((auxout = open("auxout", O_WRONLY)) == -1) {
		perror("pipe auxout");
		exit(1);
	}
}

/*
 *      Diese Funktion beendet die I/O-Handler:
 *      1. Die Dateien, die die Plattenlaufwerke simulieren,
 *         werden geschlossen.
 *      2. Die Datei "printer.cpm" wird geschlossen.
 *      3. Die named pipes "auxin" und "auxout" werden geschlossen.
 *      4. Der Emfangsprozess fuer die serielle Schnittstelle wird
 *         abgebrochen.
 */
void exit_io()
{
	register int i;

	for (i = 0; i <= 15; i++)
		if (disks[i].fd != NULL)
			close(*disks[i].fd);
	close(printer);
	close(auxin);
	close(auxout);
	kill(pid_rec, SIGHUP);
}

/*
 *      Dieser I/O-Handler wird durch die CPU-Befehle IN
 *      aufgerufen, und ruft seinerseits den Input-Handler
 *      fuer die entsprechende Port-Adresse auf
 */
BYTE io_in(adr)
register BYTE adr;
{
	return((*port[adr][0]) ());
}

/*
 *      Dieser I/O-Handler wird durch die CPU-Befehle OUT
 *      aufgerufen, und ruft seinerseits den Output-Handler
 *      fuer die entsprechende Port-Adresse auf
 */
BYTE io_out(adr, data)
register BYTE adr, data;
{
	(*port[adr][1]) (data);
}

/*
 *      I/O-Trap-Handler
 */
static BYTE io_trap()
{
	cpu_error = IOTRAP;
	cpu_state = STOPPED;
}

/*
 *      I/O-Handler fuer Konsole Status lesen:
 *      0xff : Eingabezeichen verfuegbar
 *      0x00 : keine Eingabezeichen verfuegbar
 */
static BYTE cons_in()
{
	register int flags, readed;

	fflush(stdout);
	if (last_char)
		return((BYTE) 0xff);
	if (cntl_c)
		return((BYTE) 0xff);
	if (cntl_bs)
		return((BYTE) 0xff);
	else {
		flags = fcntl(0, F_GETFL, 0);
		fcntl(0, F_SETFL, flags | O_NDELAY);
		readed = read(0, &last_char, 1);
		fcntl(0, F_SETFL, flags);
		if (readed == 1)
			return((BYTE) 0xff);
		else
			return((BYTE) 0);
	}
}

/*
 *      I/O-Handler fuer Konsole Status schreiben:
 *      keine Reaktion
 */
static BYTE cons_out(data)
register BYTE data;
{
	data = data;
}

/*
 *      I/O-Handler fuer Konsole Daten lesen:
 *      Es wird ein Zeichen vom Terminal gelesen
 *      (kein Echo und keine Zeichenumwandlungen)
 */
static BYTE cond_in()
{
	char c;

	fflush(stdout);
	aborted:
	if (last_char) {
		c = last_char;
		last_char = '\0';
	} else if (cntl_c) {
		cntl_c--;
		c = 0x03;
	} else if (cntl_bs) {
		cntl_bs--;
		c = 0x1c;
	} else if (read(0, &c, 1) != 1) {
	       goto aborted;
	}
	return((BYTE) c);
}

/*
 *      I/O-Handler fuer Konsole Daten schreiben:
 *      Es wird ein Zeichen auf das Terminal ausgegeben
 */
static BYTE cond_out(data)
register BYTE data;
{
	putchar(data & 0x7f);
	if (data == '\f')
		printf("\033H\033J");
}

/*
 *      I/O-Handler fuer Drucker Status lesen:
 *      Drucker ist immer bereit
 */
static BYTE prts_in()
{
	return((BYTE) 0xff);
}

/*
 *      I/O-Handler fuer Drucker Status schreiben:
 *      keine Reaktion
 */
static BYTE prts_out(data)
register BYTE data;
{
	data = data;
}

/*
 *      I/O-Handler fuer Drucker Daten lesen:
 *      liefert immer 0
 */
static BYTE prtd_in()
{
	return((BYTE) 0);
}

/*
 *      I/O-Handler fuer Drucker Daten schreiben:
 *      Das Zeichen wird in die Datei "printer.cpm"
 *      geschrieben.
 */
static BYTE prtd_out(data)
BYTE data;
{
	if (data != '\r')
		write(printer, (char *) &data, 1);
}

/*
 *      I/O-Handler fuer aux Status lesen
 */
static BYTE auxs_in()
{
	return((BYTE) aux_in_eof);
}

/*
 *      I/O-Handler fuer aux Status schreiben:
 *      EOF-Status veraendern
 */
static BYTE auxs_out(data)
register BYTE data;
{
	aux_in_eof = data;
}

/*
 *      I/O-Handler fuer aux Daten lesen:
 *      liefert das naechste Byte aus der pipe auxin
 */
static BYTE auxd_in()
{
	char c;

	if (read(auxin, &c, 1) == 1)
		return((BYTE) c);
	else {
		aux_in_eof = 0xff;
		return((BYTE) 0x1a);    /* CP/M EOF */
	}
}

/*
 *      I/O-Handler fuer aux Daten schreiben:
 *      schreibt das Byte in die pipe auxout
 */
static BYTE auxd_out(data)
BYTE data;
{
	if (data != '\r')
		write(auxout, (char *) &data, 1);
}

/*
 *      I/O-Handler fuer FDC drive lesen:
 *      das aktuelle Laufwerk wird geliefert
 */
static BYTE fdcd_in()
{
	return((BYTE) drive);
}

/*
 *      I/O-Handler fuer FDC drive schreiben:
 *      das gewunschte Laufwerk wird uebernommen
 */
static BYTE fdcd_out(data)
register BYTE data;
{
	drive = data;
}

/*
 *      I/O-Handler fuer FDC track lesen:
 *      die aktuelle Spur wird geliefert
 */
static BYTE fdct_in()
{
	return((BYTE) track);
}

/*
 *      I/O-Handler fuer FDC track schreiben:
 *      die gewuenschte Spur wird uebernommen
 */
static BYTE fdct_out(data)
register BYTE data;
{
	track = data;
}

/*
 *      I/O-Handler fuer FDC sector lesen:
 *      der aktuelle Sektor wird geliefert
 */
static BYTE fdcs_in()
{
	return((BYTE) sector);
}

/*
 *      I/O-Handler fuer FDC sector schreiben:
 *      der gewuenschte Sektor wird uebernommen
 */
static BYTE fdcs_out(data)
register BYTE data;
{
	sector = data;
}

/*
 *      I/O-Handler fuer FDC Command lesen:
 *      liefert immer 0, sonst keine Reaktion
 */
static BYTE fdco_in()
{
	return((BYTE) 0);
}

/*
 *      I/O-Handler fuer FDC Command schreiben:
 *      Ein Sektor wird die die gewuenschte Richtung uebertragen,
 *      0 = lesen, 1 = schreiben
 *
 *      Status-Codes:
 *        0 - ok
 *        1 - ungueltiges Laufwerk
 *        2 - ungueltige Spur
 *        3 - ungueltiger Sektor
 *        4 - Positionierungsfehler
 *        5 - Lesefehler
 *        6 - Schreibfehler
 *        7 - ungueltiger Befehl
 */
static BYTE fdco_out(data)
register BYTE data;
{
	register long pos;
	long lseek();

	if (disks[drive].fd == NULL) {
		status = 1;
		return;
	}
	if (track > disks[drive].tracks) {
		status = 2;
		return;
	}
	if (sector > disks[drive].sectors) {
		status = 3;
		return;
	}
	pos = (((long)track) * ((long)disks[drive].sectors) + sector - 1) << 7;
	if (lseek(*disks[drive].fd, pos, 0) == -1L) {
		status = 4;
		return;
	}
	switch (data) {
	case 0:                 /* lesen */
		if (read(*disks[drive].fd, (char *) ram + (dmadh << 8) + dmadl, 128) != 128)
			status = 5;
		else
			status = 0;
		break;
	case 1:                 /* schreiben */
		if (write(*disks[drive].fd, (char *) ram + (dmadh << 8) + dmadl, 128) != 128)
			status = 6;
		else
			status = 0;
		break;
	default:                /* ungueltiger Auftrag */
		status = 7;
		break;
	}
}

/*
 *      I/O-Handler fuer FDC status lesen:
 *      der Status der letzten I/O-Operation wird geliefert
 *      0 = ok, sonst Fehler
 */
static BYTE fdcx_in()
{
	return((BYTE) status);
}

/*
 *      I/O-Handler fuer FDC status schreiben:
 *      keine Reaktion
 */
static BYTE fdcx_out(data)
register BYTE data;
{
	data = data;
}

/*
 *      I/O-Handler fuer DMA low lesen:
 *      das aktuelle lower Byte der Adresse wird geliefert
 */
static BYTE dmal_in()
{
	return((BYTE) dmadl);
}

/*
 *      I/O-Handler fuer DMA low schreiben:
 *      das gewuenschte lower Byte der Adresse wird uebernommen
 */
static BYTE dmal_out(data)
register BYTE data;
{
	dmadl = data;
}

/*
 *      I/O-Handler fuer DMA high lesen:
 *      das aktuelle higher Byte der Adresse wird geliefert
 */
static BYTE dmah_in()
{
	return((BYTE) dmadh);
}

/*
 *      I/O-Handler fuer DMA high schreiben:
 *      das gewuenschte higher Byte der Adresse wird uebernommen
 */
static BYTE dmah_out(data)
register BYTE data;
{
	dmadh = data;
}

/*
 *      I/O-Handler fuer MMU initialisieren lesen:
 *      liefert die Anzahl der initialisierten
 *      Speicherbaenke der MMU
 */
static BYTE mmui_in()
{
	return((BYTE) maxbnk);
}

/*
 *      I/O-Handler fuer MMU initialisieren schreiben:
 *      nur beim ersten Aufruf werden entsprechend viele
 *      Speicherbaenke angefordert, und die Pointer auf die
 *      Speicherflaechen in die MMU eingetragen
 */
static BYTE mmui_out(data)
register BYTE data;
{
	register int i;

	if (mmu[0] != NULL)
		return;
	if (data > MAXSEG) {
		printf("Try to init %d banks, available %d banks\n", data, MAXSEG);
		exit(1);
	}
	for (i = 0; i < data; i++) {
		if ((mmu[i] = malloc(SEGSIZ)) == NULL) {
			printf("can't allocate memory for bank %d\n", i+1);
			exit(1);
		}
	}
	maxbnk = data;
}

/*
 *      I/O-Handler fuer MMU Bank Select lesen:
 *      liefert die momentan im CPU-Adressbereich
 *      eingeblendete Speicherbank
 */
static BYTE mmus_in()
{
	return((BYTE) selbnk);
}

/*
 *      I/O-Handler fuer MMU Bank Select schreiben:
 *      wenn die selektierte Bank ungleich der momentan eingeblendetet
 *      Bank ist, wird letztere in die Speicherflaeche, deren Pointer
 *      in der MMU eingetragen ist, gesichert. Dann wird die
 *      Speicherflaeche der selektierten Bank in den CPU-Adressraum
 *      kopiert, und die selektierte Bank wird zur momentanen Bank.
 */
static BYTE mmus_out(data)
register BYTE data;
{
	if (data > maxbnk) {
		printf("Try to select unallocated bank %d\n", data);
		exit(1);
	}
	if (data == selbnk)
		return;
	memcpy(mmu[selbnk], (char *) ram, SEGSIZ);
	memcpy((char *) ram, mmu[data], SEGSIZ);
	selbnk = data;
}

/*
 *      I/O-Handler fuer Clock Command lesen:
 *      liefert das letzte Clock Command
 */
static BYTE clkc_in()
{
	return(clkcmd);
}

/*
 *      I/O-Handler fuer Clock Command schreiben:
 *      uebernimmt das neue Clock Command
 */
static BYTE clkc_out(data)
register BYTE data;
{
	clkcmd = data;
}

/*
 *      I/O-Handler fuer Clock Data lesen:
 *      abhaengig vom letzten Clock Command werden folgende
 *      Daten aus der UNIX-Systemuhr geliefert:
 *              0 - Sekunden in BCD
 *              1 - Minuten in BCD
 *              2 - Stunden in BCD
 *              3 - low Byte Anzahl Tage seit 1.1.1978
 *              4 - high Byte Anzahl Tage seit 1.1.1978
 *          sonst - 0
 */
static BYTE clkd_in()
{
	register struct tm *t;
	register int val;
	extern long time();
	long Time;

	time(&Time);
	t = localtime(&Time);
	switch(clkcmd) {
	case 0:                 /* Sekunden in BCD */
		val = to_bcd(t->tm_sec);
		break;
	case 1:                 /* Minuten in BCD */
		val = to_bcd(t->tm_min);
		break;
	case 2:                 /* Stunden in BCD */
		val = to_bcd(t->tm_hour);
		break;
	case 3:                 /* low Byte Tage */
		val = get_date(t) & 255;
		break;
	case 4:                 /* high Byte Tage */
		val = get_date(t) >> 8;
		break;
	default:
		val = 0;
		break;
	}
	return((BYTE) val);
}

/*
 *      I/O-Handler fuer Clock Data schreiben:
 *      da die UNIX-Systemuhr nur vom Superuser gesetzt werden kann,
 *      erfolgt hier keine Reaktion
 */
static BYTE clkd_out(data)
register BYTE data;
{
	data = data;
}

/*
 *      Integer in BCD umwandeln
 */
static int to_bcd(val)
register int val;
{
	register int i = 0;

	while (val >= 10) {
		i += val / 10;
		i <<= 4;
		val %= 10;
	}
	i += val;
	return (i);
}

/*
 *      Anzahl Tage seit 1.1.1978 bestimmen
 */
static int get_date(t)
register struct tm *t;
{
	register int i;
	register int val = 0;

	for (i = 1978; i < 1900 + t->tm_year; i++) {
		val += 365;
		if (i % 4 == 0)
			val++;
	}
	val += t->tm_yday + 1; /* warum muss hier 1 addiert werden ? */
	return(val);
}
