/*	xcmain.c -- main module for XC
	This file uses 4-character tabstops
*/

#include <sys/types.h>
#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#ifndef COHERENT
#include <termio.h>
#else
#include <sgtty.h>
#endif
#ifdef T6000
#include <sys/ioctl.h>
#endif
#include <setjmp.h>
#include "xc.h"

#define Resume_Not_Allowed	1

short
	autoflag =	FALSE,	/* Automatic capturing */
	cismode =	FALSE,	/* Automatic response to CIS "ENQ" */
	cr_add =	TRUE,	/* Add cr to nl in B+ uploads */
	hdplxflag =	FALSE,	/* Half-duplex mode */
	nlmode =	TRUE,	/* Map nl to cr when transmitting ASCII */
	reterm =	FALSE,	/* Jumping into terminal mode */
	statflag =	FALSE,	/* Flag for status display */
	eofflag =	FALSE;	/* Flag to quit a script */
int s_cis(), s_set(), s_exit(), s_shell();
char Msg[SM_BUFF];
FILE *tfp;
#ifndef COHERENT
struct termio newmode, oldmode, sigmode;
#else
struct sgttyb newmode, oldmode, sigmode;
#endif
static char	*statfmt = "\t\t%-8s %25s %s\r\n",
			version[]="@(#)XC 3.2 JPRadley 24 Aug 1991",
			oldshell[SM_BUFF],
			*babble[] = {
				"\r\nUsage: xc [-l device] [-t] [-s file]",
				"\t-l device\tUse 'device' as the modem port",
				"\t-t\t\tEnter terminal mode immediately",
				"\t-s script\tExecute 'script' immediately",
				NULLS
			};
static s_script(), s_xmodem(), s_term(), s_help(), s_dial(), puttake(),
		SET_7bit(), SET_cr(), SET_cis(), SET_nl(), SET_xoff(),
		SET_hdplx(), SET_bps(), SET_autocapt(), SET_cfile(), SET_pfile();
extern short scriptflag;
extern void B_Transfer(), dbglog(), mattach(), terminal(), xreceive(), xsend(),
			get_ttype(), unlock_tty();
jmp_buf erret;			/* non-local error return */

struct kw {					/* Used by command parsing routines */
	char *keyword;
	int (*rtn)();
};

static struct kw cmds[] = {
	{"c",		s_cis},
	{"s",		s_script},
	{"h",		hangup},
	{"rb",		s_xmodem},
	{"rt",		s_xmodem},
	{"sb",		s_xmodem},
	{"st",		s_xmodem},
	{"set",		s_set},
	{"t",		s_term},
	{"d",		s_dial},
	{"q",		s_exit},
	{"x",		s_exit},
	{"exit",	s_exit},
	{"!",		s_shell},
	{"!!",		s_shell},
	{"$",		s_shell},
	{"%p",		puttake},
	{"%t",		puttake},
	{"help",	s_help},
	{"?",		s_help},
	{NULLS,		0}
};

static struct kw setlist[] = {
	{"7bit",	SET_7bit},
	{"auto",	SET_autocapt},
	{"bps",		SET_bps},
	{"cfile",	SET_cfile},
	{"cis",		SET_cis},
	{"cr",		SET_cr},
	{"hdplx",	SET_hdplx},
	{"nl",		SET_nl},
	{"pfile",	SET_pfile},
	{"xoff",	SET_xoff},
	{NULLS,		0}
};

/* Print the status of the program */
static void status()
{
	struct kw *ptr;
	char p[30];
	int (*fct)() = 0;

	statflag = TRUE;

	cls();
	cur_off();
	sprintf(p,"Modem Port: %s",mport(NULLS));
	drawline(0, 0, CO);
	ttgoto(1, 9);
	sprintf(Msg,"%-29s%29s",&version[4], p);
	S;
	drawline(2, 0, CO);
	ttgoto(3, 0);
	fprintf(tfp, statfmt, "Keyword", "Description", "Status");
	fprintf(tfp, statfmt, "--------", "-------------------------", "-----------");

	for (ptr = setlist; ptr->keyword != NULLS; ptr++)
		if (ptr->rtn != fct){
			fct = ptr->rtn;
			(*fct)();
		}

	ttgoto(18, 25);
	S1("Type \"help\" or ? for help");
	statflag = FALSE;
	cur_on();
}

/* Catch a signal and jump to main. Reset signal and do a longjmp */
static void catch()
{
	S2("XC: Interrupt");

	signal(SIGINT,catch);
	signal(SIGQUIT,catch);
	longjmp(erret,1);
}

static void usage()
{
	char **ptr;

	for (ptr = babble; *ptr != NULLS; ptr++)
		fprintf(tfp, "%s\r\n", *ptr);
}

main(argc, argv)
int argc;
char *argv[];
{
	char *script = NULLS;
	extern char *optarg;
	int c;
	extern int optind;

	struct kw *ptr;
	tfp = stderr;
	if (isatty(0))
		get_ttype();

#ifndef COHERENT
	ioctl(0, TCGETA, &oldmode);	/* get current tty mode	*/
#else
	ioctl(0, TIOCGETP, &oldmode);	/* get current tty mode	*/
#endif

	/* trap for SIGHUP and SIGTERM, make sure LCKfile gets killed */
	signal(SIGHUP, s_exit);
	signal(SIGTERM, s_exit);

	newmode = oldmode;

#ifndef COHERENT
	newmode.c_iflag &= ~(IXON | IXOFF | IXANY);
	newmode.c_lflag &= ~(ICANON | ISIG | ECHO);
	newmode.c_oflag = 0;
	newmode.c_cc[VMIN] = 1;
	newmode.c_cc[VTIME] = 1;

	sigmode = newmode;
	sigmode.c_lflag |= ISIG;
#else
	newmode.sg_flags |= CBREAK | RAW;
	newmode.sg_flags &= ~ECHO;

	sigmode = newmode;
#endif

	oldshell[0] = '\0';	/* set last command to blank */
	if (setjmp(erret))	/* set error handler to exit */
		exit(0);		/*  while parsing command line */
	signal(SIGINT,catch);	/* catch break & quit signals/keys */
	signal(SIGQUIT,catch);

	while ((c = getopt(argc, argv, "s:l:t")) != -1)
		switch (c){
		case 'l':	/* set modem port name */
			mport(optarg);
			break;
		case 's':	/* Execute SCRIPT file */
			script = optarg;
			break;
		case 't':	/* jump into terminal mode */
			reterm = TRUE;
			break;
		default:	/* Bad command .. print help */
			usage();
			exit(1);
		}

	setuid((int)geteuid());
	setgid((int)getegid());

	mopen();	/* opens and configures modem port, or exits */

	setuid((int)getuid());
	setgid((int)getgid());

	linkflag=2;
	do_script(STARTUP);

#ifdef DEBUG
	dbglog();
#endif

	if (!script)
		status();

	for (;;){
		setjmp(erret);
		signal(SIGQUIT, s_exit);
		isig(FALSE);

		if (script)
			do_script(script),
			script = NULLS,
			reterm = TRUE;

		if (reterm){
			s_term();
			continue;
		}

		isig(TRUE);
		show(-1,"<XC>");
		fputc(' ',tfp);

		lptr = line;
		getline();
		isig(FALSE);
		fputc('\r',tfp),
		fputc('\n',tfp);

		getword();
		lc_word(word);
		if (word[0] == '\0')		/* If blank line... reprompt */
			continue;

		for (ptr = cmds; ptr->keyword != NULLS; ptr++)
			if (!strcmp(word, ptr->keyword))
				break;

		if (ptr->keyword)
			(*ptr->rtn)();
		else
			sprintf(Msg,"Unrecognized command: %s",word),
			S;
	}
}

static s_script()
{
	getword();

	if (word[0] == '\0'){
		S1("Script file not specified");
		return;
	}

	sprintf(ddsname,"%s",word);
	do_script(ddsname);
	reterm = TRUE;
}

static s_xmodem()
{
	char d = word[0];
	char c = word[1];

	xc_setflow(FALSE);
	isig(TRUE);

	getword();
	if (word[0] == '\0')
		S1("Transfer file not specified");

	else if (d == 's')
		xsend(c);
	else
		xreceive(c);

	reterm = TRUE;

	isig(FALSE);
	xc_setflow(flowflag);
}

static s_term()
{
	for (;;){
		terminal(FALSE);
		if (cismode != 2)
			return;
		cismode = 1;
		s_cis();
	}
}

static s_dial()
{
	for (;;){
		terminal(TRUE);
		if (cismode != 2)
			return;
		cismode = 1;
		s_cis();
	}
}

s_cis()
{
	xc_setflow(FALSE);
	isig(TRUE);

	B_Transfer();
	reterm = TRUE;

	isig(FALSE);
	xc_setflow(flowflag);
}

s_shell()
{
	int stat_loc = 0;
	char c = word[0];
	static char *shell = NULLS;
	void (*oldvec)();

#ifdef NOSHELL
	return(0);
#endif
	if (word[0] == word[1])
		strcpy(wptr = word, oldshell);
	else {
		getword();
		if (*wptr)
			strcpy(oldshell, wptr);
	}

	if (!shell){
		shell = getenv("SHELL");
		if (!shell)
			shell = "/bin/sh";
	}

	fputc('\r',tfp),
	fputc('\n',tfp);
#ifndef COHERENT
	ioctl(0, TCSETAW, &oldmode);
#else
	ioctl(0, TIOCSETP, &oldmode);
#endif

	if (!forkem()){
		if (c == '$')	/* Attach modem to stdin, stdout */
			mattach();
		signal(SIGCLD,SIG_DFL);
		signal(SIGINT,SIG_DFL);
		signal(SIGQUIT,SIG_DFL);
		if (word[0] == '\0')
			execl(shell, shell, "-i", NULLS);
		else
			execl(shell, shell, "-c", wptr, NULLS);
		S1("Exec failed!");
		exit(2);
	}

	oldvec = signal(SIGINT,SIG_IGN);
	wait(&stat_loc);
	signal(SIGINT,oldvec);

	isig(FALSE);
	strcpy(oldshell, wptr);
	return(!!stat_loc);
}

static char	*cmdlist[] = {
	"\tXC Command Summary\r\n\n",
	"\tc\t\tInitiate CIS B+ File Transfer (Upload and Download)\r\n",
	"\trb file\t\tXMODEM receive (binary mode)\r\n",
	"\trt file\t\tXMODEM receive (Ascii mode)\r\n",
	"\tsb file ...\tXMODEM send (binary mode)\r\n",
	"\tst file ...\tXMODEM send (Ascii mode)\r\n",
	"\tset ...\t\tXC Parameter set\n\r\n",
	"\ts file\t\tExecute XC script file\r\n",
	"\tt\t\tEnter terminal mode\r\n",
	"\td\t\tDialing directory\r\n",
	"\th\t\tHang up the modem\r\n",
	"\tq, x or exit\tExit XC\r\n\n",
#ifndef NOSHELL
	"\t!\t\tExecute a local interactive shell\r\n",
	"\t! cmd\t\tExecute shell command string on the local system\r\n",
	"\t!!\t\tRe-execute the last shell command string\r\n",
	"\t$ cmd\t\tShell command with stdin and stdout redirected to modem\r\n",
#endif	
	"\t%p loc [rem]\tPut local file to a UNIX system\r\n",
	"\t%t rem [loc]\tTake remote file from a UNIX system\r\n",
	"\t? or help\tPrint this help screen\r\n",
	"",
	"\tSET Keywords:\r\n\n",
	"\tset\t\t\tDisplay current XC status\r\n",
	"\tset 7bit on|off\t\tSet/Reset 7-bit data mask (ignore high bits)\r\n",
	"\tset auto on|off\t\tSet/Reset automatic capturing\r\n",
	"\tset bps <value>\t\tSet Bits/Second to <value>\r\n",
	"\tset cfile name\t\tChange name of capture file\r\n",
	"\tset cis on|off\t\tSet/Reset CIS <ENQ> mode (Auto up/download)\r\n",
	"\tset cr on|off\t\tSet/Reset Carriage Return Injection mode\r\n",
	"\tset hdplx on|off\tSet/Reset half-duplex terminal mode\r\n",
	"\tset nl on|off\t\tSet/Reset newline translation\r\n",
	"\tset pfile name\t\tChange name of phonelist file\r\n",
	"\tset xoff on|off\t\tSet/Reset XON/XOFF flow control\r\n",
	"",
	"\tESCAPE sequences in TERMINAL mode:\r\n\n",
	"\t",ESC_STR," b\tSend modem break\r\n",
	"\t",ESC_STR," d\tDialing directory\r\n",
	"\t",ESC_STR," f\tSend a file through the modem (ASCII transfer)\r\n",
	"\t",ESC_STR," s\tExecute XC script file\r\n",
	"\t",ESC_STR," h\tHang up the modem\r\n",
	"\t",ESC_STR," y\tStart capturing\r\n",
	"\t",ESC_STR," t\tStop capturing\r\n",
	"\t",ESC_STR," x\tExit terminal mode back to XC command mode\r\n",
	"\t",ESC_STR," q\tQuit XC entirely\r\n",
	"",
	NULLS };

static s_help()
{
	char **ptr = cmdlist;

	cls();
	cur_off();
	for ( ; *ptr != NULLS; ptr++)
		if (**ptr)
			fprintf(tfp, "%s", *ptr);
		else
			S0("PRESS ENTER"),
			getline(),
			cls();
	status();
}

s_set()
{
	struct kw *ptr;

	getword();

	if (word[0] == '\0' && !scriptflag){
		status();
		return;
	} else if (word[0] == '\0'){
		S1("SET keyword requires an argument");
		eofflag++;
		return;
	}

	lc_word(word);

	for (ptr = setlist; ptr->keyword != NULLS; ptr++)
		if (!strcmp(ptr->keyword, word)){
			(*ptr->rtn)();
			return;
		}

	sprintf(Msg,"Invalid SET keyword: %s", word);
	S;
	eofflag++;
}

void set_onoff(flag)
short *flag;
{
	char *ptr = strdup(word);

	getword();
	lc_word(word);

	if (!strcmp(word, "on"))
		*flag = TRUE;
	else if (!strcmp(word, "off"))
		*flag = FALSE;
	else
		sprintf(Msg,"Set '%s' value must be 'on' or 'off'",ptr),
		S,
		eofflag++;

	free(ptr);
}

static SET_7bit()
{
	short val;

	if (statflag){
		fprintf(tfp, statfmt, "7bit", "Seven-bit Mask",
			bitmask == 0x7f ? "ON" : "OFF");
		return;
	}

	set_onoff(&val);
	bitmask = val ? 0x7f : 0xff;

	if (!scriptflag)
		sprintf(Msg,"%s-bit mask enabled", val ? "Seven" : "Eight"),
		S;
}

static SET_cr()
{
	if (statflag){
		fprintf(tfp, statfmt, "cr", "Carriage Return Injection",
			cr_add ? "ON" : "OFF");
		return;
	}

	set_onoff(&cr_add);

	if (!scriptflag)
		sprintf(Msg,"Carriage Returns %s injected in B+ ASCII uploads",
			cr_add ? "ARE" : "are NOT"),
		S;
}

static SET_nl()
{
	if (statflag){
		fprintf(tfp, statfmt, "nl", "Newline Translation",
			nlmode ? "ON" : "OFF");
		return;
	}

	set_onoff(&nlmode);

	if (!scriptflag)
		sprintf(Msg,"Newlines %s changed to Carriage Returns",
			nlmode ? "ARE" : "are NOT"),
		S;
}

static SET_cis()
{
	if (statflag){
		fprintf(tfp, statfmt, "cis", "CIS <ENQ> Auto Download",
			cismode ? "ON" : "OFF");
		return;
	}

	set_onoff(&cismode);

	if (!scriptflag)
		sprintf(Msg,"CIS <ENQ> Auto Download is %s", cismode ? "ON" : "OFF"),
		S;
}

static SET_xoff()
{
	if (statflag){
		fprintf(tfp, statfmt, "xoff", "Terminal mode XON/XOFF",
			flowflag ? "ON" : "OFF");
		return;
	}

	set_onoff(&flowflag);
	xc_setflow(flowflag);

	if (!scriptflag)
		sprintf(Msg,"XOFF/XON Flow control is %s", flowflag ? "ON" : "OFF"),
		S;
}

static SET_bps()
{
	if (statflag){
		char br[10];
		sprintf(br, "%d", 10*mrate(NULLS));
		fprintf(tfp, statfmt, "bps", "Bits per Second", br);
		return;
	}

	getword();
	if (word[0] == '\0')
		S1("Set Bps must have a rate");
	else if (mrate(word) < 0)
		sprintf(Msg,"Unsupported bps rate %s",word),
		S;
	eofflag++;
	if (!scriptflag)
		sprintf(Msg,"Bits/Second set to %d",10*mrate(NULLS)),
		S;
}

static SET_hdplx()
{
	if (statflag){
		fprintf(tfp, statfmt, "hdplx", "Half-duplex mode",
			hdplxflag ? "ON" : "OFF");
		return;
	}

	set_onoff(&hdplxflag);

	if (!scriptflag)
		sprintf(Msg,"Half-duplex Mode is %s", hdplxflag ? "ON" : "OFF"),
		S;
}

static SET_autocapt()
{
	if (statflag){
		fprintf(tfp, statfmt, "auto", "Auto capture",
			autoflag ? "ON" : "OFF");
		return;
	}

	set_onoff(&autoflag);

	if (!scriptflag)
		sprintf(Msg,"Auto Capture is %s", autoflag ? "ON" : "OFF"),
		S;
}

static SET_cfile()
{
	if (statflag){
		fprintf(tfp, statfmt, "cfile", "Capture file", captfile);
		return;
	}

	getword();
	if (word[0] == '\0'){
		S1("Set CFILE must have file name");
		eofflag++;
		return;
	}

	strcpy(captfile, word);

	if (!scriptflag)
		sprintf(Msg,"Capture file set to '%s'",captfile),
		S;
}

static SET_pfile()
{
	if (statflag){
		fprintf(tfp, statfmt, "pfile", "Phone number file", phonefile);
		return;
	}

	getword();
	if (word[0] == '\0'){
		S1("Set PFILE must have file name");
		eofflag++;
		return;
	}

	strcpy(phonefile, word);

	if (!scriptflag)
		sprintf(Msg,"Phone number file set to '%s'",phonefile),
		S;
}

/*	Put and Take a file to/from a UNIX-type "cu" system. Unfortunately,
	the stty command is one of those commands that always gets changed
	with different UNIX systems, so you will get (at least) a file full of
	^M on the take command for systems later than V7 or work-alikes.

	Additionally, the Take command takes a bit too much!

	Fixed a lot of this: JPRadley 89/07/27
*/

static puttake()
{
	FILE *fp;
	int i, Ch;
	char c = word[1], fname[SM_BUFF], tname[SM_BUFF], wrkbuff[SM_BUFF];

	getword();

	isig(TRUE);
	signal(SIGINT,catch);
	signal(SIGQUIT,catch);
	xc_setflow(TRUE);
	if (word[0] == '\0'){
		sprintf(Msg,"Must give a filename with the '%%%c' option",c);
		S;
		return;
	}

	strcpy(fname, word);
	getword();
	if (word[0] == '\0')
		strcpy(tname, fname);
	else
		strcpy(tname, word);
	switch (c){
	case 'p':
		if ((fp = fopen(fname, "r")) == NULLF){
			sprintf(Msg,"Can't open '%s'",fname);
			S;
		} else {
			fprintf(tfp, "\r\nPutting file '%s' to '%s' on remote UNIX\r\n",
				fname, tname);
			sprintf(wrkbuff, "stty -echo;cat >%s;sleep 1;stty echo\n", tname);
			send_string(wrkbuff);	/* send command string to remote shell */
			i = 64;
			while ((Ch = getc(fp)) != EOF){
				if (++i > 64){		/* this prevents an overload on the */
					i = 0;	 /* receiver's input buffer (64=kludge) */
#ifndef COHERENT					
					msecs((CBAUD-cbaud) * 100);
#else
					msecs(500L);
#endif					
				}
				send_mbyte(Ch);		/* send characters to cat command */
			}
			fclose(fp);
			send_mbyte(EOT);		/* send a ^D to cat */
			purge();				/* get rid of whatever was sent back */
			send_mbyte('\n');
		}
		break;

	case 't':
		strcpy(Name, tname);
		if ((fp=QueryCreate(Resume_Not_Allowed)) != NULLF){
			fprintf(tfp, "\r\nTaking file '%s' from remote UNIX to '%s'\r\n",
				fname, tname);
										/* if 'fname' has a DLE, we'll die */
			purge();
			sprintf(wrkbuff, "stty nl;cat %s;echo %c;stty -nl;sleep 1\n",
				fname, DLE);
			send_string(wrkbuff);		/* send command to remote shell */
			while (read_mbyte(3) != '\n'); /* discard up to the \n in wrkbuff */
				;
			while ((Ch=read_mbyte(0)) != -1 && /* while chars are being sent */
						Ch != DLE)			   /* and we haven't seen our DLE */
			fputc(Ch,fp);
			fclose(fp);
		}
		break;
	}
	isig(FALSE);
	xc_setflow(flowflag);
	reterm = TRUE;
}

s_exit()
{
	signal(SIGHUP,SIG_IGN);
	signal(SIGINT,SIG_IGN);
	signal(SIGQUIT,SIG_IGN);
	signal(SIGTERM,SIG_IGN);

	unlock_tty();
#ifndef COHERENT	
	ioctl(0, TCSETAF, &oldmode);
#else
	ioctl(0, TIOCSETP, &oldmode);
#endif

	exit(0);
}
