/* $Id: tua.c,v 3.3 1991/09/01 14:02:49 piggy Rel $
 * HDB-uucp Activity Analyzer. Eat all the uucp logs and extract per system,
 * user and daily stats.
 *
 *   Copyright (C) 1991  Lele Gaifax (piggy@idea.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include    <stdio.h>
#include    <time.h>
#include    "hdbstat.h"
#ifdef USG
#include    <string.h>
#else
#include    <strings.h>
#endif
#include    "patchlevel.h"
#include    "getopt.h"

#if defined(USG) && !defined(COHERENT)
#include <malloc.h>
#else
extern char *malloc ();
#endif

#ifdef __GNUC__
static inline void
error (char *str)
{
  fprintf (stderr, "Error in processing %s\n", str);
  exit (1);
}

#else
#define	error(str)  { \
			fprintf(stderr, "Error in processing %s\n", str); \
			exit(1); \
		    }
#endif

#ifdef STRDUP_MISSING

char *
strdup (string)
     char *string;

{
  char *tmp = malloc (strlen (string) + 1);

  if (tmp)
    return (strcpy (tmp, string));
  else
    return tmp;
}

#endif /* STRDUP_MISSING */


int SeparatedCommLog = FALSE;
int DoCommandReport = FALSE;
int VerboseOutput = FALSE;
int JustSomeSystem = FALSE;
int ChartSize = 10;
char *PrefixPath = "/usr/spool/uucp";
char *CurrentTime;
int CurrentMonth;
char *Since;
int DaysSinceLastClean;

static void
DisplayUsage ()
{
  fprintf (stderr, "The Uucp Analyzer (TUA) rel. %s by piggy@idea.sublink.org (Lele Gaifax)\n\nOptions:\n", RELEASE);
  fputs ("\
-S, +no-sys-rep      Do not show the report for the Systems\n\
-U, +no-user-rep     Do not show the report for the Users\n\
-D, +no-daily-rep    Do not show the day-by-day report\n\
-m, +no-monthly-act  Do not show the monthly activity report\n\
-h, +no-history      Do not show the history\n\
-y, +no-hourly-chart Do not show the hourly activity chart\n\
-C, +command-lists   Show the various command lists\n\
-O, +only-system SYS Consider just SYS in the reports\n", stderr);
  fputs ("\
-c, +separate-com    Build a separate log for the commands\n\
-k, +kill-system SYS Do not show anything about system SYS\n\
-K, +kill-user USER  Do not show anything about user USER\n\
-H, +update-hist     Update the history file\n\
-0, +reset-hist      Reset the history file\n\
-p, +prefix-path DIR Use DIR instead of /usr/spool/uucp (for debugging only)\n\
-z, +chart-size SIZE Set the size of the hourly chart to SIZE rows\n", stderr);
  fputs ("\
-o, +chart-only      Print only the hourly activity chart\n\
-v, +verbose         Print infos while processing\n\
-i, +help            Get this help\n\
", stderr);

#ifdef HDB_ERR_FACT
  fputs ("\
-f, +hdb-factor F    Set the Correction Factor for HDB to F (by default. 2.5)\n\
", stderr);
#endif

  fputs ("\n\tPlease report bugs and problems to:\n\
\t\tpiggy@idea.sublink.org\n\n", stderr);
  fputs ("\
TUA may be copied only under the terms of the GNU General Public License,\n\
a copy of which can be found with the TUA distribution package\n", stderr);
}

int
main (argc, argv)
     int argc;
     char *argv[];
{
  extern double atof ();
  extern int uname ();
  extern char *optarg;
  extern int opterr;
  int opt;
  int optidx;
  static int DoSystemReport = TRUE;
  static int DoUserReport = TRUE;
  static int DoDailyReport = TRUE;
  static int SaveHistory = FALSE;
  static int DoHistoryReport = TRUE;
  static int ResetHistory = FALSE;
  static int DoMonthlyActivityReport = TRUE;
  static int DoHourlyActivityChart = TRUE;
  static int DoChartOnly = FALSE;
  extern long time ();
  char SystemName[9];
  long Now = time ((long *) 0);
  static struct option long_options[] =
  {
    {"no-sys-rep", 0, &DoSystemReport, FALSE},	/* -S */
    {"no-user-rep", 0, &DoUserReport, FALSE},	/* -U */
    {"no-daily-rep", 0, &DoDailyReport, FALSE},	/* -D */
    {"command-lists", 0, &DoCommandReport, TRUE},	/* -C */
    {"no-monthly-act", 0, &DoMonthlyActivityReport, FALSE},	/* -m */
    {"no-history", 0, &DoHistoryReport, FALSE},	/* -h */
    {"only-system", 1, 0, 'O'},	/* -O */
#ifdef HDB_ERR_FACT
    {"hdb-factor", 1, 0, 'f'},	/* -f */
#endif
    {"separate-com", 0, &SeparatedCommLog, TRUE},	/* -c */
    {"kill-system", 1, 0, 'k'},	/* -k */
    {"kill-user", 1, 0, 'K'},	/* -K */
    {"update-hist", 0, &SaveHistory, TRUE},	/* -H */
    {"reset-hist", 0, &ResetHistory, TRUE},	/* -0 */
    {"help", 0, 0, 'i'},	/* -i */
    {"prefix-path", 1, 0, 'p'},	/* -p */
    {"chart-size", 1, 0, 'z'},	/* -z */
    {"no-hourly-chart", 0, &DoHourlyActivityChart, FALSE},	/* -y */
    {"chart-only", 0, &DoChartOnly, TRUE},	/* -o */
    {"verbose", 0, &VerboseOutput, TRUE},	/* -v */
    {0, 0, 0, 0}
  };

  CurrentTime = strdup (asctime (localtime (&Now)));
  CurrentTime[strlen (CurrentTime) - 1] = '\0';	/* Strip the ending '\0' */

  CurrentMonth = localtime (&Now)->tm_mon;

  opterr = FALSE;

#ifdef SYSTEMID
  if ((gethostname (SystemName, 9) == -1) || *SystemName == '\0')
    {
      FILE *fd;

      if ((fd = fopen (SYSTEMID, "r")) == NULL)
	strcpy (SystemName, "UNKNOWN");
      else
	{
	  int last = strlen (fgets (SystemName, 9, fd)) - 1;

	  if (last > 0)
	    {
	      if (SystemName[last] == '\n')
		SystemName[last] = '\0';
	    }
	  else
	    strcpy (SystemName, "??????");
	  fclose (fd);
	}
    }
#else
  if ((gethostname (SystemName, 9) == -1) || *SystemName == '\0')
    strcpy (SystemName, "UNKNOWN");
#endif

#ifdef HDB_ERR_FACT
#define OPTSTR "SUDCm:hck:K:H0ifp:z:yovO:"
#else
#define OPTSTR "SUDCm:hck:K:H0ip:z:yovO:"
#endif

  while ((opt = getopt_long (argc, argv, OPTSTR, long_options, &optidx)) != EOF)
    {
      if (opt == 0 && long_options[optidx].flag == 0)
	opt = long_options[optidx].val;
      switch (opt)
	{
	case 0:
	  break;
	case 'S':
	  DoSystemReport = FALSE;
	  break;
	case 'U':
	  DoUserReport = FALSE;
	  break;
	case 'D':
	  DoDailyReport = FALSE;
	  break;
	case 'C':
	  DoCommandReport = TRUE;
	  break;
	case 'm':
	  DoMonthlyActivityReport = FALSE;
	  break;
	case 'h':
	  DoHistoryReport = FALSE;
	  break;
	case 'c':
	  SeparatedCommLog = FALSE;
	  break;
	case 'k':
	  KillSys (optarg);
	  break;
	case 'K':
	  KillUser (optarg);
	  break;
	case 'H':
	  SaveHistory = TRUE;
	  break;
	case '0':
	  ResetHistory = TRUE;
	  break;
#ifdef HDB_ERR_FACT
	case 'f':
	  ERRFACT = (float) atof (optarg);
	  if (ERRFACT <= 0.0)
	    {
	      fprintf (stderr,
		       "Bad -f argument. Error factor fixed to %.1f\n",
		       HDB_ERR_FACT);
	      ERRFACT = HDB_ERR_FACT;
	    }
	  break;
#endif
	case 'p':
	  PrefixPath = optarg;
	  break;
	case 'z':
	  ChartSize = atoi (optarg);
	  break;
	case 'y':
	  DoHourlyActivityChart = FALSE;
	  break;
	case 'o':
	  DoChartOnly = TRUE;
	  break;
	case 'v':
	  VerboseOutput = TRUE;
	  break;
	case 'O':
	  (void) InsertSys (optarg);
	  JustSomeSystem = TRUE;
	  break;
	case '?':
	case 'i':
	  DisplayUsage ();
	  exit (1);
	  break;
	}
    }

  printf ("The Uucp Analyzer (TUA) rel. %s by piggy@idea.sublink.org (Lele Gaifax)\n\n", RELEASE);

  if (ReadHistory () == ERROR)
    error ("rhistory");
  if (ProcXferstats () == ERROR)
    error ("xferstats");
  if (!DoChartOnly)
    {
      if (ProcUucico () == ERROR)
	error ("uucico log");
      if (ProcUuxqt () == ERROR)
	error ("uuxqt log");
      if (ProcUucp () == ERROR)
	error ("uucp log");
      if (ProcUux () == ERROR)
	error ("uux log");

      Since = strdup (ctime (&StatStartingTime));
      Since[strlen (Since) - 1] = '\0';	/* Strip the ending '\n' */

      DaysSinceLastClean = (Now - StatStartingTime) / (3600L * 24L) + 1;
      printf ("HoneyDanBeer UUCP Analysis for ``%.9s%s'',\n  since %s to %s (%d day%c)\n\n",
	      SystemName, DOMAIN, Since, CurrentTime, DaysSinceLastClean,
	      (DaysSinceLastClean > 1 ? 's' : ' '));
      if (DoSystemReport)
	SystemsReport ();
      if (DoUserReport)
	UsersReport ();
      if (DoDailyReport)
	DailyReport ();
      if (SeparatedCommLog)
	{
	  if (DoSystemReport)
	    SystemsCommandReport ();
	  if (DoUserReport)
	    UsersCommandReport ();
	  if (DoDailyReport)
	    DailyCommandReport ();
	}
      SystemSummary ();
      SystemSummaryTable ();
      if (*TheEpoc && DoHistoryReport)
	{
	  SystemHistoryTable ();
	  if (DoMonthlyActivityReport)
	    MonthlyHistoryTable ();
	}
      if (DoHourlyActivityChart)
	HourlyActivityCharts ();
      if (SaveHistory)
	if (WriteHistory (ResetHistory) == ERROR)
	  error ("whistory: probably you need uucp's write permission");
    }
  else
    {
      printf ("HoneyDanBeer UUCP Analysis for ``%.9s%s''\n\n", SystemName, DOMAIN);
      HourlyActivityCharts ();
    }
  exit (0);
}
