/* $Id: reports.c,v 3.3 1991/09/01 14:02:43 piggy Rel $
 * Print reports.
 *
 *   Copyright (C) 1991  Lele Gaifax (piggy@idea.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * SYNOPSIS:
 * void	SystemsReport();
 * void SystemsCommandReport();
 * void UsersReport();
 * void UsersCommandReport();
 * void DailyReport();
 * void DailyCommandReport();
 * void SystemSummary();
 * void SystemSummaryTable();
 * void SystemHistoryTable();
 * void MonthlyHistoryTable();
 * void HourlyActivityCharts();
 *
 */

#include    <stdio.h>

#if defined(USG) && !defined(COHERENT)
#include    <malloc.h>
#else
extern char * malloc();
#endif

#include    "hdbstat.h"

#define BYTES_PER_KBYTE         1024.0
#define BYTES_PER_MBYTE         1024000.0

#define NotZero(rec)	\
    (rec.FilesOut || rec.FilesIn || rec.BytesOut || rec.BytesIn || \
     rec.TimeOut || rec.TimeIn)

static char *			/* Transforms seconds to "hh:mm:ss" */
TimeToString (tim)
     float tim;
{
  char *str;
  int hh, mm, ss;
  register time_t ltim = (time_t) tim;

  ltim = (time_t) tim;
  hh = (int) (ltim / 3600L);
  ltim %= 3600L;
  mm = (int) (ltim / 60L);
  ss = (int) (ltim % 60L);
  str = malloc (20);
  sprintf (str, "%4d:%02d:%02d.%02d",
           hh, mm, ss, (int) ((tim - (int) (tim)) * 100));
  return (str);
}

static void
Commands (comm)
     commrep_t comm;
{
  if (comm.Number > 1)
    printf ("\t(%3d) ", comm.Number);
  else
    printf ("\t      ");
  printf ("%s\n", comm.Command);
}

static void
SR (system)
     sysrep_t system;
{
  if (NotZero (system))
    {
      printf ("%-11.11s  %4d %8.f%s %4d   %4d %8.f%s %4d\n",
	      system.System,
	      system.FilesIn, system.BytesIn / BYTES_PER_KBYTE,
	      TimeToString (system.TimeIn),
	      (system.TimeIn ? (int) (system.BytesIn / system.TimeIn) : 0),
	      system.FilesOut, system.BytesOut / BYTES_PER_KBYTE,
	      TimeToString (system.TimeOut),
	    (system.TimeOut ? (int) (system.BytesOut / system.TimeOut) : 0));
      if (DoCommandReport && !SeparatedCommLog && system.Commands)
	{
	  puts ("\tCommands:\n\t=========");
	  EnquiryComm (system.Commands, Commands);
	  puts ("");
	}
    }
}

static void
SRnocomm (system)
     sysrep_t system;
{
  if (system.Commands)
    {
      printf ("    %s:\n", system.System);
      EnquiryComm (system.Commands, Commands);
      puts ("");
    }
}

void
SystemsReport ()
{
  puts ("By System:");
  puts ("==========");
  puts ("                     R E C E I V E D                       S E N T");
  puts ("System       Files  KBytes     Time      ATP   Files  KBytes     Time      ATP\n");
  EnquirySys (SR);
}

void
SystemsCommandReport ()
{
  if (DoCommandReport)
    {
      puts ("\nCommands By System:");
      puts ("===================");
      EnquirySys (SRnocomm);
      puts ("");
    }
}

static void
UR (user)
     userrep_t user;
{
  printf ("%-11.11s  %4d  %8.f  %s\n",
	  user.User,
          user.FilesOut, user.BytesOut / BYTES_PER_KBYTE,
          TimeToString (user.TimeOut));
  if (DoCommandReport && !SeparatedCommLog && user.Commands)
    {
      puts ("\tCommands:");
      puts ("\t=========");
      EnquiryComm (user.Commands, Commands);
      puts ("");
    }
}

static void
URnocomm (user)
     userrep_t user;
{
  if (user.Commands)
    {
      printf ("    %s:\n", user.User);
      EnquiryComm (user.Commands, Commands);
      puts ("");
    }
}

void
UsersReport ()
{
  puts ("\nBy User:");
  puts ("========");
  puts ("                        S E N T");
  puts ("User         Files   KBytes      Time\n");
  EnquiryUser (UR);
}

void
UsersCommandReport ()
{
  if (DoCommandReport)
    {
      puts ("\nBy User:");
      puts ("========");
      EnquiryUser (URnocomm);
    }
}

static void
DR (day)
     dailyrep_t day;
{
  printf ("%02d/%02d  %4d  %8.f  %s  %4d  %8.f  %s\n",
	  day.Month, day.Day,
	  day.FilesIn, day.BytesIn / BYTES_PER_KBYTE,
          TimeToString (day.TimeIn),
	  day.FilesOut, day.BytesOut / BYTES_PER_KBYTE,
          TimeToString (day.TimeOut));
  if (DoCommandReport && !SeparatedCommLog && day.Commands)
    {
      puts ("\tCommands:");
      puts ("\t=========");
      EnquiryComm (day.Commands, Commands);
      puts ("");
    }
}

static void
DRnocomm (day)
     dailyrep_t day;
{
  if (day.Commands)
    {
      printf ("    %02d/%02d:\n", day.Month, day.Day);
      EnquiryComm (day.Commands, Commands);
      puts ("");
    }
}

void
DailyReport ()
{
  puts ("\nBy Day:");
  puts ("=======");
  puts ("Date       R E C E I V E D                     S E N T");
  puts ("Mo/Dy  Files   KBytes       Time      Files   KBytes       Time\n");
  EnquiryDay (DR);
}

void
DailyCommandReport ()
{
  if (DoCommandReport)
    {
      puts ("\nBy Day:");
      puts ("=======");
      EnquiryDay (DRnocomm);
    }
}

static void
SummBySys (system)
     sysrep_t system;
{
  extern char * ctime PROTO((long *));
  
  if (NotZero (system) || system.Calls)
    {
      register idx;

      printf ("\n%s:\n", system.System);
      printf ("\tRecvd\t%10.f (%4d Files) in %s: %4d cps\n",
	      system.BytesIn, system.FilesIn, TimeToString (system.TimeIn),
	      (system.TimeIn ? (int) (system.BytesIn / system.TimeIn) : 0));
      printf ("\tSent\t%10.f (%4d Files) in %s: %4d cps\n",
	    system.BytesOut, system.FilesOut, TimeToString (system.TimeOut),
	   (system.TimeOut ? (int) (system.BytesOut / system.TimeOut) : 0));
      if (system.BytesIn > system.BytesOut)
	printf ("\tRecvd\t%10.f bytes more than sent.\n",
		system.BytesIn - system.BytesOut);
      else if (system.BytesIn < system.BytesOut)
	printf ("\tSent\t%10.f bytes more than received.\n",
		system.BytesOut - system.BytesIn);
      printf ("\tThe system has been connected for  %s\n",
	      TimeToString (system.TimeConnect));
      printf ("\tLast connection: %s", ctime(&system.LastConnection));
      for (idx = 0; idx < PHONE_PRICING_TB; idx++)
        if (system.PhoneCost[idx] != .0)
          printf ("\tTime spent transmitting in %s: %s\n",
                  PhoneCategoryNames[idx],
                  TimeToString (system.PhoneCost[idx]));
      printf ("\tThere have been %d call%s (with an average of %d Ok/day):\n\t\t\t\t\t",
	      system.Calls, (system.Calls == 1 ? "" : "s"),
	      (system.CallsOK / DaysSinceLastClean));
      if (system.CallsOK)
        {
          if (system.CallsSTOPPED)
            printf ("%3d OK (%3d IN, %3d OUT, %3d BREAKED)\n\t\t\t\t\t",
                    system.CallsOK, system.CallsIn,
                    system.CallsOut, system.CallsSTOPPED);
          else
            printf ("%3d OK (%3d IN, %3d OUT)\n\t\t\t\t\t",
                    system.CallsOK, system.CallsIn, system.CallsOut);
        }
      if (system.CallsFAIL)
	printf ("%3d FAILED\n\t\t\t\t\t", system.CallsFAIL);
    }
}

void
SystemSummary ()
{
  puts ("\nSUMMARY by System:");
  puts ("==================");
  EnquirySys (SummBySys);
  puts ("");
}

static float TotTimeIn = 0.0, TotTimeOut = 0.0;
static float TotBytesIn = 0.0, TotBytesOut = 0.0;
static int TotFilesIn = 0, TotFilesOut = 0;

static char *
BytesToString (byt)
     float byt;
{
  char *stringa = malloc (11);

  sprintf (stringa, "%9.f", byt / BYTES_PER_KBYTE);
  return stringa;
}

static void
SummBySysTable (sys)
     sysrep_t sys;
{
  float TotTime = sys.TimeIn + sys.TimeOut;
  float TotBytes = sys.BytesOut + sys.BytesIn;

  if (NotZero (sys))
    {
      TotTimeIn += sys.TimeIn;
      TotTimeOut += sys.TimeOut;
      TotBytesIn += sys.BytesIn;
      TotBytesOut += sys.BytesOut;
      TotFilesIn += sys.FilesIn;
      TotFilesOut += sys.FilesOut;
      printf ("|%-8.8s| %s :%10.10s: %5d | %s :%10.10s: %5d | %4d|\n",
	      sys.System, BytesToString (sys.BytesIn), TimeToString (sys.TimeIn), sys.FilesIn,
      BytesToString (sys.BytesOut), TimeToString (sys.TimeOut), sys.FilesOut,
	      (TotTime ? (int) (TotBytes / TotTime) : 0));
    }
}

void
SystemSummaryTable ()
{
  puts ("+--------+------------------------------+------------------------------+-----+");
  puts ("|        |        R E C E I V E D       |            S E N T           |     |");
  puts ("| SYSTEM | KiloBytes :   Time   : Files | KiloBytes :   Time   : Files | ATP |");
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
  EnquirySys (SummBySysTable);
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
  printf ("| TOTALS | %s :%10.10s: %5d | %s :%10.10s: %5d | ####|\n",
	  BytesToString (TotBytesIn), TimeToString (TotTimeIn), TotFilesIn,
       BytesToString (TotBytesOut), TimeToString (TotTimeOut), TotFilesOut);
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
}

static void
SysHistoryTable (sys)
     sysrep_t sys;
{
  float TotTime = sys.History.TimeIn + sys.History.TimeOut +
  sys.TimeIn + sys.TimeOut;
  float TotBytes = sys.History.BytesOut + sys.History.BytesIn +
  sys.BytesIn + sys.BytesOut;

  if (NotZero (sys) || NotZero (sys.History))
    {
      TotTimeIn += sys.History.TimeIn;
      TotTimeOut += sys.History.TimeOut;
      TotBytesIn += sys.History.BytesIn;
      TotBytesOut += sys.History.BytesOut;
      TotFilesIn += sys.History.FilesIn;
      TotFilesOut += sys.History.FilesOut;
      printf ("|%-8.8s| %s :%10.10s: %5d | %s :%10.10s: %5d | %4d|\n",
	      sys.System, BytesToString (sys.History.BytesIn + sys.BytesIn),
	      TimeToString (sys.History.TimeIn + sys.TimeIn),
	      sys.History.FilesIn + sys.FilesIn,
	      BytesToString (sys.History.BytesOut + sys.BytesOut),
	      TimeToString (sys.History.TimeOut + sys.TimeOut),
	      sys.History.FilesOut + sys.FilesOut,
	      (TotTime ? (int) (TotBytes / TotTime) : 0));
    }
}

void
SystemHistoryTable ()
{
  printf ("\n...and since %s\n", TheEpoc);
  puts ("+--------+------------------------------+------------------------------+-----+");
  puts ("|        |        R E C E I V E D       |            S E N T           |     |");
  puts ("| SYSTEM | KiloBytes :   Time   : Files | KiloBytes :   Time   : Files | ATP |");
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
  EnquirySys (SysHistoryTable);
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
  printf ("| TOTALS | %s :%10.10s: %5d | %s :%10.10s: %5d | ####|\n",
	  BytesToString (TotBytesIn), TimeToString (TotTimeIn), TotFilesIn,
       BytesToString (TotBytesOut), TimeToString (TotTimeOut), TotFilesOut);
  puts ("+--------+-----------:----------:-------+-----------:----------:-------+-----|");
}

static float MonthlyActivityTotals[12];

static void
MonHistoryTable (sys)
     sysrep_t sys;
{
  int idx;
  float sum = .0;

  for (idx = 0; idx <= 11; idx++)
    sum += sys.History.MonthlyActivity[idx];

  if (sum > 0.0)
    {
      printf ("\n%-6.6s", sys.System);
      for (idx = 0; idx <= 11; idx++)
	{
	  float bytes =
	  sys.History.MonthlyActivity[(idx + CurrentMonth + 1) % 12];

	  MonthlyActivityTotals[(idx + CurrentMonth + 1) % 12] += bytes;
	  if (bytes == -1.0)
	    printf ("| *** ");
	  else if (bytes < 10.0 * BYTES_PER_KBYTE)
	    printf ("|%4.0f ", bytes);
	  else if (bytes < 10.0 * BYTES_PER_MBYTE)
	    printf ("|%4.0fK", bytes / BYTES_PER_KBYTE);
	  else
	    printf ("|%4.0fM", bytes / BYTES_PER_MBYTE);
	}
    }
}

void
MonthlyHistoryTable ()
{
  static char *MonthName[] =
  {
    "Jan", "Feb", "Mar", "Apr",
    "May", "Jun", "Jul", "Ago",
    "Sep", "Oct", "Nov", "Dic"
  };
  int idx;

  puts ("\n\nLast 12 Months Activity");
  puts ("=======================\n");
  printf ("------+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----\n");
  printf ("System");
  for (idx = 0; idx <= 11; idx++)
    {
      printf ("| %s", MonthName[(idx + CurrentMonth + 1) % 12]);
      if (idx != 11)
	putchar (' ');
    }
  printf ("\n------+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----");
  EnquirySys (MonHistoryTable);
  printf ("\n------+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----\n");
  printf ("TOTALS");
  for (idx = 0; idx <= 11; idx++)
    {
      float bytes = MonthlyActivityTotals[(idx + CurrentMonth + 1) % 12];

      if (bytes < 0.0)
	printf ("| *** ");
      else if (bytes < 10.0 * BYTES_PER_KBYTE)
	printf ("|%4.0f ", bytes);
      else if (bytes < 10.0 * BYTES_PER_MBYTE)
	printf ("|%4.0fK", bytes / BYTES_PER_KBYTE);
      else
	printf ("|%4.0fM", bytes / BYTES_PER_MBYTE);
    }
  printf ("\n------+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+-----\n");
}

static float PortActivityTotals[TIMESLICES] = {0};
static int IsGlobalActivityChart = 0;
static int NumberOfPorts = 0;

static void
CountPorts (port)
     portact_t port;

{
  NumberOfPorts++;
}

static void
PortActivityChart (port)
     portact_t port;
{
  int row, slice;
  float maxval = 0.0;
  float scale;

  for (slice = 0; slice < TIMESLICES; slice++)
    {
      float curval;

      if (!IsGlobalActivityChart)
	PortActivityTotals[slice] += (curval = port.Activity[slice]);
      else
	curval = PortActivityTotals[slice];

      if (curval > maxval)
	maxval = curval;
    }
  scale = maxval / ChartSize;

  if (IsGlobalActivityChart || NumberOfPorts == 1)
    {
      printf ("\n\nGlobal Hourly Activity (on a 20 minutes basis)\n");
      puts ("==============================================\n");
    }
  else
    {
      printf ("\n\nHourly Activity (on a 20 minutes basis) on %s\n", port.Port);
      puts ("======================================================\n");
    }
  printf ("Max Value :     %.0f bytes\nScale     : '#'=%.0f bytes\n\n", maxval, scale);
  for (row = ChartSize; row > 0; row--)
    {
      float current_value = row * scale;

      printf ("     ");
      for (slice = 0; slice < 72; slice++)
	{
	  float curval = (IsGlobalActivityChart ?
			  PortActivityTotals[slice] : port.Activity[slice]);
	  if (curval >= current_value)
	    putchar ('#');
	  else
	    putchar (' ');
	}
      puts ("");
    }
  puts ("     ========================================================================");
  puts ("     0  1  2  3  4  5  6  7  8  9  10 11 12 13 14 15 16 17 18 19 20 21 22 23");
}

void
HourlyActivityCharts ()
{
  portact_t dummy;

  EnquiryPort (CountPorts);
  EnquiryPort (PortActivityChart);
  if (NumberOfPorts > 1)
    {
      IsGlobalActivityChart++;
      PortActivityChart (dummy);
    }
}
