/*{{{  Notes*/
/*
keybind.c - allow a Emacs-like definition of keybindings
Author: Wolfgang Stumvoll 1991

keybind produces 2 files:

  $HOME/.origamirc
    Contains the information, origami needs to startup
    the internal keyboardtranslation.

  $HOME/.origamibind
    Contains the help, origami displays as keybindhelp.
*/
/*}}}  */

#define ORIGAMI

/*{{{  #includes*/
#include <ctype.h>
#include <stdio.h>

#include "bool.h"
#include "argparse.h"

#include "keys.h"
#include "rcformat.h"
#include "keybind.h"
#include "scanner.h"
/*}}}  */

/*{{{F keybind.h*/
/*:::F keybind.h*/
/*}}}  */
/*{{{F scanner.h*/
/*:::F scanner.h*/
/*}}}  */
/*{{{F scanner.c*/
/*:::F scanner.c*/
/*}}}  */
/*{{{F parser.c*/
/*:::F parser.c*/
/*}}}  */

/*{{{  argument switch variables*/
char filebase[60];
bool sorting=FALSE, usage=FALSE, verbose=FALSE, list_keys=FALSE;
int keynumber;
int t_counter=0,k_counter=0;
/*}}}  */

/*{{{  display usage*/
void do_usage()
{
  fprintf(stderr,"Usage: keybind {-o<namebase>} {-s} binding-file\n");
  fprintf(stderr,"       keybind -k<key number>\n");
}
/*}}}  */
/*{{{  write procedures*/
/*{{{  write one key definition to rc file*/
write_key_rc(op,code) int op; char *code;
{
  if (*code=='\0')
  {
    fprintf(stderr,"empty code-sequence\n");
    return(1);
  }
  putc(RC_DEFKEY,rc);
  putw(op,rc);
  putw(strlen(code),rc);
  while (*code) putc(*code++,rc);
  k_counter++;
  return(0);
}
/*}}}  */
/*{{{  write binding_name to rc file*/
write_name_rc(name) char *name;
{
  int i=5;

  if (*name=='\0')
  {
    fprintf(stderr,"empty binding-name\n");
    return(1);
  }
  putc(RC_KEYNAME,rc);
  while (i--) {
    if (*name<' ') putc('\0',rc); else putc(*name,rc);
    name++;
  }
  return(0);
}
/*}}}  */
/*{{{  write one macro definition to rc file*/
write_macro_rc(tag,op,lg,code) readtags tag; TOKEN op, *code; int lg;
{
  if (lg==0) {
    fprintf(stderr,"empty code-sequence\n");
    return(1);
  }
  if (!(tag==RC_DEFMACRO || tag==RC_INITMACRO)) {
    fprintf(stderr,"wrong macrotag\n");
    return(1);
  }
  /*{{{  verbose-mode*/
  if (verbose) {
    if (tag==RC_DEFMACRO)
      fprintf(stderr,"static macrobuffer %d uses %d tokens ",op,lg);
    else
      fprintf(stderr,"defineable macrobuffer %d uses %d tokens ",op,lg);
    t_counter += lg;
  }
  /*}}}  */
  putc(tag,rc);
  putw(op,rc);
  putw(lg,rc);
  while (lg--) putw(*code++,rc);
  return(0);
}
/*}}}  */
/*{{{  write one definition to bind file*/
write_bind(name,code) char *name,*code;
{
  fprintf(bind,"%-35.*s%s\n",strlen(name),name,code);
}
/*}}}  */
/*}}}  */

/*{{{  argument parser table*/
#define argnum 6

ARG argtab[] =
{
  'k', INT_ARG, &keynumber,
  'o', STRING_ARG, filebase,
  's', BOOL_ARG, &sorting,
  '?', BOOL_ARG, &usage,
  'v', BOOL_ARG, &verbose,
  'l', BOOL_ARG, &list_keys,
};
/*}}}  */

/*{{{  main*/
main(argc,argv) int argc; char **argv;
{
char *home=getenv("HOME"),
     rcfile[128],
     bindfile[128];
bool ok;
  init_alias();
  argc=argparse(argc,argv,argtab,argnum);
  /*{{{  handle missing output argument*/
  if (*filebase=='\0') strcpy(filebase,".origami");
  /*}}}  */
  /*{{{  handle -? for usage*/
  if (usage)
  {
    do_usage();
    exit(0);
  }
  /*}}}  */
  /*{{{  handle -k<keynumber>*/
  if (keynumber)
  {
    KEY *seek;

    seek = &bindings;
    while (seek->num) {
    if ((seek->num)==keynumber)
    {
      fprintf(stderr,"%d = %s\n",keynumber,seek->name);
      break;
    }
    seek++;
    }
    if (!seek->num) {
      fprintf(stderr,"unkown key number %d!\n",keynumber);
      exit(1);
    }
    exit(0);
  }
  /*}}}  */
  /*{{{  handle -l*/
  if (list_keys) {
    sc_list_keys();
    exit(0);
  }
  /*}}}  */
  /*{{{  open files/pipe*/
  /*{{{  open input-file*/
  if ((in=fopen(argv[1],"r"))==NULL) {
    fprintf(stderr,"Can't open binding-file\n");
    exit(1);
  }
  read_in();
  /*}}}  */
  /*{{{  open .origamirc*/
  sprintf(rcfile,"%s/%src",home,filebase);
  if ((rc=fopen(rcfile,"w"))==NULL) {
    fprintf(stderr,"Can't open %s/%src\n",home,filebase);
    exit(1);
  }
  putc(RC_FORMAT,rc);
  /*}}}  */
  /*{{{  open .origamibind*/
  /*{{{  if no sorting open file*/
  if (sorting==0) {
    sprintf(bindfile,"%s/%sbind",home,filebase);
    if ((bind=fopen(bindfile,"w"))==NULL) {
      fprintf(stderr,"Can't open file %s/%sbind\n",home,filebase);
      exit(1);
    }
  /*}}}  */
  /*{{{  if sorting open pipe*/
  } else {
    sprintf(bindfile,"sort -o %s/%sbind",home,filebase);
    if ((bind=popen(bindfile,"w"))==NULL) {
      fprintf(stderr,"Can't open pipe: %s\n",bindfile);
      exit(1);
    }
  }
  /*}}}  */
  /*}}}  */
  /*}}}  */
  ok=process_file();
  /*{{{  put #vars*/
  if (int_no) {
    putc(RC_INTS,rc);
    putw(int_no,rc);
  }
  /*}}}  */
  putc(RC_ENDE,rc);
  /*{{{  close all files*/
  fclose(in);
  fclose(rc);
  if (sorting==0)
    fclose(bind);
  else
    pclose(bind);
  /*}}}  */
  if (ok) {
    fprintf(stderr,"generated files are incorrect\n");
    exit(1);
  } else if (verbose) {
    fprintf(stderr,"binding needs %d int-variables.\n",int_no);
    fprintf(stderr,"binding uses %d tokens.\n",t_counter);
    fprintf(stderr,"binding generates %d key-sequences.\n",k_counter);
  }
  exit(0);
}
/*}}}  */
