/* mtalkd - daemon for mtalk.  Handles and redistributes user messages
 * and requests.
 *
 * Copyright (c) 1991, Matthew Kimmel.  Permission granted for unlimited
 * non-commercial use and distribution.
 */
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <sys/msg.h>
#include "mtalk.h"

struct user {  /* Record of logged-on user */
  int uid;	/* if this is -1, this user slot is empty */
  int mqid;	/* user's message queue id */
  char name[31]; /* Their name */
  };

struct genmsg msgin; /* Soon-to-be-malloc'd structure we use to catch incoming messages before processing */
int ourmqid; /* Our message queue id; */
struct user users[MAX_USERS]; /* List of active users */

main()
{
  init(); /* Initialize stuff */
  server(); /* Go be the mtalk server */
  cleanup(); /* If we get here, we'd better clean things up */
}

/* Initialize various stuff--memory, message queue, signals, etc. */
init()
{
  key_t ftok();
  int handle_sigs();
  char *malloc();
  int i;

  /* Get our incoming message queue using the....magic formula! */
  if((ourmqid = msgget(ftok(DAEMON_PATH,1),(0622 | IPC_CREAT))) == -1) {
    perror("mtalkd: msgget");
    exit(1);
    }

  /* Set up to handle signals */
  signal(SIGHUP,SIG_IGN);
  signal(SIGINT,SIG_IGN);
  signal(SIGQUIT,SIG_IGN);
  signal(SIGTERM,handle_sigs);
#ifndef _I386
  signal(SIGREST,handle_sigs);
#endif

  /* Initialize user list to no users */
  for(i = 0;i < MAX_USERS;i++)
    users[i].uid = -1;
}

/* This is the main loop of mtalkd.  It waits for messages, and
   processes them.  It should spend most of its time blocked when
   nothing is happening, and save system time. */
server()
{
  for(;;) {
    if(msgrcv(ourmqid,&msgin,1024,0L,0) == -1) { /* This should NEVER happen. */
      perror("mtalkd: msgrcv");
      exit(1);
      }
    if(msgin.msgtype == M_USRMSG) { /* A public message from a user */
      usrpubmsg(msgin.contents);
      continue;
      }
    if(msgin.msgtype == M_USRCMD) { /* A command from a user */
      usrcommand(msgin.contents);
      continue;
      }
    if(msgin.msgtype == M_USRIN) { /* A user logging in */
      usrlogin(msgin.contents);
      continue;
      }
    }
}

/* Assemble and distribute a public message from a user */
usrpubmsg(mesg)
struct usrmsg *mesg;
{
  char txt[315]; /* Better safe than sorry */
  int i;

  for(i=0;i<MAX_USERS;i++)
    if(users[i].uid == mesg->uid)
      break;

  if(i == MAX_USERS) { /* Unknow uid--this should NEVER happen */
    sprintf(txt,"Message from unlogged uid %d -- tell programmer!",mesg->uid);
    sendtoall(txt);
    }

  sprintf(txt,"%s: %s",users[i].name,mesg->text);
  sendtoall(txt);
}

/* Process an incoming command from a user */
usrcommand(command)
struct usrcmd *command;
{
  switch(command->cmd) {
    case U_WHISPER : dowhisper(command->uid,command->user,command->text);
                     break;
    case U_WHO : dowho(command->uid);
                 break;
    case U_EXIT : doexit(command->uid);
                  break;
    } /* Unrecognized commands are ignored. */
}

/* Process the whisper command */
dowhisper(uid,usr,text)
int uid;  /* uid of sending user */
char *usr; /* name of receiving user */
char *text; /* Text to be whispered */
{
  int duid = -1; /* Receipient's uid */
  char *sname; /* Pointer to sender's name */
  int i;
  char buf[1024];

  for(i=0;i<MAX_USERS;i++) {
    if(users[i].uid == uid)
      sname = users[i].name;
    if(!strcmp(users[i].name,usr))
      duid = users[i].uid;
      }

  if(duid == -1) { /* User not found */
    sprintf(buf,"%s: no such user",usr);
    sendtouid(uid,buf);
    }

  /* Assemble whisper and send it */
  sprintf(buf,"%s whispers: %s",sname,text);
  sendtouid(duid,buf);
}

/* Process the who command */
dowho(uid)
int uid; /* uid of requestor */
{
  char buf[1024];
  int i;

  /* Assemble a string to be sent to the user.  This may break if */
  /* MAX_USERS is more than 30. */
  strcpy(buf,"Users currently logged on:\r\n");
  for(i=0;i<MAX_USERS;i++)
    if(users[i].uid != -1) {
      strcat(buf,"  ");
      strcat(buf,users[i].name);
      strcat(buf,"\r\n");
      }

  /* Send the string */
  sendtouid(uid,buf);
}

/* Process the exit command */
doexit(uid)
int uid; /* uid of exiter */
{
  int i;
  char buf[1024];

  /* Find user and delete him from user list, and assemble a string
     notifying other users that he has exited. */
  for(i=0;i<MAX_USERS;i++)
    if(users[i].uid == uid) {
      sprintf(buf,"%s has exited.",users[i].name);
      users[i].uid = -1;
      }

  /* Send the notification to everyone else. */
  sendtoall(buf);
}
 
/* Log a user into the system (if there's a free slot) */
usrlogin(usr)
struct usrin *usr;
{
  key_t ftok();
  int slot, tmpid;
  struct passwd *pw;
  struct {
    long mtype;
    } fullmsg;
  char announce[50];

  /* First, try to get their message queue */
  if((tmpid = msgget(ftok(CLIENT_PATH,usr->uid),0)) == -1) {
    perror("mtalkd: msgget");
    return;
    }

  /* Look for a slot for the user. */
  for(slot=0;slot<MAX_USERS;slot++)
    if(users[slot].uid == -1)
      break;

  /* Send a full message if no slots open */
  if(slot == MAX_USERS) {
    fullmsg.mtype = M_DMNFULL;
    msgsnd(tmpid,&fullmsg,0,0);
    return;
    }

  /* Otherwise, put the user in the user records. */
  users[slot].uid = usr->uid;
  users[slot].mqid = tmpid;
  setpwent();
  pw = getpwuid(usr->uid);
  strncpy(users[slot].name,pw->pw_name,30);
  endpwent();

  /* Announce that the user has logged in */
  sprintf(announce,"%s has logged in.",users[slot].name);
  sendtoall(announce);
}

/* Send a line of text to all users logged in */
sendtoall(txt)
char *txt;
{
  int i;
  struct {
    long mtype;
    char text[1024];
    } mbuf;

  /* Now do a msgsnd to all users */
  mbuf.mtype = M_DMNTEXT;
  strncpy(mbuf.text,txt,1023);
  for(i=0;i<MAX_USERS;i++)
    if(users[i].uid != -1)
      msgsnd(users[i].mqid,&mbuf,(strlen(mbuf.text)+1),0);
}

/* Send a line of text to a specific uid.  If the uid is not found in the
   list of users, the message is discarded, to save hassle. */
sendtouid(uid,txt)
int uid;
char *txt;
{
  int i;
  struct {
    long mtype;
    char text[1024];
    } mbuf;

  /* Find the uid, and send the message to that user */
  for(i=0;i<MAX_USERS;i++)
    if(users[i].uid == uid) {
      mbuf.mtype = M_DMNTEXT;
      strncpy(mbuf.text,txt,1023);
      msgsnd(users[i].mqid,&mbuf,(strlen(mbuf.text)+1),0);
      }
}

/* Handle any caught signals--at present, all caught signals make the
   program clean up and terminate. */
handle_sigs()
{
  cleanup();
  exit(0);
}

/* This function kills our message queue and frees memory when we exit.
   It leaves any users stranded, since this should never happen when
   users are logged on! */
cleanup()
{
  msgctl(ourmqid,IPC_RMID,NULL);
}

