/* Harry's Word Processor for Coherent 4.0 */

/* Header files */

#include <curses.h>
#include <stdlib.h>
#include <access.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>

/* Defines */

#define CASE 1
#define NOCASE 2

#ifndef SPOOLER
#define SPOOLER "lp"
#endif

#define LINE_LENGTH 80

#define LINES_PER_PAGE 21

#define NORMAL 0x0000

#define START_BOLD 0x0100
#define END_BOLD 0x0200

#define START_UNDERLINE 0x1000
#define END_UNDERLINE 0x2000

#define OVERSTRIKE 1
#define INSERT 2

#define BOX_CH1 (((int)'|')|A_BOLD)
#define BOX_CH2 (((int)'-')|A_BOLD)

/* Globals */

char **word_list=NULL;

int num_words=0;

short *file_buffer=NULL;

WINDOW *mystdscr,*textscr;

char filename[1024]="untitled.hwp",inputline[LINE_LENGTH+1]="";

int numlines=0,page_length=66,tab_size=8,indent=16,
	left_margin=8,right_margin=72,top_blanks=5,
	bottom_blanks=5,starting_page_no=1,version=1,
	current_page=1,current_y=0,current_x=16,current_attr=NORMAL,
	insertmode=INSERT,line_spacing=1,changed=0;

char header[LINE_LENGTH+1]="%",footer[LINE_LENGTH+1]="";

/* Forward function declarations */

void init_curses();

void end_curses();

int do_load_file();

void get_user_input();

void display_screen();

void my_exit();

void write_file();

int get_inputline();

void error_message();

int file_menu();

void do_write_file();

void load_file();

int confirm_quit();

int options_menu();

int do_menubar();

void justify();

void insert();

void print_file();

int edit_menu();

void write_ascii();

int locate_menu();

int do_search();

int spell_menu();

void get_spell_list();

void do_help();

main(argc,argv)
int argc;
char **argv;
{
	/* For now, only allow a single file can be edited; if none is given 
	   then call it 'untitled.hwp' */

	if (argc==2)
		strcpy(filename,argv[1]);
	else if (argc>2)
	{
		fprintf(stderr,"Usage: %s <filename>\n",argv[0]);

		exit(1);
	}

	if (do_load_file(filename)==0)
		exit(1);

	init_curses();

	textscr=newwin(21,80,1,0);

	keypad(textscr,TRUE);

	get_user_input();

	end_curses();
}

void init_curses()
{
	/* Initialize curses */

	mystdscr=initscr();

	if (mystdscr==NULL)
	{
		fputs("Error in initscr.\n",stderr);

		exit(1);
	}

	noecho();

	raw();

	nonl();

	keypad(mystdscr,TRUE);

	refresh();
}

void end_curses()
{
	/* reset terminal after curses */

	nocbreak();

	echo();

	nl();

	erase();

	refresh();

	endwin();
}

int do_load_file(input_filename)
char *input_filename;
{
	int current_file_fd;

	FILE *current_file;

	if (file_buffer!=NULL)
	{
		free(file_buffer);

		file_buffer=NULL;
	}

	if (!access(input_filename,AEXISTS))
	{
		if (!access(input_filename,AREAD))
		{
			if (strstr(input_filename,".hwp"))
			{
				int setup[25];

				current_file_fd=open(input_filename,O_RDONLY);

				read(current_file_fd,setup,25*sizeof(int));

				version=setup[0];

				if (version!=1)
					return 0;

				numlines=setup[1];
				page_length=setup[2];
				tab_size=setup[3];
				indent=setup[4];
				left_margin=setup[5];
				right_margin=setup[6];
				top_blanks=setup[7];
				bottom_blanks=setup[8];
				starting_page_no=setup[9];
				line_spacing=setup[10];
				current_page=setup[11];
				current_y=setup[12];
				current_x=left_margin;

				file_buffer=(short *) malloc(numlines*LINE_LENGTH*sizeof(short));

				if (file_buffer==NULL)
					return 0;

				read(current_file_fd,file_buffer,numlines*LINE_LENGTH*sizeof(short));

				close(current_file_fd);
			}
			else
			{
				int line;

				char tmpfilename[65],command[128];

				numlines=0;

				line_spacing=0;

				left_margin=0;

				indent=8;

				right_margin=79;

				top_blanks=bottom_blanks=0;

				current_x=0;

				current_y=0;

				sprintf(tmpfilename,"/tmp/hwp.%d",getpid());

				sprintf(command,"detab < %s > %s",input_filename,tmpfilename);

				system(command);

				current_file=fopen(tmpfilename,"r");

				while (fgets(inputline,LINE_LENGTH+1,current_file))
					numlines++;

				rewind(current_file);

				file_buffer=(short *) malloc(numlines*LINE_LENGTH*sizeof(short));

				if (file_buffer==NULL)
					return 0;

				for (line=0;line<numlines;line++)
				{
					int pos;

					char *tmpptr;

					fgets(inputline,LINE_LENGTH+1,current_file);

					tmpptr=strrchr(inputline,'\n');

					if (tmpptr)
						*tmpptr=0;

					tmpptr=strrchr(inputline,'\r');

					if (tmpptr)
						*tmpptr=0;

				 	for (pos=0;pos<LINE_LENGTH;pos++)
					{
						if (pos<strlen(inputline))
							file_buffer[pos+line*LINE_LENGTH]=(short)inputline[pos];
						else
							file_buffer[pos+line*LINE_LENGTH]=0;
					}
				}

				fclose(current_file);

				unlink(tmpfilename);
			}
		}
	}
	else if (!strcmp(input_filename,"untitled.hwp"))
	{
		numlines=1;

		current_y=0;

		current_page=1;

		page_length=66;

		tab_size=8;

		indent=16;

		current_x=16;

		left_margin=8;

		right_margin=72;

		top_blanks=5;

		bottom_blanks=5;

		line_spacing=1;

		file_buffer=(short *)malloc(numlines*LINE_LENGTH*sizeof(short));

		if (file_buffer==NULL)
			return 0;

		memset(file_buffer,0,numlines*LINE_LENGTH*sizeof(short));
	}
	else
		return 0;

	strcpy(filename,input_filename);

	changed=0;

	return 1;
}

void get_user_input()
{
	int quitting=0,inputch;

	attron(A_REVERSE);

	mvaddstr(0,0," F2:File    F3:Edit    F4:Locate    F5:Options    F6:Spell              F1:Help ");

	mvaddstr(22,0,"                                                                                 ");
	mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

	if (insertmode==OVERSTRIKE)
		mvprintw(22,69,"Overstrike");
	else
		mvprintw(22,69,"Insert    ");

	if (current_attr&START_BOLD)
		mvprintw(22,64,"Bold");
	else
		mvprintw(22,64,"    ");
		
	if (current_attr&START_UNDERLINE)
		mvprintw(22,54,"Underline");
	else
		mvprintw(22,54,"         ");

	attroff(A_REVERSE);

	attron(A_BOLD);

	mvaddstr(23,0," Press function key to activate menu.  Or press F10 to activate menu bar.      ");

	attroff(A_BOLD);

	refresh();

	display_screen();

	while (!quitting)
	{
		inputch=wgetch(textscr);

		switch(inputch)
		{
		case KEY_IC:	insertmode=(insertmode==INSERT)?OVERSTRIKE:INSERT;

				mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

				if (insertmode==OVERSTRIKE)
					mvprintw(22,69,"Overstrike");
				else
					mvprintw(22,69,"Insert    ");

				if (current_attr&START_BOLD)
					mvprintw(22,64,"Bold");
				else
					mvprintw(22,64,"    ");
		
				if (current_attr&START_UNDERLINE)
					mvprintw(22,54,"Underline");
				else
					mvprintw(22,54,"         ");

				break;

		case 265:
		case 266:
		case 267:
		case 268:
		case 269:
		case 270:
				attron(A_BOLD);

				mvaddstr(23,0," Use the arrow keys to move within a menu or to move to another menu.     ");

				attroff(A_BOLD);

				refresh();

				switch(inputch)
				{
					case 265:	do_help();

							break;

					case 266:	if (file_menu()==1)
								quitting++;

							break;

					case 267:	edit_menu();

							break;

					case 268:	locate_menu();

							break;

					case 269:	options_menu();

							break;

					case 270:	spell_menu();

							break;
				}

				attron(A_BOLD);

				mvaddstr(23,0," Press function key to activate menu.  Or press F10 to activate menu bar.      ");

				attroff(A_BOLD);

				attron(A_REVERSE);

				mvaddstr(0,0," F2:File    F3:Edit    F4:Locate    F5:Options    F6:Spell              F1:Help ");
				mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

				if (insertmode==OVERSTRIKE)
					mvprintw(22,69,"Overstrike");
				else
					mvprintw(22,69,"Insert    ");

				if (current_attr&START_BOLD)
					mvprintw(22,64,"Bold");
				else
					mvprintw(22,64,"    ");
		
				if (current_attr&START_UNDERLINE)
					mvprintw(22,54,"Underline");
				else
					mvprintw(22,54,"         ");

				attroff(A_REVERSE);

				refresh();

				wmove(textscr,current_y,current_x);

				wrefresh(textscr);

				break;
		
		case 27:
		case 274:	attron(A_BOLD);

				mvaddstr(23,0," Use the arrow keys to move within the menubar; Enter to select, Esc to exit.");

				attroff(A_BOLD);

				refresh();

				if (do_menubar()==1)
					quitting++;

				attron(A_BOLD);

				mvaddstr(23,0," Press function key to activate menu.  Or press F10 to activate menu bar.      ");

				attroff(A_BOLD);

				attron(A_REVERSE);

				mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

				if (insertmode==OVERSTRIKE)
					mvprintw(22,69,"Overstrike");
				else
					mvprintw(22,69,"Insert    ");

				if (current_attr&START_BOLD)
					mvprintw(22,64,"Bold");
				else
					mvprintw(22,64,"    ");
		
				if (current_attr&START_UNDERLINE)
					mvprintw(22,54,"Underline");
				else
					mvprintw(22,54,"         ");

				attroff(A_REVERSE);

				refresh();

				wmove(textscr,current_y,current_x);

				wrefresh(textscr);

				break;

		case '\t':	if (insertmode==OVERSTRIKE)
				{
					if (current_x<indent)
						current_x=indent;
					else
					{
						if (current_x%tab_size==0)
							current_x+=tab_size;
						else
							for (;current_x%tab_size;current_x++);
					}

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);
				}
				else
				{
					int new_x;

					short tmpstr[LINE_LENGTH];

					if (current_x<indent)
						new_x=indent;
					else
					{
						if (current_x%tab_size==0)
							new_x=current_x+tab_size;
						else
							for (new_x=current_x;new_x%tab_size;new_x++);
					}

					memset(tmpstr,0,LINE_LENGTH+1);

					memset(tmpstr,' ',new_x-current_x);

					insert(tmpstr,new_x-current_x,current_y);

					current_x=new_x;

					display_screen();
				}

				break;

		case '\r':
		case '\n':	if (insertmode==INSERT)
				{
					int pos;

					changed++;

					numlines+=(1+line_spacing);
			
					file_buffer=(short *) realloc(file_buffer,numlines*LINE_LENGTH*sizeof(short));
							
					if (file_buffer==NULL)
						my_exit("Realloc error on file buffer.\n");
			
					memset(file_buffer+(numlines-(1+line_spacing))*LINE_LENGTH,0,(1+line_spacing)*LINE_LENGTH*sizeof(short));
			
					if ((current_y+(current_page-1)*LINES_PER_PAGE+(1+line_spacing))<(numlines-(1+line_spacing)))
					{
						int line;

						for (line=numlines-1;line>=current_y+2*(1+line_spacing)+(current_page-1)*LINES_PER_PAGE;line-=(1+line_spacing))
							memcpy(file_buffer+line*LINE_LENGTH,file_buffer+(line-(1+line_spacing))*LINE_LENGTH,LINE_LENGTH*sizeof(short));

						memset(file_buffer+(current_y+(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,0,80*sizeof(short));
					}

					for (pos=current_x;pos<=right_margin;pos++)
					{
						file_buffer[pos+left_margin-current_x+(current_y+(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=
								file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

						file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=0;
					}

					display_screen();
				}

				current_x=right_margin;

		case KEY_RIGHT:	if (current_x<right_margin)
				{
					current_x++;

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					break;
				}
				else
				{
					current_x=left_margin;
				}
		case KEY_DOWN:	if (current_y+(1+line_spacing)+((current_page-1)*LINES_PER_PAGE)>=numlines)
				{
					numlines+=line_spacing+1;
	
					file_buffer=(short *) realloc(file_buffer,numlines*LINE_LENGTH*sizeof(short));
				
					if (file_buffer==NULL)
						my_exit("Realloc error on file buffer.\n");

					memset(file_buffer+(numlines-(1+line_spacing))*LINE_LENGTH,0,(1+line_spacing)*LINE_LENGTH*sizeof(short));
				}

				wattroff(textscr,A_BOLD);

				wattroff(textscr,A_UNDERLINE);

				if (current_y<(LINES_PER_PAGE-(1+line_spacing)))
				{
					current_y=(current_y+1+line_spacing)%LINES_PER_PAGE;

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					break;
				}

		case KEY_NPAGE:	if ((current_page*LINES_PER_PAGE)<=numlines)
				{
					current_page++;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					if (inputch==KEY_NPAGE)
					{
						if (line_spacing>0)
							current_y=((current_page-1)*LINES_PER_PAGE)%(1+line_spacing);
						else
							current_y=0;
					}
					else
						current_y=(current_y+1+line_spacing)%LINES_PER_PAGE;

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					display_screen();
				}

				break;

		case 127:
		case KEY_DC:	{
					int pos;

					changed++;

					for (pos=current_x;pos<right_margin;pos++)
						file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=file_buffer[pos+1+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

					file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=0;

					justify(current_y);
				}

				display_screen();

				break;

		case '\b':
		case KEY_BACKSPACE:
				if (current_x<=left_margin&&(current_y!=0||current_page!=1))
				{
					int pos;

					for (pos=right_margin;pos>=left_margin;pos--)
						if (file_buffer[pos+(current_y-(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
								file_buffer[pos+(current_y-(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
							break;

					current_x=-(pos+1);
						
					changed++;

					justify(current_y-(1+line_spacing));
				}
				else if (current_x>left_margin)
				{
					int pos;

					changed++;

					for (pos=current_x-1;pos<right_margin;pos++)
						file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=file_buffer[pos+1+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

					file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=0;

					justify(current_y);
				}

				display_screen();

		case KEY_LEFT:	if (current_x>left_margin)
				{
					current_x--;

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					break;
				}
				else if ((current_page>1||current_y!=0)&&current_x>0)
				{
					current_x=right_margin;
				}
				else if (current_x<0)
					current_x=-(current_x);
				else
					break;

		case KEY_UP:	if ((line_spacing==0&&current_y>0)||(line_spacing>0&&current_y>(((current_page-1)*LINES_PER_PAGE))%(1+line_spacing)))
				{
					current_y=current_y-1-line_spacing;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					break;
				}

		case KEY_PPAGE:	if (current_page>1)
				{
					current_page--;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					if (inputch==KEY_PPAGE)
					{
						if (line_spacing>0)
							current_y=((current_page)*LINES_PER_PAGE)%(1+line_spacing)-(1+line_spacing)+LINES_PER_PAGE;
#if o
							current_y=LINES_PER_PAGE-1+((current_page-1)*LINES_PER_PAGE+1)%(1+line_spacing);
#endif
						else
							current_y=LINES_PER_PAGE-1;
					}
					else
						current_y=current_y-1-line_spacing;

					if (current_y<0)
						current_y+=LINES_PER_PAGE;

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					display_screen();
				}

				break;

		case KEY_HOME:	current_x=left_margin;

				wmove(textscr,current_y,current_x);

				wrefresh(textscr);

				break;
		}

		if (isprint(0xff&inputch)&&!(inputch&0xff00))
		{
			short tempstr[LINE_LENGTH];

			int count=0,pos,word_wrap=0;

			changed++;

			if (insertmode==OVERSTRIKE)
			{
				file_buffer[current_x+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=inputch|current_attr;

				if (current_attr&START_BOLD)
					wattron(textscr,A_BOLD);

				if (current_attr&START_UNDERLINE)
					wattron(textscr,A_UNDERLINE);

				if (current_attr&END_BOLD)
					wattroff(textscr,A_BOLD);

				if (current_attr&END_UNDERLINE)
					wattroff(textscr,A_UNDERLINE);

				current_attr=NORMAL;

				waddch(textscr,inputch);

			 	if (current_x<right_margin)
				{
					current_x++;

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					continue;
				}
				else
				{
					/* do word wrap */

					for (pos=current_x;pos>=left_margin;pos--)
					{
						if (((char)file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH])==' '||(((char)file_buffer[pos+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH])==0))
							break;

						count++;
					}

					count--;

					if (pos<left_margin||pos==current_x)
						current_x=left_margin;
					else
					/* wrap word */
					{
						word_wrap=1;

						memcpy(tempstr,file_buffer+current_x-count+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,(count+1)*sizeof(short));

						memset(file_buffer+current_x-count+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,0,(count+1)*sizeof(short));

						current_x=left_margin+count+1;
					}
				}

				if (current_y+(1+line_spacing)+((current_page-1)*LINES_PER_PAGE)>=numlines)
				{
					numlines+=line_spacing+1;
	
					file_buffer=(short *) realloc(file_buffer,numlines*LINE_LENGTH*sizeof(short));
				
					if (file_buffer==NULL)
						my_exit("Realloc error on file buffer.\n");

					memset(file_buffer+(numlines-(1+line_spacing))*LINE_LENGTH,0,(1+line_spacing)*LINE_LENGTH*sizeof(short));
				}

				if (current_y<(LINES_PER_PAGE-(1+line_spacing)))
				{
					current_y+=1+line_spacing;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					if (word_wrap)
					{
						memcpy(file_buffer+left_margin+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,tempstr,(count+1)*sizeof(short));
	
						display_screen();
					}

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					continue;
				}

				if ((current_page*LINES_PER_PAGE)<=numlines)
				{
					current_y=(current_y+1+line_spacing)%LINES_PER_PAGE;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					current_page++;

					if (word_wrap)
						memcpy(file_buffer+left_margin+(current_y+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,tempstr,(count+1)*sizeof(short));

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					display_screen();
				}
			}
			else
			{
				/* build a string of one short which is the
				   character typed and the current attr; then
				   insert() that short, set the cursor position
				   appropriately and display the screen */

				tempstr[0]=inputch|current_attr;

				current_attr=NORMAL;

				insert(tempstr,1,current_y);

				display_screen();

				if (current_x<0)
					current_x=-current_x;
				else if (current_x<=right_margin)
					continue;

				if (current_y<(LINES_PER_PAGE-(1+line_spacing)))
				{
					current_y+=1+line_spacing;

					wattroff(textscr,A_BOLD);

					wattroff(textscr,A_UNDERLINE);

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					wmove(textscr,current_y,current_x);

					wrefresh(textscr);

					continue;
				}

				if ((current_page*LINES_PER_PAGE)<=numlines)
				{
					current_y=(current_y+1+line_spacing)%LINES_PER_PAGE;

					wattroff(textscr,A_BOLD);
	
					wattroff(textscr,A_UNDERLINE);

					current_page++;

					attron(A_REVERSE);

					mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

					if (insertmode==OVERSTRIKE)
						mvprintw(22,69,"Overstrike");
					else
						mvprintw(22,69,"Insert    ");
	
					if (current_attr&START_BOLD)
						mvprintw(22,64,"Bold");
					else
						mvprintw(22,64,"    ");
			
					if (current_attr&START_UNDERLINE)
						mvprintw(22,54,"Underline");
					else
						mvprintw(22,54,"         ");

					attroff(A_REVERSE);

					refresh();

					display_screen();
				}
			}
		}
	}
}

void display_screen()
{
	int row,col;

	werase(textscr);

	for (row=0;row<LINES_PER_PAGE;row++)
	{
		wattrset(textscr,A_NORMAL);

		for (col=0;col<80;col++)
		{
			if ((row+(current_page-1)*LINES_PER_PAGE)>=numlines)
				continue;

			wmove(textscr,row,col);

			if (file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&START_BOLD)
				wattron(textscr,A_BOLD);

			if (file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&END_BOLD)
				wattroff(textscr,A_BOLD);

			if (file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&START_UNDERLINE)
				wattron(textscr,A_UNDERLINE);

			if (file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&END_UNDERLINE)
				wattroff(textscr,A_UNDERLINE);

			if (file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&0xff)
				waddch(textscr,((int)file_buffer[col+(row+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]&0xff));
		}
	}
			
	wmove(textscr,current_y,current_x);

	wrefresh(textscr);
}

void my_exit(message)
char *message;
{
	erase();

	refresh();

	end_curses();

	fputs(message,stderr);

	exit(1);
}

void write_file()
{
	char output_filename[75]="";

	int retval;

	output_filename[68]=0;

	if (strstr(filename,".hwp"))
		strcpy(output_filename,filename);

	retval=get_inputline("Enter filename to write (ASCII unless ending in .hwp):",output_filename,output_filename);

	if (retval!=-1)
	{
		if (strstr(output_filename,".hwp"))
			do_write_file(output_filename);
		else
			write_ascii(output_filename);
	}
}

int get_inputline(message,prompt,output)
char *message,*prompt,*output;
{
	WINDOW *message_win;

	char tmpstr[69];

	int inputch=0,pos=0;

	memset(output+strlen(output),' ',68-strlen(output));

	if (output!=prompt)
		memset(prompt+strlen(prompt),' ',68-strlen(prompt));

	output[68]=0;

	message_win=newwin(7,70,10,5);

	keypad(message_win,TRUE);

	box(message_win,BOX_CH1,BOX_CH2);

	mvwaddstr(message_win,1,1,message);

	wattron(message_win,A_REVERSE);

	mvwaddstr(message_win,3,1,prompt);

	wattroff(message_win,A_REVERSE);

	mvwaddstr(message_win,5,25,"CTRL-C to Cancel");

	wmove(message_win,3,1);

	wrefresh(message_win);

	wattron(message_win,A_REVERSE);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=3)
	{
		inputch=wgetch(message_win);

		switch(inputch)
		{
		case KEY_IC:	insertmode=(insertmode==INSERT)?OVERSTRIKE:INSERT;

				mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

				if (insertmode==OVERSTRIKE)
					mvprintw(22,69,"Overstrike");
				else
					mvprintw(22,69,"Insert    ");

				if (current_attr&START_BOLD)
					mvprintw(22,64,"Bold");
				else
					mvprintw(22,64,"    ");
		
				if (current_attr&START_UNDERLINE)
					mvprintw(22,54,"Underline");
				else
					mvprintw(22,54,"         ");

				break;

		case KEY_LEFT:	if (pos>0)
				{
					pos--;

					wmove(message_win,3,pos+1);

					wrefresh(message_win);
				}

				break;

		case KEY_RIGHT:	if (pos<67)
				{
					pos++;

					wmove(message_win,3,pos+1);

					wrefresh(message_win);
				}

				break;

		case 127:
		case KEY_DC:	{
					int pos1;

					for (pos1=pos;pos1<68;pos1++)
						output[pos1]=output[pos1+1];
				}

				mvwaddstr(message_win,3,1,output);

				wmove(message_win,3,pos+1);

				wrefresh(message_win);

				break;

		case '\b':
		case KEY_BACKSPACE:
				{
					int pos1;

					for (pos1=pos-1;pos1<68;pos1++)
						output[pos1]=output[pos1+1];
				}

				mvwaddstr(message_win,3,1,output);

				pos--;

				wmove(message_win,3,pos+1);

				wrefresh(message_win);

				break;

		case KEY_HOME:	pos=0;

				wmove(message_win,3,pos+1);

				wrefresh(message_win);

				break;

		case 265:	wattroff(message_win,A_REVERSE);

				werase(message_win);

				wrefresh(message_win);

				do_help();

				box(message_win,BOX_CH1,BOX_CH2);

				mvwaddstr(message_win,1,1,message);

				wattron(message_win,A_REVERSE);

				mvwaddstr(message_win,3,1,output);

				wattroff(message_win,A_REVERSE);

				mvwaddstr(message_win,5,25,"CTRL-C to Cancel");

				wmove(message_win,3,1+pos);

				wrefresh(message_win);

				wattron(message_win,A_REVERSE);

				break;
		}

		if (isprint(0xff&inputch)&&!(inputch&0xff00))
		{
			if (insertmode==OVERSTRIKE)
			{
				output[pos]=inputch;

				waddch(message_win,inputch);

				if (pos<67)
					pos++;
				else
					beep();

				wmove(message_win,3,pos+1);

				wrefresh(message_win);
			}
			else
			{
				if (output[67]!=' '&&output[67]!=0)
					beep();
				else
				{
					int pos1;

					for (pos1=67;pos1>pos;pos1--)
						output[pos1]=output[pos1-1];

					output[pos]=inputch;

					if (pos<67)
						pos++;
					else
						beep();
	
					mvwaddstr(message_win,3,1,output);

					wmove(message_win,3,pos+1);
			
					wrefresh(message_win);
				}
			}
		}
	}

	wattroff(message_win,A_REVERSE);

	werase(message_win);

	wrefresh(message_win);

	delwin(message_win);

	display_screen();

	strcpy(tmpstr,output+strspn(output," "));

	for (pos=strlen(tmpstr)-1;pos>0;pos--)
		if (tmpstr[pos]==' ')
			tmpstr[pos]=0;
		else
			break;

	strcpy(output,tmpstr);

	if (inputch==3)
		return -1;
	else
		return 0;
}

void error_message(title,message)
char *title,*message;
{
	WINDOW *error_win;

	error_win=newwin(7,70,10,5);

	box(error_win,BOX_CH1,BOX_CH2);

	mvwaddstr(error_win,1,1,title);

	mvwaddstr(error_win,3,1,message);

	mvwaddstr(error_win,5,25,"Press any key to go on...");

	wrefresh(error_win);

	wgetch(error_win);

	werase(error_win);

	wrefresh(error_win);

	delwin(error_win);

	display_screen();
}

int file_menu()
{
	WINDOW *file_win;

	int inputch=0,pos=0;

	file_win=newwin(7,12,1,0);

	box(file_win,BOX_CH1,BOX_CH2);

	wattron(file_win,A_REVERSE);

	mvwaddstr(file_win,1,1,"Load File");

	wattroff(file_win,A_REVERSE);

	mvwaddstr(file_win,2,1,"Save File");

	mvwaddstr(file_win,3,1,"Print File");

	mvwaddstr(file_win,4,1,"New");

	mvwaddstr(file_win,5,1,"Quit");

	keypad(file_win,TRUE);

	wmove(file_win,1,1);

	wrefresh(file_win);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=27)
	{
		inputch=getch();

		switch(pos)
		{
			case 0:	mvwaddstr(file_win,1,1,"Load File");

				break;

			case 1: mvwaddstr(file_win,2,1,"Save File");

				break;

			case 2: mvwaddstr(file_win,3,1,"Print File");

				break;

			case 3: mvwaddstr(file_win,4,1,"New");

				break;

			case 4: mvwaddstr(file_win,5,1,"Quit");

				break;
		}

		switch(inputch)
		{
			case KEY_DOWN:	pos++;

					break;

			case KEY_UP:	pos--;

					if(pos<0)
						pos+=5;

					break;

			case KEY_RIGHT:	werase(file_win);

					wrefresh(file_win);

					delwin(file_win);

					display_screen();

					return -1;

			case KEY_LEFT:	werase(file_win);

					wrefresh(file_win);

					delwin(file_win);

					display_screen();

					return -2;
		}

		pos=pos%5;

		wattron(file_win,A_REVERSE);

		switch(pos)
		{
			case 0:	mvwaddstr(file_win,1,1,"Load File");
				wmove(file_win,1,1);

				break;

			case 1: mvwaddstr(file_win,2,1,"Save File");
				wmove(file_win,2,1);

				break;

			case 2: mvwaddstr(file_win,3,1,"Print File");
				wmove(file_win,3,1);

				break;

			case 3: mvwaddstr(file_win,4,1,"New");
				wmove(file_win,4,1);

				break;

			case 4: mvwaddstr(file_win,5,1,"Quit");
				wmove(file_win,5,1);

				break;
		}

		wattroff(file_win,A_REVERSE);

		wrefresh(file_win);
	}

	werase(file_win);

	wrefresh(file_win);

	delwin(file_win);

	display_screen();

	if (inputch==27)
		return 0;

	attron(A_REVERSE);

	mvaddstr(0,0,"                                                                        F1:Help ");

	attroff(A_REVERSE);

	refresh();

	switch(pos)
	{
		case 0:	load_file();

			display_screen();

			return 0;

		case 1:	write_file();

			return 0;

		case 2: print_file();

			return 0;

		case 3:	if (changed)
			{
				if (confirm_load()==0)
					return 0;
			}

			do_load_file("untitled.hwp");

			display_screen();

			return 0;

		case 4:	if (confirm_quit())
				return 1;
			else
				return 0;
	}
}

void do_write_file(output_filename)
char *output_filename;
{
	int write_fd,pad;

	errno=0;

	if (strlen(output_filename)>5)
	{
		if (strcmp(output_filename+strlen(output_filename)-4,".hwp"))
			strcat(output_filename,".hwp");
	}
	else
		strcat(output_filename,".hwp");

	write_fd=open(output_filename,O_WRONLY|O_CREAT,0666);

	if (write_fd==-1)
	{
		error_message("Couldn't write file because:",strerror(errno));

		return;
	}

	write(write_fd,&version,sizeof(int));
	write(write_fd,&numlines,sizeof(int));
	write(write_fd,&page_length,sizeof(int));
	write(write_fd,&tab_size,sizeof(int));
	write(write_fd,&indent,sizeof(int));
	write(write_fd,&left_margin,sizeof(int));
	write(write_fd,&right_margin,sizeof(int));
	write(write_fd,&top_blanks,sizeof(int));
	write(write_fd,&bottom_blanks,sizeof(int));
	write(write_fd,&starting_page_no,sizeof(int));
	write(write_fd,&line_spacing,sizeof(int));
	write(write_fd,&current_page,sizeof(int));
	write(write_fd,&current_y,sizeof(int));

	for (pad=0;pad<12;pad++)
		write(write_fd,0,sizeof(int));

	write(write_fd,file_buffer,numlines*LINE_LENGTH*sizeof(short));

	close(write_fd);

	strcpy(filename,output_filename);

	changed=0;
}

void load_file()
{
	char input_filename[75]="";

	int retval;

	input_filename[68]=0;

	if (changed)
	{
		if (confirm_load()==0)
			return;
	}

	retval=get_inputline("Enter filename to load:",input_filename,input_filename);

	if (retval!=-1)
	{
		if (do_load_file(input_filename)==0)
		{
			do_load_file("untitled.hwp");

			error_message("Unable to load:",input_filename);
		}
	}
}

int confirm_quit()
{
	WINDOW *quit_window;

	int inputch;

	if (!changed)
		return 1;

	quit_window=newwin(5,70,10,5);

	box(quit_window,BOX_CH1,BOX_CH2);

	mvwaddstr(quit_window,1,1,"Are you sure you wish to quit?");

	mvwaddstr(quit_window,3,1,"Press Y to quit or N to return to the word processor.");

	wrefresh(quit_window);

	inputch=wgetch(quit_window);

	werase(quit_window);

	wrefresh(quit_window);

	delwin(quit_window);

	display_screen();

	switch(inputch)
	{
		case 'Y':
		case 'y':	return 1;

		default:	return 0;
	}
}

int options_menu()
{
	WINDOW *options_win;

	int inputch=0,pos=0,tmpint;

	char tmpstr[69];

	options_win=newwin(12,19,1,35);

	box(options_win,BOX_CH1,BOX_CH2);

	wattron(options_win,A_REVERSE);

	mvwaddstr(options_win,1,1,"Line Spacing");

	wattroff(options_win,A_REVERSE);

	mvwaddstr(options_win,2,1,"Page Length");

	mvwaddstr(options_win,3,1,"Tab Size");

	mvwaddstr(options_win,4,1,"Indentation Size");

	mvwaddstr(options_win,5,1,"Left Margin");

	mvwaddstr(options_win,6,1,"Right Margin");

	mvwaddstr(options_win,7,1,"Top Blanks");

	mvwaddstr(options_win,8,1,"Bottom Blanks");

	mvwaddstr(options_win,9,1,"Starting Page #");

	mvwaddstr(options_win,10,1,"Toggle Insertmode");

	keypad(options_win,TRUE);

	wmove(options_win,1,1);

	wrefresh(options_win);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=27)
	{
		inputch=getch();

		switch(pos)
		{
			case 0:	mvwaddstr(options_win,1,1,"Line Spacing");

				break;

			case 1:	mvwaddstr(options_win,2,1,"Page Length");

				break;

			case 2:	mvwaddstr(options_win,3,1,"Tab Size");

				break;

			case 3:	mvwaddstr(options_win,4,1,"Indentation Size");

				break;

			case 4:	mvwaddstr(options_win,5,1,"Left Margin");

				break;

			case 5:	mvwaddstr(options_win,6,1,"Right Margin");

				break;

			case 6:	mvwaddstr(options_win,7,1,"Top Blanks");

				break;

			case 7:	mvwaddstr(options_win,8,1,"Bottom Blanks");

				break;

			case 8:	mvwaddstr(options_win,9,1,"Starting Page #");

				break;

			case 9:	mvwaddstr(options_win,10,1,"Toggle Insertmode");

				break;
		}

		switch(inputch)
		{
			case KEY_DOWN:	pos++;

					break;

			case KEY_UP:	pos--;

					if(pos<0)
						pos+=10;

					break;

			case KEY_RIGHT:	werase(options_win);

					wrefresh(options_win);

					delwin(options_win);

					display_screen();

					return -1;

			case KEY_LEFT:	werase(options_win);

					wrefresh(options_win);

					delwin(options_win);

					display_screen();

					return -2;
		}

		pos=pos%10;

		wattron(options_win,A_REVERSE);

		switch(pos)
		{
			case 0:	mvwaddstr(options_win,1,1,"Line Spacing");
				wmove(options_win,1,1);

				break;

			case 1:	mvwaddstr(options_win,2,1,"Page Length");
				wmove(options_win,2,1);

				break;

			case 2:	mvwaddstr(options_win,3,1,"Tab Size");
				wmove(options_win,3,1);

				break;

			case 3:	mvwaddstr(options_win,4,1,"Indentation Size");
				wmove(options_win,4,1);

				break;

			case 4:	mvwaddstr(options_win,5,1,"Left Margin");
				wmove(options_win,5,1);

				break;

			case 5:	mvwaddstr(options_win,6,1,"Right Margin");
				wmove(options_win,6,1);

				break;

			case 6:	mvwaddstr(options_win,7,1,"Top Blanks");
				wmove(options_win,7,1);

				break;

			case 7:	mvwaddstr(options_win,8,1,"Bottom Blanks");
				wmove(options_win,8,1);

				break;

			case 8:	mvwaddstr(options_win,9,1,"Starting Page #");
				wmove(options_win,9,1);

				break;

			case 9:	mvwaddstr(options_win,10,1,"Toggle Insertmode");
				wmove(options_win,10,1);

				break;
		}

		wattroff(options_win,A_REVERSE);

		wrefresh(options_win);
	}

	werase(options_win);

	wrefresh(options_win);

	delwin(options_win);

	display_screen();

	if (inputch==27)
		return 0;

	attron(A_REVERSE);

	mvaddstr(0,0,"                                                                        F1:Help ");

	attroff(A_REVERSE);

	refresh();

	switch(pos)
	{
		case 0:	sprintf(tmpstr,"%d",line_spacing+1);

			if (get_inputline("Change line spacing:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>0&&tmpint<4)
					line_spacing=tmpint-1;
				else
					error_message("Didn't change line spacing because:","Incorrect number entered - must be between 1 and 3.");
			}

			return 0;

		case 1:	sprintf(tmpstr,"%d",page_length);

			if (get_inputline("Change page length:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>0&&tmpint<201)
					page_length=tmpint;
				else
					error_message("Didn't change page length because:","Incorrect number entered - must be between 1 and 200.");
			}

			return 0;

		case 2: sprintf(tmpstr,"%d",tab_size);

			if (get_inputline("Change Tab size:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>0&&tmpint<17)
					tab_size=tmpint;
				else
					error_message("Didn't change Tab size because:","Incorrect number entered - must be between 1 and 16.");
			}

			return 0;

		case 3:	sprintf(tmpstr,"%d",indent+1);

			if (get_inputline("Change indentation size:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				tmpint--;

				if (tmpint>=left_margin&&tmpint<=right_margin)
					indent=tmpint;
				else
					error_message("Didn't change indentation size because:","Incorrect number entered - must be between left and right margin.");
			}

			return 0;

		case 4:	sprintf(tmpstr,"%d",left_margin+1);

			if (get_inputline("Change left margin:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				tmpint--;

				if (tmpint<=indent&&tmpint<right_margin&&tmpint>=0)
					left_margin=tmpint;
				else
					error_message("Didn't change left margin because:","Incorrect number entered - must be between indentation and right margin.");
			}

			return 0;

		case 5:	sprintf(tmpstr,"%d",right_margin+1);

			if (get_inputline("Change right margin:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				tmpint--;

				if (tmpint>left_margin&&tmpint<=79)
					right_margin=tmpint;
				else
					error_message("Didn't change right margin because:","Incorrect number entered - must be greater than left margin.");
			}

			return 0;

		case 6:	sprintf(tmpstr,"%d",top_blanks);

			if (get_inputline("Change top blanks:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>1&&tmpint<11)
					top_blanks=tmpint;
				else
					error_message("Didn't change top blanks because:","Incorrect number entered - must be between 2 and 10.");
			}

			return 0;

		case 7:	sprintf(tmpstr,"%d",bottom_blanks);

			if (get_inputline("Change bottom blanks:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>1&&tmpint<11)
					bottom_blanks=tmpint;
				else
					error_message("Didn't change bottom blanks because:","Incorrect number entered - must be between 2 and 10.");
			}

			return 0;

		case 8:	sprintf(tmpstr,"%d",starting_page_no);

			if (get_inputline("Change starting page #:",tmpstr,tmpstr)!=-1)
			{
				sscanf(tmpstr,"%d",&tmpint);

				if (tmpint>0&&tmpint<201)
					starting_page_no=tmpint;
				else
					error_message("Didn't change starting page # because:","Incorrect number entered - must be between 1 and 200.");
			}

			return 0;

		case 9:	insertmode=(insertmode==INSERT)?OVERSTRIKE:INSERT;

			mvprintw(22,0," Title: %s Page: %d Line: %d     ",(strrchr(filename,'/'))?strrchr(filename,'/')+1:filename,((current_page-1)*LINES_PER_PAGE+current_y)/(page_length-(top_blanks+bottom_blanks))+1,current_y+1+(current_page-1)*LINES_PER_PAGE);

			if (insertmode==OVERSTRIKE)
				mvprintw(22,69,"Overstrike");
			else
				mvprintw(22,69,"Insert    ");

			if (current_attr&START_BOLD)
				mvprintw(22,64,"Bold");
			else
				mvprintw(22,64,"    ");
		
			if (current_attr&START_UNDERLINE)
				mvprintw(22,54,"Underline");
			else
				mvprintw(22,54,"         ");

			return 0;
	}
}

int do_menubar()
{
	int pos=0,inputch=0,retval=0;

	attron(A_REVERSE);

	mvaddstr(0,0," F2:File    F3:Edit    F4:Locate    F5:Options    F6:Spell              F1:Help ");

	attroff(A_REVERSE);

	mvaddstr(0,1,"F2:File");

	move(0,1);

	refresh();

	while (inputch!=27&&retval==0)
	{
		inputch=getch();

		switch(inputch)
		{
			case KEY_RIGHT:	pos++;

					pos=pos%6;

					break;

			case KEY_LEFT:	pos--;

					if (pos<0)
						pos+=6;

					break;

			case KEY_DOWN:
			case KEY_UP:
			case '\n':
			case '\r':	switch(pos)
					{
						case 0:	switch(file_menu())
							{
								case 1:	retval=1;

									break;

								case -1:pos++;

									pos=pos%6;

									break;

								case -2:pos--;

									if (pos<0)
										pos+=6;

									break;
							}

							break;

						case 1:	switch(edit_menu())
							{
								case -1:pos++;

									pos=pos%6;

									break;

								case -2:pos--;

									if (pos<0)
										pos+=6;

									break;
							}

							break;

						case 2:	switch(locate_menu())
							{
								case -1:pos++;

									pos=pos%6;

									break;

								case -2:pos--;

									if (pos<0)
										pos+=6;

									break;
							}

							break;


						case 3:	switch(options_menu())
							{
								case -1:pos++;

									pos=pos%6;

									break;

								case -2:pos--;

									if (pos<0)
										pos+=6;

									break;
							}

							break;

						case 4:	switch(spell_menu())
							{
								case -1:pos++;

									pos=pos%6;

									break;

								case -2:pos--;

									if (pos<0)
										pos+=6;

									break;
							}

							break;


						case 5:	do_help();

							break;
					}
		}

		attron(A_REVERSE);

		mvaddstr(0,0," F2:File    F3:Edit    F4:Locate    F5:Options    F6:Spell              F1:Help ");

		attroff(A_REVERSE);

		switch(pos)
		{
			case 0:	mvaddstr(0,1,"F2:File");
				move(0,1);

				break;

			case 1:	mvaddstr(0,12,"F3:Edit");
				move(0,12);

				break;

			case 2:	mvaddstr(0,23,"F4:Locate");
				move(0,23);

				break;

			case 3:	mvaddstr(0,36,"F5:Options");
				move(0,36);

				break;

			case 4:	mvaddstr(0,50,"F6:Spell");
				move(0,50);

				break;

			case 5:	mvaddstr(0,72,"F1:Help");
				move(0,72);

				break;
		}

		refresh();
	}

	attron(A_REVERSE);

	mvaddstr(0,0," F2:File    F3:Edit    F4:Locate    F5:Options    F6:Spell              F1:Help ");

	attroff(A_REVERSE);

	refresh();

	return retval;
}

int confirm_load()
{
	WINDOW *load_window;

	int inputch;

	if (!changed)
		return 1;

	load_window=newwin(5,70,10,5);

	box(load_window,BOX_CH1,BOX_CH2);

	mvwaddstr(load_window,1,1,"The file in the buffer has not yet been saved:");

	mvwaddstr(load_window,3,1,"Press Y to load a new file or N to return to the current file.");

	wrefresh(load_window);

	inputch=wgetch(load_window);

	werase(load_window);

	wrefresh(load_window);

	delwin(load_window);

	display_screen();

	switch(inputch)
	{
		case 'Y':
		case 'y':	return 1;

		default:	return 0;
	}
}

void justify(y_val)
int y_val;
{
	int start,pos,end,saveend,next=(1+line_spacing);

	if (y_val+next+((current_page-1)*LINES_PER_PAGE)>=numlines)
		return;

	for (start=left_margin;start<indent;start++)
		if (file_buffer[start+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
				file_buffer[start+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
			break;

	if (start>=indent)
		return;

	for (pos=right_margin;pos>left_margin;pos--)
		if (file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
				file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
			break;

	for (end=start;end<=right_margin;end++)
		if (file_buffer[end+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]==0||
				file_buffer[end+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]==' ')
			break;

	end--;

	if (file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
			file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
		pos+=2;

	if ((pos+end-start)>right_margin)
		return;

	saveend=end;

	for (;end>=start;end--)
		file_buffer[pos+end-start+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=
				file_buffer[end+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

	for (end=saveend+1;end<=right_margin;end++)
		if (file_buffer[end+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
				file_buffer[end+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
			break;

	for (pos=left_margin;pos<=(right_margin-(end-start));pos++)
		file_buffer[pos+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=
				file_buffer[pos+end-start+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

	for (;pos<=right_margin;pos++)
		file_buffer[pos+(y_val+next+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=0;
		
	justify(y_val);

	justify(y_val+next);
}

void insert(str,len,y_val)
short str[LINE_LENGTH];
int len;
{
	int pos,pos1=0,saved_pos=0,start,next_line=0,offset=0;

	short tmpstr[LINE_LENGTH];

	memset(tmpstr,0,LINE_LENGTH*sizeof(short));

	for (pos=right_margin-len;pos<=right_margin;pos++)
		if (file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
				file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
			break;

	if (current_x==right_margin&&pos>right_margin)
		pos=right_margin;

	if (pos==right_margin-len)
	{
		for (;pos>=left_margin;pos--)
			if (file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]==0||
					file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]==' ')
				break;

		if (pos<=left_margin)
		{
			beep();

			return;
		}

		pos++;
	}

	if (pos<=current_x)
		next_line++;

	if (pos<=right_margin)
	{
		for (pos1=right_margin;pos1>=pos;pos1--)
			if (file_buffer[pos1+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
					file_buffer[pos1+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
				break;

		pos1+=2;

		saved_pos=pos;

		for (;pos<=pos1;pos++)
		{
			if (next_line&&pos>=current_x)
			{
				if (!offset)
					offset=pos1-pos;

				tmpstr[pos-saved_pos+len]=file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];
			}
			else
				tmpstr[pos-saved_pos]=file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

			file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=0;
		}

		if (next_line)
		{
			pos1+=len;

			for (pos=current_x;pos<current_x+len;pos++)
				tmpstr[pos-saved_pos]=str[pos-current_x];
		}

		tmpstr[pos++]=' ';

		saved_pos=pos1-saved_pos;
	}
	
	for (pos=right_margin;pos>=(current_x+len);pos--)
		file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=
				file_buffer[pos-len+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH];

	if (!next_line)
	{
		for (pos=current_x;pos<current_x+len;pos++)
			file_buffer[pos+(y_val+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]=
					str[pos-current_x];
	}

	if (saved_pos>0)
	{
		int save_x=current_x;

		if (y_val+(1+line_spacing)+((current_page-1)*LINES_PER_PAGE)>=numlines)
		{
			numlines+=(1+line_spacing);
	
			file_buffer=(short *) realloc(file_buffer,numlines*LINE_LENGTH*sizeof(short));
					
			if (file_buffer==NULL)
				my_exit("Realloc error on file buffer.\n");
	
			memset(file_buffer+(numlines-(1+line_spacing))*LINE_LENGTH,0,(1+line_spacing)*LINE_LENGTH*sizeof(short));
		}
		else
		{
			for (start=left_margin;start<indent;start++)
				if (file_buffer[start+(y_val+(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=0&&
						file_buffer[start+(y_val+(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH]!=' ')
					break;

			if (start>=indent)
			{
				numlines+=(1+line_spacing);
		
				file_buffer=(short *) realloc(file_buffer,numlines*LINE_LENGTH*sizeof(short));
						
				if (file_buffer==NULL)
					my_exit("Realloc error on file buffer.\n");
		
				memset(file_buffer+(numlines-(1+line_spacing))*LINE_LENGTH,0,(1+line_spacing)*LINE_LENGTH*sizeof(short));
		
				if ((y_val+(current_page-1)*LINES_PER_PAGE+(1+line_spacing))<(numlines-(1+line_spacing)))
				{
					int line;

					for (line=numlines-1;line>=y_val+2*(1+line_spacing)+(current_page-1)*LINES_PER_PAGE;line-=(1+line_spacing))
						memcpy(file_buffer+line*LINE_LENGTH,file_buffer+(line-(1+line_spacing))*LINE_LENGTH,LINE_LENGTH*sizeof(short));

					memset(file_buffer+(y_val+(1+line_spacing)+(current_page-1)*LINES_PER_PAGE)*LINE_LENGTH,0,80*sizeof(short));
				}
			}
		}

		current_x=left_margin;

		insert(tmpstr,saved_pos,y_val+(1+line_spacing));

		if (next_line)
			current_x=-(left_margin+saved_pos-(offset));
		else
			current_x=save_x+len;
	}
	else
	{
		current_x+=len;
	}
}

void print_file()
{
	char prn_file[65],tempstr1[LINE_LENGTH+1],tempstr2[LINE_LENGTH+1];

	FILE *prn_fd;

	int line=0,page=starting_page_no,counter,bold_on=0,underline_on=0,pos;

	sprintf(prn_file,"/tmp/hwpprn.%d",getpid());

	errno=0;

	prn_fd=fopen(prn_file,"w");

	if (prn_fd==NULL)
	{
		error_message("Couldn't open temporary print file because:",strerror(errno));

		return;
	}

	while(line<numlines)
	{
		memset(tempstr1,0,LINE_LENGTH);

		tempstr1[LINE_LENGTH]=0;

		if (strcspn(header,"%"))
			strncpy(tempstr1,header,strcspn(header,"%"));

		if (strchr(header,'%'))
		{
			sprintf(tempstr2,"%d",page);

			strcat(tempstr1,tempstr2);

			strcat(tempstr1,strchr(header,'%')+1);
		}

		memset(tempstr2,' ',LINE_LENGTH);

		tempstr2[LINE_LENGTH]=0;

		strcpy(tempstr2+LINE_LENGTH-strlen(tempstr1),tempstr1);
		strcat(tempstr2,"\n");

		fputs(tempstr2,prn_fd);

		for (counter=1;counter<top_blanks;counter++)
			fputc('\n',prn_fd);

		for (counter=0;counter<page_length-top_blanks-bottom_blanks&&line<numlines;counter++)
		{
			for (pos=0;pos<LINE_LENGTH;pos++)
			{
				if (file_buffer[pos+line*LINE_LENGTH])
					fputc(file_buffer[pos+line*LINE_LENGTH],prn_fd);
				else
					fputc(' ',prn_fd);
			}

			line++;
		}

		if (line>=numlines)
			for (;counter<page_length-top_blanks-bottom_blanks;counter++)
				fputc('\n',prn_fd);

		for (counter=1;counter<bottom_blanks;counter++)
			fputc('\n',prn_fd);

		memset(tempstr1,0,LINE_LENGTH);

		tempstr1[LINE_LENGTH]=0;

		if (strcspn(footer,"%"))
			strncpy(tempstr1,footer,strcspn(footer,"%"));

		if (strchr(footer,'%'))
		{
			sprintf(tempstr2,"%d",page);

			strcat(tempstr1,tempstr2);

			strcat(tempstr1,strchr(footer,'%')+1);
		}

		memset(tempstr2,' ',LINE_LENGTH);

		tempstr2[LINE_LENGTH]=0;

		strcpy(tempstr2+LINE_LENGTH-strlen(tempstr1),tempstr1);
		strcat(tempstr2,"\n");

		fputs(tempstr2,prn_fd);

		fflush(prn_fd);

		page++;
	}

	close(prn_fd);

	sprintf(tempstr1,"%s %s",SPOOLER,prn_file);

	system(tempstr1);

	unlink(prn_file);

	clearok(stdscr,TRUE);

	refresh();

	clearok(textscr,TRUE);

	wrefresh(textscr);
}

int edit_menu()
{
	WINDOW *edit_win;

	int inputch=0,pos=0;

	edit_win=newwin(4,13,1,11);

	box(edit_win,BOX_CH1,BOX_CH2);

	wattron(edit_win,A_REVERSE);

	mvwaddstr(edit_win,1,1,"Edit Header");

	wattroff(edit_win,A_REVERSE);

	mvwaddstr(edit_win,2,1,"Edit Footer");

	keypad(edit_win,TRUE);

	wmove(edit_win,1,1);

	wrefresh(edit_win);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=27)
	{
		inputch=getch();

		switch(pos)
		{
			case 0:	mvwaddstr(edit_win,1,1,"Edit Header");

				break;

			case 1:	mvwaddstr(edit_win,2,1,"Edit Footer");

				break;
		}

		switch(inputch)
		{
			case KEY_DOWN:	pos++;

					break;

			case KEY_UP:	pos--;

					if(pos<0)
						pos+=2;

					break;

			case KEY_RIGHT:	werase(edit_win);

					wrefresh(edit_win);

					delwin(edit_win);

					display_screen();

					return -1;

			case KEY_LEFT:	werase(edit_win);

					wrefresh(edit_win);

					delwin(edit_win);

					display_screen();

					return -2;
		}

		pos=pos%2;

		wattron(edit_win,A_REVERSE);

		switch(pos)
		{
			case 0:	mvwaddstr(edit_win,1,1,"Edit Header");
				wmove(edit_win,1,1);

				break;

			case 1:	mvwaddstr(edit_win,2,1,"Edit Footer");
				wmove(edit_win,2,1);

				break;
		}

		wattroff(edit_win,A_REVERSE);

		wrefresh(edit_win);
	}

	werase(edit_win);

	wrefresh(edit_win);

	delwin(edit_win);

	display_screen();

	if (inputch==27)
		return 0;

	attron(A_REVERSE);

	mvaddstr(0,0,"                                                                        F1:Help ");

	attroff(A_REVERSE);

	refresh();

	switch(pos)
	{
		case 0:	get_inputline("Change header (first % will be page no.):",header,header);

			return 0;

		case 1:	get_inputline("Change footer (first % will be page no.):",footer,footer);

			return 0;
	}
}

void write_ascii(ascii_file)
char *ascii_file;
{
	char tempstr1[LINE_LENGTH+1],tempstr2[LINE_LENGTH+1];

	FILE *ascii_fd;

	int line=0,page=starting_page_no,counter,bold_on=0,underline_on=0,pos;

	errno=0;

	ascii_fd=fopen(ascii_file,"w");

	if (ascii_fd==NULL)
	{
		error_message("Couldn't open ASCII file because:",strerror(errno));

		return;
	}

	while(line<numlines)
	{
		memset(tempstr1,0,LINE_LENGTH);

		tempstr1[LINE_LENGTH]=0;

		if (strcspn(header,"%"))
			strncpy(tempstr1,header,strcspn(header,"%"));

		if (strchr(header,'%'))
		{
			sprintf(tempstr2,"%d",page);

			strcat(tempstr1,tempstr2);

			strcat(tempstr1,strchr(header,'%')+1);
		}

		memset(tempstr2,' ',LINE_LENGTH);

		tempstr2[LINE_LENGTH]=0;

		strcpy(tempstr2+LINE_LENGTH-strlen(tempstr1),tempstr1);
		strcat(tempstr2,"\n");

		fputs(tempstr2,ascii_fd);

		for (counter=1;counter<top_blanks;counter++)
			fputc('\n',ascii_fd);

		for (counter=0;counter<page_length-top_blanks-bottom_blanks&&line<numlines;counter++)
		{
			for (pos=0;pos<LINE_LENGTH;pos++)
			{
				if (file_buffer[pos+line*LINE_LENGTH])
					fputc(file_buffer[pos+line*LINE_LENGTH],ascii_fd);
				else
					fputc(' ',ascii_fd);
			}

			fputc('\n',ascii_fd);

			line++;
		}

		if (line>=numlines)
			for (;counter<page_length-top_blanks-bottom_blanks;counter++)
				fputc('\n',ascii_fd);

		for (counter=1;counter<bottom_blanks;counter++)
			fputc('\n',ascii_fd);

		memset(tempstr1,0,LINE_LENGTH);

		tempstr1[LINE_LENGTH]=0;

		if (strcspn(footer,"%"))
			strncpy(tempstr1,footer,strcspn(footer,"%"));

		if (strchr(footer,'%'))
		{
			sprintf(tempstr2,"%d",page);

			strcat(tempstr1,tempstr2);

			strcat(tempstr1,strchr(footer,'%')+1);
		}

		memset(tempstr2,' ',LINE_LENGTH);

		tempstr2[LINE_LENGTH]=0;

		strcpy(tempstr2+LINE_LENGTH-strlen(tempstr1),tempstr1);
		strcat(tempstr2,"\n");

		fputs(tempstr2,ascii_fd);

		fflush(ascii_fd);

		page++;
	}

	close(ascii_fd);
}

int locate_menu()
{
	WINDOW *locate_win;

	int inputch=0,pos=0;

	static find_type=0;

	static char search_str[LINE_LENGTH+1]="";

	locate_win=newwin(5,25,1,23);

	box(locate_win,BOX_CH1,BOX_CH2);

	wattron(locate_win,A_REVERSE);

	mvwaddstr(locate_win,1,1,"Find (case sensitive)");

	wattroff(locate_win,A_REVERSE);

	mvwaddstr(locate_win,2,1,"Find (case insensitive)");

	mvwaddstr(locate_win,3,1,"Find next");

	keypad(locate_win,TRUE);

	wmove(locate_win,1,1);

	wrefresh(locate_win);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=27)
	{
		inputch=getch();

		switch(pos)
		{
			case 0:	mvwaddstr(locate_win,1,1,"Find (case sensitive)");

				break;

			case 1:	mvwaddstr(locate_win,2,1,"Find (case insensitive)");

				break;

			case 2:	mvwaddstr(locate_win,3,1,"Find next");

				break;
		}

		switch(inputch)
		{
			case KEY_DOWN:	pos++;

					break;

			case KEY_UP:	pos--;

					if(pos<0)
						pos+=3;

					break;

			case KEY_RIGHT:	werase(locate_win);

					wrefresh(locate_win);

					delwin(locate_win);

					display_screen();

					return -1;

			case KEY_LEFT:	werase(locate_win);

					wrefresh(locate_win);

					delwin(locate_win);

					display_screen();

					return -2;
		}

		pos=pos%3;

		wattron(locate_win,A_REVERSE);

		switch(pos)
		{
			case 0:	mvwaddstr(locate_win,1,1,"Find (case sensitive)");
				wmove(locate_win,1,1);

				break;

			case 1:	mvwaddstr(locate_win,2,1,"Find (case insensitive)");
				wmove(locate_win,2,1);

				break;

			case 2:	mvwaddstr(locate_win,3,1,"Find next");
				wmove(locate_win,3,1);

				break;
		}

		wattroff(locate_win,A_REVERSE);

		wrefresh(locate_win);
	}

	werase(locate_win);

	wrefresh(locate_win);

	delwin(locate_win);

	display_screen();

	if (inputch==27)
		return 0;

	attron(A_REVERSE);

	mvaddstr(0,0,"                                                                        F1:Help ");

	attroff(A_REVERSE);

	refresh();

	switch(pos)
	{
		case 0:	if (get_inputline("Enter search string:",search_str,search_str)==-1)
				return 0;

			find_type=CASE;

			do_search(search_str,find_type,1);

			display_screen();

			return 0;

		case 1:	if (get_inputline("Enter search string:",search_str,search_str)==-1)
				return 0;

			find_type=NOCASE;

			do_search(search_str,find_type,1);

			display_screen();

			return 0;

		case 2:	do_search(search_str,find_type,0);

			display_screen();

			return 0;
	}
}

int do_search(string,case_type,first)
char *string;
int case_type,first;
{
	int line,pos,cont,goodpos,len;

	len=strlen(string);

	if (case_type==NOCASE)
	{
		for (pos=0;pos<len;pos++)
		{
			if (isupper(string[pos]))
				string[pos]=tolower(string[pos]);
		}
	}

	pos=current_x+1;

	for (line=current_y+(current_page-1)*LINES_PER_PAGE;line<numlines;line++)
	{
		cont=0;

		for (;pos<LINE_LENGTH;pos++)
		{
			if (case_type==CASE)
			{
				if (string[cont]==file_buffer[line*LINE_LENGTH+pos])
				{
					if (cont==0)
						goodpos=pos;
	
					cont++;
				}
				else
					cont=0;

				if (cont>=len)
					break;
			}
			else
			{
				char tmpch;

				if (isupper(file_buffer[line*LINE_LENGTH+pos]))
					tmpch=tolower(file_buffer[line*LINE_LENGTH+pos]);
				else
					tmpch=file_buffer[line*LINE_LENGTH+pos];

				if (string[cont]==tmpch)
				{
					if (cont==0)
						goodpos=pos;
	
					cont++;
				}
				else
					cont=0;

				if (cont>=len)
					break;
			}
		}

		pos=0;

		if (cont>=len)
			break;
	}

	if (cont>=len)
	{
		current_x=goodpos;

		current_y=line%LINES_PER_PAGE;

		current_page=line/LINES_PER_PAGE+1;

		return 1;
	}

	for (line=0;line<current_y+(current_page-1)*LINES_PER_PAGE;line++)
	{
		cont=0;

		for (pos=0;pos<LINE_LENGTH;pos++)
		{
			if (case_type==CASE)
			{
				if (string[cont]==file_buffer[line*LINE_LENGTH+pos])
				{
					if (cont==0)
						goodpos=pos;
	
					cont++;
				}
				else
					cont=0;

				if (cont>=len)
					break;
			}
			else
			{
				char tmpch;

				if (isupper(file_buffer[line*LINE_LENGTH+pos]))
					tmpch=tolower(file_buffer[line*LINE_LENGTH+pos]);
				else
					tmpch=file_buffer[line*LINE_LENGTH+pos];

				if (string[cont]==tmpch)
				{
					if (cont==0)
						goodpos=pos;
	
					cont++;
				}
				else
					cont=0;

				if (cont>=len)
					break;
			}
		}

		if (cont>=len)
			break;
	}

	if (cont>=len)
	{
		current_x=goodpos;

		current_y=line%LINES_PER_PAGE;

		current_page=line/LINES_PER_PAGE+1;

		return 1;
	}
	else
	{
		if (first>0)
			error_message("Couldn't find pattern because:","It isn't present in this file.");
		else if (first==0)
			error_message("Couldn't find pattern because:","There aren't any more instances in this file.");

		return 0;
	}

	return 0;
}

int spell_menu()
{
	WINDOW *spell_win;

	int inputch=0,pos=0;

	static current_word=-1,last_x=-1,last_y=-1,last_page=-1;

	spell_win=newwin(4,21,1,49);

	box(spell_win,BOX_CH1,BOX_CH2);

	wattron(spell_win,A_REVERSE);

	mvwaddstr(spell_win,1,1,"Check Spelling");

	wattroff(spell_win,A_REVERSE);

	mvwaddstr(spell_win,2,1,"Next incorrect word");

	keypad(spell_win,TRUE);

	wmove(spell_win,1,1);

	wrefresh(spell_win);

	while(inputch!='\n'&&inputch!='\r'&&inputch!=27)
	{
		inputch=getch();

		switch(pos)
		{
			case 0:	mvwaddstr(spell_win,1,1,"Check Spelling");

				break;

			case 1:	mvwaddstr(spell_win,2,1,"Next incorrect word");

				break;
		}

		switch(inputch)
		{
			case KEY_DOWN:	pos++;

					break;

			case KEY_UP:	pos--;

					if(pos<0)
						pos+=2;

					break;

			case KEY_RIGHT:	werase(spell_win);

					wrefresh(spell_win);

					delwin(spell_win);

					display_screen();

					return -1;

			case KEY_LEFT:	werase(spell_win);

					wrefresh(spell_win);

					delwin(spell_win);

					display_screen();

					return -2;
		}

		pos=pos%2;

		wattron(spell_win,A_REVERSE);

		switch(pos)
		{
			case 0:	mvwaddstr(spell_win,1,1,"Check Spelling");
				wmove(spell_win,1,1);

				break;

			case 1:	mvwaddstr(spell_win,2,1,"Next incorrect word");
				wmove(spell_win,2,1);

				break;
		}

		wattroff(spell_win,A_REVERSE);

		wrefresh(spell_win);
	}

	werase(spell_win);

	wrefresh(spell_win);

	delwin(spell_win);

	display_screen();

	if (inputch==27)
		return 0;

	attron(A_REVERSE);

	mvaddstr(0,0,"                                                                        F1:Help ");

	attroff(A_REVERSE);

	refresh();

	switch(pos)
	{
		case 0:	get_spell_list();

			last_page=last_x=last_y=-1;

			if (num_words==-1)
				return 0;

			if (num_words==0)
			{
				error_message("No words to check because:","All words are spelled correctly.");

				return 0;
			}

			current_word=0;

			while (do_search(word_list[current_word],CASE,-1)==0)
			{
				current_word++;

				if (current_word>=num_words)
				{
					error_message("No words to check because:","All words are spelled correctly.");

					return 0;
				}
			}

			display_screen();

			return 0;

		case 1:	if (num_words<0)
			{
				error_message("Can't get next incorrect word because:","You haven't checked the spelling yet.");

				return 0;
			}

			if (current_word>=num_words)
			{
				error_message("Can't get next incorrect word because:","All words have been checked.");

				return 0;
			}

			while (1)
			{
				if (do_search(word_list[current_word],CASE,-1)==0)
				{
					current_word++;

					last_page=last_x=last_y=-1;

					if (current_word>=num_words)
					{
						error_message("Can't get next incorrect word because:","All words have been checked.");
	
						return 0;
					}
				}
				else
				{
					if (last_x==-1)
					{
						last_x=current_x;
		
						last_y=current_y;
		
						last_page=current_page;

						break;
					}
					else if (last_x==current_x&&last_y==current_y&&last_page==current_page)
					{
						last_page=last_x=last_y=-1;

						current_word++;

						if (current_word>=num_words)
						{
							error_message("Can't get next incorrect word because:","All words have been checked.");
		
								return 0;
						}
					}
					else
						break;
				}
			}

			display_screen();

			return 0;
	}
}

void get_spell_list()
{
	FILE *spell_fd;

	int line=0,page=1,counter,pos;

	char spell_file1[65],spell_file2[65],command[65],inputline[LINE_LENGTH+1];

	if (word_list!=NULL)
	{
		for (counter=0;counter<num_words;counter++)
			free(word_list[counter]);

		free(word_list);

		word_list=NULL;

		num_words=-1;
	}

	sprintf(spell_file1,"/tmp/hwpsp1.%d",getpid());
	sprintf(spell_file2,"/tmp/hwpsp2.%d",getpid());

	errno=0;

	spell_fd=fopen(spell_file1,"w");

	if (spell_fd==NULL)
	{
		error_message("Couldn't open temporary spell checking file because:",strerror(errno));

		return;
	}

	for (line=0;line<numlines;line++)
	{
		for (pos=0;pos<LINE_LENGTH;pos++)
		{
			if (file_buffer[pos+line*LINE_LENGTH])
				fputc(file_buffer[pos+line*LINE_LENGTH],spell_fd);
			else
				fputc(' ',spell_fd);
		}

		fputc('\n',spell_fd);
	}

	fflush(spell_fd);

	close(spell_fd);

	sprintf(command,"spell %s > %s",spell_file1,spell_file2);

	system(command);

	unlink(spell_file1);

	errno=0;

	spell_fd=fopen(spell_file2,"r");

	if (spell_fd==NULL)
	{
		error_message("Couldn't open temporary spell checking file because:",strerror(errno));

		return;
	}

	num_words=0;

	while (fgets(inputline,LINE_LENGTH+1,spell_fd))
		num_words++;

	rewind(spell_fd);

	word_list=(char **) malloc(num_words*sizeof(char *));

	if (file_buffer==NULL)
	{
		error_message("Couldn't spell check because:","Malloc error on word list.");

		num_words=-1;

		return;
	}

	for (line=0;line<num_words;line++)
	{
		char *tmpptr;

		fgets(inputline,LINE_LENGTH+1,spell_fd);

		tmpptr=strrchr(inputline,'\n');

		if (tmpptr)
			*tmpptr=0;

		tmpptr=strrchr(inputline,'\r');

		if (tmpptr)
			*tmpptr=0;

		word_list[line]=(char *)malloc(strlen(inputline)+1);

		if (word_list[line]==NULL)
		{
			error_message("Couldn't spell check because:","Malloc error on word in word list.");
	
			num_words=-1;
	
			return;
		}

		strcpy(word_list[line],inputline);
	}

	fclose(spell_fd);

	unlink(spell_file2);
}

void do_help()
{
	werase(textscr);

	mvwaddstr(textscr,0,4,"Help for hwp (Harry's Word Processor):");
	mvwaddstr(textscr,1,4,"**************************************");

	mvwaddstr(textscr,3,4,"To activate menubar, press F10.  To use a specific function press the");
	mvwaddstr(textscr,4,4,"appropriate function key.  Once in the menubar, use the arrow keys to");
	mvwaddstr(textscr,5,4,"navigate the menubar; use Enter, Up or Down to select a menu.  Once in");
	mvwaddstr(textscr,6,4,"a menu, use the Up and Down keys to move to the desired selection; use");
	mvwaddstr(textscr,7,4,"Enter key to select.  ESC cancels either the menubar or a menu.");
	mvwaddstr(textscr,8,4,"hwp automatically word wraps in both insert and overstrike mode.  Use");
	mvwaddstr(textscr,9,4,"the arrow keys, pageup/down and home/end to move around.  Use Delete and");
	mvwaddstr(textscr,10,4,"and Backspace to erase text.  To change the insert mode use the Insert");
	mvwaddstr(textscr,11,4,"key or choose Toggle Insertmode from the Options menu.  To change your");
	mvwaddstr(textscr,12,4,"header or footer, use the Edit menu.  To load, save, print, new a");
	mvwaddstr(textscr,13,4,"file or quit use the File menu; to save a file as ASCII use an ending");
	mvwaddstr(textscr,14,4,"other than .hwp.  To change margins, tab size, etc. use the Options");
	mvwaddstr(textscr,15,4,"menu.  To find words in the text, use the find menu.  To check the");
	mvwaddstr(textscr,16,4,"spelling of your document, use the Spell menu; choose Next Incorrect");
	mvwaddstr(textscr,17,4,"Word to go to the next misspelled word.");

	mvwaddstr(textscr,20,4,"Press any key to go on...");

	wrefresh(textscr);

	getch();

	display_screen();
}
