/*
 * FGREP.C - Durchsuche File(s) nach Muster(n)
 *
 * Version 1.08       11. May 1991
 *
 * Veraenderungen:
 *
 *   V1.00 (84/12/01)   - Beta-Test-Version
 *   V1.01 (85/01/01)   - -P-Option hinzugefuegt
 *                      - Kommandozeilen-Ueberpruefung verbessert
 *   V1.02 (85/01/06)   - "run_fsa()" und "bd_move()" veraendert
 *   V1.03 (85/02/11)   - -S-Option hinzugefuegt.
 *   V1.04 (85/09/08)   - "bd_go()" fuer UNIX korrigiert
 *   V1.05 (85/09/20)   - (Lloyd Rice, Computalker Consultants)
 *                      - Definitionen fuer Lattice C hinzugefuegt
 *                      - Zeilenzaehler fuer -CV-Option komplementiert
 *                      - Puffer fuer "stoupper()" hinzugefuegt
 *                      - Ermoeglichung von signed char - Darstellungen
 *   V1.06 (86/09/13)   - Fehler in -P-Option korrigiert.
 *   V1.07 (86/11/20)   - Definitionen fuer Microsoft C V4.0 hinzugefuegt
 *
 * Copyright 1985,1986: Ian Ashdown
 *                      byHeart Software
 *                      620 Ballantree Road
 *                      West Vancouver, B.C.
 *                      Canada V7S 1W3
 *
 *   V1.08 (91/05/11)	- Anpassung an COHERENT
 *			- Fehlermeldungen in Englisch und ohne Beep
 *
 * Copyright 1991:	Udo Munk
 *			Oberstr. 21
 *			4040 Neuss 1
 *			Germany
 *
 *  Dieses Programm darf fuer persoenliche, nicht kommerzielle Zwecke
 *  kopiert werden, vorausgesetzt, dass die obige Copyright-Bemerkung
 *  in allen Kopien des Source-Code steht. Die Erstellung einer Kopie
 *  fuer andere Zwecke ist ohne ausdrueckliche Erlaubnis des Autors
 *  nicht gestattet.
 *
 *  Anmerkungen:
 *
 *  Der in diesem Programm verwendete Algorithmus konstruiert einen
 *  deterministischen finiten Zustands-Mechanismus (FSA) zur
 *  Mustererkennung in Teilstrings. Daraufhin verwendet es den FSA zur
 *  Bearbeitung des Text-Strings in einem Durchgang. Die erforderliche
 *  Zeit fuer den Aufbau der FSA ist proportional zur Summe der Laengen
 *  in den Teilstrings. Die Anzahl der Status-Uebergaenge innerhalb des FSA
 *  ist fuer die Textberarbeitung unabhaengig von der Anzahl der Teilstrings.
 *
 *  Algorithmusquelle:
 *
 *  "Efficient String Matching: An Aid to Bibliographic Search"
 *  Alfred V. Aho & Margaret J. Corasick
 *  Communications of the ACM
 *  Seiten 333 - 340, Vol. 18 No. 6 (June '75)
 *
 * Aufrufformat: fgrep [-vclnhyefxps] [strings] <files>
 *
 * wobei:
 *
 *      -v      Alle nicht passenden Zeilen werden ausgegeben.
 *      -c      Lediglich die Anzahl der passenden Zeilen werden ausgegeben.
 *      -l      Die Namen der Files mit passenden Zeilen werden einmal
 *              - getrennt durch Zeilenvorschuebe - aufgelistet.
 *      -n      Jeder Zeile wird ihre Zeilennummer im File vorgesetzt.
 *      -h      Keine Filenamen-Header mit den Zeilen ausgeben.
 *      -y      Alle Zeichen in dem File werden vor der Mustererkennung
 *              in Grossbuchstaben umgewandelt (Dies ist die Default-
 *              einstellung, wenn der String in der Kommandozeile unter
 *              CP/M eingegeben wird, da CP/M alles in der Kommandozeile
 *              in Grossbuchstaben umgewandelt wird. Verwenden Sie die
 *              Option -f, wenn Sie sowohl grosse als auch kleine Zeichen
 *              benoetigen). Dies ist keine eigentliche UNIX-fgrep-Option
 *              (normalerweise nur unter grep vorhanden), ist jedoch
 *              zu nuetzlich, als dass wir es auslassen sollten.
 *      -e      <string>. Dasselbe wie ein String-Argument, jedoch nuetzlich,
 *              wenn der String mit einem '-' beginnt.
 *      -f      <file>. Die Strings (durch Zeilenwechsel getrennt) werden
 *              einem File entnommen. Wenn in dem File verschiedene
 *              Strings aufgelistet sind, dann wird eine Uebereinstimmung
 *              angezeigt, sofern irgendein String passt. Falls -f eingegeben
 *              wurde, wird jedes folgende Argument in der Kommandozeile
 *              als Filename uebernommen.
 *      -x      Nur vollstaendig passende Zeilen werden ausgegeben.
 *      -p      Jede passende Zeile wird von dem passenden Teilstring
 *              angefuehrt. Dies ist keine UNIX-fgrep-Option, jedoch zu
 *              nuetzlich, um sie fortzulassen.
 *      -s      Es wird keine Ausgabe vorgenommen. Nur ein Status wird
 *              zurueckgegeben. Diese Option laesst sich verwenden, wenn
 *              fgrep als Prozess laeuft, der einen Status an seinen
 *              aufrufenden Prozess uebergeben soll. Unter CP/M kann eine
 *              ueber exit() zurueckgegebene Zahl ungleich Null ein Submit-
 *              File beenden, das das Programm startete, obwohl das
 *              von der jeweiligen Anwendung abhaengt.
 *
 * RUECKGABEN:
 *
 * Der Exit-Status ist Null, wenn Muster erkannt wurden, 1 wenn nicht
 * und 2 fuer einen Fehler
 *
 * FEHLER:
 *
 * Die folgende UNIX-spezifische Option wird nicht unterstuetzt:
 *
 *      -b      Jeder Zeile wird eine Blocknummer vorgestellt.
 *
 * Zeilen sind auf 256 Zeichen begrenzt.
 *
 */

/*** Definitionen ***/

#define TRUE               1
#define FALSE              0
#define MAX_LINE         257  /* Maximale Anzahl Zeichen pro  */
			      /* Zeile plus NULL-Kennung      */
/* #define CPM */             /* Inaktiviert fuer Kompilation  */
			      /* unter UNIX oder MS-DOS       */

/*  Die folgenden Definitionen sind fuer die Compiler Lattice (Vers. 2.15)
 *  und Microsoft (Version 4.0) unter MS-DOS zustaendig. Die Lattice-
 *  und Microsoft-Libraries enthalten keine UNIX-Funktion "index()".
 *  Dennoch unterstuetzt Lattice eine aequivalente Funktion mit dem
 *  Namen "stpchr()" - bei Microsoft heisst sie "strchr()".
 */

/* #define index stpchr */    /* Entfernen Sie die Kommentarzeichen */
			      /* fuer die Lattice C Version 2.15     */
/* #define index strchr */    /* Entfernen Sie die Kommentarzeichen */
			      /* fuer die Microsoft C Version 4.0    */

#define CMD_ERR    0    /* Fehler-Codes */
#define OPT_ERR    1
#define INP_ERR    2
#define STR_ERR    3
#define MEM_ERR    4

/*** Typedefs ***/

typedef int BOOL;       /* Bool-Flag */

/*  Einige C-Compiler, einschliesslich Lattice C, unterstuetzen keinen
 *  "void"-Datentyp. Entfernen Sie die folgenden Kommentarzeichen fuer
 *  solche Compiler.
 */

/* typedef int void; */

/*** Datenstrukturen ***/

/* Schlangen-Elemente */

typedef struct queue
{
  struct state_el *st_ptr;
  struct queue *next_el;
} QUEUE;

/* Uebergangs-Element */

typedef struct transition
{
  char lchar;                   /* Uebergangszeichen       */
  struct state_el *nextst_ptr;  /* Uebergangs-Statuszeiger */
  struct transition *next_el;
} TRANSITION;

/* FSA-Status-Element */

typedef struct state_el
{
  TRANSITION *go_ls;    /* Zeiger auf den Kopf der "go"-Liste */
  TRANSITION *mv_ls;    /* Zeiger auf Kopf der "move"-Liste   */
  struct state_el *fail_state;  /* "failure"-Uebergangsstatus  */
  char *out_str;        /* Endstatus-Meldung (falls vorhanden)*/
} FSA;

/*** Globale Variablen und Strukturen ***/

/* Dummy-"failure"-Status */

FSA FAIL_STATE;

/*  Definiere eine separate Datenstruktur fuer Status Null der
 *  FSA, um die Bearbeitung bei der Eingabe in diesem Status zu
 *  beschleunigen. Da der Aho-Corasick-Algorithmus lediglich
 *  go-Uebergaenge fuer diesen Status definiert (einen fuer jedes
 *  gueltige Eingabezeichen) und keine failure-Uebergaenge oder
 *  output-Meldungen, lediglich ein Array von go-Uebergangs-Status-
 *  Nummern benoetigt wird. Auf das Array wird direkt zugegriffen, indem
 *  das Eingabezeichen als Index benutz wird.
 */

FSA *MZ[128];   /* Status 0 der FSA */

BOOL vflag = FALSE,     /* Kommandozeilen-Options-Flags */
     cflag = FALSE,
     lflag = FALSE,
     nflag = FALSE,
     hflag = FALSE,
     yflag = FALSE,
     eflag = FALSE,
     fflag = FALSE,
     xflag = FALSE,
     pflag = FALSE,
     sflag = FALSE;

/*** Include-Files ***/

#include <stdio.h>
#include <ctype.h>

/*** Hauptteil des Programmes ***/

int main(argc,argv)
int argc;
char **argv;
{
  char *temp;
  BOOL match_flag = FALSE,
       proc_file();
  void bd_go(),
       bd_move(),
       error();

  /* Teste auf die Mindestzahl an Kommandozeilen-Argumenten */

  if(argc < 2)
    error(CMD_ERR,NULL);

  /* Durchsuche die Kommandozeile auf Benutzer-Optionen */

  while(--argc && (*++argv)[0] == '-' && eflag == FALSE)
    for(temp = argv[0]+1; *temp != '\0'; temp++)    
      switch(toupper(*temp))
      {
        case 'V':
          vflag = TRUE;
          break;
        case 'C':
          cflag = TRUE;
          break;
        case 'L':
          lflag = TRUE;
          break;
        case 'N':
          nflag = TRUE;
          break;
        case 'H':
          hflag = TRUE;
          break;
        case 'Y':
          yflag = TRUE;
          break;
        case 'E':
          eflag = TRUE;
          break;
        case 'F':
          fflag = TRUE;
          break;
        case 'X':
          xflag = TRUE;
          break;
        case 'P':
          pflag = TRUE;
          break;
        case 'S':
          sflag = TRUE;
          break;
        default:
          error(OPT_ERR,NULL);
      }

  /* "pflag" kann nur TRUE sein, wenn die folgenden Flags FALSE sind: */

  if(vflag == TRUE || cflag == TRUE || lflag == TRUE ||
      xflag == TRUE || sflag == TRUE)
    pflag = FALSE;

  /* Teste auf String- (oder String-File-) Argument */

  if(!argc)
    error(CMD_ERR,NULL);

  /* Errichte die go-Uebergaenge */

  bd_go(*argv++);
  argc--;

  /* Errichte die failure- und move-Uebergaenge */

  bd_move();

  /* Bearbeite alle Eingabefiles, falls nicht stdin */

  if(argc < 2)
    hflag = TRUE;
  if(!argc)
  {
    if(proc_file(NULL,FALSE) == TRUE && match_flag == FALSE)
      match_flag = TRUE;
  }
  else
    while(argc--)
      if(proc_file(*argv++,TRUE) == TRUE && match_flag == FALSE)
        match_flag = TRUE;

  /*  Liefere den Status an den aufrufenden Prozess. Der Status ist
   *  Null, wenn Uebereinstimmungen gefunden wurden, 1 wenn nicht.
   */

  if(match_flag == TRUE)
    exit(0);
  else
    exit(1);
}

/*** Functionen und Prozeduren ***/

/* PROC_FILE() - Starte die FSA mit dem Eingabe-File "in_file". Liefere
 *               TRUE, wenn ein Muster gefunden wurde, ansonsten FALSE.
 */

BOOL proc_file(in_file,prt_flag)
char *in_file;
BOOL prt_flag;
{
  char buffer[MAX_LINE],        /* Eingabe-String-Puffer */
       *bufptr,
       *nl,
       *index(),
       *stoupper(),
       *fgets();
  static char ucbuf[MAX_LINE];  /* Grossbuchstaben-String-Puffer */
  long line_cnt = 0L,   /* Zeilenzaehler               */
       mtch_cnt = 0L;   /* Zaehler fuer passende Zeilen */
  BOOL mtch_flag,       /* Flag fuer passende Zeilen   */
       run_fsa();
  FILE *in_fd,
       *fopen();
  void error();

  if(in_file != NULL)   /* Ein File wurde als Eingabe angegeben */
  {
    if(!(in_fd = fopen(in_file,"r")))
      error(INP_ERR,in_file);
  }
  else
    in_fd = stdin;

  /* Lese zur Bearbeitung jeweils eine Zeile ein */

  while(fgets(buffer,MAX_LINE,in_fd))
  {
    if(nl = index(buffer,'\n'))  /* Entferne Zeilenenden */
      *nl = '\0';
    bufptr = buffer;
#ifdef CPM
    if(fflag == FALSE || yflag == TRUE)
#else
    if(yflag == TRUE)
#endif
    {
      strcpy(ucbuf,buffer);
      stoupper(bufptr = ucbuf);
    }
    line_cnt++;         /* Erhoehe den Zeilenzaehler           */
    if((mtch_flag = run_fsa(bufptr)) == TRUE)
      mtch_cnt++;       /* Erhoehe Zaehler fuer passende Zeilen */
    if(cflag == FALSE && lflag == FALSE && sflag == FALSE &&
        ((mtch_flag == TRUE && vflag == FALSE) ||
        (mtch_flag == FALSE && vflag == TRUE)))
    {
      if(hflag == FALSE && prt_flag == TRUE)
        printf("%s: ",in_file);
      if(nflag == TRUE)
        printf("%05ld: ",line_cnt);
      puts(buffer);
    }
  }
  if(lflag == TRUE && mtch_cnt > 0)
    printf("%s\n",in_file);
  else if(cflag == TRUE && sflag == FALSE)
  {
    if(vflag == TRUE)
      printf("%ld\n",line_cnt - mtch_cnt);
    else
      printf("%ld\n",mtch_cnt);
  }
  if(in_file != NULL)
    fclose(in_fd);
  if(mtch_cnt)          /* Muster gefunden      */
    return TRUE;
  else                  /* Kein Muster gefunden */
    return FALSE;
}

/* RUN_FSA() - Starte den finiten Zustands-Mechnismus mit dem String
 *             "str" als Eingabe. Liefere TRUE bei Uebereinstimmung,
 *             FALSE im anderen Fall.
 */

BOOL run_fsa(str)
register char *str;
{
  register FSA *st_ptr;
  BOOL msg_flag = FALSE;
  FSA *go(),
      *move();

  st_ptr = NULL;        /* Initialisiere FSA */
  if(xflag == FALSE)
  {
    /* Bearbeite das naechste Eingabezeichen im String */

    while(*str)
    {
      st_ptr = move(st_ptr,*str);

      /* Ausgabe einer End-Zustands-Meldung */

      if(st_ptr && st_ptr->out_str)
      {
        msg_flag = TRUE;
        if(pflag == TRUE)
          printf("--> %s\n",st_ptr->out_str);
        else
          break;
      }
      str++;
    }
    return msg_flag;
  }
  else  /* Erkenne nur exakte Zeilen */
  {
    while(*str)
    {
      st_ptr = go(st_ptr,*str++);
      if(!st_ptr || st_ptr == &FAIL_STATE)
	return FALSE;   /* Zeile passte nicht  */
    }
    return TRUE;        /* Zeile passte exakt  */
  }
}

/* GO() - Bearbeite "litchar" und liefere einen Zeiger auf den
 *        korrespondierenden go-Uebergangs-Zustand der FSA. Falls
 *        das Zeichen sich nicht im go-Uebergangs-Zustand der FSA
 *        befindet, dann liefere einen Zeiger auf FAIL_STATE.
 */

FSA *go(st_ptr,litchar)
FSA *st_ptr;
register char litchar;
{
  register TRANSITION *current;

  /*  Bei Zustand 0 greife auf die separate Datenstruktur fuer Zustand
   *  Null der FSA zu. Beachten Sie, dass keine failure-Zustaende fuer
   *  irgendeine Eingabe bei FSA-Zustand Null gibt.
   */

  if(!st_ptr)
    return MZ[litchar];
  else
  {
    /*  Zeige auf den Kopf der verbundenen Liste der mit dem
     *  Zustand korrespondierenden go-Uebergaenge.
     */

    current = st_ptr->go_ls;

    /*  Durchschreit die Liste und suche nach einer Uebereinstimmung
     *  mit dem Eingabezeichen.
     */

    while(current)
    {
      if(current->lchar == litchar)
        break;
      current = current->next_el;
    }

    /*  Der Wert Null fuer current kennzeichnet, dass das Ende der Liste
     *  erreicht wurde, ohne eine Uebereinstimmung mit dem Eingabe-
     *  Zeichen gefunden zu haben.
     */

    return current ? current->nextst_ptr : &FAIL_STATE;
  }
}

/* MOVE() - Bearbeite "litchar" und liefere einen Zeiger auf den
 *          korrespondierenden move-Uebergangszustand der FSA.
 */

FSA *move(st_ptr,litchar)
FSA *st_ptr;
register char litchar;
{
  register TRANSITION *current;

  /* Bei Zustand Null greife auf die getrennte Datenstruktur fuer
   * Zustand Null der FSA zu.
   */

  if(!st_ptr)
    return MZ[litchar];
  else
  {
    /*  Zeige auf den Kopf der verbundenen Liste der mit dem Zustand
     *  verbundenen move-Uebergaenge.
     */

    current = st_ptr->mv_ls;

    /*  Durchschreite die Liste und suche nach einer Uebereinstimmung
     *  mit dem Eingabezeichen.
     */

    while(current)
    {
      if(current->lchar == litchar)
        break;
      current = current->next_el;
    }

    /*  Der Wert Null fuer current kennzeichnet, dass das Ende der Liste
     *  erreicht wurde, ohne dass eine Uebereinstimmung mit dem Eingabe-
     *  Zeichen gefunden werden konnte. Der zurueckgegebene Zeiger zeigt
     *  dann auf Zustand Null.
     */

    return current ? current->nextst_ptr : NULL;
  }
}

/* BD_GO() - Errichte die go-Uebergaenge fuer jeden Zustand aus den
 *           Kommandozeilen-Argumenten.
 */ 

void bd_go(str)
char *str;
{
  register char litchar;
  char *nl,
       buffer[MAX_LINE],        /* Eingabe-String-Puffer */
       *stoupper(),
       *fgets(),
       *index();
  FILE *str_fd,
       *fopen();
  void error(),
       enter();

  /*  Initialisiere das FSA-go-Uebergangs-Array fuer Status Null so,
   *  dass jeder Aufruf von go() mit state=0 zunaechst einen Zeiger
   *  auf FAIL_STATE liefert.
   */

  for(litchar = 1; litchar & 127; litchar++)
    MZ[litchar] = &FAIL_STATE;

  /*  Falls die -f-Option selektiert wurde, hole die durch Zeilenwechsel
   *  getrennten Strings aus dem File str und gibt sie Zeile fuer Zeile
   *  in die FSA. Anderenfalls bringe den String str in die FSA.
   */

  if(fflag == TRUE)
  {
    if(!(str_fd = fopen(str,"r")))
      error(STR_ERR,str);

    while(fgets(buffer,MAX_LINE,str_fd))
    {
      if(nl = index(buffer,'\n'))       /* Entferne das Zeilenende */
        *nl = '\0';
      if(yflag == TRUE)
        stoupper(buffer);
      enter(buffer);
    }
    fclose(str_fd);
  }
  else
  {
    if(yflag == TRUE)
      stoupper(str);
    enter(str);
  }

  /*  Fuer jedes Eingabezeichen, das nicht auf einen definierten
   *  go-Uebergang von FSA-Status Null passt, setze das korrespondierende
   *  Element im go-Uebergangs-Array des Status Null, um einen
   *  go-Uebergang nach Status Null zu kennzeichnen.
   */

  for(litchar = 1; litchar & 127; litchar++)
    if(MZ[litchar] == &FAIL_STATE)
      MZ[litchar] = NULL;
}

/* ENTER() - Bringe eine String in die FSA, indem jedes Zeichen einzeln
 *           durch die momentan teilweise aufgebaute FSA geschleust
 *           wird. Wenn ein Fehler auftaucht, fuege den Rest des Strings
 *           als einen neuen Zustand pro Zeichen an die FSA an
 *           (Beachten Sie, dass '\0' niemals ein gueltiges Zeichen sein
 *           kann - C verwendet es als String-Begrenzer).
 */

void enter(str)
char *str;
{
  FSA *s,
      *go(),
      *create();
  TRANSITION *current,
             *insert();
  char *strsave();
  register char *temp;
  register FSA *st_ptr = NULL;  /* Starte bei FSA-Zustand Null */
  register FSA *nextst_ptr;
  void error();

  /*  Schleuse jedes Zeichen einzeln durch die teilweise aufgebaute
   *  FSA bis ein Fehler auftaucht.
   */  

  temp = str;
  while((s = go(st_ptr,*temp)) != &FAIL_STATE)
  {
    temp++;
    st_ptr = s;
  }

  /* Bearbeite den Rest des Strings */

  while(*temp)
  {
    /*  Falls ein neuer Zustand erforderlich ist, erzeuge einen neuen
     *  Zustand und fuege das Uebergangszeichen sowie den go-Uebergang in
     *  den aktuellen Zustand (Beachten Sie die spezielle Behandlung
     *  des FSA-Zustandes Null).
     */

    if(!st_ptr)
      nextst_ptr = MZ[*temp++] = create();
    else if(!(current = st_ptr->go_ls))
    {
      nextst_ptr = create();
      st_ptr->go_ls = insert(nextst_ptr,*temp++);
    }
    else
    {
      /*  ... oder es war das Zeichen, fuer das die FSA einen "failure"
       *  zurueckgegeben hatte. Suche das Ende der aktuellen Statusliste
       *  der go-Uebergaenge, erzeuge einen neuen Status und haenge ihn
       *  an die go-Liste der aktuellen Zustaende an.
       */

      while(current->next_el)
        current = current->next_el;
      nextst_ptr = create();
      current->next_el = insert(nextst_ptr,*temp++);
    }
    st_ptr = nextst_ptr;
  }

  /* Erzeuge den String der output-Message fuer den Endzustand */

  st_ptr->out_str = strsave(str);
}

/* INSERT() - Erzeuge einen neuen go-Uebergang und liefere einen Zeiger
 *            auf ihn.
 */

TRANSITION *insert(st_ptr,litchar)
FSA *st_ptr;
char litchar;
{
  TRANSITION *current;
  char *malloc();
  void error();
    
  if(!(current = (TRANSITION *)malloc(sizeof(TRANSITION))))
    error(MEM_ERR,NULL);
  current->lchar = litchar;
  current->nextst_ptr = st_ptr;
  current->next_el = NULL;
  return current;
}

/* CREATE() - Erzeuge einen FSA-Zustand und liefere einen Zeiger darauf. */

FSA *create()
{
  FSA *st_ptr;
  char *malloc();
  void error();

  if(!(st_ptr = (FSA *)malloc(sizeof(FSA))))
    error(MEM_ERR,NULL);
  st_ptr->go_ls = st_ptr->mv_ls = NULL;
  st_ptr->fail_state = NULL;
  st_ptr->out_str = NULL;
  return st_ptr;
}

/* BD_MOVE() - Erzeuge die failure- und move-Uebergaenge fuer jeden
 *             Zustand aus den go-Uebergaengen.
 */

void bd_move()
{
  register char litchar;
  register FSA *r,      /* Temporaerer FSA-Status-Zeiger */
               *s,
               *t;
  FSA *go(),
      *move();
  TRANSITION *current,
             *insert();
  QUEUE *first,         /* Zeiger auf den Kopf der Schlange */
	*last;          /* Zeiger auf das Ende der Schlange */
  void add_queue(),
       delete_queue();

  last = first = NULL;  /* Initialisiere die Schlange der FSA-Zustaende */

  /*  Addiere fuer jedes Eingabezeichen mit einem go-Uebergang aus
   *  FSA-Zustand Null heraus einen Zeiger auf den go-Zustand der
   *  Schlange. Beachten Sie, dass dies ebenfalls als move-Uebergangs-Liste
   *  fuer Status Null dient.
   */

  for(litchar = 1; litchar & 127; litchar++)
    if(s = go(NULL,litchar))
      add_queue(&first,&last,s);

  /* Da immer noch Status-Zeiger in der Schlange sind, fuehre folgendes
   * durch:
   */

  while(first)
  {
    /* Entferne den Zeiger auf Status r vom Kopf der Schlange */

    r = first->st_ptr;
    delete_queue(&first);

    /*  Fuehre folgende Taetigkeiten fuer jede Eingabe nach Status r durch,
     *  die einen go-Uebergang auf Status s besitzt:
     */

    for(litchar = 1; litchar & 127; litchar++)
    {
      if((s = go(r,litchar)) != &FAIL_STATE)
      {
	/* Falls in der Eingabe litchar ein definierter go-Uebergang
	 * fuer Status r existiert, fuege einen Zeiger auf Status s ans
	 * Ende der Schlange ein.
         */

        add_queue(&first,&last,s);

	/* Berechne den failure-Uebergang von Zustand s unter Verwendung
	 * des folgenden Algorithmus:
         */

        t = r->fail_state;
        while(go(t,litchar) == &FAIL_STATE)
          t = t->fail_state;
        s->fail_state = go(t,litchar);
      }
      else
      {
	/*  ... setze anderenfalls den Zeiger auf Status s auf einen
	 *  Zeiger auf den bereits berechneten move-Uebergang des
	 *  failure-Status von Zustand r fuer die Eingabe litchar.
         */

        s = move(r->fail_state,litchar);
      }

      /*  Fuege Status s als move-Uebergang fuer Status r an die Eingabe
       *  litchar nur dann an, wenn es nicht Status Null ist.
       */

      if(s)
	if(!r->mv_ls)
          current = r->mv_ls = insert(s,litchar);
	else
          current = current->next_el = insert(s,litchar);
    }
  }
}

/* ADD_QUEUE() - Fuege einen Eintrag an das Ende einer Schlange */

void add_queue(head_ptr,tail_ptr,st_ptr)
QUEUE **head_ptr,
      **tail_ptr;
FSA *st_ptr;
{
  QUEUE *pq;
  char *malloc();
  void error();

  /* Reserviere den notwendigen Speicher und setze die Variablen. */

  if(!(pq = (QUEUE *)malloc(sizeof(QUEUE))))
    error(MEM_ERR,NULL);

  pq->st_ptr = st_ptr;
  pq->next_el = NULL;

  if(!*head_ptr)        /* Erster Eintrag der Schlange? */
    *tail_ptr = *head_ptr = pq;
  else                  /* Nein ein anderer ...         */
    *tail_ptr = (*tail_ptr)->next_el = pq;
}

/* DELETE_QUEUE() - Loesche einen Eintrag vom Kopf der Schlange */

void delete_queue(head_ptr)
QUEUE **head_ptr;
{
  *head_ptr = (*head_ptr)->next_el;
}

/* STRSAVE() - Speichere einen String irgendwo in den Speicher */

char *strsave(str)
char *str;
{
  int strlen();
  char *p,
       *malloc();
  void error();

  if(p = malloc(strlen(str) + 1))
    strcpy(p,str);
  else
    error(MEM_ERR,NULL);
  return p;
}

/* STOUPPER() - Wandle den gesamten String, auf den str zeigt,
 *              in Grossbuchstaben.
 */

char *stoupper(str)
register char *str;
{
  register char *temp;

  temp = str;
  while(*temp)
  {
    *temp = toupper(*temp);
    ++temp;
  }
  return str;
}

/* ERROR() - Fehler-Meldung. Liefert einen Exit-Status von 2 an
 *           den aufrufenden Prozess
 */

void error(n,str)
int n;
char *str;
{
  switch(n)
  {
    case CMD_ERR:
      fprintf(stderr,"illegal command line");
      break;
    case OPT_ERR:
      fprintf(stderr,"illegal option");
      break;
    case INP_ERR:
      fprintf(stderr,"cannot open input file %s",str);
      break;
    case STR_ERR:
      fprintf(stderr,"cannot open string file %s",str);
      break;
    case MEM_ERR:
      fprintf(stderr,"out of memory");
      break;
    default:
      break;
  }
  fprintf(stderr,"\nusage: fgrep [-vclnhyefxps]");
  fprintf(stderr," [strings] <files>\n");
  exit(2);
}
