/*
** utilities.C - utility functions
**
** utilities.C 1.55   Delta'd: 14:56:16 2/28/92   Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/


#include <ctype.h>
#ifndef _IBMR2
#include <libc.h>
#endif
#include <osfcn.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#ifdef _IBMR2
#include <sys/access.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#ifndef COHERENT
#include <sys/errno.h>
#else
#include <errno.h>
#endif
#include <termio.h>
#include <unistd.h>

#include "classes.h"
#include "dired.h"
#include "display.h"
#include "keys.h"

#ifdef COHERENT
#define SIG_PF char *
#endif

/*
** expand_tilde - expects a string of the form "~ ...", which must
**                have been allocated via new().  Returns a new string
**                with the user's home directory in place of the `~'.
**                The passed string is deleted and a new()'d string is
**                returned.  The user's home directory is always appended
**                in the form: "/usr/staff/mjlx"; a slash is not added to
**                the end of the home directory string.  If we cannot get
**                the user's home directory, we simply return the passed
**                string.
*/
char *expand_tilde(char *str)
{
    char *home = getenv("HOME");

    if (home == NULL) {
        struct passwd *user = getpwuid(getuid());
        if (user == NULL) return str;
        home = user->pw_dir;
    }

    char *newstr = new char[strlen(str) + strlen(home)];
    (void)strcpy(newstr, home);
    (void)strcat(newstr, str+1);
    delete str;
    return newstr;
} 

/*
** fgetline(FILE *f) - returns a pointer to the start of a line read
**	     	       from fp, or the null pointer if we hit eof or get
**                     an error from fgets(). Exits if new() fails.
**                     Strips the newline from the line.
**		       Caller should free memory if desired.
*/

static const int FGETLINE_BUFSIZE = 80; // chunksize for calls to new()

char *fgetline(FILE *fp)
{
    char *buffer = new char[FGETLINE_BUFSIZE];

    char *result= fgets(buffer, FGETLINE_BUFSIZE, fp);
    if (result == 0) return 0; // either error or at eof

    if (buffer[strlen(buffer)-1] != '\n' && !feof(fp)) {
        //
        // longer line than buffer can hold
        //
        char *restofline = fgetline(fp);
        if (restofline == 0) return 0; // eof or error

        char *longline = new char[strlen(buffer)+strlen(restofline)+1];
        (void)strcat(strcpy(longline, buffer), restofline);

        delete restofline;
        delete buffer;

        if (longline[strlen(longline)-1] == '\n')
            longline[strlen(longline)-1] = '\0';

        return longline;
    }
    else {
        if (buffer[strlen(buffer)-1] == '\n')
            buffer[strlen(buffer)-1] = '\0';
        return buffer;
    }
}

/*
** This routine tries to determine the full pathname of the current
** directory. The pointer points to volatile storage.
*/

char *get_current_directory(void)
{
    int size = 50;
    char *dir = getcwd(0, size);
    while (dir == 0)
        if (errno == ERANGE) {
            size *= 2;
            dir = getcwd(0, size);
            continue;
        }
        else
            //
            // We must not be able to read the current
            // directory.  That is, we probably got an EACCES.
            //
            return 0;
    return dir;
}

/*
** display_string - prints a string to the given the display, guaranteeing not
**                  to print more than columns() characters.  If the string
**                  exceeds the width of the window, a `!' is placed in
**                  the final column.  Can be called with or without the
**                  length of the string to be printed.  In most places in
**                  the code we know the exact length of the strings we
**                  wish to print.  Note that `len' has a default value
**                  of zero defined by the declaration in "dired.h".
**                  We never call this when trying to write to the last
**                  row on the screen.  That is the dominion of message().
*/

void display_string(const char *str, size_t len)
{
    size_t string_length;
    
    len == 0 ? string_length = strlen(str) : (string_length = len);
    
    if (string_length < columns()) {
        (void)printf("%s", str);
        cursor_wrap();
    }
    else if (string_length > columns()) {
        (void)printf("%*.*s%c", columns()-1, columns()-1, str, '!');
        if (!AM || XN) cursor_wrap();
    }
    else {
        (void)printf("%s", str);
        if (!AM || XN) cursor_wrap();
    }
}

/*
** Get a long listing of the given directory.  Returns 0 if we got other
** than a "memory exhausted" error.  Exits via _new_handler if new() fails
** in fgetline();  Dirname is possible modified here because we may choose
** to strip off the head.  Returns 0 for other errors.
*/
DirList *get_directory_listing(char *dirname)
{
    const char *cmd = ls_cmd[the_sort_order()];
    char *popen_cmd = new char[strlen(cmd) + strlen(dirname) + 1];

    message("Reading directory ... ");

    (void)strcpy(popen_cmd, cmd);
    (void)strcat(popen_cmd, dirname);

    FILE *fp = popen(popen_cmd, "r");
    if (fp == 0) return 0;
    
    DirList *directory = new DirList(dirname);

    //
    // discard lines of the form:
    //
    //      total 1116
    //

    char *line = fgetline(fp);  // read and discard the `total' line
    if (fp == 0) return 0;
    
    while((line = fgetline(fp)) != 0) {
        DirLine *entry = new DirLine(line);
        directory->add(entry);
    }

    message("Reading directory ... done");

    if (feof(fp) && !ferror(fp)) {
        (void)pclose(fp);
        return directory;
    }
    else
        return 0;
}

/*
** error - Prints error message so it can be read.  This is the error
**         function we call once we've initialized the display.
*/

void error(const char *format, ...)
{
    va_list ap;
    va_start(ap, format);
    clear_display();
    move_cursor(rows()-1, 0);
    (void) vfprintf(stdout, format, ap);
    cursor_wrap();
    va_end(ap);
    synch_display();
    term_display();
    exit(EXIT_FAILURE);
}

/*
** update_modeline - this routine concatenates the two strings
**                   into the modeline.  The modeline
**                   is displayed in standout mode if possible.
**                   We never put more than columns() characters into
**                   the modeline.  The modeline is the penultimate
**                   line on the terminal screen.  It does not
**                   synch the display.
*/
void update_modeline(const char *head, const char *tail)
{
    move_to_modeline();
    enter_standout_mode();

    int len = (int)strlen(head);

    if (len < columns())
        (void)printf("%s", head);
    else {
        (void)printf("%*.*s", columns(), columns(), head);
        return;
    }
    
    //
    // write exactly columns() characters to modeline
    //
    for (int i = len; i < columns()-1 && *tail != '\0'; i++, tail++)
        putchar(*tail);

    if (i < columns()-1) {
        putchar(' ');
        for (i = columns()-i-1; i > 0; i--) putchar('-');
    }
    else if (*tail != '\0')
        //
        // The string was overly long.  Put a '!' in the last space
        // on the modeline to signify truncation.
        //
        putchar('!');
    else
        //
        // Here len == columns()-1 && *tail == '\0'.
        //
        putchar(' ');

    end_standout_mode();
}

/*
** is_directory - returns non-zero if a directory, otherwise 0.
**                Also returns zero on error.
*/

int is_directory(const char *dir)
{
    struct stat stbuf;

    if (stat(dir, &stbuf) < 0) return 0;

    return S_ISDIR(stbuf.st_mode);
}

/*
** is_regular_file - returns non-zero if a regular file, otherwise 0.
**                   Also returns zero on error.
*/

int is_regular_file(const char *file)
{
    struct stat stbuf;
    
    if (stat(file, &stbuf) < 0) return 0;

    return S_ISREG(stbuf.st_mode);
}

/*
** read_and_exec_perm - returns non-zero if we have read and execute
**                      permission on the directory, otherwise 0.
**                      Returns 0 on error.
*/

int read_and_exec_perm(const char *dir)
{
    return access(dir, R_OK | X_OK) == -1 ? 0 : 1;
}

/*
** Find the column position of the first character of the filename
** in the current line of the given DirList.
**
** The straight-forward way to do this is to walk the string from it's
** tail to it's head until we hit some whitespace.  This presumes
** that filenames don't contain whitespace.  The one special case
** to worry about is if the file is a symbolic link.  In that case we'll
** have a filename field entry of the form
**
**       Xm -> /usr/lpp/include/Xm
*/

int goal_column(DirList *l)
{
    DirLine *line = l->currLine();

    const char *tmp;

#ifndef NO_SYMLINKS
    if ((tmp = strstr(line->line(), " -> ")) != 0)
        //
        // we have a symbolic link
        //
        --tmp;
    else
#endif
        tmp = line->line() + line->length() - 1;

    while(!isspace(*tmp)) --tmp;

    return tmp - line->line() + 1;
}

/*
** get_file_name - returns the filename of the current line of the DirList.
**
**                 We always allocate the filename in `new' space which
**                 should be `delete'd by the caller.
**
**                 If we have a symbolic link, we return the link not
**                 the file pointed to.
*/
char *get_file_name(DirList *dl)
{
    const char *tmp = &(dl->currLine()->line())[goal_column(dl)];

    //
    // This `new' allocates more space than we need in the case
    // we have a symbolic link, but if we're that low on space,
    // we'll be crashing soon enough anyway :-).
    //
    char *file = new char[strlen(tmp)+1];
    (void)strcpy(file, tmp);

    //
    // do we have a symbolic link?
    //
    char *result = strstr(file, " -> ");
    if (result) *result = '\0';

    return file;
}

/*
** redisplay - this routine redisplays the DirList at the top
**             of our stack.  It assumes that the physical screen 
**             has become corrupted, clears it and then repaints it.
*/

void redisplay()
{
    DirList *dl = dir_stack->top();

    clear_display();

    DirLine *ln = dl->firstLine();
    for (int i = 0; i < rows()-2 && ln != 0; i++, ln = ln->next())
        display_string(ln->line(), ln->length());

    update_modeline(modeline_prefix, dl->name());

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());

    synch_display();
}

/*
** exec_command - execute the passed command using system(3).
**                If prompt == 1, which is the default, we prompt for
**                a key before returning.
*/

void exec_command(char *cmd, int prompt)
{
    unsetraw();
    unset_signals();
    system(cmd);
    set_signals();
    setraw();

    if (prompt) {
        enter_standout_mode();
        message("Press Any Key to Continue");
        end_standout_mode();
        getchar();
    }
}

#ifdef SIGWINCH
/*
** adjust_window - called to adjust our window after getting a SIGWINCH
*/

static void adjust_window()
{
#ifdef TIOCGWINSZ
    struct winsize w;
    if (ioctl(1, TIOCGWINSZ, (char *)&w) == 0 && w.ws_row > 0) LI = w.ws_row;
    if (ioctl(1, TIOCGWINSZ, (char *)&w) == 0 && w.ws_col > 0) CO = w.ws_col;
    //
    // is current line still on the screen?
    //
    if (dir_stack->top()->savedYPos() >= rows()-2) {
        dir_stack->top()->setCurrLine(dir_stack->top()->firstLine());
        dir_stack->top()->saveYXPos(0, goal_column(dir_stack->top()));
    }
    //
    // need to adjust lastLine()
    //
    DirLine *ln = dir_stack->top()->firstLine();
    for (int i = 0; i < rows()-2 && ln; i++, ln = ln->next()) ;
    ln ? dir_stack->top()->setLast(ln->prev()) :
         dir_stack->top()->setLast(dir_stack->top()->tail());
#endif
}
#endif

/*
** prompt - displays `msg' prompt and then collects the response.
**          The keys of the response are echoed as they're collected.  
**          The response should be delete'd when no longer needed.
**          A response of 0 indicates that the command was aborted.
**          A response can contain any graphical character.
**          C-G will abort out of a prompt.  Keyboard-generated signals
**          are ignored while in a prompt. Backspace works as expected.
**          Carriage return indicates the end of response.
**          Non-graphical characters are ignored.
*/
char *prompt(const char *msg)
{
    (void)signal(SIGINT,  SIG_IGN);
    (void)signal(SIGQUIT, SIG_IGN);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, SIG_IGN);
#endif

    size_t len = strlen(msg);

    message(msg);

    //
    // We never echo into the last position in the message window.
    // This means that we can allocate exactly enough char's to just
    // fill the window, reserving the final position for the `null'.
    //
    long space_available = columns()-len-1;
    int pos = 0;

    char *response = new char[space_available+1];

    char key;
    for (;;) {

#ifdef SIGWINCH
        if (win_size_changed) {
            win_size_changed = 0;
            adjust_window();
            redisplay();
            message(msg);
            for (int i = 0; i < pos; i++) putchar(response[i]);
            synch_display();
        }
#endif
        key = getchar();
        if (isprint(key)) {
            //
            // echo character to message window and wait for another
            //
            response[pos++] = key;
            putchar(key);
            synch_display();
            
            if (pos == space_available) {
                //
                // need to allocate more room for the response
                // note that strlen(response) == pos
                //
                char *long_response = new char[pos+columns()/2+1];
                response[pos] = 0;  // stringify response
                (void)strcpy(long_response, response);

                delete response;
                response = long_response;

                //
                // Shift prompt in message window so we
                // always have the end in view where we are
                // adding characters as they're typed.
                //
                move_to_message_line();
                display_string(&response[pos-columns()/2+1], columns()/2-1);
                synch_display();
                space_available += columns()/2;
            }
        }
        else
            switch (key) {
              case KEY_CR:
                //
                // we have the complete response
                //
                response[pos] = 0;
                clear_message_line();
                synch_display();
                set_signals();
                return response;
              case KEY_ABORT  :
                //                       
                // abort the command and reset cursor to previous position
                //
                message("Aborted");
                move_cursor(dir_stack->top()->savedYPos(),
                            dir_stack->top()->savedXPos());

                delete response;
                set_signals();
                return 0;
              case KEY_BKSP:
                //
                // back up one character
                //
                if (pos == 0)
                    break;
                else {
                    backspace();
                    DC ? delete_char_at_cursor() : clear_to_end_of_line();
                    --pos;
                    synch_display();
                }
                break;
              default:
                //
                // ignore other characters
                //
                break;
            }
    }
}

/*
** lines_displayed - returns the number of lines in the DirList
**                   currently displayed on the screen.
*/

int lines_displayed(DirList *dl)
{
    DirLine *ln = dl->firstLine();
    for (int i = 1; ln != dl->lastLine(); i++, ln = ln->next()) ;
    return i;
}

/*
** message - prints a message on the last line of the screen.
**           It is up to the calling process to put the cursor
**           back where it belongs.  Synchs the display.
*/

static char message_window_dirty = 0;

void message(const char *msg)
{
    clear_message_line();
    int len = int(strlen(msg));
    if (len < columns())
        (void)printf("%s", msg);
    else
        (void)printf("%*.*s", columns()-1, columns()-1, msg);
    synch_display();
    message_window_dirty = 1;
}

/*
** get_key - reads a key using getch() and then clears the message window,
**           if it needs to be cleared. Used only by read_commands in the
**           main switch statement so that message() doesn't need to sleep()
**           and clear() on the messages that get written.  This way, the
**           message window is cleared after each keypress within the main
**           loop, when necessary.  We also check for and deal with window
**           size changes here.
*/

int get_key(DirList *dl)
{

#ifdef SIGWINCH
    if (win_size_changed) {
        win_size_changed = 0;
        adjust_window();
        redisplay();
    }
#endif

    int key = getchar();
    if (message_window_dirty) {
        clear_message_line();
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        message_window_dirty = 0;
    }
    return key;
}
    
/*
** cleanup - cleanup and exit after a SIGHUP, SIGTERM, SIGQUIT or SIGINT
*/

void cleanup(int)
{
    term_display();
    exit(0);
}

/*
** initial_listing - prints the initial listing screen.  Called  by dired()
**                   and read_commands() when rereading the current directory.
**                   Adjusts firstLine(), lastLine() and currLine().
*/

void initial_listing(DirList *dl)
{
    clear_listing();

    DirLine *ln = dl->head();
    dl->setFirst(ln);
    dl->setCurrLine(ln);
    for (int i = 0; i < rows()-2 && ln; ln = ln->next(), i++)
        display_string(ln->line(), ln->length());

    ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
}

#ifdef SIGWINCH

int win_size_changed;

/*
** winch - set flag indicating window size changed.
*/

void winch(int)
{
    (void)signal(SIGWINCH, (SIG_PF)winch);
    win_size_changed = 1;
}

#endif    

/*
** set_signals - set up our signal handlers
*/

void set_signals()
{
    (void)signal(SIGHUP,  (SIG_PF)cleanup);
    (void)signal(SIGINT,  (SIG_PF)cleanup);
    (void)signal(SIGQUIT, (SIG_PF)cleanup);
    (void)signal(SIGTERM, (SIG_PF)cleanup);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, (SIG_PF)termstop);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, (SIG_PF)winch);
#endif
}

/*
** ignore_signals - set signals back to defaults
*/

void unset_signals()
{
    (void)signal(SIGHUP,  SIG_DFL);
    (void)signal(SIGINT,  SIG_DFL);
    (void)signal(SIGQUIT, SIG_DFL);
    (void)signal(SIGTERM, SIG_DFL);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, SIG_DFL);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, SIG_DFL);
#endif
}

/*
** leftshift_current_line - shifts the current line in DirList left until
**                          its tail is visible.
*/
void leftshift_current_line(DirList *dl)
{
        int inc = dl->currLine()->length()-columns()+1;
        move_cursor(dl->savedYPos(), 0);
        clear_to_end_of_line();
        display_string(&(dl->currLine()->line())[inc],columns()-1);
        dl->saveYXPos(dl->savedYPos(), max(goal_column(dl)-inc, 0));
        move_cursor(dl->savedYPos(), dl->savedXPos());
}

/*
** rightshift_current_line - rightshifts current line to "natural" position.
*/
void rightshift_current_line(DirList *dl)
{
    move_cursor(dl->savedYPos(), 0);
    clear_to_end_of_line();
    display_string(dl->currLine()->line(), dl->currLine()->length());
    dl->saveYXPos(dl->savedYPos(), goal_column(dl));
    move_cursor(dl->savedYPos(), dl->savedXPos());
}

#ifdef NO_STRSTR
/*
** strstr - from Henry Spencers ANSI C library suite
*/

char *strstr(const char *s, const char *wanted)
{
	register const char *scan;
	register size_t len;
	register char firstc;

	/*
	 * The odd placement of the two tests is so "" is findable.
	 * Also, we inline the first char for speed.
	 * The ++ on scan has been moved down for optimization.
	 */
	firstc = *wanted;
	len = strlen(wanted);
	for (scan = s; *scan != firstc || strncmp(scan, wanted, len) != 0; )
		if (*scan++ == '\0') return NULL;
	return (char *)scan;
}

#endif /* NO_STRSTR */

#ifdef NO_STRCHR
/*
** strchr - find first occurrence of a character in a string.  From Henry
**          Spencer's string(3) implementation.
*/

char *strchr(const char *s, char charwanted)
{
    register const char *scan;
    
    /*
     * The odd placement of the two tests is so NULL is findable.
     */
    for (scan = s; *scan != charwanted;)	/* ++ moved down for opt. */
        if (*scan++ == '\0') return 0;
    return (char *)scan;
}

/*
** strrchr - find last occurrence of a character in a string. From Henry
**           Spencer's string(3) implementation. 
*/

char *strrchr(const char *s, char charwanted)
{
	register const char *scan;
	register const char *place;

	place = NULL;
	for (scan = s; *scan != '\0'; scan++)
		if (*scan == charwanted) place = scan;
	if (charwanted == '\0') return (char *)scan;
	return (char *)place;
}

#endif /* NO_STRCHR */
