/*
** command1.C - some of the commands called from the main command loop.
**
**               Command1.C and command2.C
**               are concatenated during the make into
**               commands.C, which consists of the main
**               command loop and all commands called from
**               within that loop.
**
** command1.C 1.78   Delta'd: 14:08:16 3/9/92   Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/


#include <ctype.h>
#ifndef _IBMR2
#include <libc.h>
#endif
#include <osfcn.h>
#include <sys/types.h>

#include "classes.h"
#include "dired.h"
#include "display.h"
#include "keys.h"

/*
** scroll_up_one_line - Scroll the listing up one line.
**                      We only call this routine when we KNOW that
**                      there is at least one line below the window
**                      which can be scrolled into it and the cursor
**                      is on the last line of the screen.
*/

static void scroll_up_one_line(DirList *dl)
{
    dl->setFirst(dl->firstLine()->next());
    dl->setLast(dl->lastLine()->next());
    dl->setCurrLine(dl->lastLine());

    if (CS) {
        scroll_listing_up_one();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else if (DL || SF) {
        clear_modeline();
        scroll_screen_up_one();
        update_modeline(modeline_prefix, dl->name());
        move_cursor(rows()-3, 0);
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else
        redisplay();

    dl->saveYXPos(rows()-3, goal_column(dl));
    move_cursor(rows()-3, dl->savedXPos());
}

/*
** scroll_down_one_line - Scroll the listing down one line.
**                        We only call this routine when we KNOW
**                        that the head() of the listing is not visible
**                        and the cursor is on the first line in the window.
*/

static void scroll_down_one_line(DirList *dl)
{
    if (lines_displayed(dl) == rows()-2)
        //
        // Must update lastLine.  We previously had a screenfull of lines.
        //
        dl->setLast(dl->lastLine()->prev());

    dl->setFirst(dl->firstLine()->prev());
    dl->setCurrLine(dl->firstLine());

    if (CS) {
        scroll_listing_down_one();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else if (AL || SR) {
        clear_modeline();
        scroll_screen_down_one();
        update_modeline(modeline_prefix, dl->name());
        cursor_home();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else
        redisplay();

    dl->saveYXPos(0, goal_column(dl));
    move_cursor(0, dl->savedXPos());
}

/*
** scroll_up_full_window - scroll listing up one full window,
**                         leaving one line of overlap.  This routine
**                         is only called when we know that the tail()
**                         of the listing is not currently displayed.
*/

static void scroll_up_full_window(DirList *dl)
{
    clear_listing();
    
    DirLine *ln = dl->lastLine();
    dl->setFirst(ln);
    dl->setCurrLine(ln);
    
    for (int i = 0; i < rows()-2 && ln; i++, ln = ln->next())
        display_string(ln->line(), ln->length());
    
    ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
    
    dl->saveYXPos(0, goal_column(dl));

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(0, dl->savedXPos());
    
    synch_display();
}

/*
** scroll_down_full_window - try to scroll listing down one full window,
**                           with one line of overlap.  This routine is
**                           only called when we KNOW that there is at
**                           least one line "above" the current listing.
**                           Only change the current line if it flows off
**                           the "bottom" of the screen.  This routine is
**                           only called when we know that the head() of the
**                           listing isn't currently displayed.
*/

static void scroll_down_full_window(DirList *dl)
{
    clear_listing();
    
    DirLine *ln = dl->firstLine();
    for (int y = 0; y < rows()-3 && ln != dl->head(); y++, ln = ln->prev());
    //
    // y == # of lines preceding head() to add to screen
    //
    
    dl->setFirst(ln);
    for (int j = 0; j < rows()-2 && ln; j++, ln = ln->next())
        display_string(ln->line(), ln->length());
    
    if (ln) dl->setLast(ln->prev());
    
    if (dl->savedYPos()+y >= rows()-2) {
        dl->setCurrLine(dl->lastLine());
        dl->saveYXPos(rows()-3, goal_column(dl));
    }
    else
        dl->saveYXPos(dl->savedYPos()+y, dl->savedXPos());

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());
    
    synch_display();
}

/*
** scroll_up_half_window - scroll listing up half a window.  This routine
**                         is only called when the tail() of the listing
**                         isn't being displayed.  We try to leave the
**                         cursor on the file it was on previously,
**                         otherwise it is left on the first file in
**                         the screen.
*/

static void scroll_up_half_window(DirList *dl, int y)
{

    if (dl->currLine()->length() > columns())
        rightshift_current_line(dl);

    DirLine *ln = dl->firstLine();
    for (int i = 0; i < (rows()-2)/2; i++, ln = ln->next()) ;
    dl->setFirst(ln);
    
    if (CS || DL || SF || DLN) {
        
        if (CS)
            scroll_listing_up_N((rows()-2)/2);
        else {
            clear_modeline();
            scroll_screen_up_N((rows()-2)/2);
            update_modeline(modeline_prefix, dl->name());
        }
        
        move_cursor(rows()-2-((rows()-2)/2), 0);
        
        ln = dl->lastLine()->next();
        for (i = 0; i < (rows()-2)/2 && ln; i++, ln = ln->next())
            display_string(ln->line(), ln->length());

        ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
    }
    else {
        
        clear_listing();
        
        for (i = 0; i < rows()-2 && ln->next(); i++, ln = ln->next())
            display_string(ln->line(), ln->length());
        
        if (i != rows()-2) {
            //
            // We hit last line before outputing all that we could.
            // Must output lastLine() == tail().
            //
            display_string(ln->line(), ln->length());
            dl->setLast(ln);
        }
        else
            dl->setLast(ln->prev());
    }
    
    int pos = y - (rows()-2)/2;
    if (pos < 0) {
        pos = 0;
        dl->setCurrLine(dl->firstLine());
    }
    
    dl->saveYXPos(pos, goal_column(dl));

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(pos, dl->savedXPos());
    
    synch_display();
}

/*
** scroll_down_half_window - try to scroll listing down half a window.
**                           If `freshen' is true, which is the default,
**                           the screen is refreshed.  It is important
**                           to note that we may not be able to scroll
**                           down a complete half window, since we
**                           always anchor the head of the listing to
**                           the first line in the screen.  This routine
**                           is only called when the head() of the
**                           listing isn't being displayed.
*/

static void scroll_down_half_window(DirList *dl, int y, int freshen = 1)
{
    if (dl->firstLine() != dl->head()) {
        //
        // We can scroll down.  Try to leave the cursor on the file
        // it started out on.  Otherwise, leave it on the
        // (rows()-2)/2 line, which was the previous firstLine().
        //
        DirLine *ln = dl->firstLine();
        for (int i = 0; i < (rows()-2)/2 && ln->prev(); i++, ln = ln->prev()) ;
        dl->setFirst(ln);

        if (dl->currLine()->length() > columns())
            rightshift_current_line(dl);

        if (CS || AL || ALN || SR) {

            if (CS)
                scroll_listing_down_N(i);
            else {
                clear_modeline();
                scroll_screen_down_N(i);
                update_modeline(modeline_prefix, dl->name());
                clear_message_line();
            }

            cursor_home();

            for (int j = 0; j < i; j++, ln = ln->next())
                display_string(ln->line(), ln->length());
            
            ln = dl->firstLine();
            for (int i = 0; i < rows()-2 && ln->next(); i++, ln = ln->next()) ;
            dl->setLast(ln);
        }
        else {

            clear_listing();

            for (int i = 0; i < rows()-2 && ln->next(); i++, ln = ln->next())
                display_string(ln->line(), ln->length());
            
            if (i != rows()-2) {
                //
                // We hit last line before outputing all that we could.
                // Must output lastLine() == tail().
                //
                display_string(ln->line(), ln->length());
                dl->setLast(ln);
            }
            else
                dl->setLast(ln->prev());
        }

        int pos = i + y;
        if (pos > rows()-3) {
            pos = rows()-3;
            dl->setCurrLine(dl->lastLine());
        }

        dl->saveYXPos(pos, goal_column(dl));

        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(pos, dl->savedXPos());

        if (freshen) synch_display();
    }
}

/*
** goto_first - position cursor on first line in listing.  This routine
**              is not called if atBegOfList() is true.
*/

static void goto_first(DirList *dl)
{
    if (dl->head() != dl->firstLine())
        initial_listing(dl);
    else {
        if (dl->currLine()->length() > columns())
            rightshift_current_line(dl);
        dl->setCurrLine(dl->head());
    }

    dl->saveYXPos(0, goal_column(dl));

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(0, dl->savedXPos());

    synch_display();
}

/*
** goto_last - position cursor on last file in listing.  This routine is
**             not called if atEndOfList() is true.
*/

static void goto_last(DirList *dl)
{
    if (dl->currLine()->length() > columns())
        rightshift_current_line(dl);

    dl->setCurrLine(dl->tail());

    if (dl->tail() == dl->lastLine()) {
        //
        // only need to reposition the cursor
        //
        dl->saveYXPos(lines_displayed(dl)-1, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    else {
        //
        // redisplay end of listing & update our view
        //
        clear_listing();
        DirLine *ln = dl->tail();
        dl->setLast(ln);
        for (int i = 0; i < rows()-2; i++, ln = ln->prev()) {
            move_cursor(rows()-3-i, 0);
            display_string(ln->line(), ln->length());
        }
        dl->setFirst(ln->next());
        dl->saveYXPos(rows()-3,goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(rows()-3, dl->savedXPos());
    }

    synch_display();
}

/*
** edit_prompted_for_directory - prompt for a directory to edit and edit
**                               it, provided it is a valid directory.
**                               If the first character of the directory is
**                               tilde (`~'), the tilde is expanded to the
**                               user's home directory.
*/
static void edit_prompted_for_directory(DirList *dl)
{
    char *dir = prompt("Directory to edit: ");

    if (dir == 0) {  // command was aborted
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }

    if (*dir == '~') dir = expand_tilde(dir);

    if (!is_directory(dir)) {
        char *msgfmt = "`%s' is not a valid directory name";
        char *msgbuf = new char[strlen(dir) + strlen(msgfmt) + 1];
        (void)sprintf(msgbuf, msgfmt, dir);
        message(msgbuf);
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete dir;
        delete msgbuf;
        return;
    }

    if (!read_and_exec_perm(dir)) {
        message("need read & exec permissions to edit directory");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete dir;
        return;
    }

    dired(dir);
}
    
/*
** edit_current_file - edit the current file in the DirList.
*/

static void edit_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    if (is_directory(file)) {
        
        if (strcmp(file, ".") == 0 ||
            (strcmp(file, "..") == 0 && strcmp(dl->name(), "/") == 0)) {
            //
            // Don't reedit our current directory or if we're
            // at the root, don't reedit ourselves.
            //
            delete file;
            return;
        }
        
        if (!read_and_exec_perm(file)) {
            message("need read & exec permissions to edit directory");
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            delete file;
            return;
        }
        
        dired(file);
        
    }
    else if (is_regular_file(file)) {
        
        char *editor = getenv("EDITOR");
        if (editor == 0) editor = "vi";
 
        char *cmd = new char[strlen(editor)+strlen(file)+4];
        (void)strcpy(cmd, editor);
        (void)strcat(cmd, " '");
        (void)strcat(cmd, file);
        (void)strcat(cmd, "'");

        /*
         * Clear screen and position cursor in case the editor doesn't
         * do this itself.  This is primarily for those people who use
         * non-fullscreen editors.
         */
        clear_display();
        move_to_message_line();
        synch_display();
        
        exec_command(cmd, 0);
        
        delete cmd;

        /*
         * Re-read the listing line for this file
         * and insert into the directory listing.
         * This way, changes in file characteristics are
         * reflected in the soon-to-be updated listing.
         */
        size_t size = strlen(ls_cmd[the_sort_order()]) + strlen(file) + 3;

        cmd = new char[size];
        (void)strcpy(cmd, ls_cmd[the_sort_order()]);
        (void)strcat(cmd, "'");
        (void)strcat(cmd, file);
        (void)strcat(cmd, "'");

        FILE *fp = popen(cmd, "r");
        if (fp == 0)
            error("File %s, line %d: error on popen() in edit_current_file()",
                  __FILE__, __LINE__);
        delete cmd;

        char *new_line = fgetline(fp);
        if (fp == 0)
            error("File %s, line %d: couldn't read from popen() pipe",
                  __FILE__, __LINE__);
        (void)pclose(fp);
            
        dl->currLine()->update(new_line);

        redisplay();
    }
    else {
        message("Can only edit regular files and directories");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
    
    delete file;
}

/*
** page_current_file - attempt to page the current file in the DirList.
**                     If we have a directory, we edit it.
*/

static void page_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    if (is_directory(file)) {
        
        if (strcmp(file, ".") == 0 ||
            (strcmp(file, "..") == 0 && strcmp(dl->name(), "/") == 0)) {
            //
            // Don't reedit our current directory or if we're
            // at the root, don't reedit ourselves.
            //
            delete file;
            return;
        }
        
        if (!read_and_exec_perm(file)) {
            message("need read & exec permissions to edit directory");
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            delete file;
            return;
        }
        
        dired(file);
        
    }
    else if (is_regular_file(file)) {

        char *pager = getenv("PAGER");
        if (pager == 0) pager = "more";
        
        char *cmd = new char[strlen(pager)+strlen(file)+4];
        (void)strcpy(cmd, pager);
        (void)strcat(cmd, " '");
        (void)strcat(cmd, file);
        (void)strcat(cmd, "'");

        clear_display();
        move_to_message_line();
        synch_display();

        exec_command(cmd);
        
        delete cmd;

        redisplay();
    }
    else {
        message("Can only page through regular files and directories.");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
    
    delete file;
}

/*
** insert_line - inserts the current line into the DirList after line y.
**               Only called if y != rows()-3.  That is, this routine
**               is for the case when we don't need to scroll the screen
**               to place the line into its logical place on the screen.
*/

static void insert_line(DirList *dl, int y )
{
    if (CS) {
        insert_listing_line(y+1);
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else if(AL) {
        clear_modeline();
        insert_blank_line(y+1);
        display_string(dl->currLine()->line(), dl->currLine()->length());
        update_modeline(modeline_prefix, dl->name());
    }
    else
        redisplay();
}

/*
** insert_into_dirlist - inserts the listing line for `dest' into the
**                       DirList after line `y' in the current window.
*/

static void insert_into_dirlist(DirList *dl, const char *dest, int y)
{
    size_t size = strlen(ls_cmd[the_sort_order()]) + strlen(dest) + 3;
    char *command = new char[size];
    (void)strcpy(command, ls_cmd[the_sort_order()]);
    (void)strcat(command, "'");

    char *pos = strrchr(dest, '/');
    //
    // pos is non-zero in those cases where `dest' contains
    // `..' trickery.  Say, ../this-dir/new-file.
    //
    pos ? (void)strcat(command, pos+1) : (void)strcat(command, dest);
    (void)strcat(command, "'");
    
    FILE *fp = popen(command, "r");
    if (fp == 0)
        error("File %s, line %d: error on popen()", __FILE__, __LINE__);
    
    char *new_line = fgetline(fp);
    
    if (fp == 0)
        error("File %s, line %d: couldn't read from popen() pipe",
              __FILE__, __LINE__);
    (void)pclose(fp);
    
    delete command;

    int nlines = lines_displayed(dl);

    if (dl->currLine()->length() > columns())
        rightshift_current_line(dl);
    
    dl->setCurrLine(dl->insert(new_line));
    
    if (nlines == rows()-2) {
        if (y == rows()-3) {
            //
            // we must scroll listing up
            //
            dl->setFirst(dl->firstLine()->next());
            dl->setLast(dl->currLine());

            if (CS) {
                scroll_listing_up_one();
                display_string(dl->currLine()->line(), dl->currLine()->length());
            }
            else if (DL || SF) {
                clear_modeline();
                scroll_screen_up_one();
                update_modeline(modeline_prefix, dl->name());
                move_cursor(rows()-3, 0);
                display_string(dl->currLine()->line(), dl->currLine()->length());
            }
            else
                redisplay();

            dl->saveYXPos(y, goal_column(dl));
            move_cursor(y, dl->savedXPos());
        }
        else {
            //
            // just insert line
            //
            dl->setLast(dl->lastLine()->prev());
            insert_line(dl, y);
            dl->saveYXPos(y+1, goal_column(dl));
            move_cursor(y+1, dl->savedXPos());
        }
    }
    else {
        insert_line(dl, y);
        dl->saveYXPos(y+1, goal_column(dl));
        move_cursor(y+1, dl->savedXPos());
        if (nlines == y+1)
            //
            // The new line becomes the new lastLine.
            //
            dl->setLast(dl->currLine());
    }

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
}

/*
** yes_or_no - returns true if a 'y' or 'Y' is typed in response to
**             the msg.  Does not synch the display.
*/

static int yes_or_no(const char *msg)
{
    message(msg);
    synch_display();

    int key = getchar();
    int response = key == 'y' || key == 'Y';
    clear_message_line();
    return response;
}

/*
** copy_file - copy current file to destination.
**             Update window appropriately.
*/

static void copy_file(DirList *dl, const char *src, const char *dest)
{
    char *command = new char[strlen(src)+strlen(dest)+25];
    (void)strcpy(command, "cp '");
    (void)strcat(command, src);
    (void)strcat(command, "' '");
    (void)strcat(command, dest);
    (void)strcat(command, "' >/dev/null 2>&1");

    if (is_regular_file(dest)) {
        //
        // `dest' exists as a regular file
        //
        char *msg = new char[strlen(dest)+21];
        (void)strcpy(msg, "overwrite `");
        (void)strcat(msg, dest);
        (void)strcat(msg, "'? (y|n) ");
        
        if (yes_or_no(msg))
            //
            // Overwrite it.
            //
            if (!system(command))
                message("`copy' was successful");
            else
                message("`copy' failed");
        
        move_cursor(dl->savedYPos(), dl->savedXPos());

        delete msg;
    }
    else if (is_directory(dest)) {
        //
        // `dest' is a directory.
        //
        char *file = new char[strlen(dest)+strlen(src)+2];
        (void)strcpy(file, dest);
        if (dest[strlen(dest)-1] != '/') (void)strcat(file, "/");
        (void)strcat(file, src);
        
        char *msg = new char[strlen(file)+21];
        (void)strcpy(msg, "overwrite `");
        (void)strcat(msg, file);
        (void)strcat(msg, "'? (y|n) ");

        if (is_regular_file(file)) {
            //
            // `dest/src' exists.
            //
            if (yes_or_no(msg))
                //
                // Overwrite it.
                //
                if (!system(command))
                    message("`copy' was successful");
                else
                    message("`copy' failed");
        }
        else {
            //
            // Just do the `cp'.
            //
            if (!system(command))
                message("`copy' was successful");
            else
                message("`copy' failed");
        }

        move_cursor(dl->savedYPos(), dl->savedXPos());

        delete file;
        delete msg;
    }
    else {
        //
        // `dest' doesn't already exist
        //
        if (!system(command)) {
            message("`copy' was successful");
            //
            // Is `dest' in our current directory?
            //
            if (strchr(dest, '/') == NULL)
                //
                // It must be in our current directory.
                //
                insert_into_dirlist(dl, dest, dl->savedYPos());
            else
                //
                // Assume it isn't.  This isn't foolproof, but that
                // is what the `g' command is for :-).
                //
                move_cursor(dl->savedYPos(), dl->savedXPos());
        }
        else {
            message("`copy' failed");
            move_cursor(dl->savedYPos(), dl->savedXPos());
        }
    }

    delete command;
}

/*
** copy_current_file - attempt to copy current file to another
*/

static void copy_current_file(DirList *dl)
{
    char *src = get_file_name(dl);
    
    if (strcmp(src, ".")  == 0 || strcmp(src, "..") == 0) {
        message("`copying' of `.' and `..' is not allowed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        return;
    }
    
    char *dest = prompt("Copy to: ");
    
    if (dest == 0) {  // command was aborted
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete dest;
        delete src;
        return;
    }
    else if (*dest == 0) {
        message("not a valid file name");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        delete dest;
        delete src;
        synch_display();
        return;
    }

    if (*dest == '~') dest = expand_tilde(dest);
    
    copy_file(dl, src, dest);
    
    synch_display();
    
    delete dest;
    delete src;
}

/*
** link_file - attempt to link `file' to `dest'.
*/

static void link_file(DirList *dl, const char *file, const char *dest)
{
    if (!link(file, dest)) {
        message("`link' was successful");
        //
        // Is `dest' in our current directory?
        //
        if (strchr(dest, '/') == NULL)
            //
            // It must be in our current directory.
            //
            insert_into_dirlist(dl, dest, dl->savedYPos());
        else
            //
            // Assume it isn't.  This isn't foolproof, but that
            // is what the `g' command is for :-).
            //
            move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    else {
        message("`link' failed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
}

/*
** link_current_file - attempt to make a link to the current file
*/

static void link_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    char *link = prompt("Link to: ");
    
    if (link == 0) {  // command was aborted
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        
        delete file;
        delete link;
        
        return;
    }
    else if (*link == 0) {
        message("not a valid file name");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete file;
        delete link;
        return;
    }
    
    link_file(dl, file, link);
    
    synch_display();
    
    delete file;
    delete link;
}

#ifndef NO_SYMLINKS
/*
** symlink_file - symbolically link `file' to `dest'
*/

static void symlink_file(DirList *dl, const char *file, const char *dest)
{
    if (!symlink(file, dest)) {
        message("`symlink' was successful");
        //
        // Is `dest' in our current directory?
        //
        if (strchr(dest, '/') == NULL)
            //
            // It must be in our current directory.
            //
            insert_into_dirlist(dl, dest, dl->savedYPos());
        else
            //
            // Assume it isn't.  This isn't foolproof, but that
            // is what the `g' command is for :-).
            //
            move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    else {
        message("`symlink' failed");
        move_cursor(dl->savedYPos(), dl->savedXPos());
    }
}

/*
** symlink_current_file - attempt create a symbolic link to the current file
*/

static void symlink_current_file(DirList *dl)
{
    char *file = get_file_name(dl);
    
    char *link = prompt("Name of symbolic link: ");
    
    if (link == 0) {  // command was aborted
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete file;
        delete link;
        return;
    }
    else if (*link == 0) {
        message("not a valid file name");
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        delete file;
        delete link;
        return;
    }
    
    symlink_file(dl, file, link);
    
    synch_display();
    
    delete file;
    delete link;
}
#endif


/*
** remove_listing_line - delete the current line in the DirList
**                       and update both the screen and data
**                       structures appropriately.  `y' is the position
**                       in the window of the current line.
*/

static void remove_listing_line(DirList *dl, int y)
{
    if (dl->lastLine() != dl->tail()) {
        //
        // Last line in listing isn't in window.
        // We can just scroll up one line.
        //
        dl->deleteLine();
        dl->setLast(dl->lastLine()->next());

        if (CS || DL) {
            if (CS)
                delete_listing_line(y);
            else {
                clear_modeline();
                delete_screen_line(y);
                update_modeline(modeline_prefix, dl->name());
            }
            move_cursor(rows()-3, 0);
            display_string(dl->lastLine()->line(), dl->lastLine()->length());
        }
        else {
            clear_to_end_of_screen(y);
            move_cursor(y, 0);
            DirLine *ln = dl->currLine();
            for (int i = y; i < rows()-2; i++, ln = ln->next())
                display_string(ln->line(), ln->length());
            update_modeline(modeline_prefix, dl->name());
        }

        dl->saveYXPos(y, goal_column(dl));
    }
    else {
        //
        // Last line of listing is visible in window.
        //
        if (dl->atWindowTop() && dl->atWindowBot()) {
            //
            // The last line in the window is also the first line.
            // Since we don't allow deletion of `.' or `..', there
            // must be more viewable lines.  Scroll down half
            // a window to put more into view.
            //
            scroll_down_half_window(dl, y, 0);
            dl->deleteLine();
            DirLine *ln = dl->firstLine();
            for (int pos = 0; ln != dl->tail(); pos++, ln = ln->next()) ;
            dl->saveYXPos(pos, goal_column(dl));
            move_cursor(pos+1, 0);
            clear_to_end_of_line();
            move_cursor(pos, dl->savedXPos());
        }
        else if (dl->atWindowBot()) {
            //
            // we want to delete the last line in the window
            //
            dl->deleteLine();
            move_cursor(y, 0);
            clear_to_end_of_line();
            dl->saveYXPos(y-1, goal_column(dl));
            move_cursor(y-1, dl->savedXPos());
        }
        else {
            //
            // we are in the middle of the listing
            //
            dl->deleteLine();

            if (CS || DL) {
                if (CS)
                    delete_listing_line(y);
                else {
                    clear_modeline();
                    delete_screen_line(y);
                    update_modeline(modeline_prefix, dl->name());
                }
            }
            else {
                clear_to_end_of_screen(y);
                move_cursor(y, 0);
                for (DirLine *ln = dl->currLine(); ln; ln = ln->next())
                    display_string(ln->line(), ln->length());
                update_modeline(modeline_prefix, dl->name());
            }

            dl->saveYXPos(y, goal_column(dl));
         }
    }

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());
}
