/*
** classes.h - the declarations of the classes used in dired
**
** classes.h 1.9   Delta'd: 09:09:57 8/28/91   Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef __CLASSES_H
#define __CLASSES_H

///////////////////////////////////////////////////////////////////////////////
// DirLine - class which contains one line of the long
//           listing of a directory.
///////////////////////////////////////////////////////////////////////////////

class DirLine {
    friend class DirList;
  protected:
    char               *_line;
    int                 _length;
    DirLine            *_next;
    DirLine            *_prev;
  public:
    DirLine(char *);
    ~DirLine();

    const char *line()  const { return _line;   }
    int length()        const { return _length; }
    DirLine *next()     const { return _next;   }
    DirLine *prev()     const { return _prev;   }

    void update(char *);
};

///////////////////////////////////////////////////////////////////////////////
// DirList - class which manages a doubly-linked list of DirLines.
//           It also maintains our current notion of what is and isn't
//           visible in the curses(3) window.
///////////////////////////////////////////////////////////////////////////////

class DirList {
  protected:
    DirLine    *_head;
    DirLine    *_tail;
    char       *_name;        // full pathname of the directory
    int         _nelems;
    short       _saved_x;     // saved x cursor position
    short       _saved_y;     // saved y cursor position
    DirLine    *_firstLine;   // first viewable dirLine in curses window
    DirLine    *_lastLine;    // last  viewable dirLine in curses window
    DirLine    *_currLine;    // line cursor is on in curses window
    DirList    *_next;
    DirList    *_prev;
  public:
    DirList(char *);
    ~DirList();

    DirLine *head()              const { return _head;                   }
    DirLine *tail()              const { return _tail;                   }
    DirLine *firstLine()         const { return _firstLine;              }
    DirLine *lastLine()          const { return _lastLine;               }
    DirLine *currLine()          const { return _currLine;               }
    DirList *next()              const { return _next;                   }
    DirList *prev()              const { return _prev;                   }

    int savedXPos()              const { return _saved_x;                }
    int savedYPos()              const { return _saved_y;                }

    void setFirst(DirLine *e)          { _firstLine = e;                 }
    void setLast (DirLine *e)          { _lastLine  = e;                 }
    void setCurrLine (DirLine *ln)     { _currLine = ln;                 }

    void setNext (DirList *l)          { _next = l;                      }
    void setPrev (DirList *l)          { _prev = l;                      }

    const char *name()           const { return _name;                   }
    int nelems()                 const { return _nelems;                 }
    void saveYXPos(int y, int x)       { _saved_x = (short)x;
                                         _saved_y = (short)y;            }

    int atBegOfList()            const { return _currLine == _head;      }
    int atEndOfList()            const { return _currLine == _tail;      }

    int atWindowTop()            const { return _currLine == _firstLine; }
    int atWindowBot()            const { return _currLine == _lastLine;  }

    DirLine *insert(char *);
    void add(DirLine *);
    void deleteLine();
};

///////////////////////////////////////////////////////////////////////////////
// DirStack - class which manages a stack of DirLists.
///////////////////////////////////////////////////////////////////////////////

class DirStack {
  protected:
    DirList             *_top;
    int                  _nelems;
  public:
    DirStack()                 { _top = 0; _nelems = 0; }

    DirList *top()       const { return _top;           }
    int nelems()         const { return _nelems;        }

    void push(DirList *);
    DirList *pop();
};

#endif /* __CLASSES_H */
