/*
** classes.C - contains definitions of the member functions which
**             aren't defined in the relevant class declarations.
**
** classes.C 1.10   Delta'd: 10:41:32 10/11/91   Mike Lijewski, CNSF
**
** Copyright (c) 1991 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <string.h>
#include "classes.h"

//
// constructor for DirLine
//
DirLine::DirLine(char *line)
{
    _line   = line;
    _length = (int)strlen(line);
    _next   = _prev = 0;
}

//
// destructor for DirLine
//
DirLine::~DirLine()
{
    delete _line;
}

//
// Update the line in DirLine with a new version.  The new
// line should have been been allocated via new().
// 
void DirLine::update(char *new_line)
{
    delete _line;
    _line   = new_line;
    _length = (int)strlen(new_line);
}

//
// constructor for DirList
//
DirList::DirList(char *name)
{
    _head = _tail = 0;
    _next = _prev = 0;
    _nelems = 0;
    _name = name;
}

//
// Inserts a new DirLine, constructed from the given char*, after
// the current line in the DirList.
//
DirLine *DirList::insert(char *line)
{
    DirLine *new_line = new DirLine(line);
    DirLine *ln = currLine();

    if (atEndOfList()) {
        ln->_next = new_line;
        new_line->_prev = ln;
        _tail = new_line;
    }
    else {
        new_line->_next = ln->next();
        new_line->_prev = ln;
        ln->_next->_prev = new_line;
        ln->_next = new_line;
    }
    _nelems++;
    return new_line;
}

//
// Adds the DirLine to the listing maintained by DirList.
//
void DirList::add(DirLine *link)
{
    if (nelems()) {
        _tail->_next = link;
        _tail->_next->_prev = tail();
        _tail = link;
        _nelems++;
    }
    else {
        _head = _tail = link;
        _nelems = 1;
    }
}

//
// Delete the current listing line in the window
// and update our view.  The width of our view
// always decreases by one.  If the calling procedure
// adds more lines to the screen, they'll have to reset
// lastLine() and/or firstLine(), but currLine doesn't need to change.
//
void DirList::deleteLine()
{
    DirLine *line = currLine();

    if (atBegOfList()) {
        //
        // that is, firstLine() == head()
        //
        _head = _firstLine = _currLine = head()->next();
        _head->_prev = 0;
    }
    else if (atWindowTop()) {
        //
        // but firstLine() != head()
        //
        _firstLine = _currLine = line->next();
        line->_next->_prev = line->prev();
        line->_prev->_next = line->next();
    }
    else if (atEndOfList()) {
        //
        // lastLine() == tail()
        //
        _tail = _lastLine = _currLine = line->prev();
        _tail->_next = 0;
    }
    else {
        _currLine = line->next();
        line->_next->_prev = line->prev();
        line->_prev->_next = line->next();
    }

    _nelems--;
    delete line;
}

//
// destructor for DirList
//
DirList::~DirList()
{
    if (nelems()) {
        DirLine *tmp = tail(), *prev = tail()->prev();
        while(tmp) {
            delete tmp;
            if ((tmp = prev) != 0) prev = tmp->prev();
        }
        delete tmp;

        delete _name;
    }
}

//
// push a DirList onto our stack
//
void DirStack::push(DirList *list)
{
    if (nelems()) {
        top()->setNext(list);
        list->setPrev(top());
        _top = list;
        _nelems++;
    }
    else {
        _top    = list;
        _nelems = 1;
    }
}

//
// Pop off the top element of the stack.
//
DirList *DirStack::pop()
{
    DirList *tmp = top();

    if (nelems() > 1) {
        _top = top()->prev();
        top()->setNext(0);
    }
    else
        _top = 0;

    _nelems--;
    return tmp;
}
