#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define	bool	char
#define	TRUE	1
#define	FALSE	0

#define	MAXINDEX(array)	(sizeof(array)/sizeof(array[0]))

/*
 * bools & boolcaps:
 *	lookup translate table for boolean fields.
 */
struct	bools {
    char	*capname;	/* termcap name */
    char	*infoname;	/* terminfo name */
};
struct	bools	boolcaps[] = {
    { "bw", "bw" },	{ "am", "am" },		{ "xb", "xsb" },
    { "xs", "xhp" },	{ "xn", "xenl" },	{ "eo", "eo" },
    { "gn", "gn" },	{ "hc", "hc" },		{ "km", "km" },
    { "hs", "hs" },	{ "in", "in" },		{ "da", "da" },
    { "db", "db" },	{ "mi", "mir" },	{ "ms", "msgr" },
    { "os", "os" },	{ "es", "eslok" },	{ "xt", "xt" },
    { "hz", "hz" },	{ "ul", "ul" },		{ "xo", "xon" }
};
#define MAXBOOLS	MAXINDEX(boolcaps)

/*
 * nums & numcaps:
 *	lookup translate table for numeric capabilities.
 */
struct	nums {
    char	*capname;	/* termcap name */
    char	*infoname;	/* terminfo name */
};
struct	nums	numcaps[] = {
    { "co", "cols" },	{ "it", "it" },		{ "li", "lines" },
    { "lm", "lm" },	{ "sg", "xmc" },	{ "pb", "pb" },
    { "vt", "vt" },	{ "ws", "wsl" }
};
#define	MAXNUMS		MAXINDEX(numcaps)

/*
 * strs & strcaps:
 *	lookup translate table for string capabilities.
 */
struct	strs	{
    char	*capname;	/* termcap name */
    char	*infoname;	/* terminfo name */
    char	*dflt;		/* default value */
};
struct	strs	strcaps[] = {
    { "bt",	"cbt",		((char *)0) },
    { "bl",	"bel",		"\007" },
    { "cr",	"cr",		"\r" },
    { "cs",	"csr",		((char *)0) },
    { "ct",	"tbc",		((char *)0) },
    { "cl",	"clear",	((char *)0) },
    { "ce",	"el",		((char *)0) },
    { "cd",	"ed",		((char *)0) },
    { "ch",	"hpa",		((char *)0) },
    { "CC",	"cmdch",	((char *)0) },
    { "cm",	"cup",		((char *)0) },
    { "do",	"cud1",		"\n" },
    { "ho",	"home",		((char *)0) },
    { "vi",	"civis",	((char *)0) },
    { "le",	"cub1",		"\b" },		/* special case - check bc */
    { "CM",	"mrcup",	((char *)0) },
    { "ve",	"cnorm",	((char *)0) },
    { "nd",	"cuf1",		((char *)0) },
    { "ll",	"ll",		((char *)0) },
    { "up",	"cuu1",		((char *)0) },
    { "vs",	"cvvis",	((char *)0) },
    { "dc",	"dch1",		((char *)0) },
    { "dl",	"dl1",		((char *)0) },
    { "ds",	"dsl",		((char *)0) },
    { "hd",	"hd",		((char *)0) },
    { "as",	"smacs",	((char *)0) },
    { "mb",	"blink",	((char *)0) },
    { "md",	"bold",		((char *)0) },
    { "ti",	"smcup",	((char *)0) },
    { "dm",	"smdc",		((char *)0) },
    { "mh",	"dim",		((char *)0) },
    { "im",	"smir",		((char *)0) },
    { "mp",	"prot",		((char *)0) },
    { "mr",	"rev",		((char *)0) },
    { "mk",	"invis",	((char *)0) },
    { "so",	"smso",		((char *)0) },
    { "us",	"smul",		((char *)0) },
    { "ec",	"ech",		((char *)0) },
    { "ae",	"rmacs",	((char *)0) },
    { "me",	"sgr0",		((char *)0) },
    { "te",	"rmcup",	((char *)0) },
    { "ed",	"rmdc",		((char *)0) },
    { "ei",	"rmir",		((char *)0) },
    { "se",	"rmso",		((char *)0) },
    { "ue",	"rmul",		((char *)0) },
    { "vb",	"flash",	((char *)0) },
    { "ff",	"ff",		((char *)0) },
    { "fs",	"fsl",		((char *)0) },
    { "is",	"is1",		((char *)0) },
    { "i1",	"is2",		((char *)0) },
    { "i2",	"is3",		((char *)0) },
    { "if",	"if",		((char *)0) },
    { "ic",	"ich1",		((char *)0) },
    { "al",	"il1",		((char *)0) },
    { "ip",	"ip",		((char *)0) },
    { "kb",	"kbs",		"\b" },
    { "ka",	"ktbc",		((char *)0) },
    { "kC",	"kclr",		((char *)0) },
    { "kt",	"kctab",	((char *)0) },
    { "kD",	"kdch1",	((char *)0) },
    { "kL",	"kdl1",		((char *)0) },
    { "kd",	"kcud1",	"\n" },
    { "kM",	"krmir",	((char *)0) },
    { "kE",	"kel",		((char *)0) },
    { "kS",	"ked",		((char *)0) },
    { "k0",	"kf0",		((char *)0) },
    { "k1",	"kf1",		((char *)0) },
    { "k2",	"kf2",		((char *)0) },
    { "k3",	"kf3",		((char *)0) },
    { "k4",	"kf4",		((char *)0) },
    { "k5",	"kf5",		((char *)0) },
    { "k6",	"kf6",		((char *)0) },
    { "k7",	"kf7",		((char *)0) },
    { "k8",	"kf8",		((char *)0) },
    { "k9",	"kf9",		((char *)0) },
    { "kh",	"khome",	((char *)0) },
    { "kI",	"kich1",	((char *)0) },
    { "kA",	"kil1",		((char *)0) },
    { "kl",	"kcub1",	"\b" },
    { "kH",	"kll",		((char *)0) },
    { "kN",	"knp",		((char *)0) },
    { "kP",	"kpp",		((char *)0) },
    { "kr",	"kcuf1",	((char *)0) },
    { "kF",	"kind",		((char *)0) },
    { "kR",	"kri",		((char *)0) },
    { "kT",	"khts",		((char *)0) },
    { "ku",	"kcuu1",	((char *)0) },
    { "ke",	"rmkx",		((char *)0) },
    { "ks",	"smkx",		((char *)0) },
    { "l0",	"lf0",		((char *)0) },
    { "l1",	"lf1",		((char *)0) },
    { "l2",	"lf2",		((char *)0) },
    { "l3",	"lf3",		((char *)0) },
    { "l4",	"lf4",		((char *)0) },
    { "l5",	"lf5",		((char *)0) },
    { "l6",	"lf6",		((char *)0) },
    { "l7",	"lf7",		((char *)0) },
    { "l8",	"lf8",		((char *)0) },
    { "l9",	"lf9",		((char *)0) },
    { "mm",	"smm",		((char *)0) },
    { "mo",	"rmm",		((char *)0) },
    { "nw",	"nel",		"\r\n" },
    { "pc",	"pad",		((char *)0) },
    { "DC",	"dch",		((char *)0) },
    { "DL",	"dl",		((char *)0) },
    { "DO",	"cud",		((char *)0) },
    { "IC",	"ich",		((char *)0) },
    { "SF",	"indn",		((char *)0) },
    { "AL",	"il",		((char *)0) },
    { "LE",	"cub",		((char *)0) },
    { "RI",	"cuf",		((char *)0) },
    { "SR",	"rin",		((char *)0) },
    { "UP",	"cuu",		((char *)0) },
    { "pk",	"pfkey",	((char *)0) },
    { "pl",	"pfloc",	((char *)0) },
    { "px",	"pfx",		((char *)0) },
    { "ps",	"mc0",		((char *)0) },
    { "pf",	"mc4",		((char *)0) },
    { "po",	"mc5",		((char *)0) },
    { "rp",	"rep",		((char *)0) },
    { "rs",	"rs1",		((char *)0) },
    { "r1",	"rs2",		((char *)0) },
    { "r2",	"rs3",		((char *)0) },
    { "rf",	"rf",		((char *)0) },
    { "rc",	"rc",		((char *)0) },
    { "cv",	"vpa",		((char *)0) },
    { "sc",	"sc",		((char *)0) },
    { "sf",	"ind",		"\n" },
    { "sr",	"ri",		((char *)0) },
    { "sa",	"sgr",		((char *)0) },
    { "st",	"hts",		((char *)0) },
    { "wi",	"wind",		((char *)0) },
    { "ta",	"ht",		((char *)0) },	/* conditional - check pt */
    { "ts",	"tsl",		((char *)0) },
    { "uc",	"uc",		((char *)0) },
    { "hu",	"hu",		((char *)0) },
    { "iP",	"iprog",	((char *)0) },
    { "K1",	"ka1",		((char *)0) },
    { "K2",	"kb2",		((char *)0) },
    { "K3",	"ka3",		((char *)0) },
    { "K4",	"kc1",		((char *)0) },
    { "K5",	"kc3",		((char *)0) },
    { "pO",	"mc5p",		((char *)0) },
    { "tc",	"use",		((char *)0) }
};
#define	MAXSTRS		MAXINDEX(strcaps)

int	tgetname ();	/* get termcap name */
int	tgetflag ();	/* get termcap boolean value */
int	tgetnum ();	/* get termcap numeric value */
int	tgetstr ();	/* get termcap string value */


/*
 * print_name:
 *	print name and aliases of current termcap entry.
 */
print_name ()
{
    char	name[100];

    tgetname (name);
    printf ("%s,\n", name);
}

/*
 * print_bools:
 *	print all boolean fields of current termcap entry.
 */
print_bools ()
{
    int		i, val;
    bool	stuffprinted = FALSE;

    for (i = 0; i < MAXBOOLS; i++)
	if (val = tgetflag (boolcaps[i].capname)) {
	    if (val == 1)
		printf ("\t%s,", boolcaps[i].infoname);
	    else
		printf ("\t%s@,", boolcaps[i].infoname);
	    stuffprinted = TRUE;
	}
    if (stuffprinted)
	(void) putchar ('\n');
}

/*
 * print_nums:
 *	print all numeric fields of current termcap entry.
 */
print_nums ()
{
    int		i, capval;
    bool	stuffprinted = FALSE;

    for (i = 0; i < MAXNUMS; i++)
	if ((capval = tgetnum (numcaps[i].capname)) >= 0) {
	    printf ("\t%s#%d,", numcaps[i].infoname, capval);
	    stuffprinted = TRUE;
	}
	else if (capval == -2) {
	    printf ("\t%s@,", numcaps[i].infoname);
	    stuffprinted = TRUE;
	}
    if (stuffprinted)
	(void) putchar ('\n');
}

/*
 * print_strs:
 *	print all string fields of current termcap entry.
 */
print_strs ()
{
    int		i, count = 0;
    char	capval[100];

    for (i = 0; i < MAXSTRS; i++) {
	tgetstr (strcaps[i].capname, capval);
	if (!capval[0]) {
	    if (strcmp (strcaps[i].capname, "le") == 0) {
		tgetstr ("bc", capval);
	    }
	    else if (strcmp (strcaps[i].capname, "ta") == 0) { 
		if (tgetflag ("pt")) {
		    capval[0] = '\t';
		    capval[1] = '\0';
		}
	    }
	}
	if ((!capval[0]) && (strcaps[i].dflt))
	    (void) strcpy (capval, strcaps[i].dflt);
	if (capval[0]) {
	    if (strcmp (capval, "@") != 0) {
		printf ("\t%s=", strcaps[i].infoname);
		if (strcmp(strcaps[i].infoname,"use") != 0) {
		    put_str (capval);
		    printf (",");
		}
		else
		    printf ("%s,", capval);
	    }
	    else
		printf ("\t%s@,", strcaps[i].infoname);
	    count++;
	    if (!(count %= 3))
		putchar ('\n');
	}
    }
    if (count)
	(void) putchar ('\n');
}

/*
 * put_str:
 *	translate strings to printable format and print them.
 */
put_str (s)
char	*s;
{
    bool	rflag = FALSE;	/* % codes */
    char	*c;
    int		parm;

    if ((isdigit (*s)) || (*s == '.')) {	/* handle padding */
	printf ("$<");
	while ((isdigit (*s)) || (*s == '.')) {
	    (void) putchar (*s);
	    s++;
	}
	if (*s == '*') {
	    (void) putchar (*s);
	    s++;
	}
	(void) putchar ('>');
    }
    for (c = s; *c; c++) {	/* scan for % codes (needs work) */
	if (*c == '%') {
	    c++;
	    switch (*c) {
		case 'r':
		    rflag = TRUE;
		    break;
		default:
		    break;	/* ignore */
	    }
	}
    }
    parm = 0;
    while (*s) {		/* print the string */
	switch (*s) {
	    case '%':
		s++;
		switch (*s) {
		    case '%':
			printf ("%%%%");
			break;
		    case 'i':
			printf ("%%i");
			break;
		    case 'd':
			parm++;
			if ((rflag) && (parm <= 2)) {
			    if (parm == 1)
				printf ("%%p2%%d");
			    else
				printf ("%%p1%%d");
			}
			else
			    printf ("%%p%d%%d", parm);
			break;
		    case '2':
			parm++;
			if ((rflag) && (parm <= 2)) {
			    if (parm == 1)
				printf ("%%p2%%02d");
			    else
				printf ("%%p1%%02d");
			}
			else
			    printf ("%%p%d%%02d", parm);
			break;
		    case '3':
			parm++;
			if ((rflag) && (parm <= 2)) {
			    if (parm == 1)
				printf ("%%p2%%03d");
			    else
				printf ("%%p1%%03d");
			}
			else
			    printf ("%%p%d%%03d", parm);
			break;
		    case '.':
			parm++;
			if ((rflag) && (parm <= 2)) {
			    if (parm == 1)
				printf ("%%p2%%c");
			    else
				printf ("%%p1%%c");
			}
			else
			    printf ("%%p%d%%c", parm);
			break;
		    case '+':
			s++;
			parm++;
			if ((rflag) && (parm <= 2)) {
			    if (parm == 1)
				printf ("%%p2%%'%c'%%+%%c", *s);
			    else
				printf ("%%p1%%'%c'%%+%%c", *s);
			}
			else
			    printf ("%%p%d%%'%c'%%+%%c", parm, *s);
			break;
		    default:
			break;	/* ignore */
		}
		break;
	    case '\200':
		printf ("\\0");
		break;
	    case '\177':
		printf ("^?");
		break;
	    case ',':
		printf ("\\,");
		break;
	    case '\\':
		printf ("\\\\");
		break;
	    case '^':
		printf ("\\^");
		break;
	    case ':':
		printf ("\\:");
		break;
	    case '\033':
		printf ("\\E");
		break;
	    case '\n':
		printf ("\\n");
		break;
	    case '\r':
		printf ("\\r");
		break;
	    case '\t':
		printf ("\\t");
		break;
	    case '\b':
		printf ("\\b");
		break;
	    case '\f':
		printf ("\\f");
		break;
	    case ' ':
		printf ("\\s");
		break;
	    default:
		if (*s < ' ')
		    printf ("^%c", *s + 'A' - 1);
		else
		    (void) putchar (*s);
		break;
	}
	s++;
    }
}
