/*
 * CREF - C cross-reference utility
 *
 * Autor:	Jeff Taylor, The Toolsmith (C) Copyright 1982, 1985
 * Environment:	C; UNIX 4.2 BSD
 * Algorithmen:	Modifizierter Algorithmus T aus "The Art of Computer
 *		Programming", Vol. 1, p. 317, von D.E. Knuth
 *		verwendet in tree_walk()
 * History:	22 November 1982 - Rekursion aus lookup() entfernt
 *		16 December 1982 - Rekursion aus tree_walk() entfernt
 *		26 March    1985 - Portierung auf DECUS C
 *		27 March    1985 - Zeilennummern in zirkulaerer Liste
 *				   verwaltet: insert(), list_file()
 *				   Suche nach Schluesselworten mittels
 *				   Binaersuche: keyword()
 *		28 March    1985 - Portierung auf UNIX 4.2 BSD
 *
 * Aenderungen:	Udo Munk (C) Copyright 1990
 *		31 December 1990 - Anpassung an COHERENT, etwas Kosmetik
 *				   und Fehlerbehandlung fuer malloc()
 *		mwcbbs!umunk!udo
 */

#include <stdio.h>
#include <ctype.h>

#define lower(c)	(isupper(c) ? tolower(c) : (c))
#define streq(a, b)	(strcmp(a, b) == 0)
#define TRUE		1
#define FALSE		0
#define WIDTH		80		/* Breite des Ausgabegeraets */
#define MAXLINE		256		/* Maximale Laenge Eingabezeile */
#define ID		'a'		/* Identifier */
#define INTEGER		'0'		/* Integer */

struct instance {
	struct instance *next;
	int line;
};

union ptr {
	struct node *files;
	struct instance *lines;
};

struct node {
	struct node *right, *left;
	union ptr p;
	char *name;
};

struct node *symbol_root = NULL;
struct node *file_root = NULL;
int line_count = 0;
char *reserved[] = { "auto", "break", "case", "char", "continue",
		     "default", "do", "double", "else", "extern",
		     "float", "for", "goto", "if", "int", "long",
		     "register", "return", "short", "sizeof", "static",
		     "struct", "switch", "typedef", "union",
		     "unsigned", "void", "while"
};

/*
 * Routinen fuer Symboltabellen
 */

/* lookup - Name an der Wurzel oder unterhalb einhaengen */
struct node **lookup(root, name)
register struct node **root;
register char *name;
{
	register int cond;

#ifdef RECURSIVE
	if (*root != NULL) {
		if ((cond = lexcmp(name, (*root)->name)) != 0)
			root = lookup((cond < 0) ? &(*root)->left, : &(*root)->right, name);
	}
#else
	while (*root != NULL && (cond = lexcmp(name, (*root)->name)))
		root = (cond < 0) ? &(*root)->left : &(*root)->right;
#endif
	return(root);
}

/* add - Eintrag fuer "name" in Baum an der Stelle "root" mit aufnehmen */
struct node *add(name)
char *name;
{
	void panic();
	char *malloc();
	register struct node *r;

	if ((r = (struct node *) malloc(sizeof(struct node))) == NULL)
		panic();
	r->left = r->right = NULL;
	r->p.lines = NULL;
	r->name = name;
	return(r);
}

/* tree_walk - ruft 'ftn' fuer jeden Knoten in Inorder */
void tree_walk(root, ftn)
register struct node *root;
register void (*ftn)();
{
#ifdef RECURSIVE
	if (root != NULL) {
		tree_walk(root->left, ftn);
		(*ftn)(root);
		tree_walk(root->right, ftn);
	}
#else
	register struct node *stack, *tmp;

	stack = NULL;				/* Stack zu Beginn leer */
	for ( ; ; ) {
		if (root != NULL) {
			tmp = root;
			root = root->left;	/* nach links gehen */
			tmp->left = stack;
			stack = tmp;
		} else if (stack != NULL) {	/* Stack nicht leer */
			root = stack;		/* pop */
			stack = stack->left;
			(*ftn)(root);		/* Knoten besuchen */
			root = root->right;	/* nach rechts gehen */
		} else
			break;			/* Stack ist leer */
	}
#endif
}

/* insert - fuegt 'line_no' in die zirkulaere Liste 'origin' ein */
struct instance *insert(origin, line_no)
register struct instance *origin;
int line_no;
{
	void panic();
	char *malloc();
	register struct instance *t;

	if (origin == NULL || origin->line != line_no) {
		if ((t = (struct instance *) malloc(sizeof(struct instance))) == NULL)
			panic();
		if (origin == NULL)
			origin = t;
		t->line = line_no;
		t->next = origin->next;
		origin->next = t;
		origin = t;
	}
	return(origin);
}

/* use - vermerkt ein Vorkommnis von "name" in Datei "file" an der Stelle "line" */
void use(name, file, line)
char *name, *file;
int line;
{
	char *newcpy();
	register struct node **ft, **nt;

	if (*(nt = lookup(&symbol_root, name)) == NULL)
		*nt = add(newcpy(name));
	if (*(nt = lookup(&((*nt)->p.files), file)) == NULL) {
		if (*(ft = lookup(&file_root, file)) == NULL)
			*ft = add(newcpy(file));
		*nt = add((*ft)->name);
	}
	(*nt)->p.lines = insert((*nt)->p.lines, line);
}

/* get_name - holt Dateinamen aus der Zeile */
void get_name(line, file)
register char *line;
char *file;
{
	void copy_until();
	register char *delim;

	while (*line == ' ' || *line == '\t')
		++line;
	if (*line != '\n') {
		if (*line == '"') {
			delim = "\"\n";
			++line;
		} else if (*line == '<') {
			delim = ">\n";
			++line;
		} else
			delim = " \t\n";
		copy_until(file, line, delim);
	}
}

/* new_line - liefert Zeiger auf die naechste Zeile */
char *new_line()
{
	static char line[MAXLINE+1];

	++line_count;
	return(fgets(line, MAXLINE, stdin));
}

/* white_space - Test auf Leerzeichen, Tab und Kommentare */
int white_space(s)
register char **s;
{
	if (**s == ' ' || **s == '\t')
		return(TRUE);
	if (**s == '/' && *(*s+1) == '*') {	/* Kommentar */
		while (*++*s != '/') {
			while (*++*s != '*') {
				if (**s == '\0') {
					if ((*s = new_line()) != NULL)
						--*s;	/* wegen des Praeinkrements in der inneren Schleife */
					else {
						fprintf(stderr, "unexpected EOF\n");
						return(FALSE);
					}
				}
			}
		}
		return(TRUE);
	}
	return(FALSE);
}

/* ishex - ist 'c' eine Hexadezimalziffer? */
int ishex(c)
register char c;
{
	return(('0' <= c && c <= '9') || ('a' <= c && c <= 'f'));
}

/* get_token - entfernt fuehrendes Token aus s */
char get_token(s, t)
register char **s, *t;
{
	char esc();
	register char class;

	while (white_space(s))
		++*s;
	if (isalpha(**s) || **s == '_') {	/* Identifier */
		class = ID;
		do
			*t++ = *(*s)++;
		while (isdigit(**s) || isalpha(**s) || **s == '_');
	} else if (**s == '\"' || **s == '\'') {/* String oder Literal */
		class = **s;
		do {
			esc(s);
			++*s;
			if (**s == '\0') {
				if ((*s = new_line()) == NULL)
					goto out;
			}
		} while (**s != class);
		++*s;
	} else if (isdigit(**s)) {
		do {
			class = *++*s;
			class = lower(class);	/* lower() kann ein Makro sein */
		} while (ishex(class) || class == 'x' || class == 'l' || class == '.');
		class = INTEGER;
	} else {
		class = *(*s)++;
	}
out:
	*t = '\0';
	return(class);
}

/* keyword - ist 's' ein Schluesselwort von C? */
int keyword(s)
char *s;
{
	register int cond;	/* Bedingungscode von lexcmp() */
	register int mid;
	int hi, lo;

	/* Binaersuche; reserved[] muss sortiert sein! */
	lo = 0;
	hi = sizeof(reserved) / sizeof(char *) - 1;
	while (lo <= hi) {
		mid = (hi + lo) / 2;
		if ((cond = lexcmp(s, reserved[mid])) == 0)
			return(TRUE);
		if (cond < 0)
			hi = mid - 1;
		else
			lo = mid + 1;
	}
	return(FALSE);
}

/* xref - Cross-Referenz */
void xref(file)
register char *file;
{
	int atoi();
	register char class;
	char *s, token[MAXLINE+1];

	line_count = 0;
	while ((s = new_line()) != NULL) {
		if ((class = get_token(&s, token)) != '#') {
			while (class != '\n') {
				if (class == ID && !keyword(token))
					use(token, file, line_count);
				class = get_token(&s, token);
			}
		} else if (get_token(&s, token) == ID) {
			if (streq(token, "include")) {
				get_name(s, token);
				use(token, file, line_count);
			} else if (streq(token, "define")) {
				get_token(&s, token);
				use(token, file, line_count);
			} else if (streq(token, "ifdef") || streq(token, "ifndef")) {
				get_token(&s, token);
				use(token, file, line_count);
			} else if (streq(token, "line")) {
				if (get_token(&s, token) == INTEGER)
					line_count = atoi(token);
				else
					fprintf(stderr, "#line %s\n", token);
			} else
				;	/* ignoriere #else, #endif, etc. */
		}
	}
}

/* putp - unvollstaendige Zeile auf stdout ausgeben */
unsigned putp(s)
register char *s;
{
	register unsigned n;

	for (n = 0; *s != '\0'; ++n)
		putchar(*s++);
	return(n);
}

/* list_file - Zeilen in einer Datei drucken */
void list_file(ft)
register struct node *ft;
{
	char *itoa();
	register unsigned b;
	register struct instance *it;
	char buf[5];

	b = putp("    ");
	b += putp(ft->name);
	/* Zeilennummer drucken */
	it = ft->p.lines = ft->p.lines->next;
	do {
		if (b == 0)
			b = putp("       ");	/* diese und die vorletzte Zeile */
						/* muessen uebereinstimmen */
		b += putp("   ");
		b += itoa(it->line, buf) - buf;
		putp(buf);
		if (b > WIDTH - 8) {		/* rechten Rand lassen */
			putp("\n");
			b = 0;
		}
		it = it->next;
	} while (it != ft->p.lines);
	if (b > 6)				/* nichtleere Zeile */
		putp("\n");
}

/* print_xref - Cross-Referenzlisting auf stdout schreiben */
void print_xref(nt)
struct node *nt;
{
	putp(nt->name);
	putp("\n");
	tree_walk(nt->p.files, list_file);
}

main(argc, argv)
register int argc;
register char **argv;
{
	FILE *freopen();

	if (argc <= 1)
		xref("<stdin>");
	else {
		while (--argc > 0) {
			if (freopen(*++argv, "r", stdin) == NULL)
				fprintf(stderr, "can't open %s\n", *argv);
			else
				xref(*argv);
		}
	}
	tree_walk(symbol_root, print_xref);
}
