/*
 *   CDplayer 1.1 - command line Audio CDplayer
 *   Copyright (C) 1993,1994 Mark Buckaway (mark@datasoft.com)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/param.h>
#include <stdlib.h>
#include "config.h"

#ifndef COHERENT
#include <mntent.h>
#endif

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

/*
 * Linux stores cdrom.h in the linux include directory
 * while all others (I'm told) store cdrom.h in the sys include
 * directory
 */
#ifdef LINUX
#include <linux/cdrom.h>
#else
#include <sys/cdrom.h>
#endif

/* 
 * Command index numbers. These *MUST* correspond to the keywords below
 */
enum CMDS {CMD_PLAY=0,CMD_STOP,CMD_INFO,CMD_PAUSE,CMD_RESUME,CMD_EJECT,
           CMD_VOLUME,CMD_SKIP};

/*
 * List of possible commands that can be issued on the command line.
 * This list is NULL terminated...nasty things will happen if the NULL is left
 * out.
 */
static char *keywords[]={
	"play",		/* play an Audio CD */
	"stop",		/* stop the CD */
	"info",		/* show some info on the CD */
	"pause",	/* pause the CD */
	"resume",	/* resume the paused CD */
	"eject",	/* eject a CD on a supported CDROM */
	"volume",	/* adjust te volume */
	"skip",		/* skip to the next track */
	NULL};		/* DO NOT REMOVE THIS!!! */
/*
 * NEC CDROM's do strange things so we redefine the ioctl command to use
 * the special nec_ioctl function that came from xcdplayer
 */
#ifdef NEC_CDROM
#define ioctl nec_ioctl
#endif

/*
 * Define constants
 */

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*
 * Define global variables
 */

char cd_dev[MAXPATHLEN];

void usage(char *arg0)
{
	int i;

	fputs("CDplayer 1.1 by Mark Buckaway (mark@datasoft.com)\n",stderr);
        fputs("(C) 1993,1994 DataSoft Communications\n",stderr);	  
#ifdef NEC_CDROM
	fprintf(stderr,"Compiled to open: %s (NEC) (%s)\n",DEV_CDROM,OS);
#else
	fprintf(stderr,"Compiled to open: %s (%s)\n",DEV_CDROM,OS);
#endif
	fprintf(stderr,"Usage: %s command [track no/volume no]\n",arg0);
	fputs("Available commands: ",stderr);
	for (i=0;keywords[i]!=NULL;i++)	 
		fprintf(stderr,"%s ",keywords[i]);
	fputs("\n",stderr);
	exit(1);
}

int main(int argc, char *argv[])
{
     int i,f,cur_trk;
     struct cdrom_tochdr tochdr;
     struct cdrom_tocentry tocentry;
     struct cdrom_ti ti;
     struct mntent *mnt;
     struct cdrom_subchnl subchnl;
     struct cdrom_volctrl volctrl;	
     FILE *fp;
     unsigned int secs_at_start, secs_at_end, secs_to_play, secs_this_cut, mins_this_cut ;
     char *cd_env;

     /*
      * Check if a command was given. Unlike most UNIX commands, we tell the
      * user the available options if no command line options are given.
      */
     if (argc<2)
	usage(argv[0]);
     /*
      * Check if CD_DEVICE apparent in the environment, if not use
      * the built in default for the device to open.
      */
     if ((cd_env=getenv("CD_DEVICE"))==NULL) 
        strcpy(cd_dev,DEV_CDROM);
       else
        strcpy(cd_dev,cd_env);

#ifndef COHERNET 
     /* 
      * Check if CDROM device is already mounted, and give an error if it is.
      * This routine checks the mtab for any iso9660 file system, and then compares
      * the name of the device for that file system to the device we want to you.
      * If a match is found, the device is assumed to be already mounted.
      *
      * This check is bypassed for Coherent, as they can't yet mount iso9660
      * filesystems.
      */
     if ((fp=setmntent(MOUNTED,"r"))==NULL) {
	  fprintf(stderr,"Couldn't open %s: %s\n",MOUNTED,strerror(errno));
	  exit(1);
     }
     while ((mnt=getmntent(fp))!=NULL) {
	  if (strcmp(mnt->mnt_type,"iso9660")==0) {
               if (strcmp(mnt->mnt_fsname,cd_dev)!=0) continue;
	       fprintf(stderr,"CDROM \"%s\" appears to be already mounted. Unmount CDROM first.\n",cd_dev);
	       endmntent(fp);
	       exit(1);
	  }
     }
     endmntent(fp);
#endif

     /*
      * Open the CD device.
      */
     if ((f=open(cd_dev,O_RDONLY))==-1) {
	  perror("error opening cdrom");
	  exit(1);
     }

     /* 
      * Get table of content entries for min and max tracks and 
      * store these for future reference
      */
     if ((ioctl(f,CDROMREADTOCHDR,&tochdr))==-1) {
	  perror("ioctl(CDROMREADTOHDR)");
	  exit(1);
     }

     /*
      * Get the subchannel info (for CD status, etc) and store it for later.
      */
     subchnl.cdsc_format=CDROM_MSF;
     if ((ioctl(f,CDROMSUBCHNL,&subchnl))==-1) {
         perror("ioctl(CDROMSUBCHNL)");
         exit(1);
     }

     /*
      * CDROM device is open, let's do something with it. Check
      * argument list for a command
      */
     for (i=0;keywords[i]!=NULL;i++) 
	  if ((strcmp(argv[1],keywords[i]))==0) break;

     /*
      * Do the requested operation
      */
     switch (i) {

	    case CMD_PLAY:
	       /*
		* Check if we have a requested track to play, and play it,
                * otherwise, play starting at track 1
                */
	       if (argc==3) {
		    i=atoi(argv[2]);
		    if ((i<tochdr.cdth_trk0) || (i>tochdr.cdth_trk1)) {
			 fprintf(stderr,"Track %d is invalid. Playing from track 1.\n",i);
			 ti.cdti_trk0=tochdr.cdth_trk0;
		       } else {
			 ti.cdti_trk0=i;
		    }
		  } else {
		    ti.cdti_trk0=tochdr.cdth_trk0;
	       }
	       ti.cdti_ind0=1;
	       ti.cdti_trk1=tochdr.cdth_trk1;
	       ti.cdti_ind1=1;
	       if ((ioctl(f,CDROMPLAYTRKIND,&ti))==-1) {
		    perror("ioctl(CDROMPLAYTRKIND)");
		    exit(1);
	       }
	       printf("Audio CD playing. (Start track %d)\n",ti.cdti_trk0);
	       break;

	  case CMD_STOP:
          case CMD_EJECT:
               /*
                * Stop the CD from playing! We must pause the CDROM drive
                * and then stop it otherwise some drives complain.
                */
	       if ((ioctl(f,CDROMPAUSE))==-1) {
		    perror("ioctl(CDROMPAUSE)");
		    exit(1);
	       }
	       if ((ioctl(f,CDROMSTOP))==-1) {
		    perror("ioctl(CDROMSTOP)");
		    exit(1);
	       }
	       puts("Audio CD stopped.");
               if (i==CMD_EJECT) {
	          if ((ioctl(f,CDROMEJECT))==-1) {
		       perror("ioctl(CDROMEJECT)");
		       exit(1);
	          }
	       }  
	       break;

	   case CMD_INFO:
               /*
		* All this does is grab the total time for the CD and 
                * figures out the time per track as well. It also shows
		* the CDROM status and track in play (if playing).
		* 
                */
               printf("Audio CD Information on %s:\n",cd_dev);
	       printf("Total Tracks: %u\n",tochdr.cdth_trk1);
	       tocentry.cdte_track=CDROM_LEADOUT;
	       tocentry.cdte_format=CDROM_MSF;
	       if ((ioctl(f,CDROMREADTOCENTRY,&tocentry))==-1) {
		    perror("ioctl(CDROMREADTOCENTRY)");
		    exit(1);
	       }
	       printf("Total Playing Time: %u min %u sec\n",
                     tocentry.cdte_addr.msf.minute,
                     tocentry.cdte_addr.msf.second);
               printf("Track  Time\n");

               /*
                * Loop through each of the table of contents entries and pull
                * pull some info about each track. We compare this to the
                * subchannel info (current CD position) to find the current
                * track in play if playing
                */
	       for (i=tochdr.cdth_trk0;i<=tochdr.cdth_trk1;i++) {
		    tocentry.cdte_track=i;
		    tocentry.cdte_format=CDROM_MSF;
		    if ((ioctl(f,CDROMREADTOCENTRY,&tocentry))==-1) {
			 perror("ioctl(CDROMREADTOCENTRY)");
			 exit(1);
		    }
                    if ((subchnl.cdsc_audiostatus==CDROM_AUDIO_PLAY) &&
                        (subchnl.cdsc_absaddr.msf.minute*60+subchnl.cdsc_absaddr.msf.second>=
                         tocentry.cdte_addr.msf.minute*60+tocentry.cdte_addr.msf.second)) 
                          cur_trk=i; 
                    if (i==tochdr.cdth_trk0)
                        secs_at_start = 0 ;
                     else 
                        secs_at_start = secs_at_end ;
                    secs_at_end = tocentry.cdte_addr.msf.minute * 60 +
                                     tocentry.cdte_addr.msf.second ;
                    secs_to_play = secs_at_end - secs_at_start ;

                    mins_this_cut = secs_to_play / 60 ;
                    secs_this_cut = secs_to_play % 60 ;
                    
		    printf("%2d    %2u:%02u\n",i, mins_this_cut, secs_this_cut );
	       }

               /*
                * Print the status of the CDROM drive and the current track in play
                */
	       printf("Status: ");
	       switch (subchnl.cdsc_audiostatus) {
	              case CDROM_AUDIO_INVALID:
	                 puts("Audio status not supported");
		         break;
                      case CDROM_AUDIO_PLAY:
                         printf("Audio CD playing. Current track: %d\n",cur_trk);
                         break;
                      case CDROM_AUDIO_PAUSED:
                         puts("Audio CD paused");
                         break;
                      case CDROM_AUDIO_COMPLETED:
                         puts("Audio CD completed");
                         break;
                      case CDROM_AUDIO_ERROR:
                         puts("Audio CD stopped due to error");
                         break;
                      case CDROM_AUDIO_NO_STATUS:
                         puts("No status to return: Drive stopped or open.");
               }
	       break;

	    case CMD_PAUSE:
               /*
                * Pause a playing Audio CD
                */
               if (subchnl.cdsc_audiostatus!=CDROM_AUDIO_PLAY) {
                     printf("Audio CD not current playing. Command ignored.\n");
                     exit(1);
               }
	       if ((ioctl(f,CDROMPAUSE))==-1) {
		    perror("ioctl(CDROMPAUSE)");
		    exit(1);
	       }
	       printf("Audio CD paused. Use \"%s resume\" to continue.\n",argv[0]);
               break;

            case CMD_RESUME:
            /*
             * Check if the Audio CD is currently paused, and resume play.
             */
               if (subchnl.cdsc_audiostatus!=CDROM_AUDIO_PAUSED) {
                     printf("Audio CD not current paused. Command ignored.\n");
                     exit(1);
               }
	       if ((ioctl(f,CDROMRESUME))==-1) {
		    perror("ioctl(CDROMRESUME)");
		    exit(1);
	       }
               puts("Audio CD resumed.");
	       break;

	    case CMD_VOLUME:
               /*
                * Adjust the volume of the CDROM. This is not supported
		* on all drives and Linux drivers                
		*/
               if (subchnl.cdsc_audiostatus!=CDROM_AUDIO_PLAY) {
                     printf("Audio CD not current playing. Command ignored.\n");
                     exit(1);
               }
	       if (argc==3) {
		    i=atoi(argv[2]);
                    if ((i<0) || (i>255)) {
                       fputs("Illegal Volume value. Use 0 - 255. Command ignored.\n",stderr);
                       exit(1);
                    }
                    volctrl.channel0=i;
                    volctrl.channel1=i;
                    volctrl.channel2=i;
                    volctrl.channel3=i;
	            if ((ioctl(f,CDROMVOLCTRL,&volctrl))==-1) {
		       perror("ioctl(CDROMVOLCTRL)");
		       exit(1);
	            }
	        } else {
	          fputs("Volume requires a parameter. Command ignored.\n",stderr);
		  exit(1);	
	       }
               printf("Audio CD Volume set to %d.\n",i);
   	       break;

            case CMD_SKIP:
	       /*
                * Figure out the current track and skip to the next one.
                * If this is the last track, play the first track.
                */
               if (subchnl.cdsc_audiostatus!=CDROM_AUDIO_PLAY) {
                     printf("Audio CD not current playing. Command ignored.\n");
                     exit(1);
               }
	       tocentry.cdte_track=CDROM_LEADOUT;
	       tocentry.cdte_format=CDROM_MSF;
	       if ((ioctl(f,CDROMREADTOCENTRY,&tocentry))==-1) {
		    perror("ioctl(CDROMREADTOCENTRY)");
		    exit(1);
	       }
	       for (i=tochdr.cdth_trk0;i<=tochdr.cdth_trk1;i++) {
		    tocentry.cdte_track=i;
		    tocentry.cdte_format=CDROM_MSF;
		    if ((ioctl(f,CDROMREADTOCENTRY,&tocentry))==-1) {
			 perror("ioctl(CDROMREADTOCENTRY)");
			 exit(1);
		    }
                    if ((tocentry.cdte_addr.msf.minute*60+tocentry.cdte_addr.msf.second) >=
                        (subchnl.cdsc_absaddr.msf.minute*60+subchnl.cdsc_absaddr.msf.second)) {
                         break;
                    }
              }
              cur_trk=i;
              if (cur_trk>tochdr.cdth_trk1) cur_trk=1;
	      ti.cdti_trk0=cur_trk;
	      ti.cdti_ind0=1;
	      ti.cdti_trk1=tochdr.cdth_trk1;
	      ti.cdti_ind1=1;
	      if ((ioctl(f,CDROMPLAYTRKIND,&ti))==-1) {
		   perror("ioctl(CDROMPLAYTRKIND)");
		   exit(1);
	      }
	      printf("Audio CD skipped to track %d.\n",cur_trk);
              break; 
	  default:
               usage(argv[0]);
     }
     close(f);
     return(0);
}
