/*@z10.c:Cross References:CrossInit(), CrossMake()@***************************/
/*                                                                           */
/*  THE LOUT DOCUMENT FORMATTING SYSTEM (VERSION 3.04)                       */
/*  COPYRIGHT (C) 1994 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@cs.su.oz.au)                                   */
/*  Basser Department of Computer Science                                    */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 1, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*                                                                           */
/*  FILE:         z10.c                                                      */
/*  MODULE:       Cross References                                           */
/*  EXTERNS:      CrossInit(), CrossMake(), GallTargEval(), CrossAddTag(),   */
/*                CrossExpand(), CrossSequence(), CrossClose()               */
/*                                                                           */
/*****************************************************************************/
#include "externs"
#define	CROSS_LIT	CROSS_TARG
#define	NO_TARGET	0
#define	SEEN_TARGET	1
#define	WRITTEN_TARGET	2
#define INIT_CROSSREF_NUM	100

static OBJECT RootCross = nilobj;		/* header for all crs        */

/*****************************************************************************/
/*                                                                           */
/*  CROSSREF_TABLE                                                           */
/*                                                                           */
/*  A symbol table permitting access to cross reference generated tags by    */
/*  a mapping (symbol x file) -> current tag.                                */
/*                                                                           */
/*     crtab_getnext(sym, fnum, S)   Get next value associated with sym,fnum */
/*     crtab_debug(S, fp)            Debug print of table S to file fp       */
/*                                                                           */
/*****************************************************************************/

typedef struct crossref_rec
{ struct crossref_rec	*crtab_next;
  OBJECT		crtab_sym;
  FILE_NUM		crtab_fnum;
  int			crtab_value;
} *CROSSREF_ENTRY;

typedef struct
{ int tab_size;					/* size of table             */
  int tab_count;				/* number of entries held    */
  CROSSREF_ENTRY tab_chains[1];			/* the chains of entries     */
} *CROSSREF_TABLE;

#define	crtab_size(S)	(S)->tab_size
#define	crtab_count(S)	(S)->tab_count
#define	crtab_chain(S,i) (S)->tab_chains[i]

#define hash(pos, sym, fnum, S)						\
{ pos = ( ((int) sym) + fnum ) % crtab_size(S);				\
}

static CROSSREF_TABLE crtab_new(int newsize)
{ CROSSREF_TABLE S;  int i;
  ifdebug(DMA, D, DebugRegisterUsage(MEM_CROSSREF, 1,
    2*sizeof(int) + newsize*sizeof(CROSSREF_ENTRY)));
  S = (CROSSREF_TABLE)
    malloc(2*sizeof(int) + newsize*sizeof(CROSSREF_ENTRY));
  if( S == (CROSSREF_TABLE) NULL )
    Error(10, 1, "run out of memory enlarging crossref table", FATAL, no_fpos);
  crtab_size(S) = newsize;
  crtab_count(S) = 0;
  for( i = 0;  i < newsize;  i++ )
    crtab_chain(S, i) = (CROSSREF_ENTRY) nilobj;
  return S;
} /* end crtab_new */

static CROSSREF_TABLE crtab_rehash(CROSSREF_TABLE S, int newsize)
{ CROSSREF_TABLE NewS;  int i, newpos;  CROSSREF_ENTRY p, q;
  NewS = crtab_new(newsize);
  for( i = 0;  i < crtab_size(S);  i++ )
  { p = crtab_chain(S, i);
    while( p != NULL )
    { q = p->crtab_next;
      hash(newpos, p->crtab_sym, p->crtab_fnum, NewS);
      p->crtab_next = crtab_chain(NewS, newpos);
      crtab_chain(NewS, newpos) = p;
      crtab_count(NewS)++;
      p = q;
    }
  }
  ifdebug(DMA, D, DebugRegisterUsage(MEM_CROSSREF, -1,
   -(2*sizeof(int) + crtab_size(S)*sizeof(CROSSREF_ENTRY))));
  free(S);
  return NewS;
} /* end crtab_rehash */

static int crtab_getnext(OBJECT sym, FILE_NUM fnum, CROSSREF_TABLE *S)
{ CROSSREF_ENTRY x;  int pos;

  /* if S is NULL, create a new table */
  if( *S == NULL )  *S = crtab_new(INIT_CROSSREF_NUM);

  /* if (sym, fnum) exists, increment its value and return it */
  hash(pos, sym, fnum, *S);
  for( x = crtab_chain(*S, pos);  x != NULL;  x = x->crtab_next )
  { if( x->crtab_sym == sym && x->crtab_fnum == fnum )
    return ++x->crtab_value;
  }

  /* if table is full, rehash */
  if( crtab_count(*S) == crtab_size(*S) )
  { *S = crtab_rehash(*S, 2*crtab_size(*S));
    hash(pos, sym, fnum, *S);
  }

  /* insert a new entry for (sym, fnum) with value 1 */
  x = (CROSSREF_ENTRY) GetMem(sizeof(struct crossref_rec), no_fpos);
  x->crtab_sym = sym;
  x->crtab_fnum = fnum;
  x->crtab_next = crtab_chain(*S, pos);
  crtab_chain(*S, pos) = x;
  crtab_count(*S)++;
  return x->crtab_value = 1;

} /* end crtab_getnext */

#if DEBUG_ON
static void crtab_debug(CROSSREF_TABLE S, FILE *fp)
{ int i;  CROSSREF_ENTRY x;
  if( S == NULL )
  { fprintf(fp, "  null table\n");
    return;
  }
  fprintf(fp, "  table size: %d;  current count: %d\n",
    crtab_size(S), crtab_count(S));
  for( i = 0;  i < crtab_size(S);  i++ )
  { fprintf(fp, "crtab_chain(S, %d) =", i);
    for( x = crtab_chain(S, i);  x != NULL;  x = x->crtab_next )
    { fprintf(fp, " %s:%s,%d",
	SymName(x->crtab_sym), FileName(x->crtab_fnum), x->crtab_value);
    }
    fprintf(fp, "\n");
  }
} /* end crtab_debug */
#endif

static CROSSREF_TABLE crossref_tab = NULL;


/*@@**************************************************************************/
/*                                                                           */
/*  CrossInit(sym)     Initialize cross_sym(sym).                            */
/*                                                                           */
/*****************************************************************************/

void CrossInit(OBJECT sym)
{ OBJECT cs = New(CROSS_SYM);
  target_state(cs) = NO_TARGET;  target_seq(cs) = 0;
  cr_file(cs) = NO_FILE;
  gall_seq(cs) = 0;  gall_tag(cs) = nilobj;
  gall_tfile(cs) = NO_FILE;
  symb(cs) = sym;  cross_sym(sym) = cs;
  if( RootCross == nilobj )  RootCross = New(CR_ROOT);  Link(RootCross, cs);
}


/*****************************************************************************/
/*                                                                           */
/*  OBJECT CrossMake(sym, val, ctype)                                        */
/*                                                                           */
/*  Make a cross-reference with the given sym and tag value (NB no fpos).    */
/*                                                                           */
/*****************************************************************************/

OBJECT CrossMake(OBJECT sym, OBJECT val, int ctype)
{ OBJECT v1, res;
  debug3(DCR, DD, "CrossMake(%s, %s, %s)", SymName(sym),
    EchoObject(val), Image(ctype));
  res = New(CROSS);  cross_type(res) = ctype;  threaded(res) = FALSE;
  v1 = New(CLOSURE);  actual(v1) = sym;
  Link(res, v1);  Link(res, val);
  debug1(DCR, DD, "CrossMake returning %s", EchoObject(res));
  return res;
}

/*@::GallTargEval(), CrossGenTag()@*******************************************/
/*                                                                           */
/*  OBJECT GallTargEval(sym, dfpos)                                          */
/*                                                                           */
/*  Produce a suitable cross-reference for a galley target.                  */
/*                                                                           */
/*****************************************************************************/

OBJECT GallTargEval(OBJECT sym, FILE_POS *dfpos)
{ OBJECT cs, res;
  FULL_CHAR buff[MAX_BUFF], *str;
  debug2(DCR, DD, "GallTargEval( %s,%s )", SymName(sym), EchoFilePos(dfpos));
  if( cross_sym(sym) == nilobj )  CrossInit(sym);
  cs = cross_sym(sym);
  if( file_num(*dfpos) != gall_tfile(cs) )
  { gall_tfile(cs) = file_num(*dfpos);
    gall_seq(cs)   = 0;
  }
  str = FileName(gall_tfile(cs));
  ++gall_seq(cs);
  if( StringLength(str) + 6 >= MAX_BUFF )
    Error(10, 2, "automatically generated tag %s&%d is too long",
	FATAL, dfpos, str, gall_seq(cs));
  StringCopy(buff, str);
  StringCat(buff, AsciiToFull("&"));
  StringCat(buff, StringInt(gall_seq(cs)));
  res = CrossMake(sym, MakeWord(WORD, buff, dfpos), GALL_TARG);
  debug1(DCR, DD, "GallTargEval returning %s", EchoObject(res));
  return res;
} /* end GallTargEval */


/*****************************************************************************/
/*                                                                           */
/*  static OBJECT CrossGenTag(x)                                             */
/*                                                                           */
/*  Generate a tag suitable for labelling closure x, in such a way that      */
/*  the same tag is likely to be generated on subsequent runs.               */
/*                                                                           */
/*****************************************************************************/

static OBJECT CrossGenTag(OBJECT x)
{ FULL_CHAR buff[MAX_BUFF], *str1, *str2;
  OBJECT sym, cs, res;  FILE_NUM fnum;
  int seq;
  debug1(DCR, DD, "CrossGenTag( %s )", SymName(actual(x)));
  sym = actual(x);
  if( cross_sym(sym) == nilobj )  CrossInit(sym);
  cs = cross_sym(sym);
  fnum = file_num(fpos(x));
  str1 = SymName(sym);
  str2 = FileName(fnum);
  seq = crtab_getnext(sym, fnum, &crossref_tab);
  debug3(DCR, DDD, "%d = crtab_getnext(%s, %s, S); S =",
	seq, SymName(sym), FileName(fnum));
  ifdebug(DCR, DDD, crtab_debug(crossref_tab, stderr));
  if( StringLength(str1) + StringLength(str2) + 10 >= MAX_BUFF )
    Error(10, 3, "automatically generated tag %s.%s.%d is too long",
	FATAL, no_fpos, str1, str2, seq);
  /* *** make shorter, little risk
  StringCopy(buff, str1);
  StringCat(buff, AsciiToFull("."));
  *** */
  StringCopy(buff, StringInt(line_num(fpos(sym))));
  StringCat(buff, AsciiToFull("."));
  StringCat(buff, str2);
  StringCat(buff, AsciiToFull("."));
  StringCat(buff, StringInt(seq));
  res = MakeWord(QWORD, buff, &fpos(x));
  debug1(DCR, DD, "CrossGenTag returning %s", string(res));
  return res;
} /* end CrossGenTag */


/*@::CrossAddTag()@***********************************************************/
/*                                                                           */
/*  CrossAddTag(x)                                                           */
/*                                                                           */
/*  Add an automatically generated @Tag parameter to closure x if required.  */
/*                                                                           */
/*****************************************************************************/

void CrossAddTag(OBJECT x)
{ OBJECT link, par, ppar, y;
  debug1(DCR, DD, "CrossAddTag( %s )", EchoObject(x));

  /* search the parameter list of x for a @Tag parameter */
  for( link = Down(x);  link != x;  link = NextDown(link) )
  { Child(par, link);
    if( type(par) == PAR && is_tag(actual(par)) )
    {
      /* has tag, but if value is empty object, delete it */
      Child(y, Down(par));
      if( is_word(type(y)) && StringEqual(string(y), STR_EMPTY) )
      { DisposeChild(link);
	link = x;
      }
      break;
    }
  }
  if( link == x )
  { 
      /* search the definition of x for name of its @Tag parameter */
      ppar = nilobj;
      for( link=Down(actual(x));  link != actual(x);  link = NextDown(link) )
      {	Child(y, link);
	if( is_par(type(y)) && is_tag(y) )
	{ ppar = y;
	  break;
	}
      }
      if( ppar != nilobj ) /* should always hold */
      {
	/* prepare new PAR containing generated tag */
	par = New(PAR);
	actual(par) = ppar;
	y = CrossGenTag(x);
	Link(par, y);

	/* find the right spot, then link it to x */
	switch( type(ppar) )
	{
	  case LPAR:	link = Down(x);
			break;

	  case NPAR:	link = Down(x);
			if( Down(x) != x )
			{ Child(y, Down(x));
			  if( type(y) == PAR && type(actual(y)) == LPAR )
				link = NextDown(link);
			}
			break;

	  case RPAR:	for( link = Down(x); link != x; link = NextDown(link) )
			{ Child(y, link);
			  if( type(y) != PAR )  break;
			}
			break;
	}
	Link(link, par);
      }
  }
  debug1(DCR, DD, "CrossAddTag returning %s", EchoObject(x));
} /* end CrossAddTag */


/*@::CrossExpand()@***********************************************************/
/*                                                                           */
/*  OBJECT CrossExpand(x, env, style, crs_wanted, crs, res_env)              */
/*                                                                           */
/*  Return the value of cross-reference x, with environment *res_env.  If x  */
/*  has a non-literal tag, it must be tracked, so an object is added to *crs */
/*  for this purpose if crs_wanted.  Result replaces x, which is disposed.   */
/*                                                                           */
/*****************************************************************************/
static OBJECT nbt[2] = { nilobj, nilobj };
static OBJECT nft[2] = { nilobj, nilobj };
static OBJECT ntarget = nilobj;

OBJECT CrossExpand(OBJECT x, OBJECT env, STYLE *style, BOOLEAN crs_wanted,
OBJECT *crs, OBJECT *res_env)
{ OBJECT sym, res, tag, y, cs, link, db, tmp, index;
  int ctype, count, i;  FULL_CHAR buff[MAX_BUFF], seq[MAX_BUFF], *str;
  FILE_NUM fnum, dfnum;  BOOLEAN tagerror = FALSE;
  long cont, dfpos;
  assert( type(x) == CROSS, "CrossExpand: x!" );
  debug3(DCR, D, "[ CrossExpand( %s, %s, %s )",
    EchoObject(x), EchoObject(*crs), bool(crs_wanted));
  assert( NextDown(Down(x)) == LastDown(x), "CrossExpand: #args!" );

  /* manifest and tidy the right parameter */
  Child(tag, LastDown(x));
  debug0(DOM, D, "  [ calling Manifest from CrossExpand");
  tag = Manifest(tag, env, style, nbt, nft, &ntarget, crs, FALSE, FALSE);
  debug0(DOM, D, "  ] returning from Manifest");
  tag = ReplaceWithTidy(tag, TRUE);   /* && */

  /* extract sym (the symbol name) and tag (the tag value) from x */
  Child(y, Down(x));
  assert( type(y) == CLOSURE, "ClosureExpand: type(y) != COLSURE!" );
  sym = actual(y);
  ctype = !is_word(type(tag)) ? 1 :
	  StringEqual(string(tag), STR_EMPTY) ? 2 :
	  StringEqual(string(tag), KW_PRECEDING) ? CROSS_PREC :
	  StringEqual(string(tag), KW_FOLLOWING) ? CROSS_FOLL : CROSS_LIT;

  res = nilobj;
  switch( ctype )
  {

    case 1:

      Error(10, 5, "value of right parameter of %s is not a simple word",
	WARN, &fpos(tag), KW_CROSS);
      break;


    case 2:
    
      Error(10, 6, "value of right parameter of %s is an empty word",
	WARN, &fpos(tag), KW_CROSS);
      break;


    case CROSS_LIT:
    
      if( cross_sym(sym) == nilobj )  CrossInit(sym);
      cs = cross_sym(sym);
      if( sym == MomentSym && StringEqual(string(tag), KW_NOW) )
      {	/* this is a request for the current time */
	res = StartMoment();
      }
      else
      { if( !has_tag(sym) )
	{ Error(10, 7, "symbol %s used in cross reference has no %s parameter",
	    WARN, &fpos(x), SymName(sym), KW_TAG);
	  tagerror = TRUE;
	}
	for( link = NextUp(Up(cs));  link != cs;  link = NextUp(link) )
        { Parent(db, link);
	  assert( is_word(type(db)), "CrossExpand: db!" );
	  if( DbRetrieve(db, FALSE, sym, string(tag),seq,&dfnum,&dfpos,&cont) )
	  { SwitchScope(nilobj);
	    count = 0;
	    SetScope(env, &count);
	    res = ReadFromFile(dfnum, dfpos);
	    for( i = 1;  i <= count;  i++ )  PopScope();
	    UnSwitchScope(nilobj);
	    if( db != OldCrossDb )  AttachEnv(env, res);
	    break;
	  }
	}
      }
      break;


    case CROSS_PREC:
    case CROSS_FOLL:
    
      if( has_tag(sym) )
      { if( cross_sym(sym) == nilobj )  CrossInit(sym);
        cs = cross_sym(sym);
        assert( cs != nilobj, "CrossExpand/CROSS_FOLL: cs == nilobj!" );
        assert( type(cs) == CROSS_SYM, "CrossExpand/CROSS_FOLL: type(cs)!" );
        fnum = file_num(fpos(tag));
        if( fnum != cr_file(cs) )
        { cr_file(cs) = fnum;
	  cr_seq(cs) = 0;
        }
        str = FileName(fnum);
        ++cr_seq(cs);
        if( StringLength(str) + 5 >= MAX_BUFF )
	  Error(10, 8, "automatically generated tag %s_%d is too long",
	    FATAL, &fpos(x), str, cr_seq(cs));
        StringCopy(buff, str);
        StringCat(buff, AsciiToFull("_"));
        StringCat(buff, StringInt(cr_seq(cs)));
        tmp = CrossMake(sym, MakeWord(WORD, buff, &fpos(tag)), ctype);
        index = New(ctype);
        actual(index) = tmp;
        Link(index, tmp);
        if( crs_wanted )
        { if( *crs == nilobj )  *crs = New(CR_LIST);
	  link = Link(*crs, index);
        }
        else Error(10, 9, "%s or %s tag not allowed here",
	  FATAL, &fpos(x), KW_PRECEDING, KW_FOLLOWING);
        if( AllowCrossDb &&
	    DbRetrieve(OldCrossDb, FALSE, sym, buff, seq,&dfnum,&dfpos,&cont) )
	{
	  SwitchScope(nilobj);
	  res = ReadFromFile(dfnum, dfpos);
	  UnSwitchScope(nilobj);
	}
      }
      else
      {	Error(10, 10, "symbol %s used in cross reference has no %s parameter",
	  WARN, &fpos(x), SymName(sym), KW_TAG);
	tagerror = TRUE;
      }
      break;


    default:
    
      Error(10, 11, "CrossExpand: %d", INTERN, no_fpos, ctype);
      break;


  } /* end switch */
  if( res == nilobj )
  { OBJECT envt;
    /* *** only reporting this now if crs_wanted (i.e object is "real")
    if( ctype > 1 && !tagerror )
    *** */
    if( ctype > 1 && !tagerror && crs_wanted )
    { debug3(DCR, D, "  reporting unresolved cross reference %s%s%s",
	SymName(sym), KW_CROSS, string(tag));
      Error(10, 12, "unresolved cross reference %s%s%s",
	WARN, &fpos(x), SymName(sym), KW_CROSS, string(tag));
    }

    /* build dummy result with environment attached */
    /* nb at present we are not adding dummy import closures to this! */
    res = New(CLOSURE);  actual(res) = sym;
    y = res;
    debug1(DCR, DD, "First y = %s", SymName(actual(y)));
    while( enclosing(actual(y)) != StartSym )
    { tmp = New(CLOSURE);
      actual(tmp) = enclosing(actual(y));
      debug0(DCR, DDD, "  calling SetEnv from CrossExpand (a)");
      envt = SetEnv(tmp, nilobj);
      AttachEnv(envt, y);
      y = tmp;
      debug1(DCR, DD, "Later y = %s", SymName(actual(y)));
    }
    envt = New(ENV);  Link(y, envt);
  }

  /* set environment, replace x by res, debug and exit */
  *res_env = DetachEnv(res);
  ReplaceNode(res, x);
  DisposeObject(x);
  assert( type(res) == CLOSURE, "CrossExpand: type(res) != CLOSURE!" );
  assert( actual(res) == sym, "CrossExpand: actual(res) != sym!" );
  debug1(DCR, D, "] CrossExpand returning %s", EchoObject(res));
  debug1(DCR, DD, "  *crs = %s", EchoObject(*crs));
  debug1(DCR, DD, "  *res_env = %s", EchoObject(*res_env));
  return res;
} /* end CrossExpand */


/*@::CrossSequence()@*********************************************************/
/*                                                                           */
/*  CrossSequence(x)                                                         */
/*                                                                           */
/*  Object x is an insinuated cross-reference that has just been popped off  */
/*  the top of the root galley.  Resolve it with the sequence of others.     */
/*                                                                           */
/*****************************************************************************/

void CrossSequence(OBJECT x)
{ OBJECT sym, tag, val, tmp, cs, par, key, hold_key, link, y, env, hold_env;
  unsigned ctype;  FULL_CHAR buff[MAX_BUFF], *str, *seq;
  FILE_NUM dfnum;  int dfpos;

  /* if suppressing cross-referencing, dispose x and quit */
  if( !AllowCrossDb )
  { if( Up(x) == x )  DisposeObject(x);
    debug0(DCR, D, "CrossSequence returning (!AllowCrossDb).");
    return;
  }

  /* get interesting fragments from x */
  debugcond1(DCR, D, type(x) != CROSS, "  type(x) = %s, x =", Image(type(x)));
  ifdebugcond(DCR, D, type(x) != CROSS, DebugObject(x));
  assert( type(x) == CROSS, "CrossSequence: type(x)!" );
  ctype = cross_type(x);
  Child(tmp, Down(x));
  assert( type(tmp) == CLOSURE, "CrossSequence: type(tmp)!" );
  sym = actual(tmp);
  if( cross_sym(sym) == nilobj )  CrossInit(sym);
  cs = cross_sym(sym);
  assert( type(cs) == CROSS_SYM, "CrossSequence: cs!" );

  /* debug output */
  debug2(DCR, D, "CrossSequence %s %s", Image(ctype), EchoObject(x));
  debug1(DCR, DD, "  x = %s", EchoObject(x));
  ifdebug(DCR, DD, DebugObject(cs));

  /* delete as much of x as possible */
  Child(tag, NextDown(Down(x)));
  DeleteLink(NextDown(Down(x)));
  if( Up(x) == x )  DisposeObject(x);

  switch( ctype )
  {
    case GALL_FOLL:
    case GALL_PREC:

      /* find key of the galley, if any */
      /* *** this code is obsolete; or rather, it was always bizarrely naive
      val = tag;  key = nilobj;
      for( link = Down(val);  link != val;  link = NextDown(link) )
      {	Child(par, link);
	if( type(par) == PAR && (is_key(actual(par)) || is_tag(actual(par))) )
	{ assert( Down(par) != par, "CrossSequence: PAR child!" );
	  Child(key, Down(par));
	  key = ReplaceWithTidy(key, TRUE);
	}
      }
      *** */

      /* find the value of key of the galley, if any */
      val = tag;  key = hold_key = nilobj;
      assert( type(val) == CLOSURE, "CrossSequence/GALL_FOLL: type(val)!" );
      if( has_key(actual(val)) )
      { for( link=Down(actual(val)); link != actual(val); link=NextDown(link) )
	{ Child(y, link);
	  if( is_key(y) )
	  { OBJECT nbt[2], nft[2], crs, ntarget;
	    nbt[COL] = nft[COL] = nbt[ROW] = nft[ROW] = crs = ntarget = nilobj;
	    key = New(CLOSURE);
	    actual(key) = y;
	    hold_key = New(ACAT);
	    Link(hold_key, key);
	    env = New(ENV);
	    Link(env, val);
	    hold_env = New(ACAT);
	    Link(hold_env, env);
	    debug0(DOM, D, "  [ calling Manifest from CrossSequence");
	    key = Manifest(key, env, &save_style(val), nbt, nft,
	      &ntarget, &crs, FALSE, TRUE);
	    debug0(DOM, D, "  ] returning from Manifest");
	    key = ReplaceWithTidy(key, TRUE);
	    DeleteLink(Down(env));
	    DisposeObject(hold_env);
	  }
	}
      }

      /* write out the galley */
      str = FileName(file_num(fpos(val)));
      if( StringEndsWith(str, DATA_SUFFIX) )
	dfnum = file_num(fpos(val));
      else
      { dfnum = FileNum(str, DATA_SUFFIX);
        if( dfnum == NO_FILE )
	  dfnum = DefineFile(str, DATA_SUFFIX, &fpos(val),
	    DATABASE_FILE, SOURCE_PATH);
      }
      AppendToFile(val, dfnum, &dfpos);

      /* determine the sequence number or string of this galley */
      if( key == nilobj )
      {	++gall_seq(cs);
	StringCopy(buff, StringFiveInt(gall_seq(cs)));
	seq = buff;
      }
      else if( !is_word(type(key)) )
      {	Error(10, 13, "%s parameter is not a word", WARN, &fpos(key), KW_KEY);
	debug1(DCR, D, "key = %s", EchoObject(key));
	seq = STR_BADKEY;
      }
      else if( StringEqual(string(key), STR_EMPTY) )
      {	Error(10, 14, "%s parameter is an empty word", WARN,&fpos(key),KW_KEY);
	seq = STR_BADKEY;
      }
      else seq = string(key);

      /* either write out the index immediately or store it for later */
      if( ctype == GALL_PREC )
      {	if( gall_tag(cs) == nilobj )
	{ Error(10, 15, "no %s precedes this %s%s%s", WARN, &fpos(val),
	    SymName(sym), SymName(sym), KW_CROSS, KW_PRECEDING);
	  debug0(DCR, DD, "  ... so substituting \"none\"");
	  gall_tag(cs) = MakeWord(WORD, STR_NONE, &fpos(val));
	}
	assert( is_word(type(gall_tag(cs))) &&
		!StringEqual(string(gall_tag(cs)), STR_EMPTY),
			"CrossSequence: gall_tag!" );
	debug3(DCR, DD, "  inserting galley (prec) %s&%s %s", SymName(sym),
	  string(gall_tag(cs)), seq);
	DbInsert(NewCrossDb, TRUE, sym, string(gall_tag(cs)), no_fpos, seq,
			dfnum, (long) dfpos, FALSE);
      }
      else
      {	tmp = MakeWord(WORD, seq, &fpos(val));
	gall_rec(tmp) = TRUE;
	file_num(fpos(tmp)) = dfnum;
	gall_pos(tmp) = dfpos;
	Link(cs, tmp);
	debug2(DCR, DD, "  saving galley (foll) %s&? %s", SymName(sym), seq);
      }
      DisposeObject(val);
      if( hold_key != nilobj )  DisposeObject(hold_key);
      break;


    case GALL_TARG:

      if( gall_tag(cs) != nilobj )  DisposeObject(gall_tag(cs));
      if( !is_word(type(tag)) || StringEqual(string(tag), STR_EMPTY) )
      {
	debug2(DCR, DD, "  GALL_TARG %s put none for %s",
	  SymName(sym), EchoObject(tag));
	DisposeObject(tag);
	gall_tag(cs) = MakeWord(WORD, STR_NONE, no_fpos);
      }
      else gall_tag(cs) = tag;
      debug2(DCR, DD, "  have new %s gall_targ %s", SymName(sym),
	  EchoObject(gall_tag(cs)));
      for( link = Down(cs);  link != cs;  link = NextDown(link) )
      {	Child(y, link);
	assert( is_word(type(y)) && !StringEqual(string(y), STR_EMPTY),
				"CrossSequence: GALL_TARG y!" );
	if( gall_rec(y) )
	{
	  debug3(DCR, DD, "  inserting galley (foll) %s&%s %s", SymName(sym),
	    string(gall_tag(cs)), string(y));
	  DbInsert(NewCrossDb, TRUE, sym, string(gall_tag(cs)), no_fpos,
	    string(y), file_num(fpos(y)), (long) gall_pos(y), FALSE);
	  link = PrevDown(link);
	  DisposeChild(NextDown(link));
	}
      }
      break;


    case CROSS_PREC:

      if( target_state(cs) == NO_TARGET )
      {	Error(10, 16, "no %s precedes this %s%s%s", WARN, &fpos(tag),
	  SymName(sym), SymName(sym), KW_CROSS, KW_PRECEDING);
	break;
      }
      if( target_state(cs) == SEEN_TARGET )
      {
	debug2(DCR, DD, "  inserting %s cross_targ %s",
	  SymName(sym), target_val(cs));
	AppendToFile(target_val(cs), target_file(cs), &target_pos(cs));
	DisposeObject(target_val(cs));
	target_val(cs) = nilobj;
	target_state(cs) = WRITTEN_TARGET;
      }
      if( !is_word(type(tag)) || StringEqual(string(tag), STR_EMPTY) )
      {
	debug2(DCR, DD, "  GALL_TARG %s put none for %s", SymName(sym),
		EchoObject(tag));
	DisposeObject(tag);
	tag = MakeWord(WORD, STR_NONE, no_fpos);
      }
      debug3(DCR, DD, "  inserting cross (prec) %s&%s %s", SymName(sym),
	    string(tag), "0");
      DbInsert(NewCrossDb, FALSE, sym, string(tag), &fpos(tag), STR_ZERO,
	target_file(cs), (long) target_pos(cs), TRUE);
      DisposeObject(tag);
      break;


    case CROSS_FOLL:

      if( !is_word(type(tag)) )
      {	Error(10, 17, "tag of %s is not a simple word",
	  WARN, &fpos(tag), SymName(symb(cs)));
	debug1(DCR, DD, "  tag = %s", EchoObject(tag));
      }
      else if( StringEqual(string(tag), STR_EMPTY) )
      {
        debug1(DCR, DD, "  ignoring cross (foll) %s (empty tag)", SymName(sym));
      }
      else
      { Link(cs, tag);
	gall_rec(tag) = FALSE;
        debug3(DCR, DD, "  storing cross (foll) %s&%s %s", SymName(sym),
	    string(tag), "?");
      }
      break;


    case CROSS_TARG:

      /* get rid of old target, if any, and add new one */
      if( target_state(cs) == SEEN_TARGET )
      {
	debug2(DCR, D, "  disposing unused %s cross_targ %s", SymName(sym),
	  target_val(cs));
	DisposeObject(target_val(cs));
      }
      debug2(DCR, D, "  remembering new %s cross_targ %s", SymName(sym),
	EchoObject(tag));
      target_val(cs) = tag;
      assert( Up(tag) == tag, "CrossSeq: Up(tag)!" );
      str = FileName(file_num(fpos(tag)));
      if( StringEndsWith(str, DATA_SUFFIX) )
	target_file(cs) = file_num(fpos(tag));
      else
      { target_file(cs) = FileNum(str, DATA_SUFFIX);
        if( target_file(cs) == NO_FILE )
	  target_file(cs) = DefineFile(str, DATA_SUFFIX, &fpos(tag),
					DATABASE_FILE, SOURCE_PATH);
      }
      target_state(cs) = SEEN_TARGET;

      /* store tag of the galley, if any */
      tag = nilobj;
      assert( type(target_val(cs)) == CLOSURE, "CrossSequence: target_val!" );
      link = Down(target_val(cs));
      for( ;  link != target_val(cs);  link = NextDown(link) )
      {	Child(par, link);
	if( type(par) == PAR && is_tag(actual(par)) )
	{ assert( Down(par) != par, "CrossSequence: Down(PAR)!" );
	  Child(tag, Down(par));
	  tag = ReplaceWithTidy(tag, TRUE);  /* && */
	  if( !is_word(type(tag)) )
	  { Error(10, 18, "tag of %s is not a simple word",
	      WARN, &fpos(tag), SymName(actual(target_val(cs))));
	    debug1(DCR, DD, "  tag = %s", EchoObject(tag));
	  }
	  else if( StringEqual(string(tag), STR_EMPTY) )
	  {
            debug1(DCR, DD, "  ignoring cross (own tag) %s (empty tag)",
		SymName(sym));
	  }
	  else
	  { Link(cs, tag);
	    gall_rec(tag) = FALSE;
            debug3(DCR, DD, "  storing cross (own tag) %s&%s %s", SymName(sym),
		string(tag), "?");
	  }
	  break;
	}
      }

      /* if new target is already writable, write it */
      if( Down(cs) != cs )
      {
	debug2(DCR, D, "  writing %s cross_targ %s", SymName(sym),
		EchoObject(target_val(cs)));
	AppendToFile(target_val(cs), target_file(cs), &target_pos(cs));
	DisposeObject(target_val(cs));
	for( link = Down(cs);  link != cs;  link = NextDown(link) )
	{ Child(tag, link);
	  assert( is_word(type(tag)) && !StringEqual(string(tag), STR_EMPTY),
			"CrossSeq: non-WORD or empty tag!" );
	  if( !gall_rec(tag) )
	  {
	    debug3(DCR, DD, "  inserting cross (foll) %s&%s %s", SymName(sym),
	      string(tag), "0");
	    DbInsert(NewCrossDb, FALSE, sym, string(tag), &fpos(tag), 
	      STR_ZERO, target_file(cs), (long) target_pos(cs), TRUE);
	    link = PrevDown(link);
	    DisposeChild(NextDown(link));
	  }
	}
	target_state(cs) = WRITTEN_TARGET;
      }
      break;


    default:

      Error(10, 19, "CrossSequence: %s", INTERN, &fpos(tag), Image(ctype));
      break;

  } /* end switch */
  debug0(DCR, D, "CrossSequence returning.");
  debug0(DCR, DD, "   cs =");
  ifdebug(DCR, DD, DebugObject(cs));
} /* end CrossSequence */


/*@::CrossClose()@************************************************************/
/*                                                                           */
/*  CrossClose()                                                             */
/*                                                                           */
/*  Check for dangling forward references, and convert old cross reference   */
/*  database to new one.                                                     */
/*                                                                           */
/*****************************************************************************/

void CrossClose(void)
{ OBJECT link, cs, ylink, y, sym;  BOOLEAN g;  int len, count;
  FILE_NUM dfnum;  long dfpos, cont;
  FULL_CHAR buff[MAX_BUFF], seq[MAX_BUFF], tag[MAX_BUFF];
  debug0(DCR, D, "CrossClose()");
  ifdebug(DCR, DD, if( RootCross != nilobj ) DebugObject(RootCross));

  /* if suppressing cross referencing, return */
  if( !AllowCrossDb )
  { debug0(DCR, D, "CrossClose returning (!AllowCrossDb).");
    return;
  }

  /* check for dangling forward references and dispose cross ref structures */
  if( RootCross != nilobj )
  { for( link = Down(RootCross);  link != RootCross;  link = NextDown(link) )
    { Child(cs, link);
      assert( type(cs) == CROSS_SYM, "CrossClose: type(cs)!" );
      count = 0;  ylink = Down(cs);
      while( ylink != cs && count <= 5 )
      {	Child(y, ylink);
	Error(10, 20, "no %s follows this %s%s%s", WARN, &fpos(y),
	  SymName(symb(cs)), SymName(symb(cs)), KW_CROSS, KW_FOLLOWING);
	debug2(DCR, D, "gall_rec(y) = %s, y = %s",
	  bool(gall_rec(y)), EchoObject(y));
	if( gall_rec(y) )
	  DbInsert(NewCrossDb, TRUE, symb(cs), STR_NONE, no_fpos,
	    string(y), file_num(fpos(y)), (long) gall_pos(y), FALSE);
	count++;  ylink = NextDown(ylink);
      }
      if( ylink != cs )
	Error(10, 21, "and more undefined %s%s%s", WARN, no_fpos,
	  SymName(symb(cs)), KW_CROSS, KW_FOLLOWING);
      ifdebug(ANY, D,
	if( target_state(cs) == SEEN_TARGET )  DisposeObject(target_val(cs));
	if( gall_tag(cs) != nilobj )  DisposeObject(gall_tag(cs));
      );
    }
    ifdebug(ANY, D, DisposeObject(RootCross); );
  }

  /* add to NewCrossDb those entries of OldCrossDb from other source files */
  /* but set check to FALSE so that we don't worry about duplication there */
  cont = 0L;  len = StringLength(DATA_SUFFIX);
  while( DbRetrieveNext(OldCrossDb, &g, &sym, tag, seq, &dfnum, &dfpos, &cont) )
  { if( g ) continue;
    StringCopy(buff, FileName(dfnum));
    StringCopy(&buff[StringLength(buff) - len], STR_EMPTY);
    if( FileNum(buff, STR_EMPTY) == NO_FILE )
      DbInsert(NewCrossDb, FALSE, sym, tag, no_fpos, seq, dfnum, dfpos, FALSE);
  }

  /* close OldCrossDb's .li file so that NewCrossDb can use its name */
  DbClose(OldCrossDb);

  /* make NewCrossDb readable, for next run */
  DbConvert(NewCrossDb, TRUE);

  debug0(DCR, D, "CrossClose returning.");
  ifdebug(DCR, DD, crtab_debug(crossref_tab, stderr));
} /* end CrossClose */
