/*
 * C general utilities library.
 * qsort()
 * ANSI 4.10.5.2.
 * Quicksort.
 * Algorithm from C. A. R. Hoare, cf. Knuth Vol. 3, pp. 114 ff.
 */

/*
 * qsort chooses a pivot element (Knuth recommends random choice),
 * partitions the input into sets less than and greater than the pivot,
 * puts the pivot into the correct slot in the sorted set,
 * and recursively sorts the two subpartitions.
 * For small numbers of elements, a linear insertion sort is more efficient.
 * The obvious recursive implementation of qsort can easily use a lot of
 * stack space.  The nonrecursive implementation here is less obvious
 * but more efficient.  It is based on two observations:
 * (1) Using ordinary recursion on the smaller of the subpartitions and
 * tail recursion on the larger reduces recursion nesting to at most
 * log2(nmemb) levels.  This reduces stack usage enormously.
 * (2) Since the maximum number of recursion levels is known by (1),
 * the recursion can be faked with auto arrays of fixed size.
 * This provides additional efficiency because the constant
 * "size" and "compar" args need not be passed recursively.
 */

#include <stdlib.h>
#include <limits.h>
#include <string.h>

#define	SSIZE	(CHAR_BIT * sizeof(size_t))	/* max recursion stack size */
typedef	struct	{
	Void *	s_base;
	size_t	s_nmemb;
} STACK;

/* Use quicksort for M or more elements, linear insertion sort for fewer. */
/* The value is arrived at empirically and may differ for other processors. */
#define	M	8

void
qsort(base, nmemb, size, compar)
register Void *base;
size_t nmemb, size;
int (*compar)();
{
	register char *bot, *top;
	register size_t n;
	STACK stack[SSIZE], *sp;

	/* Initialize the stack of base and nmemb args to be sorted. */
	sp = &stack[0];
	sp->s_base = base;
	sp->s_nmemb = nmemb;

	while (sp >= &stack[0]) {

		/* Pop a base and nmemb pair from the stack. */
		base = sp->s_base;
		nmemb = sp->s_nmemb;
		sp--;

		while (nmemb >= M) {

			bot = (char *)base;
			top = bot + nmemb * size;

			/* Put middle element into *base.  Helps on sorted input. */
			_memxchg(bot, bot + (nmemb/2)*size, size);

			/* Partition into sets less than and greater than *base. */
			for (;;) {
				while ((*compar)((Void *)(top -= size), base)>=0 && top > base)
					;
				while ((*compar)(base, (Void *)(bot += size))>=0 && bot < top)
					;
				if (bot < top)
					_memxchg(bot, top, size);
				else
					break;
			}

			/* Put *base into its correct place. */
			_memxchg(top, (char *)base, size);

			/*
			 * "Recusively" sort partitions.
			 * Fake recursion on local stack for the larger,
			 * tail recursion for the smaller.
			 */
			n = (top - (char *)base)/size;
			nmemb -= (bot - (char *)base) / size;
			if (n > nmemb) {
				/* qsort(base, n, size, compar); */
				sp++;
				sp->s_base = base;
				sp->s_nmemb = n;
				/* qsort(bot, nmemb, size, compar); */
				base = (Void *)bot;
			} else {
				/* qsort(bot, nmemb, size, compar); */
				sp++;
				sp->s_base = bot;
				sp->s_nmemb = nmemb;
				/* qsort(base, n, size, compar); */
				nmemb = n;
			}
		}

		/* Use linear insertion sort for less than M elements. */
		while (nmemb-- > 1) {
			top = bot = (char *)base;
			for (n = nmemb; n--; )
				if ((*compar)((Void *)(top += size), (Void *)bot) < 0)
					bot = top;
			if (bot != (char *)base)
				_memxchg(bot, (char *)base, size);
			base = (Void *)(((char *)base) + size);
		}
	}
}
