#
# tech follow-up (tfup).  Record tech encounters, and their resolution.
#
GLOBALS "globals0.4gl"

#
# some static thingies to hold display data
#
DEFINE t_customer RECORD LIKE customer.*
DEFINE t_address RECORD LIKE address.*
DEFINE t_tech_fup RECORD LIKE tech_fup.*
DEFINE t_shipreg RECORD LIKE shipreg.*
DEFINE t_email RECORD LIKE email.*
DEFINE t_hardware RECORD LIKE hardware.*

MAIN
	DEFER INTERRUPT

	CALL get_usr()
		RETURNING user_id

	IF user_id = "unknown" THEN
		DISPLAY "Cannot find person using this program"
		EXIT PROGRAM
	END IF

	CALL tfup_menu()
END MAIN

#
# invoice menu, asks add, find, delete, modify and report
#
FUNCTION tfup_menu()
	OPEN FORM tech FROM "tech"

	DISPLAY FORM tech

	CALL blank_fields()

	MESSAGE ""

	MENU "TECH"
		COMMAND "Customer" "Select a Customer"
			CALL select_customer()

		COMMAND "Add Customer" "Add a new customer"
			CALL add_customer()

		COMMAND "Fix Address" "Update Customer's Address & Phone"
			IF t_customer.cust_no IS NULL THEN
				ERROR "You must select a customer first"
				CONTINUE MENU
			END IF
			CALL update_address()

		COMMAND "Hardware" "Add or Update Customer's Hardware"
			IF t_customer.cust_no IS NULL THEN
				ERROR "You must select a customer first"
				CONTINUE MENU
			END IF
			CALL add_hardware()

		COMMAND "New" "Add a New Problem"
			IF t_customer.cust_no IS NULL THEN
				ERROR "You must select a customer first"
				CONTINUE MENU
			END IF
			CALL add_problem()

		COMMAND "Update" "Update the Current Problem"
			IF t_customer.cust_no IS NULL THEN
				ERROR "You must select a customer first"
				CONTINUE MENU
			END IF
			CALL update_problem()

		COMMAND "Exit" ""
			EXIT MENU
	END MENU
	CLOSE FORM tech
END FUNCTION

#
# Get customer record
#
FUNCTION select_customer()
DEFINE crunch LIKE customer.crunch
DEFINE answer CHAR(1)
DEFINE counter INTEGER

	LET crunch = NULL

	CALL blank_fields()

	INPUT BY NAME	t_shipreg.ser_no,
			t_customer.cust_no,
			t_customer.contact,
			t_tech_fup.tf_serial_id

	ON KEY (INTERRUPT)
		LET int_flag = 0
		CALL blank_fields()
		EXIT INPUT

	BEFORE FIELD ser_no
		MESSAGE "Enter the serial number of this copy of Coherent"

	AFTER FIELD ser_no
		IF t_shipreg.ser_no IS NULL THEN
			NEXT FIELD cust_no
		END IF

		SELECT customer.*
			INTO	t_customer.*
			FROM	shipreg, customer
			WHERE	shipreg.ser_no = t_shipreg.ser_no
					AND
				shipreg.cust_no = customer.cust_no

		IF (status = NOTFOUND) THEN
			ERROR "No copy of Coherent has this number"
			CALL blank_fields()
			NEXT FIELD ser_no
		END IF

		CALL display_customer()
		CALL display_address()

		EXIT INPUT

	BEFORE FIELD cust_no
		MESSAGE "Enter customer number of customer"

	AFTER FIELD cust_no
 		IF t_customer.cust_no IS NULL THEN
			NEXT FIELD contact
		END IF

		SELECT *
			INTO	t_customer.*
			FROM	customer
			WHERE	customer.cust_no = t_customer.cust_no

		IF (status = NOTFOUND) THEN
			ERROR "No customer has this number"
			CALL blank_fields()
			NEXT FIELD cust_no
		END IF

		CALL display_customer()
		CALL display_address()

		EXIT INPUT

	BEFORE FIELD contact
		MESSAGE "Enter customer's name"

	AFTER FIELD contact
		IF t_customer.contact IS NULL THEN
			NEXT FIELD tf_serial_id
		END IF

		CALL cruncher (t_customer.contact)
			RETURNING crunch

		SELECT count(*)
			INTO counter
			FROM customer
			WHERE customer.crunch = crunch

		IF counter = 0 THEN
			MESSAGE "No customer by this name"
			CALL blank_fields()
			NEXT FIELD contact
		END IF

		IF counter = 1 THEN
			SELECT	*
				INTO 	t_customer.*
				FROM	customer
				WHERE	customer.crunch = crunch

			CALL display_customer()
			CALL display_address()

			EXIT INPUT
		END IF

# if we get this far, more than one person has this "crunch" code
		DECLARE contact_curs SCROLL CURSOR FOR
			SELECT cust_no, contact, phone_no
				FROM customer
				WHERE customer.crunch = crunch

		OPEN contact_curs

		FETCH FIRST contact_curs
			INTO	t_customer.cust_no,
				t_customer.contact,
				t_customer.phone_no

		CALL display_customer()
		CALL display_address()

		MENU "SELECT"
			COMMAND "Select" "Select This Customer"
				CLOSE contact_curs
				EXIT INPUT

			COMMAND "Next" "View Next Customer"
				FETCH NEXT contact_curs INTO
					t_customer.cust_no,
					t_customer.contact,
					t_customer.phone_no
				IF (status = NOTFOUND) THEN
					CALL msg ("No more in this direction")
				ELSE
					CALL display_customer()
					CALL display_address()
				END IF

			COMMAND "Previous" "View Previous Customer"
				FETCH PREVIOUS contact_curs INTO
					t_customer.cust_no,
					t_customer.contact,
					t_customer.phone_no
				IF (status = NOTFOUND) THEN
					CALL msg("No more in this direction")
				ELSE
					CALL display_customer()
					CALL display_address()
				END IF

			COMMAND "Abort" "Abort this name; try again"
				CALL blank_fields()
				EXIT MENU
		END MENU
		CLOSE contact_curs
		MESSAGE "More than one person has this name"
		NEXT FIELD contact

	BEFORE FIELD tf_serial_id
		MESSAGE "Enter problem number"

	AFTER FIELD tf_serial_id
		IF t_tech_fup.tf_serial_id IS NULL THEN
			NEXT FIELD ser_no
		END IF

		SELECT	*
			INTO	t_tech_fup.*
			FROM	tech_fup
			WHERE	tech_fup.tf_serial_id = t_tech_fup.tf_serial_id

		IF status = NOTFOUND THEN
			ERROR "No problem has this number"
			CALL blank_fields()
			NEXT FIELD tf_serial_id
		END IF

# a problem can never refer to a non-existent customer
		SELECT *
			INTO	t_customer.*
			FROM	customer
			WHERE	customer.cust_no = t_tech_fup.tf_cust_no

		CALL display_customer()
		CALL display_address()
		CALL display_problem()
		CALL display_hardware()
		EXIT INPUT

# deflect operation of the ESC key
	AFTER INPUT
		NEXT FIELD cust_no

	END INPUT
END FUNCTION

#
# Add a new customer
#
FUNCTION add_customer()
DEFINE answer CHAR(1)

	CALL blank_fields()

	INPUT BY NAME	t_customer.contact,
			t_address.line1,
			t_address.line2,
			t_address.line3,
			t_address.city,
			t_address.state,
			t_address.zipcode,
			t_address.country,
			t_customer.phone_no,
			t_email.em_address

	ON KEY (INTERRUPT)
		LET int_flag = 0
		CALL blank_fields()
		EXIT INPUT

	BEFORE FIELD contact
		MESSAGE "Enter customer's name"

	AFTER FIELD contact
		IF t_customer.contact IS NULL THEN
			ERROR "You must enter the customer's name"
			NEXT FIELD contact
		END IF

		CALL cruncher (t_customer.contact)
			RETURNING t_customer.crunch
		NEXT FIELD line1

	BEFORE FIELD line1
		MESSAGE "Enter first line of customer's address"

	AFTER FIELD line1
		IF t_address.line1 IS NULL THEN
			ERROR "You must enter the customer's street address"
			NEXT FIELD line1
		END IF
		NEXT FIELD line2

	BEFORE FIELD line2
		MESSAGE "Enter second line of customer's address"

	AFTER FIELD line2
		NEXT FIELD line3

	BEFORE FIELD line3
		MESSAGE "Enter third line of customer's address"

	AFTER FIELD line3
		NEXT FIELD city

	BEFORE FIELD city
		MESSAGE "Enter customer's city"

	AFTER FIELD city
		IF t_address.city IS NULL THEN
			ERROR "You must enter the customer's city"
			NEXT FIELD city
		END IF
		NEXT FIELD state

	BEFORE FIELD state
		MESSAGE "Enter customer's state"

	AFTER FIELD state
		IF t_address.state IS NULL THEN
			ERROR "You must enter the customer's state"
			NEXT FIELD state
		END IF

	BEFORE FIELD zipcode
		MESSAGE "Enter customer's ZIP or postal code"

	AFTER FIELD zipcode
		IF t_address.zipcode IS NULL THEN
			ERROR "You must enter the customer's ZIP or postal code"
			NEXT FIELD zipcode
		END IF

		NEXT FIELD country

	BEFORE FIELD country
		MESSAGE "Enter customer's country; press <return> if USA"

	AFTER FIELD country
		NEXT FIELD phone_no

	BEFORE FIELD phone_no
		MESSAGE "Enter customer's phone number"

	AFTER FIELD phone_no
		IF t_customer.phone_no IS NULL THEN
			ERROR "You must enter the customer's phone number"
			NEXT FIELD phone_no
		END IF

		NEXT FIELD em_address

	AFTER INPUT
		IF t_customer.contact IS NULL THEN
			ERROR "You must enter the customer's name"
			NEXT FIELD contact
		END IF

		IF t_address.line1 IS NULL THEN
			ERROR "You must enter the customer's street address"
			NEXT FIELD line1
		END IF

		IF t_address.city IS NULL THEN
			ERROR "You must enter the customer's city"
			NEXT FIELD city
		END IF

		IF t_address.state IS NULL THEN
			ERROR "You must enter the customer's state"
			NEXT FIELD state
		END IF

		IF t_address.zipcode IS NULL THEN
			ERROR "You must enter the customer's ZIP or postal code"
			NEXT FIELD zipcode
		END IF

		IF t_customer.phone_no IS NULL THEN
			ERROR "You must enter the customer's phone number"
			NEXT FIELD phone_no
		END IF

		LET t_customer.cust_no = 0
		LET t_customer.all_nfo = "y"
		LET t_customer.user_id = user_id
		LET t_customer.cust_date = TODAY

		INSERT INTO customer VALUES (t_customer.*)

		IF sqlca.sqlcode < 0 THEN
			ERROR "Error: Could not insert customer record into data base"
			EXIT INPUT
		END IF

		LET t_customer.cust_no = sqlca.sqlerrd[2]
		DISPLAY BY NAME t_customer.cust_no

		LET t_address.cust_no = t_customer.cust_no
		LET t_address.add_type = "x"

		INSERT INTO address VALUES (t_address.*)

		IF sqlca.sqlcode < 0 THEN
			ERROR "Error: Could not insert address record into data base"
			EXIT INPUT
		END IF

		IF t_email.em_address IS NOT NULL THEN
			LET t_email.em_cust_no = t_customer.cust_no
			INSERT INTO email VALUES (t_email.*)

			IF sqlca.sqlcode < 0 THEN
				ERROR "Error: Could not insert email record into data base"
				EXIT INPUT
			END IF
		END IF

		CALL add_hardware()

		EXIT INPUT
	END INPUT
END FUNCTION

#
# Add information about a customer's hardware
#
FUNCTION add_hardware()
DEFINE answer CHAR(1)

	INPUT BY NAME
		t_hardware.hw_processor,
		t_hardware.hw_sx_dx,
		t_hardware.hw_coproc,
		t_hardware.hw_proc_speed,
		t_hardware.hw_ram,
		t_hardware.hw_video_card,
		t_hardware.hw_video_ram,
		t_hardware.hw_hd,
		t_hardware.hw_hd_space

	ON KEY (INTERRUPT)
		LET int_flag = 0
		CALL blank_hardware()
		EXIT INPUT

	BEFORE FIELD hw_processor
		MESSAGE "Enter the processor (2, 3, 4, or P)"

	AFTER FIELD hw_processor
		IF t_hardware.hw_processor IS NULL THEN
			NEXT FIELD hw_processor
		END IF

		IF	t_hardware.hw_processor != "2" AND
			t_hardware.hw_processor != "3" AND
			t_hardware.hw_processor != "4" AND
			t_hardware.hw_processor != "P" THEN

			ERROR "Enter 2, 3, 4, or P"
			NEXT FIELD hw_processor
		END IF

		IF t_hardware.hw_processor = "P" THEN
			LET t_hardware.hw_coproc = "Y"
			DISPLAY BY NAME t_hardware.hw_coproc
			NEXT FIELD hw_proc_speed
		END IF

		NEXT FIELD hw_sx_dx	

	BEFORE FIELD hw_sx_dx
		MESSAGE "Enter S for SX, or D for DX"

	AFTER FIELD hw_sx_dx
		IF	t_hardware.hw_sx_dx != "S" AND
			t_hardware.hw_sx_dx != "D" THEN
			ERROR "Enter S or D"
			NEXT FIELD hw_sx_dx
		END IF

		IF t_hardware.hw_processor = "4"
				AND t_hardware.hw_sx_dx = "D" THEN
			LET t_hardware.hw_coproc = "Y"
			DISPLAY BY NAME t_hardware.hw_coproc
			NEXT FIELD hw_proc_speed
		END IF
		NEXT FIELD hw_coproc

	BEFORE FIELD hw_coproc
		MESSAGE "Does user have math coprocessor (Y/N)?"

	AFTER FIELD hw_coproc
		IF t_hardware.hw_coproc IS NULL THEN
			NEXT FIELD hw_coproc
		END IF

		IF	t_hardware.hw_coproc != "Y" AND
			t_hardware.hw_coproc != "N" THEN
			ERROR "Enter Y or N"
			NEXT FIELD hw_coproc
		END IF
		NEXT FIELD hw_proc_speed

	BEFORE FIELD hw_proc_speed
		MESSAGE "Enter processor speed"

	AFTER FIELD hw_proc_speed
		IF t_hardware.hw_proc_speed IS NULL THEN
			NEXT FIELD hw_proc_speed
		END IF
		NEXT FIELD hw_ram

	BEFORE FIELD hw_ram
		MESSAGE "Enter RAM, in megabytes"

	AFTER FIELD hw_ram
		IF t_hardware.hw_ram IS NULL THEN
			NEXT FIELD hw_ram
		END IF

		NEXT FIELD hw_video_card

	BEFORE FIELD hw_video_card
		MESSAGE "Enter user's video chipset"

	AFTER FIELD hw_video_card
		IF t_hardware.hw_video_card IS NULL THEN
			NEXT FIELD hw_video_card
		END IF

		NEXT FIELD hw_video_ram

	BEFORE FIELD hw_video_ram
		MESSAGE "Enter video RAM, in megabytes"

	AFTER FIELD hw_video_ram
		IF t_hardware.hw_video_ram IS NULL THEN
			NEXT FIELD hw_video_ram
		END IF

		NEXT FIELD hw_hd

	BEFORE FIELD hw_hd
		MESSAGE "A=AT, S=SCSI, B=both"

	AFTER FIELD hw_hd
		IF t_hardware.hw_hd IS NULL THEN
			NEXT FIELD hw_hd
		END IF

		IF	t_hardware.hw_hd != "A" AND
			t_hardware.hw_hd != "S" AND
			t_hardware.hw_hd != "B" THEN
			ERROR "Enter A, S, or B"
			NEXT FIELD hw_hd
		END IF

		NEXT FIELD hw_hd_space

	BEFORE FIELD hw_hd_space
		MESSAGE "Enter customer's total disk space, in megabytes"

	AFTER FIELD hw_hd_space
		IF t_hardware.hw_hd_space IS NULL THEN
			NEXT FIELD hw_hd_space
		END IF

	AFTER INPUT
		IF t_hardware.hw_processor IS NULL THEN
			ERROR "You must identify the processor"
			NEXT FIELD hw_processor
		END IF

		IF t_hardware.hw_coproc IS NULL THEN
			ERROR "You must enter whether user has co-processor"
			NEXT FIELD hw_coproc
		END IF

		IF t_hardware.hw_proc_speed IS NULL THEN
			ERROR "You must enter processor speed"
			NEXT FIELD hw_proc_speed
		END IF

		IF t_hardware.hw_ram IS NULL THEN
			ERROR "You must enter the amount of RAM"
			NEXT FIELD hw_ram
		END IF

		IF t_hardware.hw_video_card IS NULL THEN
			ERROR "You must enter type of video card"
			NEXT FIELD hw_video_card
		END IF

		IF t_hardware.hw_video_ram IS NULL THEN
			ERROR "You must enter amount of video RAM"
			NEXT FIELD hw_video_ram
		END IF

		IF t_hardware.hw_hd IS NULL THEN
			ERROR "You must enter the type of hard disk"
			NEXT FIELD hw_hd
		END IF

		IF t_hardware.hw_hd_space IS NULL THEN
			ERROR "You must enter the amount of hard-disk space"
			NEXT FIELD hw_hd_space
		END IF

		LET t_hardware.hw_cust_no = t_customer.cust_no

		INSERT INTO hardware VALUES (t_hardware.*)

		IF sqlca.sqlcode < 0 THEN
			ERROR "Error: Could not insert hardware record into data base"
		END IF
		EXIT INPUT
	END INPUT
END FUNCTION

#
# add a problem to this customer's problem list
#
FUNCTION add_problem()
	DEFINE answer CHAR(1)

	CALL blank_problem()
	CALL display_problem()

	LET t_tech_fup.tf_call_date = TODAY
	LET t_tech_fup.tf_user_id = user_id
	LET t_tech_fup.tf_cust_no = t_customer.cust_no

# find the next serial id, so support person can tell customer right away
	SELECT max(tf_serial_id)
		INTO	t_tech_fup.tf_serial_id
		FROM	tech_fup

	IF t_tech_fup.tf_serial_id IS NULL THEN
		LET t_tech_fup.tf_serial_id = 0
	END IF

	LET t_tech_fup.tf_serial_id = t_tech_fup.tf_serial_id + 1

	DISPLAY BY NAME
		t_tech_fup.tf_serial_id,
		t_tech_fup.tf_call_date,
		t_tech_fup.tf_user_id

	INPUT BY NAME
		t_tech_fup.tf_problem1,
		t_tech_fup.tf_problem2,
		t_tech_fup.tf_problem3,
		t_tech_fup.tf_solution1,
		t_tech_fup.tf_solution2,
		t_tech_fup.tf_solution3

	ON KEY (INTERRUPT)
		LET int_flag = 0
		CALL blank_problem()
		CALL display_problem()
		EXIT INPUT

	AFTER FIELD tf_solution3
		NEXT FIELD tf_problem1

	AFTER INPUT
		INSERT INTO tech_fup VALUES (t_tech_fup.*)
		MESSAGE "Problem recorded"
	END INPUT
END FUNCTION

#
# Update the current problem
#
FUNCTION update_problem()
	DEFINE temp_tech_fup RECORD LIKE tech_fup.*
	DEFINE answer CHAR(1)

	IF t_tech_fup.tf_serial_id IS NULL THEN
		CALL select_problem()
	ELSE
		CALL yn("Update this problem? [y/n]:")
			RETURNING answer
		IF answer = "n" THEN
			CALL select_problem()
		END IF
	END IF 

	IF t_tech_fup.tf_serial_id IS NULL THEN
		RETURN
	END IF

	LET temp_tech_fup.* = t_tech_fup.*

	INPUT BY NAME
		temp_tech_fup.tf_problem1,
		temp_tech_fup.tf_problem2,
		temp_tech_fup.tf_problem3,
		temp_tech_fup.tf_solution1,
		temp_tech_fup.tf_solution2,
		temp_tech_fup.tf_solution3
		WITHOUT DEFAULTS

	ON KEY (INTERRUPT)
		LET int_flag = 0
		CALL display_problem()
		EXIT INPUT

	AFTER FIELD tf_solution3
		NEXT FIELD tf_problem1

	AFTER INPUT
		LET t_tech_fup.* = temp_tech_fup.*

		UPDATE tech_fup
			SET	tf_problem1 = t_tech_fup.tf_problem1,
				tf_problem2 = t_tech_fup.tf_problem2,
				tf_problem3 = t_tech_fup.tf_problem3,
				tf_solution1 = t_tech_fup.tf_solution1,
				tf_solution2 = t_tech_fup.tf_solution2,
				tf_solution3 = t_tech_fup.tf_solution3
			WHERE	tf_serial_id = t_tech_fup.tf_serial_id
		MESSAGE "Problem updated"
	END INPUT
END FUNCTION

#
# Update the customer's address and phone number
#
FUNCTION update_address()
DEFINE tmp_address RECORD LIKE address.*
DEFINE tmp_phone LIKE customer.phone_no
DEFINE tmp_email LIKE email.em_address
DEFINE answer CHAR(1)
DEFINE update_flag CHAR(1)
	LET tmp_address.* = t_address.*
	LET tmp_phone = t_customer.phone_no
	LET tmp_email = t_email.em_address

	IF t_email.em_cust_no IS NOT NULL THEN
		LET update_flag = "Y"
	ELSE
		LET update_flag = "N"
	END IF

	DISPLAY BY NAME
		tmp_address.line1,
		tmp_address.line2,
		tmp_address.line3,
		tmp_address.city,
		tmp_address.state,
		tmp_address.zipcode,
		tmp_address.country
	DISPLAY tmp_phone TO phone_no
	DISPLAY tmp_email TO em_address

	INPUT	tmp_address.line1,
		tmp_address.line2,
		tmp_address.line3,
		tmp_address.city,
		tmp_address.state,
		tmp_address.zipcode,
		tmp_address.country,
		tmp_phone,
		tmp_email
	 WITHOUT DEFAULTS FROM
		line1,
		line2,
		line3,
		city,
		state,
		zipcode,
		country,
		phone_no,
		em_address

	ON KEY (INTERRUPT)
		LET int_flag = 0
		DISPLAY BY NAME
			t_address.line1,
			t_address.line2,
			t_address.line3,
			t_address.city,
			t_address.state,
			t_address.zipcode,
			t_address.country,
			t_customer.phone_no,
			t_email.em_address
		EXIT INPUT

	AFTER INPUT
		LET t_address.* = tmp_address.*
		LET t_customer.phone_no = tmp_phone
		LET t_email.em_address = tmp_email
		LET t_email.em_cust_no = t_customer.cust_no

		UPDATE address
			SET	line1 = tmp_address.line1,
				line2 = tmp_address.line2,
				line3 = tmp_address.line3,
				city = tmp_address.city,
				state = tmp_address.state,
				zipcode = tmp_address.zipcode,
				country = tmp_address.country
			WHERE	cust_no = tmp_address.cust_no

		UPDATE customer
			SET phone_no = tmp_phone
			WHERE cust_no = t_customer.cust_no

		IF update_flag = "Y" THEN
			UPDATE email
				SET em_address = tmp_email
				WHERE cust_no = t_customer.cust_no
		ELSE
			INSERT INTO email VALUES (t_email.*)
		END IF

		EXIT INPUT
	END INPUT
END FUNCTION

#
# Select a problem
#
FUNCTION select_problem()
DEFINE counter INTEGER
	SELECT count(*)
		INTO	counter
		FROM	tech_fup
		WHERE	tf_cust_no = t_customer.cust_no

	IF counter = 0 THEN
		ERROR "Customer has no recorded problems"
		RETURN
	END IF

	IF counter = 1 THEN
		SELECT *
			FROM	tech_fup
			WHERE	tf_cust_no = t_customer.cust_no

		CALL display_problem()
		RETURN
	END IF

	DECLARE problem_curs SCROLL CURSOR FOR
		SELECT *
			FROM	tech_fup
			WHERE	tf_cust_no = t_customer.cust_no

	OPEN problem_curs

	FETCH FIRST problem_curs
		INTO t_tech_fup.*

	CALL display_problem()

	MENU "SELECT"
		COMMAND "Select" "Select This Problem"
			CLOSE problem_curs
			EXIT MENU

		COMMAND "Next" "View Next Problem"
			FETCH NEXT problem_curs INTO
				t_tech_fup.*
			IF (status = NOTFOUND) THEN
				CALL msg ("No more in this direction")
			ELSE
				CALL display_problem()
			END IF

		COMMAND "Previous" "View Previous Problem"
			FETCH PREVIOUS problem_curs INTO
				t_tech_fup.*
			IF (status = NOTFOUND) THEN
				CALL msg("No more in this direction")
			ELSE
				CALL display_problem()
			END IF

		COMMAND "Abort"
			CALL blank_problem()
			EXIT MENU
	END MENU
	CLOSE problem_curs
END FUNCTION

#
# display customer information
#
FUNCTION display_customer()
	DISPLAY t_customer.cust_no,
		t_customer.contact,
		t_customer.phone_no
	TO	cust_no, contact, phone_no
END FUNCTION

#
# fetch user's home & email addresses, and display
#
FUNCTION display_address()
	SELECT *
		INTO	t_address.*
		FROM	address
		WHERE	address.cust_no = t_customer.cust_no
				AND
			(address.add_type = "b"
				OR
			 address.add_type = "x")

	IF status = NOTFOUND THEN
		RETURN
	END IF

	DISPLAY BY NAME
		t_address.line1,
		t_address.line2,
		t_address.line3,
		t_address.city,
		t_address.state,
		t_address.zipcode,
		t_address.country

	SELECT *
		INTO	t_email.*
		FROM	email
		WHERE	email.em_cust_no = t_customer.cust_no

	IF t_email.em_address IS NOT NULL THEN
		DISPLAY BY NAME t_email.em_address
	END IF
END FUNCTION

#
# display a problem.  Note that unlike the other "display*" functions,
# this does not select the problem - problems will always be selected
# elsewhere.
#
FUNCTION display_problem()
	IF t_tech_fup.tf_serial_id IS NULL THEN
		RETURN
	END IF

	DISPLAY BY NAME
		t_tech_fup.tf_serial_id,
		t_tech_fup.tf_call_date,
		t_tech_fup.tf_user_id,
		t_tech_fup.tf_problem1,
		t_tech_fup.tf_problem2,
		t_tech_fup.tf_problem3,
		t_tech_fup.tf_solution1,
		t_tech_fup.tf_solution2,
		t_tech_fup.tf_solution3
END FUNCTION

#
# select and display a customer's hardware configuration
#
FUNCTION display_hardware()
	SELECT *
		INTO	t_hardware.*
		FROM	hardware
		WHERE	hardware.hw_cust_no = t_customer.cust_no

	IF t_hardware.hw_cust_no IS NOT NULL THEN
		DISPLAY BY NAME
			t_hardware.hw_processor,
			t_hardware.hw_coproc,
			t_hardware.hw_ram,
			t_hardware.hw_video_card,
			t_hardware.hw_video_ram,
			t_hardware.hw_hd,
			t_hardware.hw_hd_space
	END IF
END FUNCTION

#
# blank out all fields
#
FUNCTION blank_fields()
	CALL blank_ser_no()
	CALL blank_customer()
	CALL blank_address()
	CALL blank_problem()
	CALL blank_email()
	CALL blank_hardware()
END FUNCTION

#
# blank out the field that holds the Coherent serial-number
#
FUNCTION blank_ser_no()
	INITIALIZE t_shipreg.* TO NULL
	DISPLAY BY NAME t_shipreg.ser_no
END FUNCTION

#
# blank out customer fields
#
FUNCTION blank_customer()
	INITIALIZE t_customer.* TO NULL

	DISPLAY BY NAME
		t_customer.cust_no,
		t_customer.contact,
		t_customer.phone_no
END FUNCTION

#
# blank out address fields
#
FUNCTION blank_address()
	INITIALIZE t_address.* TO NULL

	DISPLAY BY NAME
		t_address.line1,
		t_address.line2,
		t_address.line3,
		t_address.city,
		t_address.state,
		t_address.zipcode,
		t_address.country
END FUNCTION

#
# blank out problem fields
#
FUNCTION blank_problem()
	INITIALIZE t_tech_fup.* TO NULL
	DISPLAY BY NAME
		t_tech_fup.tf_serial_id,
		t_tech_fup.tf_call_date,
		t_tech_fup.tf_user_id,
		t_tech_fup.tf_problem1,
		t_tech_fup.tf_problem2,
		t_tech_fup.tf_problem3,
		t_tech_fup.tf_solution1,
		t_tech_fup.tf_solution2,
		t_tech_fup.tf_solution3
END FUNCTION

#
# blank out email field
#
FUNCTION blank_email()
	INITIALIZE t_email.* TO NULL
	DISPLAY BY NAME t_email.em_address
END FUNCTION

#
# blank out hardware fields
#
FUNCTION blank_hardware()
	INITIALIZE t_hardware.* TO NULL
	DISPLAY BY NAME
		t_hardware.hw_processor,
		t_hardware.hw_coproc,
		t_hardware.hw_ram,
		t_hardware.hw_video_card,
		t_hardware.hw_video_ram,
		t_hardware.hw_hd,
		t_hardware.hw_hd_space
END FUNCTION

#
# Display a prompt and get a y or n
#
FUNCTION yn(msg)
DEFINE msg CHAR(50)
DEFINE ans CHAR(1)
	WHILE (1)
		PROMPT msg CLIPPED, " " FOR CHAR ans
		IF (ans = "n" OR ans = "N") THEN
			RETURN "n"
		END IF
		IF (ans = "y" OR ans = "Y") THEN
			RETURN "y"
		END IF
	END WHILE
END FUNCTION

#
# display a message briefly
#
FUNCTION msg (msg_txt)
DEFINE msg_txt CHAR(50)
	ERROR msg_txt CLIPPED
END FUNCTION

#
# end of tech.4gl
