/* Fork a Unix child process, and set up to debug it, for GDB.
   Copyright 1990, 1991, 1992 Free Software Foundation, Inc.
   Contributed by Cygnus Support.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "frame.h"  /* required by inferior.h */
#include "inferior.h"
#include "target.h"
#include "wait.h"
#include "gdbcore.h"
#include "terminal.h"		/* For #ifdef TIOCGPGRP and new_tty */

#include <signal.h>

#ifdef SET_STACK_LIMIT_HUGE
#include <sys/time.h>
#include <sys/resource.h>

extern int original_stack_limit;
#endif /* SET_STACK_LIMIT_HUGE */

extern char **environ;

/* Start an inferior Unix child process and sets inferior_pid to its pid.
   EXEC_FILE is the file to run.
   ALLARGS is a string containing the arguments to the program.
   ENV is the environment vector to pass.  Errors reported with error().  */

#ifndef SHELL_FILE
#define SHELL_FILE "/bin/sh"
#endif

/*
 * Under COHERENT we have to use execvpe(), this are the max number
 * of arguments for the inferior, should be enough for now
 */

#define MAX_ARGS	128

void
fork_inferior (exec_file, allargs, env, traceme_fun, init_trace_fun)
     char *exec_file;
     char *allargs;
     char **env;
     void (*traceme_fun) PARAMS ((void));
     void (*init_trace_fun) PARAMS ((int));
{
#ifndef COHERENT
  char *shell_command;
  char *shell_file;
  static char default_shell_file[] = SHELL_FILE;
  int len;
#else
  int i;
  char *argv[MAX_ARGS];
#endif
  int pid;
  int pending_execs;
  /* Set debug_fork then attach to the child while it sleeps, to debug. */
  static int debug_fork = 0;
  /* This is set to the result of setpgrp, which if vforked, will be visible
     to you in the parent process.  It's only used by humans for debugging.  */
  static int debug_setpgrp = 657473;
  char **save_our_env;

  /* If no exec file handed to us, get it from the exec-file command -- with
     a good, common error message if none is specified.  */
  if (exec_file == 0)
    exec_file = get_exec_file(1);

#ifndef COHERENT
  /* The user might want tilde-expansion, and in general probably wants
     the program to behave the same way as if run from
     his/her favorite shell.  So we let the shell run it for us.
     FIXME, this should probably search the local environment (as
     modified by the setenv command), not the env gdb inherited.  */
  shell_file = getenv ("SHELL");
  if (shell_file == NULL)
    shell_file = default_shell_file;
  
  len = 5 + strlen (exec_file) + 1 + strlen (allargs) + 1 + /*slop*/ 10;
  /* If desired, concat something onto the front of ALLARGS.
     SHELL_COMMAND is the result.  */
#ifdef SHELL_COMMAND_CONCAT
  shell_command = (char *) alloca (strlen (SHELL_COMMAND_CONCAT) + len);
  strcpy (shell_command, SHELL_COMMAND_CONCAT);
#else
  shell_command = (char *) alloca (len);
  shell_command[0] = '\0';
#endif
  strcat (shell_command, "exec ");
  strcat (shell_command, exec_file);
  strcat (shell_command, " ");
  strcat (shell_command, allargs);
#else
  /* Under COHERENT build the argument vector for execvpe() */
  argv[0] = exec_file;
  argv[1] = strtok (allargs, " ");
  for (i = 1; i < MAX_ARGS && argv[i] != NULL;)
    argv[++i] = strtok (NULL, " ");
  if (i == MAX_ARGS)
    fprintf(stderr, "Too many arguments\n");
#endif

  /* exec is said to fail if the executable is open.  */
  close_exec_file ();

  /* Retain a copy of our environment variables, since the child will
     replace the value of  environ  and if we're vforked, we have to 
     restore it.  */
  save_our_env = environ;

  /* Tell the terminal handling subsystem what tty we plan to run on;
     it will just record the information for later.  */

  new_tty_prefork (inferior_io_terminal);

  /* It is generally good practice to flush any possible pending stdio
     output prior to doing a fork, to avoid the possibility of both the
     parent and child flushing the same data after the fork. */

  fflush (stdout);
  fflush (stderr);

#if defined(USG) && !defined(HAVE_VFORK)
  pid = fork ();
#else
  if (debug_fork)
    pid = fork ();
  else
    pid = vfork ();
#endif

  if (pid < 0)
    perror_with_name ("vfork");

  if (pid == 0)
    {
      if (debug_fork) 
	sleep (debug_fork);

#ifdef TIOCGPGRP
      /* Run inferior in a separate process group.  */
#ifdef NEED_POSIX_SETPGID
      debug_setpgrp = setpgid (0, 0);
#else
#if defined(USG) && !defined(SETPGRP_ARGS)
      debug_setpgrp = setpgrp ();
#else
      debug_setpgrp = setpgrp (getpid (), getpid ());
#endif /* USG */
#endif /* NEED_POSIX_SETPGID */
      if (debug_setpgrp == -1)
	 perror("setpgrp failed in child");
#endif /* TIOCGPGRP */

#ifdef SET_STACK_LIMIT_HUGE
      /* Reset the stack limit back to what it was.  */
      {
	struct rlimit rlim;

	getrlimit (RLIMIT_STACK, &rlim);
	rlim.rlim_cur = original_stack_limit;
	setrlimit (RLIMIT_STACK, &rlim);
      }
#endif /* SET_STACK_LIMIT_HUGE */

      /* Ask the tty subsystem to switch to the one we specified earlier
	 (or to share the current terminal, if none was specified).  */

      new_tty ();

      /* Changing the signal handlers for the inferior after
	 a vfork can also change them for the superior, so we don't mess
	 with signals here.  See comments in
	 initialize_signals for how we get the right signal handlers
	 for the inferior.  */

      /* "Trace me, Dr. Memory!" */
      (*traceme_fun) ();

#ifndef COHERENT
      /* There is no execlpe call, so we have to set the environment
	 for our child in the global variable.  If we've vforked, this
	 clobbers the parent, but environ is restored a few lines down
	 in the parent.  By the way, yes we do need to look down the
	 path to find $SHELL.  Rich Pixley says so, and I agree.  */
      environ = env;
      execlp (shell_file, shell_file, "-c", shell_command, (char *)0);
      fprintf (stderr, "Cannot exec %s: %s.\n", shell_file,
	       safe_strerror (errno));
#else
      /* Under COHERENT use execvpe() */
      execvpe (exec_file, argv, env);
      fprintf (stderr, "Cannot exec %s: %s.\n", exec_file,
	       safe_strerror (errno));
#endif

      fflush (stderr);
      _exit (0177);
    }

  /* Restore our environment in case a vforked child clob'd it.  */
  environ = save_our_env;

  /* Now that we have a child process, make it our target, and
     initialize anything target-vector-specific that needs initializing.  */
  (*init_trace_fun)(pid);

#ifdef CREATE_INFERIOR_HOOK
  CREATE_INFERIOR_HOOK (pid);
#endif  

/* The process was started by the fork that created it,
   but it will have stopped one instruction after execing the shell.
   Here we must get it up to actual execution of the real program.  */

  inferior_pid = pid;		/* Needed for wait_for_inferior stuff below */

  clear_proceed_status ();

  /* We will get a trace trap after one instruction.
     Continue it automatically.  Eventually (after shell does an exec)
     it will get another trace trap.  Then insert breakpoints and continue.  */

#ifdef START_INFERIOR_TRAPS_EXPECTED
  pending_execs = START_INFERIOR_TRAPS_EXPECTED;
#else
  pending_execs = 2;
#endif

  init_wait_for_inferior ();

  /* Set up the "saved terminal modes" of the inferior
     based on what modes we are starting it with.  */
  target_terminal_init ();

  /* Install inferior's terminal modes.  */
  target_terminal_inferior ();

  while (1)
    {
      stop_soon_quietly = 1;	/* Make wait_for_inferior be quiet */
      wait_for_inferior ();
      if (stop_signal != SIGTRAP)
	{
	  /* Let shell child handle its own signals in its own way */
	  /* FIXME, what if child has exit()ed?  Must exit loop somehow */
	  resume (0, stop_signal);
	}
      else
	{
	  /* We handle SIGTRAP, however; it means child did an exec.  */
	  if (0 == --pending_execs)
	    break;
	  resume (0, 0);		/* Just make it go on */
	}
    }
  stop_soon_quietly = 0;

  /* We are now in the child process of interest, having exec'd the
     correct program, and are poised at the first instruction of the
     new program.  */
#ifdef SOLIB_CREATE_INFERIOR_HOOK
  SOLIB_CREATE_INFERIOR_HOOK (pid);
#endif
}
