/* 
 * freewb.c
 *
 * Example code showing the technique to close the Workbench screen in
 * a manner suitable for use by startup-sequence programs (typically games)
 * that do not wish to load Workbench.
 *
 * This allows these programs to reclaim the space taken by the Workbench
 * bitplanes.
 *
 * What you should do is use the code shown here inside your actual program.
 * Use a method of making your program detach from the CLI.  Most compilers
 * supply such a method (SAS/C's cback.o detach is used in the example).
 * Your startup-sequence should run whatever is necessary, then it should
 * execute your program, and then call 'endcli'.  For this example, a
 * startup-sequence might be:
 *
 *     c:setpatch >NIL:
 *     freewb
 *     endcli >NIL:
 *
 * Stealing the Workbench bitplanes instead of closing the Workbench screen
 * is illegal, and is not guaranteed to work under future versions of the OS.
 *
 * Note that if there is a chance that DOS requesters can appear, you
 * should redirect them to your window's screen (shown in the example).
 * Otherwise, the appearance of a DOS requester will re-open the Workbench
 * screen.
 *
 * This code is valid under Kickstart V33 (1.2 ROM) and higher.
 *
 *     lc freewb
 *     blink lib:cback.o freewb.o to freewb lib lib:lc.lib lib:amiga.lib
 *
 *
 * (C) Copyright 1991, Commodore-Amiga, Inc.
 *
 * Executables based on this information may be used in software
 * for Commodore Amiga computers.  All other rights reserved.
 * This information is provided "as is"; no warranties are made.  All
 * use is at your own risk. No liability or responsibility is assumed.
 *
 */

#include <intuition/intuition.h>
#include <libraries/dosextens.h>

#include <clib/exec_protos.h>
#include <clib/intuition_protos.h>

struct Library *IntuitionBase;

/* Number of seconds to wait for the Workbench to close before we give up */
#define MAX_TIMEOUT (5)

void _main()
{
    long done = 0;
    long timeout = 0;

    if ( IntuitionBase = OpenLibrary( "intuition.library", 33L ) )
    {
	/* Sit here and wait for Workbench to close... */
	while ( !done )
	{
	    /* Try to close Workbench */
	    if ( CloseWorkBench() )
	    {
		/* We succeeded */
		done = 1;
	    }
	    else
	    {
		if ( timeout++ > (MAX_TIMEOUT*5) )
		{
		    /* Give up */
		    done = 2;
		}
		else
		{
		    /* Wait a bit and try again */
		    Delay( 10 );
		}
	    }
	}

	/* We succeeded! */
	if ( done == 1 )
	{
	    /* Put the body of your program here
	     * (For example purposes, we'll open a screen and window...)
	     */

	    static struct NewScreen newscr =
		{
		0, 0, 640, STDSCREENHEIGHT, 2, 0, 1, HIRES,
		CUSTOMSCREEN, NULL, NULL, NULL, NULL
		};

	    static struct NewWindow newwin =
		{
		0, 0, 250, 100, -1, -1, CLOSEWINDOW,
		ACTIVATE | WINDOWDEPTH | WINDOWDRAG | WINDOWCLOSE,
		NULL, NULL, "<- Click to close", NULL,
		NULL, 50, 50, -1, -1, CUSTOMSCREEN
		};

	    struct Screen *screen;
	    struct Window *window;

	    if ( screen = OpenScreen( &newscr ) )
	    {
		newwin.Screen = screen;
		if ( window = OpenWindow( &newwin ) )
		{
		    /* Force DOS requesters to our screen */
		    ( (struct Process *)FindTask( NULL ) )->
			pr_WindowPtr = window;

		    WaitPort( window->UserPort );

		    ( (struct Process *)FindTask( NULL ) )->
			pr_WindowPtr = NULL;

		    CloseWindow( window );
		}
		CloseScreen( screen );
	    }
	}
	else
	{
	    /* We failed.  Note that we cannot printf(), since we've
	     * detached.  For example's sake, we'll use DisplayBeep()
	     * to show failure.
	     */
	    DisplayBeep( NULL );
	}

	/* Note:  even if we open the Workbench screen, there are
	 * no CLI's available.  Unless the Workbench application is
	 * running, the user will be left with a blank screen.
	 * This is OK, since this code is intended for startup-sequence
	 * games, which generally don't exit.
	 */

	OpenWorkBench();

	CloseLibrary( IntuitionBase );
    }
}
