/* appshell.h
 * Copyright (C) 1990 Commodore-Amiga, Inc.
 * written by David N. Junod
 *
 * header file for the Amiga AppShell
 *
 */

#include <exec/types.h>
#include <exec/memory.h>
#include <workbench/startup.h>
#include <libraries/dos.h>

#include <clib/macros.h>
#include <clib/alib_protos.h>
#include <clib/dos_protos.h>
#include <clib/exec_protos.h>
#include <clib/gadtools_protos.h>
#include <clib/graphics_protos.h>
#include <clib/icon_protos.h>
#include <clib/intuition_protos.h>
#include <clib/layers_protos.h>
#include <clib/wb_protos.h>
#include <clib/utility_protos.h>

#include <intuition/classes.h>
#include <intuition/classusr.h>
#include <intuition/sghooks.h>
#include <string.h>

/* Usually allocate one large block of memory for a group of items and then
 * divy out to the appropriate pointers.  Use with caution---must be
 * consistent with field types! */
#define MEMORY_FOLLOWING(ptr) ((void *)((ptr)+1))
#define MEMORY_N_FOLLOWING(ptr,n)  ((void *)( ((ULONG)ptr) + n ))

/* Used to cast an pointer to a void pointer */
#define V(x) ((void *)x)

/* BPTR compatibility */
#define TO_BPTR(x) ((BPTR)(((ULONG)(x)) >> 2))

/* Maximum number of arguments in a command string.  Don't bother changing. */
#define	MAXARG 64

/* The Funcs structure contains base information for a command.  This
 * information gets translated into a Function Table entry and gets appended
 * to the Function Table list. */
struct Funcs
{
    UBYTE *fe_Name;		/* Name of function */
    VOID (*fe_Func)(struct AppInfo *, STRPTR, struct TagItem *);
    ULONG fe_ID;		/* ID of function */
    ULONG fe_Key;		/* HotKey assigned to function */
    ULONG fe_HelpID;		/* index into the the text catalogue for help */
    ULONG fe_Flags;		/* Status of function */
    STRPTR fe_Params;		/* optional parameters for function */
    ULONG *fe_GroupID;		/* ~0 terminated array of group ID's */
};

/* no internal function defined for event */
#define	NO_FUNCTION	NULL

/* function table entry flags */
#define	APSH_FF_DISABLED (1L<<1)/* function is disabled */
#define	APSH_FF_PRIVATE (1L<<2)	/* function can't be called from command line */
#define	APSH_FF_ALIAS (1L<<3)	/* function is an alias */

/* The BasePrefs structure contains preference information that all AppShell
 * applications honor.  Envision a preference editor and a global preference
 * file to contain this information. */
struct BasePrefs
{
    ULONG bp_Flags;		/* misc. preference flags */
};

/* Set this flag to enable saving of icons with files. This feature is
 * not implemented. */
#define	APSHF_USEICONS (1L<<1)

/* The AppShell defaults to using a console window for the Command Shell.
 * By setting the following flag, the AppShell will use a Scrolling List
 * gadget to show history and use a Text gadget for command entry.
 * This feature is not implemented. */
#define APSHF_LISTVIEW (1L<<2)	/* use listview & text gadget */


/* The Project structure is used to maintain information on a project set.
 * Project sets are not limited to the main project list, but also include
 * such things as lists of components for a project. */
struct Project
{
    struct List p_ProjList;	/* Project list */
    struct ProjNode *p_CurProj;	/* Current project */
    LONG p_NumProjs;		/* Number of projects */
    LONG p_MaxID;		/* Next available ID */
    LONG p_State;		/* Listview state */
    LONG p_TopView;		/* Listview top line */
    LONG p_CurLine;		/* Listview current line */
    ULONG p_Flags;		/* Project flags */
    APTR p_UserData;		/* User data extension */
    APTR p_SysData;		/* System data extension */
};

/* When this flag is set, the project information is being displayed by
 * the Project List data entry requester. */
#define	APSHF_PROJVIEW (1L<<1)


/* Within the Project structure is an Exec list called p_ProjList.  This
 * is a list of ProjNode structures.  This structure contains information
 * on projects or project components.  This information can be obtained at
 * startup time, from AppWindow/AppIcon messages, ASL file requester or
 * through application constructs. */
struct ProjNode
{
    struct Node pn_Node;	/* embedded Exec node */

    /* AppShell information.  Read only for application */
    struct DateStamp pn_Added;	/* date stamp when added to list */
    BPTR pn_ProjDir;		/* lock on project directory */
    STRPTR pn_ProjPath;		/* pointer to the projects' complete name */
    STRPTR pn_ProjName;		/* pointer to the projects' name */
    STRPTR pn_ProjComment;	/* pointer to the projects' comment */
    struct DiskObject *pn_DObj;	/* pointer to the projects' icon */
    LONG pn_ID;			/* user selected order */
    APTR pn_SysData;		/* System data extension */

    /* Application information */
    ULONG pn_Status;		/* status of project */
    ULONG pn_ProjID;		/* project ID */
    UBYTE pn_Name[32];		/* project name */
    APTR pn_UserData;		/* UserData for project */
    BOOL pn_Changed;		/* has project been modified? */
};

/* AppInfo structure that contains an AppShell application's global
 * variables.  This structure is variable length, so NEVER embedd this
 * within your own structures.  If you must, then reference it by a
 * pointer. */
struct AppInfo
{
    /* control information */
    UBYTE *ai_TextRtn;		/* Text return string */
    LONG ai_Pri_Ret;		/* Primary error (severity) */
    LONG ai_Sec_Ret;		/* Secondary error (actual) */
    BOOL ai_Done;		/* Done with main loop? */

    /* base application information */
    struct BasePrefs *ai_BasePrefs;/* base user preferences */
    ULONG ai_PrefSize;		/* sizeof (struct BasePrefs) */
    BPTR ai_ProgDir;		/* base directory for application */
    BPTR ai_ConfigDir;		/* configuration file directory */
    struct DiskObject *ai_ProgDO;/* application tool icon */
    STRPTR ai_ProgName;		/* pointer to application base name */
    STRPTR ai_AppName;		/* pointer to application name */
    STRPTR ai_AppVersion;	/* pointer to version string */
    STRPTR ai_AppCopyright;	/* pointer to copyright notice */
    STRPTR ai_AppAuthor;	/* pointer to author */
    STRPTR ai_AppMsgTitle;	/* pointer to title for messages */

    /* project information */
    struct Project ai_Project;	/* embedded Project structure */

    /* application information */
    APTR ai_UserData;		/* UserData */

    /* READ ONLY Intuition-specific information */
    struct TextFont *ai_Font;	/* Font for screen */
    struct Screen *ai_Screen;	/* Active screen */
    struct Window *ai_Window;	/* Active window */
    struct Gadget *ai_Gadget;	/* Active gadget */
    struct MHObject *ai_CurObj;	/* Active object (gadget) */
    struct DrawInfo *ai_DI;	/* Intuition DrawInfo */
    VOID *ai_VI;		/* GadTools VisualInfo */
    WORD ai_MouseX;		/* Position at last IDCMP message */
    WORD ai_MouseY;		/* Position at last IDCMP message */
    UWORD ai_TopLine;		/* top line */
};

/*--------------------------------------------------------------------------*/
/* Following is information used by the main portion of the AppShell        */
/*--------------------------------------------------------------------------*/

/* Each message handler gets a base ID from which all of their commands are
 * offset.  The main functions are offset from zero. */
#define APSH_MAIN_ID 0L

/* Following are ID's for the functions that are implemented by the AppShell
 * whether there are any message handlers or not. */
#define MAIN_Dummy	APSH_MAIN_ID
#define AliasID		(MAIN_Dummy+1L) /* set up function w/parameters     */
#define DebugID		(MAIN_Dummy+2L) /* general debugging                */
#define DisableID	(MAIN_Dummy+3L) /* Disable a function               */
#define EditID		(MAIN_Dummy+4L) /* Edit an object                   */
#define EnableID	(MAIN_Dummy+5L) /* Enable a function                */
#define ExecMacroID	(MAIN_Dummy+6L) /* Execute the internal macro       */
#define FaultID		(MAIN_Dummy+7L) /* Return error text                */
#define GetID		(MAIN_Dummy+8L) /* Get object attribute             */
#define GroupID		(MAIN_Dummy+9L) /* Maintain object groups           */
#define HelpID		(MAIN_Dummy+10L)/* Help                             */
#define LearnID		(MAIN_Dummy+11L)/* Learn macro function             */
#define StopLearnID	(MAIN_Dummy+12L)/* Stop learn macro function        */
#define LoadMacroID	(MAIN_Dummy+13L)/* Load a macro into the internal memory */
#define PriorityID	(MAIN_Dummy+14L)/* Set an objects priority (process) */
#define	RemoveID	(MAIN_Dummy+15L)/* remove an object (project)       */
#define SaveMacroID	(MAIN_Dummy+16L)/* Save the internal macro to disk  */
#define SelectID	(MAIN_Dummy+17L)/* select an object (project)       */
#define SelectTopID	(MAIN_Dummy+18L)/* select first object (project)    */
#define SelectNextID	(MAIN_Dummy+19L)/* select next object (project)     */
#define SelectPrevID	(MAIN_Dummy+20L)/* select previous object (project) */
#define SelectBottomID	(MAIN_Dummy+21L)/* select last object (project)     */
#define SetID		(MAIN_Dummy+22L)/* Set object attributes            */
#define StatusID	(MAIN_Dummy+23L)/* Give status of an object         */
#define StopID		(MAIN_Dummy+24L)/* Stop an operation                */
#define StubID		(MAIN_Dummy+25L)/* NOP function                     */
#define	VersionID	(MAIN_Dummy+26L)/* Version                          */

/* data transmission function ID's via OpenSIPC, SIPCPrintf, CloseSIPC */
#define	LoginID		(MAIN_Dummy+100L)/* request data transmission session */
#define	DataStreamID	(MAIN_Dummy+101L)/* data transmission */
#define	SuspendID	(MAIN_Dummy+102L)/* temporarily suspend session */
#define	ResumeID	(MAIN_Dummy+103L)/* resume data session */
#define	LogoutID	(MAIN_Dummy+104L)/* signal end of session */

/* Following are ID's for functions that should be implemented by the
 * application to help promote a consistent set of functions for the
 * end user. */

/* standard project functions */
#define	NewID		(MAIN_Dummy+500L)/* new project/process */
#define	ClearID		(MAIN_Dummy+501L)/* clear current project */
#define	OpenID		(MAIN_Dummy+502L)/* open an existing project */
#define	SaveID		(MAIN_Dummy+503L)/* save project to existing name */
#define	SaveAsID	(MAIN_Dummy+504L)/* save project to a new name */
#define	RevertID	(MAIN_Dummy+505L)/* revert project to last saved */
#define	PrintID		(MAIN_Dummy+506L)/* print the current project */
#define	PrintAsID	(MAIN_Dummy+507L)/* define print configuration */
#define	AboutID		(MAIN_Dummy+508L)/* display application information */
#define	InfoID		(MAIN_Dummy+509L)/* display project information */
#define	QuitID		(MAIN_Dummy+510L)/* exit from the application */

/* application standard edit functions */
#define	MarkID		(MAIN_Dummy+520L)
#define	CutID		(MAIN_Dummy+521L)
#define	CopyID		(MAIN_Dummy+522L)
#define	PasteID		(MAIN_Dummy+523L)
#define	EraseID		(MAIN_Dummy+524L)
#define	UndoID		(MAIN_Dummy+525L)
#define	OpenClipID	(MAIN_Dummy+526L)
#define	SaveClipID	(MAIN_Dummy+527L)
#define	PrintClipID	(MAIN_Dummy+528L)

/* application standard search functions */
#define	GotoID		(MAIN_Dummy+540L)
#define	FindID		(MAIN_Dummy+541L)
#define	NextID		(MAIN_Dummy+542L)
#define	ReplaceID	(MAIN_Dummy+543L)


/*--------------------------------------------------------------------------*/
/* Following is information used by the ARexx message handler               */
/*--------------------------------------------------------------------------*/

/* ID assigned to the ARexx message handler */
#define APSH_AREXX_ID 5000L

/* Following are ID's for the functions that are implemented by the ARexx
 * message handler. */
#define	AREXX_Dummy	APSH_AREXX_ID
#define	RXID		(AREXX_Dummy+1L)/* Execute an ARexx command */
#define	WhyID		(AREXX_Dummy+2L)/* Return information on the last error */

/* ID for an ARexx low-level message handler function */
#define	AH_SENDCMD 4

/*--------------------------------------------------------------------------*/
/* Following is information used by the Command Shell message handler       */
/*--------------------------------------------------------------------------*/

#define	APSH_DOS_ID 6000L

enum
{
    DOS_Dummy = APSH_DOS_ID,
    CMDShellID,			/* Command shell control */
};

/*--------------------------------------------------------------------------*/
/* Following is information used by the Intuition message handler           */
/*--------------------------------------------------------------------------*/

#define APSH_IDCMP_ID 7000L

enum
{
    IDCMP_Dummy = APSH_IDCMP_ID,
    ActivateID,			/* Make window available */
    ButtonID,			/* Edit gadget */
    HotKeyID,			/* Edit HotKey */
    MenuID,			/* Edit menu */
    ToFrontID,			/* bring current window to front */
    ToBackID,			/* send current window to back */
    WindowID,			/* open/close window */
};

/* extended MenuItem structure */
struct EMenuItem
{
    struct MenuItem emi_Item;	/* embedded MenuItem structure */
    ULONG emi_MenuID;		/* ID used for function number to perform */
    APTR emi_UserData;		/* UserData (like other Intuition structs) */
};

/* extended Gadget structure */
struct EGadget
{
    struct Gadget eg_Gadget;		/* embedded Gadget structure */
    ULONG eg_Funcs[10];			/* Function ID array */
};


/* Extended Gadget function array pointers */
#define	EG_DOWNPRESS	0
#define	EG_HOLD		1
#define	EG_RELEASE	2
#define	EG_DBLCLICK	3
#define	EG_ABORT	4
#define	EG_ALTHIT	5
#define	EG_SHIFTHIT	6

/* Intuition object types */
enum
{
    OBJ_Dummy = 0L,

    /* GadTools gadgets */
    OBJ_Generic,
    OBJ_Button,
    OBJ_Checkbox,
    OBJ_Integer,
    OBJ_Listview,
    OBJ_MX,
    OBJ_Number,
    OBJ_Cycle,
    OBJ_Palette,
    OBJ_Scroller,
    OBJ_reserved,		/* reserved for system use */
    OBJ_Slider,
    OBJ_String,
    OBJ_Text,

    /* other gadgets */
    OBJ_Display,
    OBJ_Select,
    OBJ_Dropbox,
    OBJ_GImage,

    /* images */
    OBJ_Image,

    /* borders */
    OBJ_Plain,
    OBJ_BevelIn,
    OBJ_BevelOut,
    OBJ_DblBevelIn,
    OBJ_DblBevelOut,

    /* other object types */
    OBJ_Screen,			/* screen information */
    OBJ_Window,			/* window information */
    OBJ_Group,			/* group information */
    OBJ_VFill,			/* Vertical fill */
    OBJ_HFill,			/* Horizontal fill */

    OBJ_LAST
};

/* Intuition user-interface object (gadget, border, text, etc... Currently
 * the AppShell has stolen the UserData fields of both gadgets and windows.
 * Hopefully the next release of AppShell will give these fields back,
 * or provide a way to use them. */
struct Object
{
    struct Object *o_NextObject;/* next object in array */
    ULONG o_Group;		/* group that object belongs in */
    ULONG o_Type;		/* type */
    ULONG o_ObjectID;		/* ID */
    ULONG o_Flags;		/* see defines below */
    UBYTE o_Key;		/* hotkey */
    UBYTE *o_Name;		/* name */
    ULONG o_LabelID;		/* label index into text catalogue */
    struct Rectangle o_Outer;	/* size w/label */
    struct TagItem *o_Tags;	/* tags for object */
    APTR o_UserData;		/* user data for object */
};

/* To indicate that a gadget is used to close a window, set the following
 * flag.  The window is closed after any functions for the gadget have
 * completed. */
#define	APSH_OBJF_CLOSEWINDOW	(1L<<1)

/* The following flags are for layout purposes and are currently
 * unimplemented. */
#define	APSH_OBJF_NOSCALE	(1L<<2)	/* Adjust top-left, don't scale */
#define	APSH_OBJF_CENTER	(1L<<3)	/* Center, don't scale */
#define	APSH_OBJF_RELRIGHT	(1L<<4)	/* object is relative to right */
#define	APSH_OBJF_RELBOTTOM	(1L<<5)	/* object is relative to bottom */
#define	APSH_OBJF_RELWIDTH	(1L<<6)	/* object width is relative */
#define	APSH_OBJF_RELHEIGHT	(1L<<7)	/* object height is relative */

/*--- KEYBOARD RELATED ITEMS ---*/
#define	SPECIAL 255

/* some useful defines */
#define	TAB      9
#define	RETURN  13
#define	ESC	27
#define	DELETE	127
#define	HELP	(SPECIAL+'?')
#define	FUNC1	(SPECIAL+'0')
#define	FUNC2	(SPECIAL+'1')
#define	FUNC3	(SPECIAL+'2')
#define	FUNC4	(SPECIAL+'3')
#define	FUNC5	(SPECIAL+'4')
#define	FUNC6	(SPECIAL+'5')
#define	FUNC7	(SPECIAL+'6')
#define	FUNC8	(SPECIAL+'7')
#define	FUNC9	(SPECIAL+'8')
#define	FUNC10	(SPECIAL+'9')
#define	UP	(SPECIAL+'A')
#define	DOWN	(SPECIAL+'B')
#define	RIGHT	(SPECIAL+'C')
#define	LEFT	(SPECIAL+'D')

/* component for our keyboard command array */
struct KeyboardCMD
{
    ULONG kbc_Key;		/* key */
    ULONG kbc_FuncID;		/* function ID */
};

/*--------------------------------------------------------------------------*/
/* Following is information used by the Simple IPC message handler          */
/*--------------------------------------------------------------------------*/

#define	APSH_SIPC_ID 10000L

/* This structure is used by the AppShell to communicate with tools and
 * other AppShell applications.
 *
 * If sipc_DType equal NULL, then the function ID in sipc_Type is performed
 * with no arguments.
 *
 *    PerfFunc (ai, sipc_Type, NULL, NULL);
 *
 * If sipc_DType equal APSH_SDT_TagList, then the function ID in sipc_Type is
 * performed with sipc_Data as the tag list arguments.
 *
 *    PerfFunc (ai, sipc_Type, NULL, sipc_Data);
 *
 * If sipc_DType equal APSH_SDT_Data, then the function ID in sipc_Type is
 * performed with with the following tags as arguments:
 *
 *    APSH_SIPCData,       sipc_Data
 *    APSH_SIPCDataLength, sipc_DSize
 *
 * If sipc_DType equal APSH_SDT_Command, then the string command line
 * passed in the sipc_Data field is performed:
 *
 *    PerfFunc (ai, NULL, sipc_Data, NULL);
 *
 */
struct SIPCMessage
{
    struct Message sipc_Msg;	/* Embedded Exec message structure */
    ULONG sipc_Type;		/* Type of message */
    APTR sipc_Data;		/* Pointer to message data */
    ULONG sipc_DSize;		/* Size of message data */
    ULONG sipc_DType;		/* Type of message data */
    ULONG sipc_Pri_Ret;		/* Primary return value */
    ULONG sipc_Sec_Ret;		/* Secondary return value */
    APTR sipc_Extens1;		/* *** PRIVATE *** SYSTEM USE ONLY! */
    APTR sipc_Extens2;		/* *** PRIVATE *** SYSTEM USE ONLY! */
};

/* These flags are used in the sipc_DType field to indicate what type of
 * information is in the sipc_Data field. */
#define	APSH_SDT_Command (1L<<1)/* Data is a STRPTR */
#define	APSH_SDT_TagList (1L<<2)/* Data is a list of TagItem's */
#define	APSH_SDT_Data    (1L<<3)/* Data is a pointer to a data block */
#define	APSH_SDT_Text    (1L<<4)/* text transmissions via sprintf */

/* Public SIPC port name given to the AppShell remote debugger.  Accessed
 * using OpenSIPC, SIPCPrintf and CloseSIPC. */
#define	DEBUGGER "AppShell_Debugger"

/*--------------------------------------------------------------------------*/
/* Following is information used by the Tool message handler                */
/*--------------------------------------------------------------------------*/

#define	APSH_TOOL_ID 11000L

typedef VOID (*F_PTR)(APTR, struct MsgPort *);

#define	TOOL_ACTIVATE (1L<<1)
#define	TOOL_LIB_PORT "public_tool_librarian"

/* structure for tool table entry */
struct Tools
{
    struct Node te_Node;	/* Node for tool entry */
    VOID (*te_Func)(VOID *);	/* Address of function */
    ULONG te_ID;		/* ID of function */
    ULONG te_Flags;		/* Status of function */
    ULONG te_HitCnt;		/* Access count */
    ULONG te_Stack;		/* Stack requirements for function */
    ULONG te_Pri;		/* Default priority for function */
    ULONG te_UseCnt;		/* Current use count of function */
    ULONG te_MaxCnt;		/* Maximum instances of function */
    UBYTE *te_Port;		/* Port name of owner */
};

/* tool functions */
enum
{
    TOOL_Dummy = APSH_TOOL_ID,
    LIB_AddEntry,		/* add an entry */
    LIB_EditEntry,		/* edit an entry */
    LIB_DelEntry,		/* delete an entry */
    LIB_InqEntry,		/* inquire entry */
    LIB_IncEntry,		/* increment use count of entry */
    LIB_DecEntry,		/* decrement use count of entry */
    LIB_RunEntry,		/* start a process */
    LIB_DispTable,		/* DEBUG: display table via kprintf */

    StartupMsgID,		/* Startup message */
    ActivateToolID,		/* Activate tool */
    ShutdownToolID,		/* Shutdown tool */
    ShutdownMsgID,		/* Shutdown message */
};

/*--------------------------------------------------------------------------*/
/* Following is information used by the Workbench message handler           */
/*--------------------------------------------------------------------------*/

#define	APSH_WB_ID 12000L

/* APSH_CmdFlags */
#define	APSH_WBF_DISPLAY (1L<<1)	/* maintain display box for icon */
#define	APSH_WBF_PROJLIST (1L<<2)	/* add the WBArgs to the project list */
#define	APSH_WBF_NOLIST (1L<<3)		/* don't add the WBArgs to a list */

/*--------------------------------------------------------------------------*/
/* Following is information for use by the Application                      */
/*--------------------------------------------------------------------------*/

/* base tag for application functions */
#define	APSH_USER_ID 100000L


/*--------------------------------------------------------------------------*/
/* Following is low-level message handler information                       */
/*--------------------------------------------------------------------------*/

/* message handler object node */
struct MHObject
{
    struct Node mho_Node;	/* embedded Exec node */
    struct List mho_ObjList;	/* embedded List of children objects */
    struct MHObject *mho_Parent;/* pointer to parent object */
    struct MHObject *mho_CurNode;/* pointer to current child object */
    ULONG mho_ID;		/* numeric ID of object */
    ULONG mho_Status;		/* status of object */
    APTR mho_SysData;		/* message handler data */
    APTR mho_UserData;		/* application data */
    APTR mho_Extens1;		/* *** PRIVATE *** */
    UBYTE mho_Name[1];		/* name of object */
};

/* message handler node */
struct MsgHandler
{
    struct MHObject mh_Header;	/* embedded MHObject structure */

    struct MsgPort *mh_Port;	/* message port for handler */
    STRPTR mh_PortName;		/* port name, if public */
    ULONG mh_SigBits;		/* signal bits to watch for */

    /* handler functions */
    WORD mh_NumFuncs;		/* number of functions in handler */
    BOOL (**mh_Func)(struct AppInfo *,struct MsgHandler *,struct TagItem *);

    STRPTR *mh_DefText;		/* Default text catalogue */
    APTR mh_Catalogue;		/* *** PRIVATE *** */

    APTR mh_Extens1;		/* *** PRIVATE *** */
    APTR mh_Extens2;		/* *** PRIVATE *** */
};

/*--- interface function array pointers ---*/
/* make a message handler active */
#define	MH_OPEN		0
/* handle messages */
#define	MH_HANDLE	1
/* make a message handler inactive */
#define	MH_CLOSE	2
/* free resources and delete message handler */
#define	MH_SHUTDOWN	3

/*--- node types ---*/
/* message handler node */
#define	MH_HANDLER_T	100
/* data node */
#define	MH_DATA_T	101

/*--- message handler object types */
#define	MHO_WINDOW	110		/* Intuition window */
#define	MHO_INTOBJ	111		/* AppShell Intuition object */
#define	MHO_TOOL	120		/* Tool */

/*--- node priorities ---*/
/* message handler node default priority */
#define	MH_HANDLER_P	10
/* data node default priority */
#define	MH_DATA_P	-10

/*--- message handler status ---*/
#define	MHS_OPEN	0x0001
#define	MHS_CLOSE	0x0002
#define	MHS_ENABLED	0x0004
#define	MHS_DISABLED	0x0008

/*--- overall status ---*/
#define	P_ACTIVE	0x0001
#define	P_INACTIVE	0x0002
#define	P_SINGLE	0x0010
#define	P_MULTIPLE	0x0020
#define	REQUIRED TRUE
#define OPTIONAL FALSE

/*--------------------------------------------------------------------------*/
/* The AppShell uses the following tags.  Reserved TAG_USER+24000L - 25999L */
/*--------------------------------------------------------------------------*/

/* Tags */
#define	APSH_Dummy		TAG_USER+24000L

/* library management */
#define	APSH_OpenLibraries	(APSH_Dummy+1L)		/* open libraries */
#define	APSH_LibNameTag		(APSH_Dummy+2L)		/* library name tag */
#define	APSH_LibName		(APSH_Dummy+3L)		/* library name */
#define	APSH_LibVersion		(APSH_Dummy+4L)		/* library version */
#define	APSH_LibStatus		(APSH_Dummy+5L)		/* required/optional */
#define	APSH_LibReserved5	(APSH_Dummy+10L)	/* RESERVED FOR SYSTEM USE */
#define	APSH_LibBase		(APSH_Dummy+11L)	/* library base */
#define	APSH_ARexxSys		(APSH_Dummy+12L)	/* rexxsyslib.library */
#define	APSH_ARexxSup		(APSH_Dummy+13L)	/* rexxsupport.library */
#define	APSH_ASL		(APSH_Dummy+14L)	/* asl.library */
#define	APSH_Commodities	(APSH_Dummy+15L)	/* commodities.library */
#define	APSH_DiskFont		(APSH_Dummy+16L)	/* diskfont.library */
#define	APSH_DOS		(APSH_Dummy+17L)	/* dos.library */
#define	APSH_GadTools		(APSH_Dummy+18L)	/* gadtools.library */
#define	APSH_Gfx		(APSH_Dummy+19L)	/* graphics.library */
#define	APSH_Icon		(APSH_Dummy+20L)	/* icon.library */
#define	APSH_Intuition		(APSH_Dummy+21L)	/* intuition.library */
#define	APSH_Layers		(APSH_Dummy+22L)	/* layers.library */
#define	APSH_IFF		(APSH_Dummy+23L)	/* iffparse.library */
#define	APSH_Translate		(APSH_Dummy+24L)	/* translator.library */
#define	APSH_Utility		(APSH_Dummy+25L)	/* utility.library */
#define	APSH_Workbench		(APSH_Dummy+26L)	/* workbench.library */

/* main AppShell tags */
#define	APSH_NumArgs		(APSH_Dummy+40L)	/* Number of Shell arguments */
#define	APSH_ArgList		(APSH_Dummy+41L)	/* Shell arguments */
#define	APSH_WBStartup		(APSH_Dummy+42L)	/* Workbench arguments */
#define	APSH_ControlPort	(APSH_Dummy+43L)	/* SIPC Control port for a cloned AppShell */
#define	APSH_AppName		(APSH_Dummy+44L)	/* pointer to the application's name */
#define	APSH_AppVersion		(APSH_Dummy+45L)	/* pointer to the application's version */
#define	APSH_AppCopyright	(APSH_Dummy+46L)	/* pointer to the application's (c) notice */
#define	APSH_AppAuthor		(APSH_Dummy+47L)	/* pointer to the application's author */
#define	APSH_AppMsgTitle	(APSH_Dummy+48L)	/* pointer to message title */
#define	APSH_FuncTable		(APSH_Dummy+55L)	/* function table for application */
#define	APSH_DefText		(APSH_Dummy+56L)	/* Default text catalogue */
#define	APSH_AppInit		(APSH_Dummy+57L)	/* Custom application init function ID */
#define	APSH_AppExit		(APSH_Dummy+58L)	/* Custom application shutdown function ID */
#define	APSH_SIG_C		(APSH_Dummy+59L)	/* SIG_BREAK_C function ID */
#define	APSH_SIG_D		(APSH_Dummy+60L)	/* SIG_BREAK_D function ID */
#define	APSH_SIG_E		(APSH_Dummy+61L)	/* SIG_BREAK_E function ID */
#define	APSH_SIG_F		(APSH_Dummy+62L)	/* SIG_BREAK_F function ID */
#define	APSH_ProjInfo		(APSH_Dummy+63L)	/* pointer to a Project structure */

/* message handler routines */
#define	APSH_AddHandler		(APSH_Dummy+80L)	/* add a message handler to application */
#define	APSH_Setup		(APSH_Dummy+81L)	/* setup function */
#define	APSH_Status		(APSH_Dummy+82L)	/* active, inactive, multiple, etc... */
#define	APSH_Rating		(APSH_Dummy+83L)	/* optional/required, etc... */
#define	APSH_Port		(APSH_Dummy+84L)	/* name of the message port */
#define	APSH_Handler		(APSH_Dummy+85L)	/* Handler ID */
#define	APSH_CmdData		(APSH_Dummy+86L)	/* Command data */
#define	APSH_CmdDataLength	(APSH_Dummy+87L)	/* Length of command data */
#define	APSH_CmdID		(APSH_Dummy+88L)	/* Command ID (function) */
#define	APSH_CmdString		(APSH_Dummy+89L)	/* Command string */
#define	APSH_CmdTagList		(APSH_Dummy+90L)	/* Command tag list */
#define	APSH_Command		(APSH_Dummy+91L)	/* Handler command */
#define	APSH_NameTag		(APSH_Dummy+92L)	/* Name Tag for object */
#define	APSH_CmdFlags		(APSH_Dummy+93L)	/* Command Flags */
#define	APSH_TextID		(APSH_Dummy+94L)	/* Text ID */
#define	APSH_BaseID		(APSH_Dummy+95L)	/* Base ID */

/* ARexx information */
#define	APSH_Extens		(APSH_Dummy+120L)	/* ARexx macro name extension */
#define	APSH_ARexxError		(APSH_Dummy+121L)	/* ARexx command ERROR function ID */
#define	APSH_ARexxOK		(APSH_Dummy+122L)	/* ARexx command OK function ID */

/* command shell */
#define	APSH_CloseMsg		(APSH_Dummy+140L)	/* Closing message */
#define	APSH_CMDWindow		(APSH_Dummy+141L)	/* Command window spec */
#define	APSH_Prompt		(APSH_Dummy+142L)	/* Command window prompt */

/* window information */
#define	APSH_WindowEnv		(APSH_Dummy+160L)	/* Window Environment */
#define	APSH_TextAttr		(APSH_Dummy+161L)	/* Text Attributes */
#define	APSH_NewScreen		(APSH_Dummy+162L)	/* NewScreen structure */
#define	APSH_NewScreenTags	(APSH_Dummy+163L)	/* Tags for new screen */
#define	APSH_Palette		(APSH_Dummy+164L)	/* Color Palette */
#define	APSH_NewWindow		(APSH_Dummy+165L)	/* NewWindow structure */
#define	APSH_NewWindowTags	(APSH_Dummy+166L)	/* Tags for new window */
#define	APSH_HotKeys		(APSH_Dummy+167L)	/* HotKey command array */
#define	APSH_Menu		(APSH_Dummy+168L)	/* Intuition-style Menu array */
#define	APSH_Gadgets		(APSH_Dummy+169L)	/* Intuition-style Gadget array */
#define	APSH_GTMenu		(APSH_Dummy+170L)	/* GadTools-style Menu array */
#define	APSH_GTGadgets		(APSH_Dummy+171L)	/* GadTools-style NewGadget array */
#define	APSH_GTFlags		(APSH_Dummy+172L)	/* flags for GadTools objects */
#define	APSH_Objects		(APSH_Dummy+173L)	/* Object array */
#define	APSH_ObjDown		(APSH_Dummy+174L)	/* Gadget downpress function ID */
#define	APSH_ObjHold		(APSH_Dummy+175L)	/* Gadget hold function ID */
#define	APSH_ObjRelease		(APSH_Dummy+176L)	/* Gadget release function ID */
#define	APSH_ObjDblClick	(APSH_Dummy+177L)	/* Gadget double-click function ID */
#define	APSH_ObjAbort		(APSH_Dummy+178L)	/* Gadget abort function ID */
#define	APSH_ObjAltHit		(APSH_Dummy+179L)	/* Gadget ALT hit function ID */
#define	APSH_ObjShiftHit	(APSH_Dummy+180L)	/* Gadget SHIFT hit function ID */
#define	APSH_ObjData		(APSH_Dummy+181L)	/* Gadget image or data */
#define	APSH_ObjInner		(APSH_Dummy+182L)	/* Inner rectangle */
#define	APSH_ObjPointer		(APSH_Dummy+183L)	/* pointer name prefix */
#define	APSH_DefWinFlags	(APSH_Dummy+184L)	/* Default window flags */
#define	APSH_ObjName		(APSH_Dummy+185L)	/* Object name */
#define	APSH_WinName		(APSH_Dummy+186L)	/* Window name */
#define	APSH_WinPointer		(APSH_Dummy+188L)	/* pointer to window */

/* IDCMP messages */
#define	APSH_SizeVerify		(APSH_Dummy+220L)	/* SIZEVERIFY function ID */
#define	APSH_NewSize		(APSH_Dummy+221L)	/* NEWSIZE function ID */
#define	APSH_RefreshWindow	(APSH_Dummy+222L)	/* REFRESHWINDOW function ID */
#define	APSH_MouseButtons	(APSH_Dummy+223L)	/* MOUSEBUTTONS function ID */
#define	APSH_ReqSet		(APSH_Dummy+224L)	/* REQSET function ID */
#define	APSH_CloseWindow	(APSH_Dummy+225L)	/* CLOSEWINDOW  function ID */
#define	APSH_ReqVerify		(APSH_Dummy+226L)	/* REQVERIFY function ID */
#define	APSH_ReqClear		(APSH_Dummy+227L)	/* REQCLEAR function ID */
#define	APSH_MenuVerify		(APSH_Dummy+228L)	/* MENUVERIFY function ID */
#define	APSH_DiskInserted	(APSH_Dummy+229L)	/* DISKINSERTED function ID */
#define	APSH_DiskRemoved	(APSH_Dummy+230L)	/* DISKREMOVED function ID */
#define	APSH_ActiveWindow	(APSH_Dummy+231L)	/* ACTIVEWINDOW function ID */
#define	APSH_InactiveWindow	(APSH_Dummy+232L)	/* INACTIVEWINDOW function ID */

/* real or simulated IntuiMessage fields */
#define	APSH_MsgClass		(APSH_Dummy+260L)	/* message class */
#define	APSH_MsgCode		(APSH_Dummy+261L)	/* message code */
#define	APSH_MsgQualifier	(APSH_Dummy+262L)	/* message qualifier */
#define	APSH_MsgIAddress	(APSH_Dummy+263L)	/* item address */
#define	APSH_MsgMouseX		(APSH_Dummy+264L)	/* mouse X coordinate */
#define	APSH_MsgMouseY		(APSH_Dummy+265L)	/* mouse Y coordinate */
#define	APSH_MsgSeconds		(APSH_Dummy+266L)	/* seconds */
#define	APSH_MsgMicros		(APSH_Dummy+267L)	/* micros */
#define	APSH_MsgWindow		(APSH_Dummy+268L)	/* window for event */

/* SIPC message */
#define	APSH_SIPCData		(APSH_Dummy+300L)	/* Pointer the data passed by a SIPC message */
#define	APSH_SIPCDataLength	(APSH_Dummy+301L)	/* Length of the SIPC data */

/* standard tool information */
#define	APSH_Tool		(APSH_Dummy+320L)	/* Name of tool */
#define	APSH_ToolAddr		(APSH_Dummy+321L)	/* Address of tool */
#define	APSH_ToolData		(APSH_Dummy+322L)	/* Data for tool */
#define	APSH_ToolStack		(APSH_Dummy+323L)	/* Stack requirements of tool */
#define	APSH_ToolPri		(APSH_Dummy+324L)	/* Priority of tool */

/* Workbench tags */
#define	APSH_AppWindowEnv	(APSH_Dummy+400L)	/* AppWindow information */
#define	APSH_AppIconEnv		(APSH_Dummy+401L)	/* AppIcon information */
#define	APSH_AppMenuEnv		(APSH_Dummy+402L)	/* AppMenuItem information */
#define	APSH_WBArg		(APSH_Dummy+420L)	/* pointer to WBArg */

/* Workbench tags for function ID's */
#define	APSH_AppOpen		(APSH_Dummy+403L)	/* After App... is added */
#define	APSH_AppBDrop		(APSH_Dummy+404L)	/* Before icons are processed */
#define	APSH_AppDDrop		(APSH_Dummy+405L)	/* For each icon in the list */
#define	APSH_AppADrop		(APSH_Dummy+406L)	/* After icons added to project list */
#define	APSH_AppClose		(APSH_Dummy+407L)	/* Before App... closed */
#define	APSH_AppRemove		(APSH_Dummy+408L)	/* Before App... deleted */
#define	APSH_AppDblClick	(APSH_Dummy+409L)	/* When icon double-clicked */

#define	APSH_NEXT_TAG		(APSH_Dummy+500L)	/* remember... */

/*--------------------------------------------------------------------------*/
/* Following are ID's to use to access the AppShell text table              */
/*--------------------------------------------------------------------------*/
enum
{
    APSH_PAD = 0,
    APSH_NOT_AN_ICON,		/* %s is not an icon. */
    APSH_NOT_AVAILABLE,		/* %s is not available */
    APSH_PORT_ACTIVE,		/* %s port already active */
    APSH_PORT_X_ACTIVE,		/* port, %s, already active */
    APSH_NOT_AN_IFF,		/* %s is not an IFF file */
    APSH_NOT_AN_IFF_X,		/* %1$s is not an IFF %2$s file */
    APSH_CLOSE_ALL_WINDOWS,	/* Close all windows */
    APSH_CMDSHELL_PROMPT,	/* Cmd> */
    APSH_CLDNT_CREATE_X,	/* Could not create %s */
    APSH_CLDNT_CREATE_PORT,	/* Could not create port, %s */
    APSH_CLDNT_CREATE_OBJ,	/* Could not create object */
    APSH_CLDNT_CREATE_OBJ_X,	/* Could not create object, %s */
    APSH_CLDNT_CREATE_FILE,	/* Could not create file */
    APSH_CLDNT_CREATE_FILE_X,	/* Could not create file, %s */
    APSH_CLDNT_INIT_X,		/* Could not initialize %s */
    APSH_CLDNT_INIT_MSGH,	/* Could not initialize %s message handler */
    APSH_CLDNT_LOCK,		/* Could not lock %s */
    APSH_CLDNT_LOCK_DIR,	/* Could not lock directory */
    APSH_CLDNT_LOCK_DIR_X,	/* Could not lock directory, %s */
    APSH_CLDNT_LOCK_PUB,	/* Could not lock public screen */
    APSH_CLDNT_LOCK_PUB_X,	/* Could not lock public screen, %s */
    APSH_CLDNT_OBTAIN,		/* Could not obtain %s */
    APSH_CLDNT_OPEN,		/* Could not open %s */
    APSH_CLDNT_OPEN_FILE,	/* Could not open file */
    APSH_CLDNT_OPEN_FILE_X,	/* Could not open file, %s */
    APSH_CLDNT_OPEN_FONT_X,	/* Could not open font, %s */
    APSH_CLDNT_OPEN_MACRO,	/* Could not open macro file, %s */
    APSH_CLDNT_OPEN_PREF,	/* Could not open preference file, %s */
    APSH_CLDNT_OPEN_SCREEN,	/* Could not open screen */
    APSH_CLDNT_OPEN_WINDOW,	/* Could not open window */
    APSH_SETUP_TIMER,		/* Could not set up timer event */
    APSH_SETUP_HOTKEYS,		/* Could not set up HotKeys */
    APSH_START_PROCESS,		/* Could not start process */
    APSH_START_TOOL,		/* Could not start tool */
    APSH_START_TOOL_X,		/* Could not start tool, %s */
    APSH_WRITE_FILE,		/* Could not write to file */
    APSH_WRITE_FILE_X,		/* Could not write to file, %s */
    APSH_WRITE_MACRO,		/* Could not write to macro file */
    APSH_CMDSHELL_WIN,		/* CON:0/150/600/50/Command Shell/CLOSE */
    APSH_NO_NAMETAG_WIN,	/* No name given for window */
    APSH_NO_PORT,		/* No port name specified */
    APSH_NOT_ENOUGH_MEMORY,	/* Not enough memory */
    APSH_WAITING_FOR_MACRO,	/* Waiting for macro return */
    APSH_DISABLED,		/* %s is disabled */
    APSH_IOERR,			/* IoErr #%ld */
    APSH_INVALID_NAMETAG,	/* Invalid name tag. */
    APSH_OKAY_TXT,		/* Okay */
    APSH_CANCEL_TXT,		/* Cancel */
    APSH_CONTINUE_TXT,		/* Continue */
    APSH_DONE_TXT,		/* Done */
    APSH_ABORT_TXT,		/* Abort */
    APSH_QUIT_TXT,		/* Quit */
    APSH_UNNAMED,		/* Unnamed */
    APSH_LAST_MESSAGE
};
