#define WINVER 0x0501

#include <windows.h>
#include <stdio.h>
#include "getopt.h"
#include "winuser.h"
#include "resource.h"
#include "ver.h"

/*
 * Command line option token values.
 *  0x0000-0x00ff   Single characters
 *  0x1000-0x1fff   Long switches (no arg)
 *  0x2000-0x2fff   Long settings (arg required)
 */
#define OPT_H           0x0001

#define OPT_HELP        0x1000
#define OPT_SAVE        0x1001
#define OPT_LIST        0x1002
#define OPT_COMPAT      0x1003
#define OPT_BEST        0x1004
#define OPT_TEST        0x1005
#define OPT_CURRENT     0x1006
#define OPT_LISTD       0x1007
#define OPT_LISTAD      0x1008

#define OPT_KBDINFO     0x1100

#define OPT_MOUSEINFO   0x1200

#define OPT_X           0x2000
#define OPT_Y           0x2001
#define OPT_BPP         0x2002
#define OPT_VSYNC       0x2003
#define OPT_FSMOOTH     0x2004
#define OPT_DISP        0x2005
#define OPT_DORIENT     0x2006
//#define OPT_CLEARTYPE   0x2005

#define OPT_KBDSPEED    0x2100
#define OPT_KBDDELAY    0x2101
#define OPT_KBDLANG     0x2102
#define OPT_KBDLANG_LAYOUT   0x2103

#define OPT_MOUSESPEED  0x2200
#define OPT_MOUSELEFT   0x2201

#ifndef DM_DISPLAYORIENTATION
#define DM_DISPLAYORIENTATION   0x00000080L
#endif //DM_DISPLAYORIENTATION

/* DEVMODE dmDisplayOrientation specifiations */
#ifndef DMDO_DEFAULT
#define DMDO_DEFAULT    0
#define DMDO_90         1
#define DMDO_180        2
#define DMDO_270        3
#endif //DMDO_DEFAULT

#ifndef DISP_CHANGE_BADDUALVIEW
#define DISP_CHANGE_BADDUALVIEW     -6
#endif //DISP_CHANGE_BADDUALVIEW

#ifndef EDS_ROTATEDMODE
#define EDS_RAWMODE                   0x00000002
#define EDS_ROTATEDMODE               0x00000004
#endif //EDS_ROTATEDMODE

struct option long_options[] = {
    { "?", no_argument, NULL, OPT_H },
    { "h", no_argument, NULL, OPT_H },
    { "help", no_argument, NULL, OPT_HELP },
    { "save", no_argument, NULL, OPT_SAVE },
    { "s", no_argument, NULL, OPT_SAVE },
    { "list-display", no_argument, NULL, OPT_LISTD },
    { "list-displays", no_argument, NULL, OPT_LISTD },
    { "ld", no_argument, NULL, OPT_LISTD },
    { "list-all-displays", no_argument, NULL, OPT_LISTAD },
    { "lda", no_argument, NULL, OPT_LISTAD },
    { "list", no_argument, NULL, OPT_LIST },
    { "l", no_argument, NULL, OPT_LIST },
    { "compat", no_argument, NULL, OPT_COMPAT },
    { "c", no_argument, NULL, OPT_COMPAT },
    { "optimal", no_argument, NULL, OPT_BEST },
    { "o", no_argument, NULL, OPT_BEST },
    { "test", no_argument, NULL, OPT_TEST },
    { "t", no_argument, NULL, OPT_TEST },
    { "current", no_argument, NULL, OPT_CURRENT},
    { "cur", no_argument, NULL, OPT_CURRENT},
    // 
    { "x", required_argument, NULL, OPT_X },
    { "y", required_argument, NULL, OPT_Y },
    { "vsync", required_argument, NULL, OPT_VSYNC },
    { "ref", required_argument, NULL, OPT_VSYNC },
    { "r", required_argument, NULL, OPT_VSYNC },
    { "bp", required_argument, NULL, OPT_BPP },
    { "d", required_argument, NULL, OPT_DISP },
    { "do", required_argument, NULL, OPT_DORIENT },
    { "orientation", required_argument, NULL, OPT_DORIENT },
    { "display", required_argument, NULL, OPT_DISP },

    { "fontsmooth", required_argument, NULL, OPT_FSMOOTH },
    { "fs", required_argument, NULL, OPT_FSMOOTH },
//    { "cleartype", required_argument, NULL, OPT_CLEARTYPE },
//    { "ct", required_argument, NULL, OPT_CLEARTYPE },

    { "k", no_argument, NULL, OPT_KBDINFO },
    { "kbd", no_argument, NULL, OPT_KBDINFO },
    { "kbdinfo", no_argument, NULL, OPT_KBDINFO },
    { "ks", required_argument, NULL, OPT_KBDSPEED },
    { "kbdspeed", required_argument, NULL, OPT_KBDSPEED },
    { "kd", required_argument, NULL, OPT_KBDDELAY },
    { "kbddelay", required_argument, NULL, OPT_KBDDELAY },
    { "kls", required_argument, NULL, OPT_KBDLANG },
    { "kbdlangswitch", required_argument, NULL, OPT_KBDLANG },
    { "kll", required_argument, NULL, OPT_KBDLANG },
    { "kbdlayoutswitch", required_argument, NULL, OPT_KBDLANG_LAYOUT },

    { "m", no_argument, NULL, OPT_MOUSEINFO },
    { "mouse", no_argument, NULL, OPT_MOUSEINFO },
    { "ms", required_argument, NULL, OPT_MOUSESPEED },
    { "mousespeed", required_argument, NULL, OPT_MOUSESPEED },
    { "ml", required_argument, NULL, OPT_MOUSELEFT },
    { "mouseleft", required_argument, NULL, OPT_MOUSELEFT },

    { NULL, no_argument, NULL, NULL }
};

ULONG opt_x = -1;
ULONG opt_y = -1;
ULONG opt_vsync = -1;
ULONG opt_bpp = -1;
ULONG opt_fs = -1;
ULONG opt_do = -1;
ULONG opt_disp = 1;
//ULONG opt_cleartype = -1;

BOOLEAN opt_save = FALSE;
BOOLEAN opt_list = FALSE;
BOOLEAN opt_compat = FALSE;
BOOLEAN opt_best = FALSE;
BOOLEAN opt_test = FALSE;
BOOLEAN opt_current = FALSE;
BOOLEAN opt_listd = FALSE;
BOOLEAN opt_listad = FALSE;

BOOLEAN opt_kbd = FALSE;

ULONG opt_kbdspeed = -1;
ULONG opt_kbddelay = -1;
ULONG opt_kbdlang  = -1;
ULONG opt_kbdlang_layout  = -1;

BOOLEAN opt_mouse = FALSE;

ULONG opt_mousespeed = -1;
ULONG opt_mouseleft = -1;

CHAR gDisplayDevName[128] = "\\\\.\\Display1";

#define CUR_DISPLAY_NAME_TPL  "\\\\.\\" "Display%d"
#define CUR_DISPLAY_NAME      (&(gDisplayDevName[0])+4)
#define CUR_DISPLAY_DEV_NAME  gDisplayDevName

void
usage(void)
{
    fprintf(stderr, "UI Control utility, Version %d.%d (%x)\n"
        "Copyright (C) by AlterWare, " COPYRIGHT_YEARS "\n"
        "Home site: http://www.alter.org.ua\n", VERSION_MAJOR,
        VERSION_MINOR, VERSION_BUILD);
    fprintf(stderr,
        "Usage:\n"
        "\tuictl <options> \n"
        "General options:\n"
        "\t--help, --h, --?      display this screen\n"
        "\t--save, --s           save new settings in registry (system-wide change)\n"
        "Display options:\n"
        "\t--d=NUM or\n"
        "\t  --display=NUM       display ID (1 - first)\n"
        "\t--x=NUM               X-resolution\n"
        "\t--y=NUM               Y-resolution\n"
        "\t--ref=NUM or\n"
        "\t  --vsync=NUM         Vertical Sync. (Hz)\n"
        "\t--bp=NUM              bits per pixel\n"
        "\t--orientation=NUM or\n"
        "\t  --do=NUM            display orientation, 0 (default), 90/cw, 180/flip, 270/ccw (Win2000+)\n"
        "\t--list, --l           display all available modes (compatible with --ld, --lad)\n"
        "\t--list-displays or\n"
        "\t  --ld                display all active display devices\n"
        "\t--list-all-displays or\n"
        "\t  --lad               display all display devices\n"
        "\t--compat, --c         display display-compatible modes only (Win2000+)\n"
        "\t--optimal, --o        try to set 'optimal' mode ;)\n"
        "\t--test, --t           test requested mode\n"
        "\t--fontsmooth=NUM or\n"
        "\t  --fs=NUM            enable or disable font smooth option (1/0 - enable/disable)\n"
//        "\t--cleartype=NUM or\n"
//        "\t  --ct=NUM            enable or disable cleartype option (1/0 - enable/disable)\n"
        "Keyboard options:\n"
        "\t--kbdinfo, --k        display keyboard settings\n"
        "\t--kbdspeed=NUM or\n"
        "\t  --ks=NUM            set keyboard speed (0 - 31)\n"
        "\t--kbddelay=NUM or\n"
        "\t  --kd=NUM            set keyboard delay (0 - 3)\n"
        "\t--kbdlangswitch=NUM or\n"
        "\t  --kls=NUM           set keyboard language switch keys\n"
        "\t                         1 - Alt-Shift, 2 - Ctrl-Shift, 3 - none\n"
        "\t--kbdlayoutswitch=NUM or\n"
        "\t  --kll=NUM           set keyboard layout switch keys (Win2000+)\n"
        "\t                         1 - Alt-Shift, 2 - Ctrl-Shift, 3 - none\n"
        "Mouse options:\n"
        "\t--mouseinfo, --m      display mouse settings (Win2000+)\n"
        "\t--mousespeed=NUM or\n"
        "\t  --ms=NUM            set mouse speed (1 - 20)\n"
        "\t--mouseleft=NUM or\n"
        "\t  --ml=NUM            switch mouse buttons to left-handed mode (0/1 - normal/swapped)\n"
        "\n"

        "Example:\n"
        "\tuictl --x=320 --y=240               set 320x200\n"
        "\tuictl --x=1600 --y=1280 --ref=85    set 1600x1280 85Hz\n"
        "\tuictl --do=90                       set portrait orientation, CW90\n"
        "\tuictl --ks=31 --kd=0                set maximum keyboard speed\n"
        "\tuictl --ms=15 --ml=1                set high mouse speed and swap left/rigth buttons\n"
    );
}

void
parse_args(int argc, char *argv[])
{
    int retval;
    while (argc > 1) {
            retval = getopt_long(argc, argv, "V", long_options, NULL);
        switch (retval)
        {
            case OPT_HELP:
            case OPT_H:
                usage();
                exit(0);
                break;

            case OPT_X:
                opt_x = strtoul(optarg, 0, 0);
                break;

            case OPT_Y:
                opt_y = strtoul(optarg, 0, 0);
                break;

            case OPT_DORIENT:
				if(!stricmp(optarg, "cw") ||
				   !stricmp(optarg, "cw90") ||
				   !stricmp(optarg, "portrait")) {
				    opt_do = 90;
				} else
				if(!stricmp(optarg, "flip")) {
				    opt_do = 180;
				} else
				if(!stricmp(optarg, "default")) {
				    opt_do = 0;
				} else
				if(!stricmp(optarg, "ccw") ||
				   !stricmp(optarg, "cwc90") ||
				   !stricmp(optarg, "portrait-flip")) {
				    opt_do = 270;
				} else {
                    opt_do = strtoul(optarg, 0, 0);
				}
				switch(opt_do) {
				case 0:
				case 90:
				case 180:
				case 270:
					break;
				default:
					usage();
					exit(0);
				}
                break;

            case OPT_VSYNC:
                opt_vsync = strtoul(optarg, 0, 0);
                break;

            case OPT_BPP:
                opt_bpp = strtoul(optarg, 0, 0);
                break;

            case OPT_DISP:
                opt_disp = strtoul(optarg, 0, 0);
                sprintf(gDisplayDevName, CUR_DISPLAY_NAME_TPL, opt_disp);
                break;

            case OPT_FSMOOTH:
                opt_fs = strtoul(optarg, 0, 0);
                break;
/*
            case OPT_CLEARTYPE:
                opt_cleartype = strtoul(optarg, 0, 0);
                break;
*/
            case OPT_SAVE:
                opt_save = TRUE;
                break;

            case OPT_LIST:
                opt_list = TRUE;
                break;

            case OPT_LISTD:
                opt_listd = TRUE;
                break;

            case OPT_LISTAD:
                opt_listd = TRUE;
                opt_listad = TRUE;
                break;

            case OPT_COMPAT:
                opt_compat = TRUE;
                break;

            case OPT_BEST:
                opt_best = TRUE;
                break;

            case OPT_TEST:
                opt_test = TRUE;
                break;

            case OPT_CURRENT:
                opt_list = TRUE;
                opt_current = TRUE;
                break;

            case OPT_KBDINFO:
                opt_kbd = TRUE;
                break;

            case OPT_KBDSPEED:
                opt_kbdspeed = strtoul(optarg, 0, 0);
                break;

            case OPT_KBDDELAY:
                opt_kbddelay = strtoul(optarg, 0, 0);
                break;

            case OPT_KBDLANG:
                opt_kbdlang = strtoul(optarg, 0, 0);
                break;

            case OPT_KBDLANG_LAYOUT:
                opt_kbdlang_layout = strtoul(optarg, 0, 0);
                break;

            case OPT_MOUSEINFO:
                opt_mouse = TRUE;
                break;

            case OPT_MOUSESPEED:
                opt_mousespeed = strtoul(optarg, 0, 0);
                break;

            case OPT_MOUSELEFT:
                opt_mouseleft = strtoul(optarg, 0, 0);
                break;

            case EOF:
                return;

            default:
                fprintf(stderr, "Try `uictl --help' "
                    "for more information\n");
                exit(-2);
        }
    }

}

typedef
BOOL
(WINAPI *ptrEnumDisplaySettingsEx) (
    LPCTSTR lpszDeviceName,  // display device
    DWORD iModeNum,          // graphics mode
    LPDEVMODE lpDevMode,     // graphics mode settings
    DWORD dwFlags            // options
    );

ptrEnumDisplaySettingsEx _EnumDisplaySettingsEx = NULL;

typedef
LONG
(WINAPI *ptrChangeDisplaySettingsEx) (
    LPCTSTR lpszDeviceName,  // name of display device
    LPDEVMODE lpDevMode,     // graphics mode
    HWND hwnd,               // not used; must be NULL
    DWORD dwflags,           // graphics mode options
    LPVOID lParam            // video parameters (or NULL)
    );

ptrChangeDisplaySettingsEx _ChangeDisplaySettingsEx = NULL;

LONG
MyChangeDisplaySettingsEx(
    LPCTSTR lpszDeviceName,  // name of display device
    LPDEVMODE lpDevMode,     // graphics mode
    HWND hwnd,               // not used; must be NULL
    DWORD dwflags,           // graphics mode options
    LPVOID lParam            // video parameters (or NULL)
    )
{
    if(_ChangeDisplaySettingsEx) {
        return _ChangeDisplaySettingsEx(lpszDeviceName, lpDevMode, hwnd, 
            dwflags, lParam);
    } else {
        return ChangeDisplaySettings(lpDevMode, dwflags);
    }
} // end MyChangeDisplaySettingsEx()

LONG
MyEnumDisplaySettingsEx(
    LPCTSTR lpszDeviceName,  // display device
    DWORD iModeNum,          // graphics mode
    LPDEVMODE lpDevMode,     // graphics mode settings
    DWORD dwFlags            // options
    )
{
    if(_EnumDisplaySettingsEx) {
		lpDevMode->dmSize = sizeof(DEVMODE);
		lpDevMode->dmDriverExtra = 0;
        return _EnumDisplaySettingsEx(lpszDeviceName, iModeNum, lpDevMode, dwFlags);
    } else {
        return EnumDisplaySettings(lpszDeviceName, iModeNum, lpDevMode);
    }
} // end MyEnumDisplaySettingsEx()

typedef
BOOL
(WINAPI *ptrEnumDisplayDevices) (
	LPCSTR           lpDevice,
	DWORD            iDevNum,
	PDISPLAY_DEVICEA lpDisplayDevice,
	DWORD            dwFlags
    );

ptrEnumDisplayDevices _EnumDisplayDevices = NULL;

LONG
MyEnumDisplayDevices(
	LPCSTR           lpDevice,
	DWORD            iDevNum,
	PDISPLAY_DEVICE  lpDisplayDevice,
	DWORD            dwFlags
    )
{
	memset(lpDisplayDevice, 0, sizeof(DISPLAY_DEVICE));
    if(_EnumDisplayDevices) {
		lpDisplayDevice->cb = sizeof(DISPLAY_DEVICE);
        return _EnumDisplayDevices(lpDevice, iDevNum, lpDisplayDevice, dwFlags);
    } else {
		if(lpDevice) {
			strcpy(lpDisplayDevice->DeviceName, lpDevice);
		} else {
			sprintf(lpDisplayDevice->DeviceName, CUR_DISPLAY_NAME_TPL, iDevNum+1);
		}
		return TRUE;
    }
} // end MyEnumDisplaySettingsEx()

__inline DWORD GetOrientation(
    PDEVMODE pDisplayMode
    )
{
    if(!(pDisplayMode->dmFields & DM_DISPLAYORIENTATION)) {
        return DMDO_DEFAULT;
    }
    DWORD* pdmDisplayOrientation = (DWORD*)((PVOID)&(pDisplayMode->dmScale));
//    DWORD* pdmDisplayOrientation = (DWORD*)((PVOID)&(pDisplayMode->dmDisplayOrientation));
    return *pdmDisplayOrientation;
} // end GetOrientation()

__inline VOID SetOrientation(
    PDEVMODE pDisplayMode,
    DWORD dmDisplayOrientation
    )
{
    DWORD* pdmDisplayOrientation = (DWORD*)((PVOID)&(pDisplayMode->dmScale));
//    DWORD* pdmDisplayOrientation = (DWORD*)((PVOID)&(pDisplayMode->dmDisplayOrientation));
    *pdmDisplayOrientation = dmDisplayOrientation;
;
} // end SetOrientation()

void
find_best(
    PDEVMODE pDisplayMode,
    BOOLEAN  max_vsync=FALSE
    );

void
find_best(
    PDEVMODE pDisplayMode,
    BOOLEAN  max_vsync
    )
{
    DEVMODE DisplayMode;
    ULONG flags = EDS_ROTATEDMODE;
    ULONG i=0;

    i=0;
    while(TRUE) {
        if(opt_compat && _EnumDisplaySettingsEx) {
            if(!_EnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, i, &DisplayMode, flags))
                break;
        } else {
            if(!EnumDisplaySettings(CUR_DISPLAY_DEV_NAME, i, &DisplayMode))
                break;
        }

        i++;

        if(max_vsync) {
            if(DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth &&
               DisplayMode.dmPelsHeight == pDisplayMode->dmPelsHeight &&
               DisplayMode.dmBitsPerPel == pDisplayMode->dmBitsPerPel &&
               DisplayMode.dmDisplayFrequency < pDisplayMode->dmDisplayFrequency) {
                *pDisplayMode = DisplayMode;
            }
            continue;
        }

        if(i==1) {
            *pDisplayMode = DisplayMode;
            continue;
        }

        // colors
        if(DisplayMode.dmBitsPerPel  > pDisplayMode->dmBitsPerPel) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmBitsPerPel  < pDisplayMode->dmBitsPerPel) {
            continue;
        }

        // freq
        if(DisplayMode.dmDisplayFrequency  > pDisplayMode->dmDisplayFrequency) {
            if(DisplayMode.dmDisplayFrequency >= 100) {
                if(DisplayMode.dmPelsWidth >= 1024) {
                    *pDisplayMode = DisplayMode;
                    continue;
                }
            } else
            if(DisplayMode.dmDisplayFrequency > 60) {
                if(DisplayMode.dmPelsWidth >= 800) {
                    *pDisplayMode = DisplayMode;
                    continue;
                }
            } else {
                *pDisplayMode = DisplayMode;
                continue;
            }
/*
            if(DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth &&
               DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth) {
                *pDisplayMode = DisplayMode;
                continue;
            }
*/

        } else
        if(DisplayMode.dmDisplayFrequency  < pDisplayMode->dmDisplayFrequency) {
            continue;
        }

        if(opt_do != -1) {
            DWORD dmDisplayOrientation = GetOrientation(&DisplayMode);
            if(dmDisplayOrientation * 90  != opt_do) {
                continue;
            }
            /*
            switch(dmDisplayOrientation) {
            case DMDO_90:
                if(opt_do != 90)
                    continue;
                break;
            case DMDO_180:
                if(opt_do != 180)
                    continue;
                break;
            case DMDO_270:
                if(opt_do != 270)
                    continue;
                break;
            case DMDO_DEFAULT:
                if(opt_do != 0)
                    continue;
                break;
            }*/
        }

        // x-size
        if(DisplayMode.dmPelsWidth  > pDisplayMode->dmPelsWidth) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmPelsWidth  < pDisplayMode->dmPelsWidth) {
            continue;
        }

        // y-size
        if(DisplayMode.dmPelsHeight  > pDisplayMode->dmPelsHeight) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmPelsHeight  < pDisplayMode->dmPelsHeight) {
            continue;
        }
    }
    if(!max_vsync) {
        //find_best(pDisplayMode, TRUE);
    }
} // end find_best()

VOID
CenterWindow(
    HWND hDlg
    )
{
    RECT            aRt;

    // center the dialog box
    GetWindowRect( hDlg, &aRt );
    OffsetRect( &aRt, -aRt.left, -aRt.top );
    MoveWindow( hDlg,
    		((GetSystemMetrics( SM_CXSCREEN ) -
    			aRt.right ) / 2 + 4) & ~7,
      		(GetSystemMetrics( SM_CYSCREEN ) -
    			aRt.bottom) / 2,
    		aRt.right, aRt.bottom, 0 );
} // end CenterWindow()

ULONG gCounter;
DEVMODE gDisplayMode;

VOID
SetOrigInfo(
    HWND hDlg
    )
{
    CHAR txt[256];
    sprintf(txt, "Original mode %dx%d %dbit %dHz\nwill be restored in %d seconds",
           gDisplayMode.dmPelsWidth,
           gDisplayMode.dmPelsHeight,
           gDisplayMode.dmBitsPerPel,
           gDisplayMode.dmDisplayFrequency,
           gCounter
           );
    SetDlgItemText(hDlg, IDC_ORIG, txt);
    return;
} // end SetOrigInfo()


INT_PTR
APIENTRY
QuestionDialog(
    HWND hDlg,
    UINT message,
    WPARAM wParam,
    LPARAM lParam
    )
{
    switch (message) {
    case WM_INITDIALOG :

    	// center the window
    	CenterWindow(hDlg);

    	gCounter = 10;
        SetOrigInfo(hDlg);
    	// Start up timer to exit after timeout
    	SetTimer( hDlg,	1, 1000/*ms*/, NULL );
    	break;

    case WM_COMMAND:
    	switch (LOWORD(wParam)) {
    	case IDOK:
    	case IDCANCEL:

            EndDialog(hDlg, LOWORD(wParam));
            return TRUE;
        }
    	break; 

    case WM_TIMER:

        SetOrigInfo(hDlg);
        gCounter--;
        if(gCounter)
            break;

    case WM_CLOSE:	

        EndDialog(hDlg, IDCANCEL);
        return TRUE;
    }
    return FALSE;
} // end QuestionDialog()

char*
get_lng_sw_str(
    int num
    )
{
    switch(num) {
    case 1:
        return "Alt-Shift";
    case 2:
        return "Ctrl-Shift";
    case 3:
        return "None";
    }
    return "unknown";
} // end get_lng_sw_str()

void main(int argc, char* argv[]) {

    DEVMODE DisplayMode;
    DEVMODE DisplayMode0;
	DISPLAY_DEVICE DisplayDevice;
    ULONG flags = 0;
    ULONG enumFlags = EDS_ROTATEDMODE;
    ULONG i=0;
    HMODULE hMod = NULL; // == hInstance
//    ULONG vsync_list[128];
//    ULONG old_vsync_list[128];
//    ULONG vsync_list_sz;
//    ULONG old_vsync_list_sz;
//    ULONG last_i=0;
    
    ULONG old_x = -1;
    ULONG old_y = -1;
//    ULONG old_vsync = -1;
	ULONG old_freq = -1;
    ULONG old_bpp = -1;
    ULONG old_kbdspeed = -1;
    ULONG old_kbddelay = -1;
    ULONG old_mousespeed = -1;
    BOOLEAN printed;
//    BOOLEAN scan = TRUE;
    HKEY hKey = NULL;
    ULONG res;

    if ( argc < 2 ) {
        usage();
        exit(-1);
    }
    parse_args(argc, argv);

    hMod = GetModuleHandle("User32.dll");
    if(hMod) {
        _EnumDisplaySettingsEx = (ptrEnumDisplaySettingsEx)GetProcAddress(hMod, "EnumDisplaySettingsEx");
        if(!_EnumDisplaySettingsEx) {
            _EnumDisplaySettingsEx = (ptrEnumDisplaySettingsEx)GetProcAddress(hMod, "EnumDisplaySettingsExA");
    	}
        _ChangeDisplaySettingsEx = (ptrChangeDisplaySettingsEx)GetProcAddress(hMod, "ChangeDisplaySettingsEx");
    	if(!_ChangeDisplaySettingsEx) {
            _ChangeDisplaySettingsEx = (ptrChangeDisplaySettingsEx)GetProcAddress(hMod, "ChangeDisplaySettingsExA");
    	}
		_EnumDisplayDevices = (ptrEnumDisplayDevices)GetProcAddress(hMod, "EnumDisplayDevicesA");
    }

    if(opt_kbd) {
        SystemParametersInfo( SPI_GETKEYBOARDSPEED,
                              0,
                              &old_kbdspeed,
                              0);
        SystemParametersInfo( SPI_GETKEYBOARDDELAY,
                              0,
                              &old_kbddelay,
                              0);
        printf("Keyboard speed: %d (0-31)\n", old_kbdspeed);
        printf("Keyboard delay: %d (0-3)\n", old_kbddelay);
    }


    if(opt_kbdspeed != -1) {
        if(SystemParametersInfo( SPI_SETKEYBOARDSPEED,
                              opt_kbdspeed,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Keyboard speed changed to %d\n", opt_kbdspeed);
        } else {
            printf("Keyboard speed not set\n");
        }
        opt_kbd = TRUE;
    }

    if(opt_kbddelay != -1) {
        if(SystemParametersInfo( SPI_SETKEYBOARDDELAY,
                              opt_kbddelay,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Keyboard delay changed to %d\n", opt_kbddelay);
        } else {
            printf("Keyboard delay not set\n");
        }
        opt_kbd = TRUE;
    }

    if(opt_kbdlang != -1 ||
       opt_kbdlang_layout != -1) {

        res = RegOpenKeyEx(HKEY_CURRENT_USER, "Keyboard Layout\\Toggle", 0, KEY_ALL_ACCESS, &hKey);
        if(res == ERROR_SUCCESS) {
            if(opt_kbdlang != -1) {
                res = RegSetValueEx(hKey, "Hotkey", NULL, REG_DWORD, (PUCHAR)(&opt_kbdlang), sizeof(DWORD));
                res = RegSetValueEx(hKey, "Language Hotkey", NULL, REG_DWORD, (PUCHAR)(&opt_kbdlang), sizeof(DWORD));
            }
            if(opt_kbdlang_layout != -1) {
                res = RegSetValueEx(hKey, "Layout Hotkey", NULL, REG_DWORD, (PUCHAR)(&opt_kbdlang_layout), sizeof(DWORD));
            }
            RegCloseKey(hKey);
        }
        if((res == ERROR_SUCCESS) &&
           SystemParametersInfo( SPI_SETLANGTOGGLE,
                              0,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Keyboard language is now switched with %s\n", get_lng_sw_str(opt_kbdlang));
        } else {
            printf("Keyboard language switch hotkey is not adjusted\n");
        }
        opt_kbd = TRUE;
    }

    if(opt_mouse) {
        if(SystemParametersInfo( SPI_GETMOUSESPEED,
                              0,
                              &old_mousespeed,
                              0)) {
            printf("Mouse speed: %d (1-20)\n", old_mousespeed);
        } else {
            printf("Mouse speed: information unavailable\n");
        }
    }

    if(opt_mousespeed != -1) {
        if(SystemParametersInfo( SPI_SETMOUSESPEED,
                              0,
                              (PVOID)opt_mousespeed,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Mouse speed: %d\n", opt_mousespeed);
        } else {
            printf("Mouse speed not set\n");
        }
        opt_mouse = TRUE;
    }

    if(opt_mouseleft != -1) {
        if(SystemParametersInfo( SPI_SETMOUSEBUTTONSWAP,
                              opt_mouseleft,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Mouse buttons switched to %s mode\n", opt_mouseleft ? "left-handed" : "normal");
        } else {
            printf("Mouse button swap not adjusted\n");
        }
        opt_mouse = TRUE;
    }

    if(opt_listd) {
        opt_disp = 1;
    }
    while(opt_listd || opt_list) {
        printed = FALSE;
        //sprintf(gDisplayDevName, CUR_DISPLAY_NAME_TPL, opt_disp);
		i = MyEnumDisplayDevices(NULL, opt_disp-1, &DisplayDevice, 0);
		if(i) {
		    strcpy(gDisplayDevName, DisplayDevice.DeviceName);
		}

        if(!i ||
           !MyEnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, ENUM_CURRENT_SETTINGS, &DisplayMode, enumFlags)) {
            if(opt_listad && MyEnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, 0, &DisplayMode, enumFlags)) {
                printf("--- %s (inactive):\n", CUR_DISPLAY_NAME);
            } else
            if(!opt_listd || (opt_disp > 16)) {
                if(opt_list) {
                    exit(1);
                }
                break;
            }
        } else {
			printf("*** %s %s:\n", CUR_DISPLAY_NAME, DisplayDevice.DeviceString);
			switch(GetOrientation(&DisplayMode)) {
			case DMDO_90:
				printf("  CW90\n"); break;
			case DMDO_180:
				printf("  FLIP\n"); break;
			case DMDO_270:
				printf("  CCW90\n"); break;
			}
			if(DisplayMode.dmPosition.x || DisplayMode.dmPosition.y) {
				printf("   V-pos %d,%d\n", DisplayMode.dmPosition.x, DisplayMode.dmPosition.y);
			}

        }
        if(opt_list) {

            if(opt_best || opt_current) {
                if(opt_current) {
                    MyEnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, ENUM_CURRENT_SETTINGS, &DisplayMode, 0);
                } else {
                    find_best(&DisplayMode);
                }
                printf("%s mode: %dx%d %dbit %dHz",
                       opt_current ? "Current" : "Optimal",
                       DisplayMode.dmPelsWidth,
                       DisplayMode.dmPelsHeight,
                       DisplayMode.dmBitsPerPel,
                       DisplayMode.dmDisplayFrequency
                       );
                if(DisplayMode.dmFields & DM_DISPLAYORIENTATION) {
                    switch(GetOrientation(&DisplayMode)) {
                    case DMDO_90:
                        printf(" CW90"); break;
                    case DMDO_180:
                        printf(" FLIP"); break;
                    case DMDO_270:
                        printf(" CCW90"); break;
                    }
                }
                if(SystemParametersInfo( SPI_GETFONTSMOOTHING,
                                      0,
                                      &opt_fs,
                                      0)) {
                    printf(" Font smooth is %s\n", opt_fs ? "ON" : "OFF");
                }
    /*
                if(SystemParametersInfo( SPI_GETCLEARTYPE,
                                      0,
                                      &opt_cleartype,
                                      0)) {
                    printf("Font smooth is %s\n", opt_cleartype ? "ON" : "OFF");
                }
    */

                if(opt_listd) {
                    opt_disp++;
                    continue;
                }
                exit(1);
            }

            i=0;
            while(TRUE) {
                if(opt_compat && _EnumDisplaySettingsEx) {
                    if(!_EnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, i, &DisplayMode, 0))
                        break;
                } else {
                    if(!MyEnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, i, &DisplayMode, enumFlags))
                        break;
                }
                i++;
                if(opt_x != -1 &&
                   DisplayMode.dmPelsWidth != opt_x)
                    continue;
                if(opt_y != -1 &&
                   DisplayMode.dmPelsHeight != opt_y)
                    continue;
                if(opt_vsync != -1 &&
                   DisplayMode.dmDisplayFrequency != opt_vsync)
                    continue;
                if(opt_bpp != -1 &&
                   DisplayMode.dmBitsPerPel != opt_bpp)
                    continue;

                if(((DisplayMode.dmPelsWidth  == old_x &&
                   DisplayMode.dmPelsHeight == old_y) ||
				    (DisplayMode.dmPelsWidth  == old_y &&
                   DisplayMode.dmPelsHeight == old_x)) &&
                   DisplayMode.dmBitsPerPel == old_bpp) {
    //                if(!scan)
					if(old_freq != DisplayMode.dmDisplayFrequency) {
                        printf("/%d", DisplayMode.dmDisplayFrequency);
						old_freq = DisplayMode.dmDisplayFrequency;
					}
    //                vsync_list_sz++;
                } else {
                    if(printed)
                        printf("Hz\n");
    //                vsync_list_sz = 0;
                    printf("%dx%d %dbit %d",
                           DisplayMode.dmPelsWidth,
                           DisplayMode.dmPelsHeight,
                           DisplayMode.dmBitsPerPel,
                           DisplayMode.dmDisplayFrequency
                           );
                    printed = TRUE;
                    old_x   = DisplayMode.dmPelsWidth;
                    old_y   = DisplayMode.dmPelsHeight;
                    old_bpp = DisplayMode.dmBitsPerPel;
                    old_freq = DisplayMode.dmDisplayFrequency;
                }
            }
            if(printed)
                printf("Hz\n");
        }
        if(opt_listd) {
            opt_disp++;
        } else {
            exit(1);
        }
    }
    if(opt_listd) {
        exit(1);
    }

    if(opt_best) {
        find_best(&DisplayMode);
        printf("Seting %dx%d %dbit %dHz",
               DisplayMode.dmPelsWidth,
               DisplayMode.dmPelsHeight,
               DisplayMode.dmBitsPerPel,
               DisplayMode.dmDisplayFrequency
               );
        opt_x     = DisplayMode.dmPelsWidth;
        opt_y     = DisplayMode.dmPelsHeight;
        if(opt_vsync == -1) {
            opt_vsync = DisplayMode.dmDisplayFrequency;
        }
        opt_bpp   = DisplayMode.dmBitsPerPel;
    }
    i = MyEnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, ENUM_CURRENT_SETTINGS, &DisplayMode, enumFlags);
	if(!i) {
		printf("Can't read current mode\n");
		exit(0);
	}
    //if(opt_test) {
        DisplayMode0 = DisplayMode;
    //}

    if(opt_fs != -1) {
        if(SystemParametersInfo( SPI_SETFONTSMOOTHING,
                              opt_fs,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Font smooth is %s\n", opt_fs ? "ON" : "OFF");
        } else {
            printf("Font smooth not adjusted\n");
        }
    }
/*
    if(opt_cleartype != -1) {
        if(SystemParametersInfo( SPI_SETCLEARTYPE,
                              opt_cleartype,
                              0,
                              (opt_save ? SPIF_UPDATEINIFILE : 0) | SPIF_SENDWININICHANGE )) {
            printf("Cleartype is %s\n", opt_fs ? "ON" : "OFF");
        } else {
            printf("Cleartype not adjusted\n");
        }
    }
*/
    if(opt_x == -1 && opt_y == -1 && opt_vsync == -1 && opt_bpp == -1 && opt_do == -1) {

        if(opt_kbd || opt_mouse || opt_fs!=-1) {
            exit(1);
        }

        //printf("Display mode not specified\n");
        usage();
        exit(0);
    }

	DisplayMode.dmDisplayFlags = 0;
    // Set X-resolution
    if(opt_x != -1) {
        DisplayMode.dmPelsWidth = opt_x;
		flags |= DM_PELSWIDTH;
    }
    // Set Y-resolution
    if(opt_y != -1) {
        DisplayMode.dmPelsHeight = opt_y;
		flags |= DM_PELSHEIGHT;
    }
    // optionally set v-sync
    if(opt_vsync != -1) {
        DisplayMode.dmDisplayFrequency = opt_vsync;
        flags |= DM_DISPLAYFREQUENCY;
    }
    // optionally set color depth
    if(opt_bpp != -1) {
        DisplayMode.dmBitsPerPel = opt_bpp;
        flags |= DM_BITSPERPEL;
    }
    if(opt_do != -1) {
    	DWORD dmDisplayOrientation, dmDisplayOrientation0;
    	switch(opt_do) {
    	case 90:
    		dmDisplayOrientation = DMDO_90; break;
    	case 180:
    		dmDisplayOrientation = DMDO_180; break;
    	case 270:
    		dmDisplayOrientation = DMDO_270; break;
    	case 0:
    		dmDisplayOrientation = DMDO_DEFAULT; break;
    	default:
    		exit(0);
    	}
    	dmDisplayOrientation0 = GetOrientation(&DisplayMode);
    	SetOrientation(&DisplayMode, dmDisplayOrientation);

    	if(opt_x != -1 && opt_y != -1) {
    	    if(dmDisplayOrientation0 == DMDO_270 || dmDisplayOrientation0 == DMDO_270) {
    	        if(opt_x > opt_y) {
                    DisplayMode.dmPelsWidth = opt_y;
                    DisplayMode.dmPelsHeight = opt_x;
    	        }
    	    }
    	} else
    	if((dmDisplayOrientation0 == DMDO_DEFAULT || dmDisplayOrientation0 == DMDO_180) ^ 
    		(dmDisplayOrientation == DMDO_DEFAULT || dmDisplayOrientation == DMDO_180))  {
            flags |= DM_PELSWIDTH;
            flags |= DM_PELSHEIGHT;
            DisplayMode.dmPelsWidth = DisplayMode0.dmPelsHeight;
            DisplayMode.dmPelsHeight = DisplayMode0.dmPelsWidth;
    	}
    flags |= DM_DISPLAYORIENTATION;
    	//DisplayMode.dmDisplayFlags = DM_DISPLAYORIENTATION;
    }
    DisplayMode.dmFields = DM_DISPLAYFLAGS | flags;

    i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
        CDS_SET_PRIMARY, NULL);
	switch(i) {
	case DISP_CHANGE_SUCCESSFUL:
		printf("OK\n"); break;
	case DISP_CHANGE_BADDUALVIEW:
		printf("Dual veiw error\n"); break;
	case DISP_CHANGE_BADFLAGS:
		printf("Bad flags\n"); break;
	case DISP_CHANGE_BADMODE:
		printf("Bad mode\n"); break;
	case DISP_CHANGE_BADPARAM:
		printf("Bad param\n"); break;
	case DISP_CHANGE_NOTUPDATED:
		printf("Can't update registry\n"); break;
	case DISP_CHANGE_RESTART:
		printf("Need restart\n"); break;

	}
    if(opt_test) {
        gDisplayMode = DisplayMode;
        if(IDOK != DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_QUESTION), NULL, QuestionDialog)) {
            DisplayMode = DisplayMode0;
            i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
                CDS_SET_PRIMARY, NULL);
        }
    }
    if(opt_save && (i==DISP_CHANGE_SUCCESSFUL || i==DISP_CHANGE_RESTART)) {
        i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
            CDS_UPDATEREGISTRY, NULL);
    }
    exit(1);
} // end main()
