#include <windows.h>
#include <stdio.h>
#include "getopt.h"
#include "winuser.h"
#include "resource.h"
#include "ver.h"

/*
 * Command line option token values.
 *  0x0000-0x00ff   Single characters
 *  0x1000-0x1fff   Long switches (no arg)
 *  0x2000-0x2fff   Long settings (arg required)
 */
#define OPT_H           0x0001

#define OPT_HELP        0x1000
#define OPT_SAVE        0x1001
#define OPT_LIST        0x1002
#define OPT_COMPAT      0x1003
#define OPT_BEST        0x1004
#define OPT_TEST        0x1005
#define OPT_CURRENT     0x1006

#define OPT_X           0x2000
#define OPT_Y           0x2001
#define OPT_BPP         0x2002
#define OPT_VSYNC       0x2003

struct option long_options[] = {
    { "?", no_argument, NULL, OPT_H },
    { "h", no_argument, NULL, OPT_H },
    { "help", no_argument, NULL, OPT_HELP },
    { "save", no_argument, NULL, OPT_SAVE },
    { "s", no_argument, NULL, OPT_SAVE },
    { "list", no_argument, NULL, OPT_LIST },
    { "l", no_argument, NULL, OPT_LIST },
    { "compat", no_argument, NULL, OPT_COMPAT },
    { "c", no_argument, NULL, OPT_COMPAT },
    { "optimal", no_argument, NULL, OPT_BEST },
    { "o", no_argument, NULL, OPT_BEST },
    { "test", no_argument, NULL, OPT_TEST },
    { "t", no_argument, NULL, OPT_TEST },
    { "current", no_argument, NULL, OPT_CURRENT},
    { "cur", no_argument, NULL, OPT_CURRENT},
    // Logical Volume Descriptor
    { "x", required_argument, NULL, OPT_X },
    { "y", required_argument, NULL, OPT_Y },
    { "vsync", required_argument, NULL, OPT_VSYNC },
    { "ref", required_argument, NULL, OPT_VSYNC },
    { "r", required_argument, NULL, OPT_VSYNC },
    { "bp", required_argument, NULL, OPT_BPP },

    { NULL, no_argument, NULL, NULL }
};

ULONG opt_x = -1;
ULONG opt_y = -1;
ULONG opt_vsync = -1;
ULONG opt_bpp = -1;

BOOLEAN opt_save = FALSE;
BOOLEAN opt_list = FALSE;
BOOLEAN opt_compat = FALSE;
BOOLEAN opt_best = FALSE;
BOOLEAN opt_test = FALSE;
BOOLEAN opt_current = FALSE;

#define CUR_DISPLAY_NAME     "Display1"
#define CUR_DISPLAY_DEV_NAME "\\\\.\\" CUR_DISPLAY_NAME

void
usage(void)
{
    fprintf(stderr, "Display Mode Control utility, Version %d.%d (%x)\n"
        "Copyright (C) by AlterWare, " COPYRIGHT_YEARS "\n"
        "Home site: http://www.alter.org.ua", VERSION_MAJOR,
        VERSION_MINOR, VERSION_BUILD);
    fprintf(stderr,
        "Usage:\n"
        "\tvmctl <options> \n"
        "Options:\n"
        "\t--help, --h, --?      display this screen\n"
        "\t--x=NUMBER            X-resolution\n"
        "\t--y=NUMBER            Y-resolution\n"
        "\t--ref=NUMBER\n"
        "\t  or\n"
        "\t--vsync=NUMBER        Vertical Sync. (Hz)\n"
        "\t--bp=NUMBER           bits per pixel\n"
        "\t--save, --s           save new settings in registry (system-wide change)\n"
        "\t--list, --l           display all available modes\n"
        "\t--compat, --c         display display-compatible modes only (Win2000+)\n"
        "\t--optimal, --o        try to set 'optimal' mode ;)\n"
        "\t--test, --t           test requested mode\n"
        "\n"

        "Example:\n"
        "\tvmctl --x=320 --y=240\n"
        "\tvmctl --x=1600 --y=1280 --ref=85\n"
    );
}

void
parse_args(int argc, char *argv[])
{
    int retval;
    while (argc > 1) {
            retval = getopt_long(argc, argv, "V", long_options, NULL);
        switch (retval)
        {
            case OPT_HELP:
            case OPT_H:
                usage();
                exit(0);
                break;

            case OPT_X:
                opt_x = strtoul(optarg, 0, 0);
                break;

            case OPT_Y:
                opt_y = strtoul(optarg, 0, 0);
                break;

            case OPT_VSYNC:
                opt_vsync = strtoul(optarg, 0, 0);
                break;

            case OPT_BPP:
                opt_bpp = strtoul(optarg, 0, 0);
                break;

            case OPT_SAVE:
                opt_save = TRUE;
                break;

            case OPT_LIST:
                opt_list = TRUE;
                break;

            case OPT_COMPAT:
                opt_compat = TRUE;
                break;

            case OPT_BEST:
                opt_best = TRUE;
                break;

            case OPT_TEST:
                opt_test = TRUE;
                break;

            case OPT_CURRENT:
                opt_list = TRUE;
                opt_current = TRUE;
                break;

            case EOF:
                return;

            default:
                fprintf(stderr, "Try `vmctl --help' "
                    "for more information\n");
                exit(-2);
        }
    }

}

typedef
BOOL
(*ptrEnumDisplaySettingsEx) (
    LPCTSTR lpszDeviceName,  // display device
    DWORD iModeNum,          // graphics mode
    LPDEVMODE lpDevMode,     // graphics mode settings
    DWORD dwFlags            // options
    );

ptrEnumDisplaySettingsEx _EnumDisplaySettingsEx = NULL;

typedef
LONG
(*ptrChangeDisplaySettingsEx) (
    LPCTSTR lpszDeviceName,  // name of display device
    LPDEVMODE lpDevMode,     // graphics mode
    HWND hwnd,               // not used; must be NULL
    DWORD dwflags,           // graphics mode options
    LPVOID lParam            // video parameters (or NULL)
    );

ptrChangeDisplaySettingsEx _ChangeDisplaySettingsEx = NULL;

LONG
MyChangeDisplaySettingsEx(
    LPCTSTR lpszDeviceName,  // name of display device
    LPDEVMODE lpDevMode,     // graphics mode
    HWND hwnd,               // not used; must be NULL
    DWORD dwflags,           // graphics mode options
    LPVOID lParam            // video parameters (or NULL)
    )
{
    if(_ChangeDisplaySettingsEx) {
        return _ChangeDisplaySettingsEx(lpszDeviceName, lpDevMode, hwnd, 
            dwflags, lParam);
    } else {
        return ChangeDisplaySettings(lpDevMode, dwflags);
    }
} // end MyChangeDisplaySettingsEx()

void
find_best(
    PDEVMODE pDisplayMode,
    BOOLEAN  max_vsync=FALSE
    );

void
find_best(
    PDEVMODE pDisplayMode,
    BOOLEAN  max_vsync
    )
{
    DEVMODE DisplayMode;
    ULONG flags = 0;
    ULONG i=0;

    i=0;
    while(TRUE) {
        if(opt_compat && _EnumDisplaySettingsEx) {
            if(!_EnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, i, &DisplayMode, 0))
                break;
        } else {
            if(!EnumDisplaySettings(CUR_DISPLAY_DEV_NAME, i, &DisplayMode))
                break;
        }

        i++;

        if(max_vsync) {
            if(DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth &&
               DisplayMode.dmPelsHeight == pDisplayMode->dmPelsHeight &&
               DisplayMode.dmBitsPerPel == pDisplayMode->dmBitsPerPel &&
               DisplayMode.dmDisplayFrequency < pDisplayMode->dmDisplayFrequency) {
                *pDisplayMode = DisplayMode;
            }
            continue;
        }

        if(i==1) {
            *pDisplayMode = DisplayMode;
            continue;
        }

        // colors
        if(DisplayMode.dmBitsPerPel  > pDisplayMode->dmBitsPerPel) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmBitsPerPel  < pDisplayMode->dmBitsPerPel) {
            continue;
        }

        // freq
        if(DisplayMode.dmDisplayFrequency  > pDisplayMode->dmDisplayFrequency) {
            if(DisplayMode.dmDisplayFrequency >= 100) {
                if(DisplayMode.dmPelsWidth >= 1024) {
                    *pDisplayMode = DisplayMode;
                    continue;
                }
            } else
            if(DisplayMode.dmDisplayFrequency > 60) {
                if(DisplayMode.dmPelsWidth >= 800) {
                    *pDisplayMode = DisplayMode;
                    continue;
                }
            } else {
                *pDisplayMode = DisplayMode;
                continue;
            }
/*
            if(DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth &&
               DisplayMode.dmPelsWidth  == pDisplayMode->dmPelsWidth) {
                *pDisplayMode = DisplayMode;
                continue;
            }
*/

        } else
        if(DisplayMode.dmDisplayFrequency  < pDisplayMode->dmDisplayFrequency) {
            continue;
        }

        // x-size
        if(DisplayMode.dmPelsWidth  > pDisplayMode->dmPelsWidth) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmPelsWidth  < pDisplayMode->dmPelsWidth) {
            continue;
        }

        // y-size
        if(DisplayMode.dmPelsHeight  > pDisplayMode->dmPelsHeight) {
            *pDisplayMode = DisplayMode;
            continue;
        } else
        if(DisplayMode.dmPelsHeight  < pDisplayMode->dmPelsHeight) {
            continue;
        }
    }
    if(!max_vsync) {
        //find_best(pDisplayMode, TRUE);
    }
} // end find_best()

VOID
CenterWindow(
    HWND hDlg
    )
{
    RECT            aRt;

    // center the dialog box
    GetWindowRect( hDlg, &aRt );
    OffsetRect( &aRt, -aRt.left, -aRt.top );
    MoveWindow( hDlg,
    		((GetSystemMetrics( SM_CXSCREEN ) -
    			aRt.right ) / 2 + 4) & ~7,
      		(GetSystemMetrics( SM_CYSCREEN ) -
    			aRt.bottom) / 2,
    		aRt.right, aRt.bottom, 0 );
} // end CenterWindow()

ULONG gCounter;
DEVMODE gDisplayMode;

VOID
SetOrigInfo(
    HWND hDlg
    )
{
    CHAR txt[256];
    sprintf(txt, "Original mode %dx%d %dbit %dHz\nwill be restored in %d seconds",
           gDisplayMode.dmPelsWidth,
           gDisplayMode.dmPelsHeight,
           gDisplayMode.dmBitsPerPel,
           gDisplayMode.dmDisplayFrequency,
           gCounter
           );
    SetDlgItemText(hDlg, IDC_ORIG, txt);
    return;
} // end SetOrigInfo()

INT
APIENTRY
QuestionDialog(
    HWND hDlg,
    UINT message,
    UINT wParam,
    LONG lParam
    )
{
    switch (message) {
    case WM_INITDIALOG :

    	// center the window
    	CenterWindow(hDlg);

    	gCounter = 10;
        SetOrigInfo(hDlg);
    	// Start up timer to exit after timeout
    	SetTimer( hDlg,	1, 1000/*ms*/, NULL );
    	break;

    case WM_COMMAND:
    	switch (LOWORD(wParam)) {
    	case IDOK:
    	case IDCANCEL:

            EndDialog(hDlg, LOWORD(wParam));
            return TRUE;
        }
    	break; 

    case WM_TIMER:

        SetOrigInfo(hDlg);
        gCounter--;
        if(gCounter)
            break;

    case WM_CLOSE:	

        EndDialog(hDlg, IDCANCEL);
        return TRUE;
    }
    return FALSE;
} // end QuestionDialog()

void main (int argc, char* argv[]) {

    DEVMODE DisplayMode;
    DEVMODE DisplayMode0;
    ULONG flags = 0;
    ULONG i=0;
    HMODULE hMod = NULL; // == hInstance
//    ULONG vsync_list[128];
//    ULONG old_vsync_list[128];
//    ULONG vsync_list_sz;
//    ULONG old_vsync_list_sz;
//    ULONG last_i=0;
    
    ULONG old_x = -1;
    ULONG old_y = -1;
//    ULONG old_vsync = -1;
    ULONG old_bpp = -1;
    BOOLEAN printed = FALSE;
//    BOOLEAN scan = TRUE;

    if ( argc < 2 ) {
        usage();
        exit(-1);
    }
    parse_args(argc, argv);

    hMod = GetModuleHandle("User32.dll");
    if(hMod) {
        _EnumDisplaySettingsEx = (ptrEnumDisplaySettingsEx)GetProcAddress(hMod, "EnumDisplaySettingsEx");
        _ChangeDisplaySettingsEx = (ptrChangeDisplaySettingsEx)GetProcAddress(hMod, "ChangeDisplaySettingsEx");
    }

    if(opt_list) {

        if(opt_best || opt_current) {
            if(opt_current) {
                EnumDisplaySettings(CUR_DISPLAY_DEV_NAME, ENUM_CURRENT_SETTINGS, &DisplayMode);
            } else {
                find_best(&DisplayMode);
            }
            printf("%s mode: %dx%d %dbit %dHz",
                   opt_current ? "Current" : "Optimal",
                   DisplayMode.dmPelsWidth,
                   DisplayMode.dmPelsHeight,
                   DisplayMode.dmBitsPerPel,
                   DisplayMode.dmDisplayFrequency
                   );
            exit(1);
        }

        i=0;
        while(TRUE) {
            if(opt_compat && _EnumDisplaySettingsEx) {
                if(!_EnumDisplaySettingsEx(CUR_DISPLAY_DEV_NAME, i, &DisplayMode, 0))
                    break;
            } else {
                if(!EnumDisplaySettings(CUR_DISPLAY_DEV_NAME, i, &DisplayMode))
                    break;
            }
            i++;
            if(opt_x != -1 &&
               DisplayMode.dmPelsWidth != opt_x)
                continue;
            if(opt_y != -1 &&
               DisplayMode.dmPelsHeight != opt_y)
                continue;
            if(opt_vsync != -1 &&
               DisplayMode.dmDisplayFrequency != opt_vsync)
                continue;
            if(opt_bpp != -1 &&
               DisplayMode.dmBitsPerPel != opt_bpp)
                continue;

            if(DisplayMode.dmPelsWidth  == old_x &&
               DisplayMode.dmPelsHeight == old_y &&
               DisplayMode.dmBitsPerPel == old_bpp) {
//                if(!scan)
                    printf("/%d", DisplayMode.dmDisplayFrequency);
//                vsync_list_sz++;
            } else {
                if(printed)
                    printf("Hz\n");
//                vsync_list_sz = 0;
                printf("%dx%d %dbit %d",
                       DisplayMode.dmPelsWidth,
                       DisplayMode.dmPelsHeight,
                       DisplayMode.dmBitsPerPel,
                       DisplayMode.dmDisplayFrequency
                       );
                printed = TRUE;
                old_x   = DisplayMode.dmPelsWidth;
                old_y   = DisplayMode.dmPelsHeight;
                old_bpp = DisplayMode.dmBitsPerPel;
            }
        }
        if(printed)
            printf("Hz\n");
        exit(1);
    }

    if(opt_best) {
        find_best(&DisplayMode);
        printf("Seting %dx%d %dbit %dHz",
               DisplayMode.dmPelsWidth,
               DisplayMode.dmPelsHeight,
               DisplayMode.dmBitsPerPel,
               DisplayMode.dmDisplayFrequency
               );
        opt_x     = DisplayMode.dmPelsWidth;
        opt_y     = DisplayMode.dmPelsHeight;
        if(opt_vsync == -1) {
            opt_vsync = DisplayMode.dmDisplayFrequency;
        }
        opt_bpp   = DisplayMode.dmBitsPerPel;
    }
    EnumDisplaySettings(CUR_DISPLAY_DEV_NAME, ENUM_CURRENT_SETTINGS, &DisplayMode);
    if(opt_test) {
        DisplayMode0 = DisplayMode;
    }

    if(opt_x == -1 && opt_y == -1 && opt_vsync == -1 && opt_bpp == -1) {
        printf("Display mode not specified\n");
        exit(0);
    }

    // Set X-resolution
    if(opt_x != -1) {
        DisplayMode.dmPelsWidth = opt_x;
    }
    flags |= DM_PELSWIDTH;
    // Set Y-resolution
    if(opt_y != -1) {
        DisplayMode.dmPelsHeight = opt_y;
    }
    flags |= DM_PELSHEIGHT;
    // optionally set v-sync
    if(opt_vsync != -1) {
        DisplayMode.dmDisplayFrequency = opt_vsync;
        flags |= DM_DISPLAYFREQUENCY;
    }
    // optionally set color depth
    if(opt_bpp != -1) {
        DisplayMode.dmBitsPerPel = opt_bpp;
        flags |= DM_BITSPERPEL;
    }
    DisplayMode.dmFields = DM_DISPLAYFLAGS | flags;

    i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
        CDS_SET_PRIMARY, NULL);
    if(opt_test) {
        gDisplayMode = DisplayMode;
        if(IDOK != DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_QUESTION), NULL, QuestionDialog)) {
            DisplayMode = DisplayMode0;
            i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
                CDS_SET_PRIMARY, NULL);
        }
    }
    if(opt_save && (i==DISP_CHANGE_SUCCESSFUL || i==DISP_CHANGE_RESTART)) {
        i = MyChangeDisplaySettingsEx(CUR_DISPLAY_DEV_NAME,  &DisplayMode, NULL, 
            CDS_UPDATEREGISTRY, NULL);
    }
    exit(1);
} // end main()
