//version by gopher://shibboleths.org random 1.1

function generateRandomStage() {
  const objects = [];
  const step = 16; // Grid step

  // Base (eagle) in the center of the lower part of the map
  const baseX = 224, baseY = 400;
  objects.push(`Base(${baseX},${baseY})`);

  // Defensive wall around the base in the shape of the letter П (like in the original)
  const fencePositions = [
    {x: 208, y: 416}, // Additional block at the bottom
    {x: 208, y: 400}, // Left-center
    {x: 208, y: 384}, // Left-top
    {x: 224, y: 384}, // Center-top
    {x: 240, y: 384}, // Right-top
    {x: 256, y: 384}, // Further right-top
    {x: 256, y: 400}, // Right-center
    {x: 256, y: 416}  // Right-bottom
  ];
  fencePositions.forEach(pos => {
    objects.push(`BrickWall(${pos.x},${pos.y})`);
  });

  // Occupied positions: base and defensive wall
  const occupied = new Set();
  occupied.add(`${baseX},${baseY}`);
  fencePositions.forEach(pos => {
    occupied.add(`${pos.x},${pos.y}`);
  });

  // Occupied positions for player respawn: only the central part (tank spawn location), so blocks can generate around it but not on the spot itself
  // Original zone x32-96/y352-416, but we block only the center x64-80/y384-400 (4x2 blocks, where the tank spawns exactly)
  for (let x = 64; x <= 80; x += step) {
    for (let y = 384; y <= 400; y += step) {
      occupied.add(`${x},${y}`);
    }
  }

  // Occupied positions for enemy respawns: three points (left edge, center, right edge) with a 32x32 zone around the center (like tank size)
  // Left: center x=48, y=32 -> zone x32-64, y16-48
  for (let x = 32; x <= 64; x += step) {
    for (let y = 16; y <= 48; y += step) {
      occupied.add(`${x},${y}`);
    }
  }
  // Central: center x=224, y=32 -> zone x208-240, y16-48
  for (let x = 208; x <= 240; x += step) {
    for (let y = 16; y <= 48; y += step) {
      occupied.add(`${x},${y}`);
    }
  }
  // Right: center x=400, y=32 -> zone x384-416, y16-48
  for (let x = 384; x <= 416; x += step) {
    for (let y = 16; y <= 48; y += step) {
      occupied.add(`${x},${y}`);
    }
  }

  // Map boundaries: x32-432, y16-416 (like in the original, considering SteelWall up to 432)
  const minX = 32, maxX = 432, minY = 16, maxY = 416;
  const gridWidth = Math.floor((maxX - minX) / step) + 1;
  const gridHeight = Math.floor((maxY - minY) / step) + 1;

  // Randomized limits: number of blocks from 150 to 250, max Water/Trees from 2 to 6
  const numBlocks = 150 + Math.floor(Math.random() * 101); // 150-250
  const maxWater = 2 + Math.floor(Math.random() * 5); // 2-6
  const maxTrees = 2 + Math.floor(Math.random() * 5); // 2-6
  let waterCount = 0, treesCount = 0;

  for (let i = 0; i < numBlocks; i++) {
    // Random position on the grid, multiple of step
    let x = minX + Math.floor(Math.random() * (gridWidth - 1)) * step;
    let y = minY + Math.floor(Math.random() * (gridHeight - 1)) * step;

    // With 20% probability, make a single block (1x1), otherwise 2x2
    const isSingleBlock = Math.random() < 0.20;
    const blockPositions = isSingleBlock ? [{x, y}] : [
      {x, y},
      {x: x + step, y},
      {x, y: y + step},
      {x: x + step, y: y + step}
    ];

    // Check if positions are free (not occupied by base, fence, player respawn or enemies)
    const isFree = blockPositions.every(pos => 
      pos.x <= maxX && pos.y <= maxY && !occupied.has(`${pos.x},${pos.y}`)
    );
    if (!isFree) continue;

    // Type selection: probabilities based on original maps (more BrickWall), with limits on Water and Trees
    let type;
    const rand = Math.random();
    if (rand < 0.05 && waterCount < maxWater) { // 5% Water, max random
      type = "Water";
      waterCount++;
    } else if (rand < 0.15 && treesCount < maxTrees) { // 10% Trees, max random (next 10% after Water)
      type = "Trees";
      treesCount++;
    } else if (rand < 0.80) { // 70% BrickWall (next 70%)
      type = "BrickWall";
    } else { // 15% SteelWall
      type = "SteelWall";
    }

    // Add block positions
    blockPositions.forEach(pos => {
      objects.push(`${type}(${pos.x},${pos.y})`);
      occupied.add(`${pos.x},${pos.y}`);
    });
  }

  // Join the map into a string
  const map = objects.join(";");

  // Randomized number of tanks: from 15 to 25
  const numTanks = 15 + Math.floor(Math.random() * 11); // 15-25
  const tanks = [];
  for (let i = 0; i < numTanks; i++) {
    // Light randomization of probabilities: basic percentages with small spread
    const basicProb = 0.20 + Math.random() * 0.10; // 20-30% BASIC
    const fastProb = 0.20 + Math.random() * 0.10;  // 20-30% FAST
    const armorProb = 0.30 + Math.random() * 0.10; // 30-40% ARMOR
    // POWER: remainder (about 20-30%)

    let rand = Math.random();
    if (rand < basicProb) {
      tanks.push(Tank.Type.BASIC);
    } else if (rand < basicProb + fastProb) {
      tanks.push(Tank.Type.FAST);
    } else if (rand < basicProb + fastProb + armorProb) {
      tanks.push(Tank.Type.ARMOR);
    } else {
      tanks.push(Tank.Type.POWER);
    }
  }

  return { map, tanks };
}

// Generate 50 random levels (each with unique map and tanks)
Globals.stages = [];
for (let level = 0; level < 50; level++) {
  Globals.stages.push(generateRandomStage());
}