Program stick (output);

{ This file has no copyright assigned and is placed in the Public Domain }

{ This sample reads a sector from a USB memory stick using the "bulk
 only" protocol. Adapted from 'stick.bas' by Joe da Silva, to suit
 various real-mode compilers, such as Turbo Pascal 5.5 (available
 for free download from "http://bdn.borland.com/museum").  However,
 for TP/BP 7.0, be sure to use a non-Borland version of the CRT unit
 (or not at all), else you may encounter the infamous "RTE 200" bug. }

{ Note, there are some minor functional differences between this
 code and the original 'stick.bas' code. This is mostly by design. }

Uses
   dos,  {provides the "registers" type & "intr" routine}
   crt;  {provides the "clrscr" & "gotoxy" routines}

Const
   LUN = 0;
   buffer_len = 64;
   DosUsbInt = $65;        {interrupt used by DosUSB/DosUHCI driver}
   ignore_error = false;   {DosUHCI call error handling}
   {transaction_token values ...}
   control_token = $2D;
   in_token = $69;
   out_token = $E1;
   command_token = $FF;
   {command values ...}
   reset_dev = 1;
   enable_debug = 2;
   disable_debug = 3;
   display_urb = 4;
   clr_toggle = 5;
   check_presence = 6;
   ident_ctrl_type = 7;
   restart_dev = 8;
   ident_ctrl_chip = 9;
   {SCSI commands ...}
   inquiry = $12;
   request_sense = $03;
   test_ready = $00;
   read_sectors = $28;
   {misc. constants ...}
   CR = ^M;
   LF = ^J;
   BEL = ^G;
   {for preset values, see second 'const' block below}

Type
   dword = longint;

   URB_type =
      packed record
         transaction_token : byte; {see token values above}
         chain_end_flag  : byte; {1 = another URB follows this one in memory}
         dev_add         : byte;
         end_point       : byte;
         error_code      : byte;
         status          : byte; {returned by dosuhci}
         transaction_flags : word; {reserved}
         buffer_off      : word; {for in/out}
         buffer_seg      : word; {for in/out}
         buffer_length   : word; {for in/out}
         actual_length   : word; {for in/out}
         setup_buffer_off : word; {for control}
         setup_buffer_seg : word; {for control}
         start_frame     : word; {reserved}
         nr_of_packets   : word; {reserved - iso}
         int_interval    : byte; {reserved}
         error_count     : byte; {reserved}
         timeout         : word; {reserved}
         next_urb_off    : word; {reserved}
         next_urb_seg    : word; {reserved}
      end; {record, 32 bytes long}

   setup_type =
      packed record
         {refer USB 1.1/2.0 specification @ section 9.3}
         bmRequestType  : byte;
         bRequest	: byte;
         wValue		: word;
         wIndex		: word;
         wLength	: word;
      end; {record}

   read_command =
      packed record
         {first CBW fields 15 bytes}
         {refer USB Mass Storage Class Bulk-Only Transport spec. @ 5.1}
         signature      : dword;
         tag            : dword;
         tlength        : dword;
         flags          : byte;
         cbwlun         : byte;
         clength        : byte;
         {now command fields 16 bytes}
         {refer SCSI Reduced Block Commands spec. @ section 5.2}
         operation_code : byte;
         reserved1      : byte;
         lba            : dword; {big endian}
         reserved2      : byte;
         blocks         : word; {big endian}
         controls       : byte;
         padding1       : word;
         padding2       : word;
         padding3       : word;
      end; {record, 31 bytes long}

   inquiry_command =
      packed record
         {first CBW fields 15 bytes}
         {refer USB Mass Storage Class Bulk-Only Transport spec. @ 5.1}
         signature      : dword;
         tag            : dword;
         tlength        : dword;
         flags          : byte;
         cbwlun         : byte;
         clength        : byte;
         {now command fields 16 bytes}
         {refer SCSI Primary Commands - 2 spec. @ 7.3, 7.20, 7.24}
         operation_code : byte;
         en_vp_dat      : byte;
         vp_dat_page    : byte;
         alloc_len      : word; {big endian}
         controls       : byte;
         padding1       : word;
         padding2       : word;
         padding3       : word;
         padding4       : word;
         padding5       : word;
      end; {record, 31 bytes long}

   hexstring = string[8];
   buffer_type = packed array [1..buffer_len] of char;

Var
   urb : URB_type;
   device_request : setup_type;
   read_command_request : read_command;
   inquiry_command_request : inquiry_command;
   buffer : buffer_type;
   sbuffer : packed array [1..1256] of char;
   i : integer;

Const {preset default values}
   lba : dword = 0; {543}      {sector to read}
   devadd : byte = 2; {1}      {device address assigned by DosUSB/DosUHCI}
   in_endpoint : byte = 1;     {was 2}
   out_endpoint : byte = 2;    {was 1}

(***************************************************************)

Procedure quit (ecode : word; eMess : string);
begin {quit}
   writeln(CR,LF,eMess);
   halt(ecode);
end; {quit}

Function getarg (param : integer; max : dword; eMess : string) : dword;
{Get numeric command line parameter. Note, max = 0 means "don't care"}
var
   temp : dword;
   ecode : integer;
begin {getarg}
   val(ParamStr(param), temp, ecode);
   if (ecode = 0) and ((max = 0) or (temp <= max))
   then
      getarg := temp
   else
      quit(param, eMess);
end; {getarg}

Function swap4endian (number : dword) : dword;
type
   dual_word =
      packed record
         first, second : word;
      end; {record}
var
   temp : dual_word;
begin {swap4endian}
   temp.first := swap(dual_word(number).second);
   temp.second := swap(dual_word(number).first);
   swap4endian := dword(temp);
end; {swap4endian}

Procedure writechars (var first : char; len : word);
{display a block of characters, verbatim}
type
   bigstring = packed array [1..32000] of char;
var
   charptr : ^bigstring;
   i : integer;
begin {writechars}
   charptr := addr(first);
   for i := 1 to len do
      write(charptr^[i]);
end; {writechars}

Procedure WriteCharsLn (var first : char; len : word);
{display a block of characters, substitute BEL, add new line}
type
   bigstring = packed array [1..32000] of char;
var
   charptr : ^bigstring;
   i : integer;
begin {WriteCharsLn}
   charptr := addr(first);
   for i := 1 to len do
      if charptr^[i] = BEL
      then
         write('?')
      else
         write(charptr^[i]);
   writeln;
end; {WriteCharsLn}

Function hex2char (nibble : byte) : char;
{Form hex character from nibble value}
begin {hex2char}
   if nibble < 10
   then
      hex2char := chr(ord('0')+nibble)
   else
      hex2char := chr(ord('A')+nibble-10)
end; {hex2char}

Function hexstr (num : dword; digits : word) : hexstring;
var
   st : array [1..8] of char;
   i : word;
begin {hexstr}
   for i := digits downto 1 do
      begin {for}
         st[i] := hex2char(num and $F);
         num := num shr 4
      end; {for}
   hexstr := copy(st,1,digits);
end; {hexstr}

Procedure DosUHCI (var urb : URB_type; cmd_code : word);
{Call DosUHCI/DosUSB driver API.   Note, 'cmd_code' only
 applies when 'transaction_token' = command_token ($FF).}
var
   regs : registers;
   continue : boolean;
begin {DosUHCI}
   urb.error_code := 0;
   regs.AX := cmd_code;
   regs.DS := seg(urb);
   regs.DX := ofs(urb);
   intr(DosUsbInt, regs);
   continue := ignore_error;
   case urb.error_code of
      0 : continue := true;
      1 : if cmd_code = check_presence
          then
             continue := true
          else
             writeln('Error, invalid device address.');
      2 : writeln('Error, internal error.');
      3 : writeln('Error, invalid transaction_type.');
      4 : writeln('Error, invalid buffer length.')
   else
      writeln('Error, code = ',urb.error_code);
   end; {case}
   if not continue
   then
      halt(urb.error_code);
end; {DosUHCI}

Function DosUHCIpresent : boolean;
const
   confirmation = $47;
var
   urb : URB_type;
   vector : pointer;
begin {DosUHCIpresent}
   {prepare for DosUHCI call, also initializes return code}
   urb.transaction_token := command_token;
   urb.dev_add := 0;  {driver-sensitive item, empirically determined}
   urb.buffer_length := 0; {driver-sensitive item, empirically determined}
   {call API, but first ensure int. vector is plausible}
   GetIntVec(DosUsbInt,vector);
   if seg(vector^) > $100
   then
      DosUHCI(urb,check_presence);
   DosUHCIpresent := (urb.transaction_token = confirmation);
end; {DosUHCIpresent}

Procedure EnableDebug (dev_addr : byte);
var
   urb : URB_type;
begin {EnableDebug}
   urb.transaction_token := command_token;
   urb.dev_add := dev_addr;
   DosUHCI(urb,enable_debug);
end; {EnableDebug}

Procedure DisableDebug (dev_addr : byte);
var
   urb : URB_type;
begin {DisableDebug}
   urb.transaction_token := command_token;
   urb.dev_add := dev_addr;
   DosUHCI(urb,disable_debug);
end; {DisableDebug}

Procedure InRequest (dev_id: byte; var urb: URB_type; var buf: buffer_type;
                     clear_buffer: boolean);
begin {InRequest}
   if clear_buffer
   then
      fillchar(buf,buffer_len,chr(0)); {return data here}

   {set up In request}
   urb.transaction_token := in_token;
   urb.chain_end_flag := 0;
   urb.dev_add := dev_id;
   urb.end_point := in_endpoint;
   urb.error_code := 0;
   urb.status := 0;
   urb.transaction_flags := 0;
   urb.buffer_off := ofs(buf);
   urb.buffer_seg := seg(buf);
   urb.buffer_length := buffer_len;
   urb.actual_length := buffer_len;
   urb.setup_buffer_off := 0;
   urb.setup_buffer_seg := 0;
   urb.start_frame := 0;
   urb.nr_of_packets := 0;
   urb.int_interval := 0;
   urb.error_count := 0;
   urb.timeout := 0;
   urb.next_urb_off := 0;
   urb.next_urb_seg := 0;

   {now call DosUHCI/DosUSB}
   DosUHCI(urb,0);
end; {InRequest}

Procedure OutRequest (dev_id: byte; var urb: URB_type; var buf: buffer_type);
begin {OutRequest}
   {set up Out request}
   urb.transaction_token := out_token;
   urb.chain_end_flag := 0;
   urb.dev_add := dev_id;
   urb.end_point := out_endpoint;
   urb.error_code := 0;
   urb.status := 0;
   urb.transaction_flags := 0;
   urb.buffer_off := ofs(buf);
   urb.buffer_seg := seg(buf);
   urb.buffer_length := 31;
   urb.actual_length := 31; {64}
   urb.setup_buffer_off := 0;
   urb.setup_buffer_seg := 0;
   urb.start_frame := 0;
   urb.nr_of_packets := 0;
   urb.int_interval := 0;
   urb.error_count := 0;
   urb.timeout := 0;
   urb.next_urb_off := 0;
   urb.next_urb_seg := 0;

   {now call DosUHCI/DosUSB}
   DosUHCI(urb,0);
end; {OutRequest}

Procedure ControlInRequest (dev_id: byte; var urb: URB_type;
                      var buf: buffer_type; var dev_req : setup_type);
begin {ControlInRequest}
   {set up Control/In request}
   urb.transaction_token := control_token;
   urb.chain_end_flag := 0;
   urb.dev_add := dev_id;
   urb.end_point := 0;
   urb.error_code := 0;
   urb.status := 0;
   urb.transaction_flags := 0;
   urb.buffer_off := ofs(buf);
   urb.buffer_seg := seg(buf);
   urb.buffer_length := 1;  {requested descriptor length}
   urb.actual_length := 8;  {maximum packet length}
   urb.setup_buffer_off := ofs(dev_req);
   urb.setup_buffer_seg := seg(dev_req);
   urb.start_frame := 0;
   urb.nr_of_packets := 0;
   urb.int_interval := 0;
   urb.error_count := 0;
   urb.timeout := 0;
   urb.next_urb_off := 0;
   urb.next_urb_seg := 0;

   {now call DosUHCI/DosUSB}
   DosUHCI(urb,0);
end; {ControlInRequest}

Procedure inquiry_request (opcode : byte; tr_len : word);
{send 'inquiry' or similar SCSI command request}
begin {inquiry_request}
   {set up inquiry command incl. cbw}
   inquiry_command_request.signature := $43425355;
   inquiry_command_request.tag := $5753433A;
   inquiry_command_request.tlength := tr_len;
   inquiry_command_request.flags := $80;
   inquiry_command_request.cbwlun := LUN;
   inquiry_command_request.clength := 6; {12}
   {now command fields}
   inquiry_command_request.operation_code := opcode;
   inquiry_command_request.en_vp_dat := 0;
   inquiry_command_request.vp_dat_page := 0;
   inquiry_command_request.alloc_len := swap(tr_len); {big endian}
   inquiry_command_request.controls := 0;
   inquiry_command_request.padding1 := 0;
   inquiry_command_request.padding2 := 0;
   inquiry_command_request.padding3 := 0;
   inquiry_command_request.padding4 := 0;
   inquiry_command_request.padding5 := 0;

   move(inquiry_command_request,buffer,sizeof(inquiry_command_request));

   OutRequest(devadd, urb, buffer);
end; {inquiry_request}

Procedure do_read (lbanumber : dword);
{send 'read sectors' SCSI command request & retrieve the data}
var
   i : integer;
   midpos : integer;
begin {do_read}
   {set up bulk read command incl. cbw}
   read_command_request.signature := $43425355;
   read_command_request.tag := $5753433A;
   read_command_request.tlength := $200; {512 bytes}
   read_command_request.flags := $80;
   read_command_request.cbwlun := LUN;
   read_command_request.clength := 10; {12}
   {now command fields}
   read_command_request.operation_code := read_sectors;
   read_command_request.reserved1 := 0; {32} {1}
   read_command_request.lba := swap4endian(lbanumber); {0} {big endian}
   {write('LBA:', hexstr(read_command_request.lba,8), ' ');}
   read_command_request.reserved2 := 0;
   read_command_request.blocks := swap($01); {00} {big endian}
   {writeln('Blocks:', hexstr(read_command_request.blocks,4));}
   read_command_request.controls := 0;
   read_command_request.padding1 := 0;
   read_command_request.padding2 := 0;
   read_command_request.padding3 := 0;

   move(read_command_request,buffer[1],sizeof(read_command_request));

   OutRequest(devadd, urb, buffer);

   i := 0;
   while i < 8 {leave if 8 ( 8*64=512 )} do
      begin {while}
         InRequest(devadd, urb, buffer, true);
         if urb.status = 0
         then
            begin {then}
               inc(i);
               midpos := 1+(i-1)*64;
               move(buffer,sbuffer[midpos],buffer_len);
            end {then}
         else
            if urb.status <> 88
            then
               exit; {do not incr. if 88}
      end; {while}
end; {do_read}

(***************************************************************)

Begin {stick}

   {revert console I/O from CRT to standard}
   assign(input,'');
   reset(input);
   assign(output,'');
   rewrite(output);

   {initialization start}
   if not DosUHCIpresent
   then
      quit(99,'Error : DosUHCI/DosUSB driver isn''t loaded.');
   {EnableDebug(devadd);}
   {DisableDebug(devadd);}

   fillchar(sbuffer,sizeof(sbuffer),chr(0)); {return data here}

   {load optional command line parameters}
   if ParamCount >= 1
   then
      lba := getarg(1,0,'Usage: STICK [lba_num dev_addr in_endp out_endp]');

   if ParamCount >= 2
   then
      devadd := getarg(2, 127, 'Error: Invalid device address value : ' +
                               ParamStr(2));
   if ParamCount >= 3
   then
      in_endpoint := getarg(3, 15, 'Error: Invalid in_endpoint value : ' +
                                   ParamStr(3));
   if ParamCount >= 4
   then
      out_endpoint := getarg(4, 15, 'Error: Invalid out_endpoint value : ' +
                                    ParamStr(4));
   ClrScr;
   writeln('Parameters: LBA = ', lba, ', USB device = ', devadd,
           ', In endpoint = ',in_endpoint,', Out endpoint = ',out_endpoint);
   {initialization end}

(* optional code ... *)
   {get max lun}
   device_request.bmRequestType := $A1;
   device_request.bRequest := $FE;
   device_request.wValue := $0000;
   device_request.wIndex := $0000;
   device_request.wLength := $0001;

   ControlInRequest(devadd, urb, buffer, device_request);

   {transaction error?}
   if urb.status > 1
   then
      begin {then}
         ClrScr;
         gotoxy(14,23);
         writeln('Device does not respond');
         halt(urb.status);
      end; {then}

   if urb.actual_length = 1
   then
      writeln(CR, LF, 'Maximum logical unit: ', ord(buffer[1]),
              '  (using unit: ', LUN, ')')
   else
      writeln(CR,LF,'Cannot determine logical units.');
(* ... optional code *)

   {send 'inquiry' SCSI command}
   inquiry_request(inquiry,$24);

   {In request - to read inquiry result}
   InRequest(devadd, urb, buffer,true);

   write(CR,LF,'[frag=',urb.actual_length,'] Inquiry: ');
   WriteCharsLn(buffer[9],urb.actual_length-8);

   {In request - to read CSW}
   InRequest(devadd, urb, buffer,false);

   write('[frag=',urb.actual_length,'] CSW: ');
   WriteCharsLn(buffer[1],urb.actual_length);

   {send 'request sense' SCSI command}
   inquiry_request(request_sense,$12);

   {In request - to read request sense result}
   InRequest(devadd, urb, buffer,true);

   write(CR,LF,'[frag=',urb.actual_length,'] Request sense: ');
   for i := 1 to urb.actual_length do
      write(hexstr(ord(buffer[i]),2), ' ');
   writeln;

   {In request - to read CSW}
   InRequest(devadd, urb, buffer,false);

   write('[frag=',urb.actual_length,'] CSW: ');
   WriteCharsLn(buffer[1],urb.actual_length);

   {send 'test unit ready' SCSI command}
   inquiry_request(test_ready,$00);

   writeln(CR,LF,'Test unit ready.');

   {In request - to read CSW}
   InRequest(devadd, urb, buffer,true);

   write('[frag=',urb.actual_length,'] CSW: ');
   WriteCharsLn(buffer[1],urb.actual_length);

   {Read the sector data!}
   do_read(lba);

   if urb.error_code > 0
   then
      begin
         ClrScr;
         gotoxy(14,23);
         writeln('Buffer length exceeded');
      end;

   writeln(CR,LF,'[frag=',urb.actual_length,'] Sector (LBA) ',lba,':');
   WriteCharsLn(sbuffer[1],512);

(* optional code ...
   if lba = 0
   then
      begin
         write(CR,LF,'Buffer in hex: ');
         for i := $1BD to 512 {urb.actual_length} do
            begin
               if (i-$1BD) mod 16 = 0
               then
                  writeln;
               write(hexstr(ord(sbuffer[i]),2), ' ');
            end;
         writeln;
      end;
... optional code *)

   {In request - to read CSW}
   InRequest(devadd, urb, buffer,false);

   write('[frag=',urb.actual_length,'] CSW: ');
   WriteCharsLn(buffer[1],urb.actual_length);
End. {stick}
