; This file has no copyright assigned and is placed in the Public Domain
;****************************************************************
;*    This is a Serial Port Device Driver for the DosUSB driver *
;*                                                              *
;*    Compile: tasm serial1.asm                                 *
;*    Link:    tlink serial1.obj /t,serial1.sys                 *
;****************************************************************

     cseg	segment   para public    'code'
     org 0000h

     serialport proc      far
		assume    cs:cseg, es:cseg, ds:cseg

;structures ****************************************************

rh		struc		;request header 
rh_len		db	?	;len of packet
rh_unit		db	?     	;unit code 
				;(block devices only)
rh_cmd		db	?     	;device driver command
rh_status	dw	?     	;returned by device driver 
rh_res1		dd	?     	;reserved
rh_res2		dd	?     	;reserved
rh		ends	      	;

rh0		struc		;Initialization (command 0)
rh0_rh		db	size rh dup (?)	;fixed portion
rh0_nunits	db	?     	;number of units 
				;(block devices only)
rh0_brk_ofs	dw	?     	;offset address for break
rh0_brk_seg	dw	?     	;segment address for break
rh0_bpb_tbo	dw	?     	;offset address of pointer
				;to BPB array
rh0_bpb_tbs	dw	?     	;segment address of pointer 
				;to BPB array
rh0_drv_ltr	db	?     	;first available drive 
				;(DOS 3+) (block only)
rh0		ends	      	;

rh3		struc		;IOCTL_INPUT (command 3)
rh3_rh		db	size rh dup(?)	;fixed portion
rh3_media	db	?     	;media descriptor from DPB
rh3_buf_ofs  	dw	?     	;offset address of 
				;data transfer area
rh3_buf_seg  	dw	?     	;segment address of 
				;data transfer area
rh3_count 	dw	?     	;transfer count 
				;(sectors for block) 
				;(bytes for character)
rh3_start	dw	?     	;start sector number 
				;(block only)
rh3		ends	      	;

rh4             struc   ;request header for INPUT         (command 4)
rh4_rh          db      size rh dup(?)  ;fixed request header portion
rh4_media       db      ?     ;media descriptor from DPB
rh4_buf_ofs     dw      ?     ;offset address of data transfer area
rh4_buf_seg     dw      ?     ;segment address of data transfer area
rh4_count       dw      ?     ;transfer count (sectors for block)
                              ;(bytes for character)
rh4_start       dw      ?     ;start sector number (block only)
rh4             ends          ;


rh8		struc		;OUTPUT (command 8)
rh8_rh		db	size rh dup(?)	;fixed portion
rh8_media	db	?     	;media descriptor from DPB
rh8_buf_ofs  	dw	?     	;offset address of 
				;data transfer area
rh8_buf_seg  	dw	?     	;segment address of 
				;data transfer area
rh8_count 	dw	?     	;transfer count 
				;(sectors for block) 
				;(bytes for character)
rh8_start	dw	?     	;start sector number 
				;(block only)
rh8		ends	      	;

rh9		struc		;OUTPUT_VERIFY (command 9)
rh9_rh		db	size rh dup(?)	;fixed portion
rh9_media	db	?     	;media descriptor from DPB
rh9_buf_ofs  	dw	?     	;offset address of 
				;data transfer area
rh9_buf_seg  	dw	?     	;segment address of 
				;data transfer area
rh9_count 	dw	?     	;transfer count 
				;(sectors for block) 
				;(bytes for character)
rh9_start	dw	?     	;start sector number (block only)
rh9		ends	      	;

rh10		struc		;Output_Status (command 10)
rh10_len	db	?     	;len of packet
rh10_unit	db	?     	;unit code 
				;(block devices only)
rh10_cmd	db	?     	;device driver command
rh10_status	dw	?     	;returned by device driver 
rh10_res1	dd	?     	;reserved
rh10_res2	dd	?     	;reserved
rh10		ends	      	;

rh12		struc		;IOCTL_OUTPUT (command 12)
rh12_rh		db	size rh dup(?)	;fixed portion
rh12_media	db	?     	;media descriptor from DPB
rh12_buf_ofs  	dw	?     	;offset address of 
				;data transfer area
rh12_buf_seg  	dw	?     	;segment address of 
				;data transfer area
rh12_count 	dw	?     	;transfer count 
				;(sectors for block) 
				;(bytes for character)
rh12_start	dw	?     	;start sector number 
				;(block only)
rh12		ends	      	;

rh16		struc		;OUTPUT_BUSY (command 16)
rh16_rh		db	size rh dup (?)	;fixed portion
rh16_media	db	?     	;media descriptor
rh16_buf_ofs 	dw	?     	;offset address of 
				;data transfer area
rh16_buf_seg 	dw	?     	;segment address of 
				;data transfer area
rh16_count	dw	?     	;byte count returned 
				;from device driver
rh16		ends	      	;

;commands that do not have unique portions to the request header:
;	INPUT_STATUS 	(command 6)
;	INPUT_FLUSH	(command 7)
;	OUTPUT_STATUS	(command 10)
;	OUTPUT_FLUSH	(command 11)
;	OPEN		(command 13)
;	CLOSE		(command 14)
;	REMOVEABLE	(command 15)
;	 


;****************************************************************
;*      MAIN PROCEDURE CODE                                     *
;****************************************************************

     begin:

;****************************************************************
;*      DEVICE HEADER REQUIRED BY DOS                           *
;****************************************************************

next_dev       dd   -1             ;no other drivers following
attribute      dw   08000h         ;char
strategy       dw   dev_strategy   ;Strategy routine address
interrupt      dw   dev_interrupt  ;Interrupt routine address
dev_name       db   'SERIAL1 '     ;name of our Serial driver

;****************************************************************
;*      WORK SPACE FOR OUR DEVICE DRIVER                        *
;****************************************************************

rh_ofs	dw   	?    	;offset address of the request header
rh_seg	dw   	?	;segment address of the request header

devicenr   db 1
int_endpointnr db 0 ;1
in_endpointnr  db 0 ;3
out_endpointnr db 0 ;2
int_endpoint_flag db 0
in_endpoint_flag db 0
out_endpoint_flag db 0

baudrate db 7 ;38400

bytes_to_print   dw 0
bytes_done       dw 0

retry_count dw 0

tb_buffer db 64 dup(0)

maxpacketsize dw 64

init_usb_flag db 0

showptr    dw 0
serial_on  db 0

;define commands*****************
get_dev_dis   db  80h,06h 
	      dw  0100h,0000h,0008h

get_conf_dis  db  80h,06h
              dw  0200h,0000h,0040h ;just request 64 bytes

;define data type of URB
urbstruc struc
  transaction_token db ? ;control, in, out 
  chain_end_flag  db ?  
  dev_add         db ?
  end_point       db ?
  error_code      db ?  ;error codes returned by dosuhci
  status          db ?  ;status byte returned by controller
  transaction_flags dw ?  ;reserved
  buffer_off      dw ?  ;for in/out
  buffer_seg      dw ?  ;for in/out
  buffer_length   dw ?  ;for in/out
  actual_length   dw ?  ;for in/out
  setup_buffer_off dw ? ;for control
  setup_buffer_seg dw ? ;for control
  start_frame     dw ?  ;reserved 
  nr_of_packets   dw ?  ;iso
  int_interval    db ?  ;int
  error_count     db ?  ;reserved 
  timeout         dw ?  ;reserved
  next_urb_off    dw ?  ;reserved
  next_urb_seg    dw ?  ;reserved
urbstruc ends ;32 byte long 

;define urb with this type
urb     urbstruc <> 

;define commands*****************
set_line_coding   db  21h,20h 
                  dw  0000h,0000h,0007h


set_control_line  db  21h,22h 
                  dw  0003h,0000h,0000h

set_control_flag db 0

class_req   db  0A1h,01h 
            dw  0000h,0000h,0001h

;                      ________________ 00009600h = 38400d
;                     |
;                     |         _______ one stop bit
;                     |         |
;                     |         | _____ no parity
;                     |         | |  __ 8 data bits
;                     |         | | |
line_coding_data  db  0,96h,0,0,0,0,8 ;38400 baud, 8 bits, no parity

line_coding_done_flag db 0

temp_buffer2 db 64 dup(0)

temp_buffer db  8 dup(0)

prtstatus db 0

;****************************************************************
;*      THE STRATEGY PROCEDURE                                  *
;****************************************************************

dev_strategy:  mov  cs:rh_seg,es   ;save the segment address
               mov  cs:rh_ofs,bx   ;save the offset address
               ret                 ;return to DOS

;****************************************************************
;*      THE INTERRUPT PROCEDURE                                 *
;****************************************************************

;device interrupt handler - 2nd call from DOS

dev_interrupt:

        pushf
        cld                     ;save machine state on entry
        push    ds
        push    es
        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si

        mov     ax,cs:rh_seg    ;restore ES as saved by STRATEGY call
        mov     es,ax           ;
        mov     bx,cs:rh_ofs    ;restore BX as saved by STRATEGY call

;jump to appropriate routine to process command

        mov     al,es:[bx].rh_cmd       ;get request header header command

        ;rol     al,1                    ;times 2 for index into word table
        shl     al,1                    ;times 2 for index into word table
        lea     di,cmdtab               ;function (command) table address
        mov     ah,0                    ;clear hi order
        add     di,ax                   ;add the index to start of table
        jmp     word ptr[di]            ;jump indirect

;CMDTAB is the command table that contains the word address
;for each command. The request header will contain the
;command desired. The INTERRUPT routine will jump through an
;address corresponding to the requested command to get to
;the appropriate command processing routine.

CMDTAB  label   byte            ;* = char devices only
        dw      INITIALIZATION  ; initialization
        dw      done            ; media check (block only)
        dw      done            ; build bpb
        dw      done            ; ioctl in
        dw      INPUT           ; input (read)
        dw      done            ;*non destructive input no wait
        dw      done            ;*input status
        dw      done            ;*input flush
        dw      OUTPUT          ; output (write)
        dw      OUTPUT          ; output (write) with verify
        dw      done            ;*output status
        dw      done            ;*output flush
        dw      done            ; ioctl output
        dw      done            ; device open
        dw      done            ; device close
        dw      done            ; removeable media
        dw      OUTPUT          ; output til busy

;****************************************************************
;*      YOUR LOCAL PROCEDURES                                   *
;****************************************************************

;***************************************************************
;do get endpoints
;
;there can be different packet sizes for the control transfer and
;the bulk transfer to endpoints 1+2 
;
initusb proc near

push es
push bx

;read device descriptor to get maxpacket size
mov cs:retry_count,0
repeat_maxpacket:
;set up URB for ctrl+in transaction
mov cs:urb.transaction_token,2Dh
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov cs:urb.end_point,0 ;control e.p.
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,offset cs:tb_buffer ;load temp buffer offset address
mov cs:urb.buffer_off,ax
mov ax,cs                 ;load segment address
mov cs:urb.buffer_seg,ax
mov cs:urb.buffer_length,8
mov cs:urb.actual_length,8
mov ax,offset cs:get_dev_dis 
mov cs:urb.setup_buffer_off,ax
mov ax,cs
mov cs:urb.setup_buffer_seg,ax
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUSB
push cs
pop ds
mov dx,offset cs:urb
int 65h

;read successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,5      ;already four retries?
jb initusblabel1 ;ok
jmp initusb_l_transaction_err ;exit with STC
initusblabel1:
cmp cs:urb.status,0       ;transaction OK?
je  maxpacket_transaction_ok ;yes
cmp cs:urb.status,88h      ;NAK?
jne initusblabel3          ;no, different error - exit without retry
jmp repeat_maxpacket       ;got NAK, retry
initusblabel3:
jmp initusb_l_transaction_err ;exit with STC

maxpacket_transaction_ok:
mov ax,word ptr cs:tb_buffer[7]
mov cs:maxpacketsize,ax

;get config descriptor to read endpoints
mov cs:retry_count,0
repeat_inoutep:
;set up URB for ctrl+in transaction
mov cs:urb.transaction_token,2Dh
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov cs:urb.end_point,0 ;control e.p.
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,offset cs:tb_buffer ;load temp buffer offset address
mov cs:urb.buffer_off,ax
mov ax,cs                 ;load segment address
mov cs:urb.buffer_seg,ax
mov cs:urb.buffer_length,64 ;try multiple
mov ax,cs:maxpacketsize
mov cs:urb.actual_length,ax
mov ax,offset cs:get_conf_dis 
mov cs:urb.setup_buffer_off,ax
mov ax,cs
mov cs:urb.setup_buffer_seg,ax
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUSB
push cs
pop ds
mov dx,offset cs:urb
int 65h

;read successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,5      ;already four retries?
jb initusblabel2
jmp initusb_l_transaction_err ;exit with STC
initusblabel2:
cmp cs:urb.status,0       ;transaction OK?
je  inoutep_transaction_ok ;yes
cmp cs:urb.status,88h      ;NAK?
jne initusblabel4           ;no, different error - exit without retry
jmp repeat_inoutep         ;got NAK, retry
initusblabel4:
jmp initusb_l_transaction_err ;exit with STC

inoutep_transaction_ok:

mov al,cs:tb_buffer[20]  ;read Endpoint address field
cmp al,7Fh ;in-endpoint? - bit 7 should be set - jbe below!
jbe tryout1
and al,07h ;mask all but low 3 bits with endpoint address
cmp cs:tb_buffer[21],3 ;this in a interrupt endpoint?
je int_endp_1

mov cs:in_endpointnr,al ;1 ;7Fh read
mov cs:in_endpoint_flag,1 ;is set
jmp endp2

int_endp_1:
mov cs:int_endpointnr,al ;1 ;7Fh read
mov cs:int_endpoint_flag,1 ;is set
jmp endp2

tryout1:
cmp al,0 ;out-endpoint?
je endp2 ;no further
and al,07h ;mask all but low 3 bits with endpoint address
mov cs:out_endpointnr,al ;1
mov cs:out_endpoint_flag,1 ;is set
endp2:

mov al,cs:tb_buffer[27]
cmp al,7Fh ;in-endpoint? - bit 7 should be set - jbe below!
jbe tryout2
and al,07h ;mask all but low 3 bits with endpoint address
cmp cs:tb_buffer[28],3 ;this in a interrupt endpoint?
je int_endp_2

mov cs:in_endpointnr,al ;2
mov cs:in_endpoint_flag,1 ;is set
jmp endp3

int_endp_2:
mov cs:int_endpointnr,al ;1 ;7Fh read
mov cs:int_endpoint_flag,1 ;is set
jmp endp3

tryout2:
cmp al,0 ;out-endpoint?
je endp3 ;no further
and al,07h ;mask all but low 3 bits with endpoint address
mov cs:out_endpointnr,al ;2
mov cs:out_endpoint_flag,1 ;is set
endp3:

;some have only two endpoints, so check wether in and out have been set already
mov al,cs:in_endpoint_flag
add al,cs:out_endpoint_flag
cmp al,2 ;both set already
je in_out_both_set

mov al,cs:tb_buffer[34]
cmp al,7Fh ;in-endpoint? - bit 7 should be set - jbe below!
jbe tryout3
and al,07h ;mask all but low 3 bits with endpoint address
cmp cs:tb_buffer[35],3 ;this in an interrupt endpoint?
je int_endp_3

mov cs:in_endpointnr,al ;3
jmp endp4

int_endp_3:
mov cs:int_endpointnr,al ;1 ;7Fh read
mov cs:int_endpoint_flag,1 ;is set
jmp endp4

tryout3:
cmp al,0 ;out-endpoint?
je endp4 ;no further, don't test for interrupt
and al,07h ;mask all but low 3 bits with endpoint address
mov cs:out_endpointnr,al ;3
endp4:

in_out_both_set:

clc
mov cs:retry_count,0
pop bx
pop es
ret

initusb_l_transaction_err: ;also called from maxpacket and get descriptor!
stc
mov cs:retry_count,0
pop bx
pop es
ret

initusb endp

;**************************************************************


do_set_line_coding proc near
;if set_control_flag = 1 then set control signals instead

if 1
;set baud rate in line coding data
push cx
push ax
mov ax,300
mov cl,cs:baudrate
shl ax,cl

cmp ax,11264 ;result of 8 - 57600
jne try_next_baud
mov ax,0E100h
try_next_baud:
cmp ax,22528 ;result of 9 - 115200
jne try_next_baud2
mov ax,0C200h
mov byte ptr cs:line_coding_data[2],1
try_next_baud2:
mov word ptr cs:line_coding_data[0],ax
pop ax
pop cx
endif

push es
push bx

;set line coding request - first control followed by an out and an in separately
mov cs:retry_count,0
repeat_slcr1:
;set up URB for ctrl+in transaction
mov cs:urb.transaction_token,2Dh
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov cs:urb.end_point,0 ;control e.p.
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,0
mov cs:urb.buffer_off,ax
mov ax,0
mov cs:urb.buffer_seg,ax
mov cs:urb.buffer_length,0
mov cs:urb.actual_length,0
mov ax,offset cs:set_line_coding
cmp set_control_flag,0
je  no_set_control_flag
mov ax,offset cs:set_control_line
no_set_control_flag:
mov cs:urb.setup_buffer_off,ax
mov ax,cs
mov cs:urb.setup_buffer_seg,ax
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUSB
push cs
pop ds
mov dx,offset cs:urb
int 65h

;read successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,5      ;already four retries?
jb slcr2 ;ok
jmp slcr_err ;exit with STC

slcr2:
cmp cs:urb.status,0       ;transaction OK?
je  slcr1_ok              ;yes
cmp cs:urb.status,88h     ;NAK?
jne slcr3                 ;no, different error - exit without retry
jmp repeat_slcr1          ;got NAK, retry
slcr3:
jmp slcr_err              ;exit with STC

slcr1_ok:

cmp set_control_flag,0
je no_set_control_flag2
jmp slcr_err ;no error, just exit

no_set_control_flag2:

;now output line coding data - do_in and do_out are set for dos buffers

push es:[bx].rh4_count
push es:[bx].rh4_buf_ofs
push es:[bx].rh4_buf_seg
        mov     es:[bx].rh4_count,7     ;load output count
        mov     ax,offset cs:line_coding_data
        mov     es:[bx].rh4_buf_ofs,ax  ;load offset address
        mov     ax,cs
        mov     es:[bx].rh4_buf_seg,ax  ;load segment address 

;mov ah,cs:devicenr
mov ah,cs:out_endpointnr
;push ax ;save both
mov al,cs:in_endpointnr
push ax

;mov cs:devicenr,0
mov cs:out_endpointnr,0
call do_out

;mov cs:devicenr,0
mov cs:in_endpointnr,0
        mov     es:[bx].rh4_count,65     ;load input count
        mov     ax,offset cs:temp_buffer2
        mov     es:[bx].rh4_buf_ofs,ax  ;load offset address
        mov     ax,cs
        mov     es:[bx].rh4_buf_seg,ax  ;load segment address 

call do_in

pop ax
mov cs:in_endpointnr,al
;pop ax ;restore both
mov cs:out_endpointnr,ah
;mov cs:devicenr,ah

pop es:[bx].rh4_buf_seg
pop es:[bx].rh4_buf_ofs
pop es:[bx].rh4_count

slcr_err:

pop bx
pop es

ret
do_set_line_coding endp
;**********************************************************************
do_in proc near

mov cs:bytes_done,0
mov cs:retry_count,0

bytes_ok_in:               ;also retry label!

;set up URB for in transaction
mov cs:urb.transaction_token,069h
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov al,cs:in_endpointnr
mov cs:urb.end_point,al
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,es:[bx].rh4_buf_ofs  ;load offset address
;add ax,cs:bytes_done           ;start at this offset if 64 max
mov cs:urb.buffer_off,ax
mov ax,es:[bx].rh4_buf_seg  ;load segment address
mov cs:urb.buffer_seg,ax
mov ax,es:[bx].rh4_count ;load input count
cmp ax,64
jbe not_greater_64
mov ax,64 ;max packet size - return no more even if requested
not_greater_64:
mov cs:urb.buffer_length,ax
mov cs:urb.actual_length,64
mov cs:urb.setup_buffer_off,0
mov cs:urb.setup_buffer_seg,0
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUSB
push cs
pop ds
mov dx,offset cs:urb
int 65h
push cs
pop ds

;transaction successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,5 ;32000  ;allow for 32000 NAK's 
;jae error_in           ;yes, report error to DOS

cmp cs:urb.status,0       ;transaction OK?
je  transaction_ok_in  ;yes

cmp cs:urb.status,88h     ;NAK?
jne error_in           ;no, different error - exit without retry
mov bytes_done,0
ret ;exit function

;jmp bytes_ok_in        ;got NAK, retry

transaction_ok_in:
mov cs:retry_count,0      ;reset error counter

mov ax,cs:urb.actual_length ;bytes received are returned in actual_length
mov cs:bytes_done,ax     ;return bytes received
mov cs:urb.status,0
ret  ;exit function

error_in:  ;if urb.status is <> 0, an error will be reported to DOS 
mov ax,800Bh  ;return read error
ret

do_in endp

;****************************************************************


do_out proc near

mov cs:bytes_done,0
mov cs:retry_count,0

repeat_out:
mov ax,es:[bx].rh8_count ;load output count
cmp ax,cs:bytes_done ;all done?
ja not_all_done
mov cs:urb.status,0
ret  ;exit function

not_all_done:
sub ax,cs:bytes_done       ;how many left?
mov cs:bytes_to_print,ax   ;bytes_to_print
cmp cs:bytes_to_print,64
jbe bytes_ok            ;less or equal the max
mov cs:bytes_to_print,64   ;max packet size for bulk

bytes_ok:               ;also retry label!

;set up URB for out transaction
mov cs:urb.transaction_token,0E1h
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov al,cs:out_endpointnr
mov cs:urb.end_point,al
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,es:[bx].rh8_buf_ofs  ;load offset address
add ax,cs:bytes_done           ;start at this offset if 64 max
mov cs:urb.buffer_off,ax
mov ax,es:[bx].rh8_buf_seg  ;load segment address
mov cs:urb.buffer_seg,ax
;mov ax,es:[bx].rh8_count ;load output count
mov ax,cs:bytes_to_print
mov cs:urb.buffer_length,ax
mov cs:urb.actual_length,64
mov cs:urb.setup_buffer_off,0
mov cs:urb.setup_buffer_seg,0
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUHCI
push cs
pop ds
mov dx,offset urb
int 65h
push cs
pop ds

;transaction successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,10 ;32000  ;allow for 32000 NAK's 
jae error_out          ;yes, report error to DOS
cmp cs:urb.status,0       ;transaction OK?
je  transaction_ok     ;yes
cmp cs:urb.status,88h     ;NAK?
jne error_out          ;no, different error - exit without retry
jmp cs:bytes_ok           ;got NAK, retry

transaction_ok:
mov cs:retry_count,0      ;reset error counter
mov ax,cs:bytes_to_print
add cs:bytes_done,ax     ;count how many done
jmp repeat_out

error_out:  ;if urb.status is <> 0, an error will be reported to DOS 
mov ax,800Ah  ;return write error

ret

do_out endp

;***************************************************************
;query printer for errors
;returns status read in prtstatus and DOS-return word in AX reg
;set carry if status not read successfully

readstatus proc near
push es
push bx

mov cs:retry_count,0

repeat_get_status:
;set up URB for ctrl+in+out transaction
mov cs:urb.transaction_token,2Dh
mov cs:urb.chain_end_flag,0
mov al,cs:devicenr
mov cs:urb.dev_add,al
mov cs:urb.end_point,0 ;control e.p.
mov cs:urb.error_code,0
mov cs:urb.status,0
mov cs:urb.transaction_flags,0
mov ax,offset cs:temp_buffer ;load temp buffer offset address
mov cs:urb.buffer_off,ax
mov ax,cs                 ;load segment address
mov cs:urb.buffer_seg,ax
mov cs:urb.buffer_length,8
mov cs:urb.actual_length,8
mov ax,offset cs:class_req 
mov cs:urb.setup_buffer_off,ax
mov ax,cs
mov cs:urb.setup_buffer_seg,ax
mov cs:urb.start_frame,0
mov cs:urb.nr_of_packets,0
mov cs:urb.int_interval,0
mov cs:urb.error_count,0
mov cs:urb.timeout,0
mov cs:urb.next_urb_off,0
mov cs:urb.next_urb_seg,0

;now call DosUHCI
push cs
pop ds
mov dx,offset cs:urb
int 65h
push cs
pop ds

;read successful?
inc cs:retry_count        ;always inc error counter
cmp cs:retry_count,5      ;already four retries?
jae get_status_err
cmp cs:urb.status,0       ;transaction OK?
je  get_status_ok ;yes
cmp cs:urb.status,88h      ;NAK?
jne get_status_err ;no, different error - exit without retry
jmp repeat_get_status     ;got NAK, retry

get_status_ok:
mov al,byte ptr temp_buffer[0]
mov cs:prtstatus,al

xor ax,ax ;clear

test cs:prtstatus,00100000b ;paper empty bit
jz test_for_select ;jmp if bit in prtstatus NOT set
mov ax,8009h  ;paper empty
jmp prtstatus_check_done

test_for_select:
test cs:prtstatus,00010000b ;selected bit set = no error
jnz test_for_error ;jmp if bit in prtstatus IS set
mov ax,8002h  ;device not ready
jmp prtstatus_check_done

test_for_error:
test cs:prtstatus,00001000b ;is that bit set = no error! ?
jnz prtstatus_check_done ;jmp if bit in prtstatus IS set
mov ax,800Ah  ;write error

prtstatus_check_done:
clc
mov cs:retry_count,0
pop bx
pop es
ret

get_status_err:
xor ax,ax ;clear
stc
mov cs:retry_count,0
pop bx
pop es
ret

readstatus endp

;**********************************************************
do_check proc near
        ;check if interrupt 65h is set or uninitalized
        ;return dx=1 if installed
        push es
        xor dx,dx ;zero = default
        xor ax,ax
        mov es,ax ;set to zero
        mov bx,es:194h ;read offset into bx
        mov ax,es:196h ;read segment
        mov es,ax      ;segment in es
        mov     ax,es  ;test if 0:0 vector -> no driver
        or      ax,bx
        pop es
        jz      not_loaded
        push es
        xor ax,ax
        mov es,ax ;set to zero
        mov     ax,es:[bx]
        cmp     al,0CFh ;IRET?
        pop es
        je      not_loaded
        mov dx,1 ;installed

        cmp init_usb_flag,1
        je not_loaded ;it is loaded, just use that label
        mov init_usb_flag,1 ;do just once - or via ioctrl
        call initusb

        cmp cs:line_coding_done_flag,1
        je not_loaded ;it is loaded, just use that label
        mov cs:line_coding_done_flag,1 ;do just once - or via ioctrl
        call do_set_line_coding ;set to 38400baud,8bit
        mov cs:set_control_flag,1
        call do_set_line_coding ;enable signals
        mov cs:set_control_flag,0

not_loaded:

ret
do_check endp
;***************************************************************************
read_commandline proc near
;determine length of command line first
        ;rh0_bpb_tbo     dw      ?       ;offset address of pointer
        ;rh0_bpb_tbs     dw      ?       ;segment address of pointer 
        cld
        push es
        mov di,es:[bx].rh0_bpb_tbo  ;position of command_string
        mov cx,es:[bx].rh0_bpb_tbs  ;segment
        mov es,cx
        mov al,0Dh  ;CR
        xor cx,cx
        mov cl,80   ;read 80 max
 repne  scasb
        pop es
        mov dx,di
        sub dx,es:[bx].rh0_bpb_tbo
        ;check if valid length read
        cmp dx,0
        ja chl0
        jmp read_commandline_done
chl0:
        cmp dx,80
        jb chl1
        jmp read_commandline_done
chl1:
        cld
        push es
        mov di,es:[bx].rh0_bpb_tbo  ;position of command_string
        mov cx,es:[bx].rh0_bpb_tbs  ;segment
        mov es,cx
        mov al,'/' 
        xor cx,cx
        mov cl,dl ;length in dx
 repne  scasb
        cmp cx,0 ;not found?
        jne nc0f ;jmp if found
        pop es   ;therefore no jcxz
        jmp nextcommand1  ;not found
nc0f:
        mov ax,es:[di] ;now read that byte
        cmp al,42h ;B?
        jne nc0a
        mov ax,es:[di+1] ;read baud rate
        sub ax,30h ;just one number max 9 incl. 0!
        mov cs:baudrate,al
        jmp nc0c
nc0a:
        cmp al,44h ;D?
        jne nc0b
        mov ax,es:[di+1] ;read device number
        sub ax,30h ;just one number max 9
        mov cs:devicenr,al
        jmp nc0c
nc0b:

nc0c:
        pop es

        ;now test for next command
nextcommand1:
        cld
        push es
        ;leave di and continue there
        ;mov di,es:[bx].rh0_bpb_tbo  ;position of command_string
        mov cx,es:[bx].rh0_bpb_tbs  ;segment
        mov es,cx
        mov al,'/' 
        xor cx,cx
        mov cl,dl ;length 
 repne  scasb
        cmp cx,0 ;not found?
        jne nc1f ;jmp if found
        pop es   ;therefore no jcxz
        jmp nextcommand2  ;not found
nc1f:
        mov ax,es:[di] ;now read that byte
        cmp al,42h ;B?
        jne nc1a
        mov ax,es:[di+1] ;read baudrate
        sub ax,30h ;just one number max 9 incl. 0!
        mov cs:baudrate,al
        jmp nc1c
nc1a:
        cmp al,44h ;D?
        jne nc1b
        mov ax,es:[di+1] ;read device number
        sub ax,30h ;just one number max 9
        mov cs:devicenr,al
        jmp nc1c
nc1b:

nc1c:
        pop es

        ;now test for next command
nextcommand2:

;check if values read ok
cmp cs:devicenr,0
ja dnext1      ;ok
mov cs:devicenr,1 ;default
dnext1:
cmp cs:devicenr,10
jb dnext2      ;ok
mov cs:devicenr,1 ;default
dnext2:

cmp cs:baudrate,0
;ja bnext1        ;ok
;mov cs:baudrate,5 ;default
bnext1:
cmp cs:baudrate,10
jb bnext2        ;ok
mov cs:baudrate,7 ;default
bnext2:

read_commandline_done:

ret
read_commandline endp

;******************************************************
show proc near ;show number in cl
	      ;pusha                 ; Register sichern
	      push  di
	      push  ax
	      push  es
	      push  cx
	      push  ax

	      push  0B800h          ; Video-Segment
	      pop   es              ; ES auf Video-RAM
	      mov   di, 450         ; StartOffset
	      xor   ch,ch           ; clear high byte
	      inc showptr
	      inc showptr
	      mov ax,showptr
	      add di,ax

	      cld
	      mov   ax, es:[di]     ; Farbe ermitteln
	      mov   al,cl           ; write number to show
	      add   al,48           ; convert to ascii
	      stosw                 ; write to es:di

	      pop   ax
	      pop   cx
	      pop   es
	      pop   ax
	      pop   di
	      ;popa
ret
show endp

;****************************************************************
;*      DOS COMMAND PROCESSING                                  *
;****************************************************************

;command 0      Initialization  *********************************
Initialization:
        call    read_commandline
        call    initial                 ;display message
        lea     ax,initial              ;set Break Addr. at initial
        mov     es:[bx].rh0_brk_ofs,ax  ;store offset address
        mov     es:[bx].rh0_brk_seg,cs  ;store segment address

;end of initialization - restore es:bx exit
	mov     ax,cs:rh_seg            ;move request header
	mov     es,ax                   ; segment to es
	mov     bx,cs:rh_ofs            ; offset to bx

        jmp     done                    ;set done status and exit

;command 4      Input *******************************************   
Input:

	mov	cx,es:[bx].rh4_count	;load input count 
	mov	di,es:[bx].rh4_buf_ofs	;load offset address
	mov	ax,es:[bx].rh4_buf_seg	;load segment address 
	mov	es,ax			; move to es
        ;is cx zero = rh4_count
        mov     cs:bytes_done,cx        ;set to zero - return 0 if cx=0
        jcxz    in_done                ;no bytes to print - leave

        push    cs
        pop     ds   ;set ds to cs

;check if usb driver installed
        push es
        push bx
        call    do_check
        pop bx
        pop es
        cmp dx,1 ;    urb.transaction_token,47h
        je check_ok_in

        mov     ax,800Bh                ;read error
        mov     es:[bx].rh4_count,0     ;return output count 
	jmp	load_status		;load status & exit

check_ok_in:

in_continue:

        push    cs
        pop     ds   ;set ds to cs
        push es
        push bx
	mov     ax,cs:rh_seg    ;move request header
	mov     es,ax           ; segment to es
	mov     bx,cs:rh_ofs    ; offset to bx
        call do_in ;now seg/off/bytes read from header in do_in proc
        pop bx
        pop es
        cmp cs:urb.status,0 ;ok?
        je in_done
        cmp cs:urb.status,88h ;NAK?
        je in_done

        ;now return error
        ;ax is filled with error return code for DOS

	mov     ax,cs:rh_seg            ;move request header
	mov     es,ax                   ; segment to es
	mov     bx,cs:rh_ofs            ; offset to bx
        mov     es:[bx].rh4_count,0     ;return zero input count
        jmp	load_status		;load status & exit

in_done:

	mov     ax,cs:rh_seg            ;move request header
	mov     es,ax                   ; segment to es
	mov     bx,cs:rh_ofs            ; offset to bx

        mov     ax,cs:bytes_done
        mov     es:[bx].rh4_count,ax    ;return bytes done as count
        mov     ax,0                    ;no error
	jmp	load_status		;load status & exit

;command 8      Output  *****************************************
Output:

	mov	cx,es:[bx].rh8_count	;load output count
        jcxz    out_done                ;no bytes to print - leave

        push    cs
        pop     ds   ;set ds to cs

;check if usb driver installed
        push es
        push bx
        call    do_check
        pop bx
        pop es
        cmp dx,1 ;    urb.transaction_token,47h
        je check_ok
        mov     ax,800Ah                ;write error
        mov     cx,0 ;cs:bytes_done
        mov     es:[bx].rh8_count,cx    ;return output count 
	jmp	load_status		;load status & exit

check_ok:

        jmp out_continue ;if readstatus not done above!

        mov     es:[bx].rh8_count,0     ;return output count 0
        jmp	load_status		;load status & exit

out_continue:
        push    cs
        pop     ds   ;set ds to cs
        push es
        push bx
        call do_out ;now seg/off/bytes read from header in do_out proc
        pop bx
        pop es
        cmp cs:urb.status,0 ;ok?
        je out_done
        ;now return error
        ;ax is filled with error return code for DOS

        mov     cx,cs:bytes_done        ;have to return these
        mov     es:[bx].rh8_count,cx    ;return output count
        jmp	load_status		;load status & exit

out_done:  
        mov     ax,0                    ;no error
	jmp	load_status		;load status & exit

;****************************************************************
;*      ERROR EXIT                                              *
;****************************************************************

isunknown: 
	or	es:[bx].rh_status,8003h	;set error bit and error code
	jmp	done			;set done and exit

;****************************************************************
;*      COMMON EXIT                                             *
;****************************************************************
load_status:
	mov	cx,cs:rh_seg		;restore request header 
	mov	es,cx			; segment to es
	mov	cx,cs:rh_ofs		;restore offset also
	xchg	bx,cx			;switch them
        mov     es:[bx].rh_status,ax    ;return status, ax filled above
        ;mov     es:[bx].rh8_count,cx    ;return output count - leave it
	jmp	done			;set done bit and exit

done:   or      es:[bx].rh_status,0100h ;set done bit

	pop	si			;restore all registers
	pop	di
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
        popf
	ret				;return to DOS

;****************************************************************
;*      END OF PROGRAM                                          *
;****************************************************************

;this procedure is called from the Initialization command and
;is executed only once. We tell DOS that the next available
;memory location (Break Address) is here. This allows DOS to over
;write this code; we save space.

initial	proc	near	;display message on console
	lea	dx,msg1	;message to be displayed
	mov	ah,9	;display
	int	21h	;DOS call

        mov  ah,02          ;
        mov  dl,devicenr    ;
        add  dl,48          ;convert to ascii
        int  21h            ;

        lea     dx,msg2 ;message to be displayed
	mov	ah,9	;display
	int	21h	;DOS call

        lea     dx,crlf ;message to be displayed
	mov	ah,9	;display
	int	21h	;DOS call

        call do_check   ;will do_set_line_coding if dosusb already loaded

	ret		;return to caller
initial	endp

msg1    db      0dh,0ah,"Serial port device driver for DosUSB, Version 1.0",0dh,0ah
        db      "Device number: $"
msg2    db      " | Default baud rate: 38000,8,N,1 $"
crlf    db      0dh,0ah,"(can be set with /DX and /BX -> /B0=300, /B5=9600,",0dh,0ah
        db      "/B6=19200, /B7=38400, /B8=57600 /B9=115200)"
        db      0dh,0ah,0ah,"$"
;crlf    db      0dh,0ah,"(can be set with /DX, /IX and /OX)",0dh,0ah,0ah,"$"

serialport endp            ;end of printer procedure
cseg	ends		;end of cseg segment
	end	begin	;end of program

