; E100PKT, packet driver for DOS
; Copyright (C) 2018, Seth Simon (sethsimon@sdf.org)
;
; This program is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program.  If not, see
; <https://www.gnu.org/licenses/>.

section .bss
    eeprom_dump: resb 256 * 2       ; Max 256 16-bit registers
section .text

EESK EQU 1  ; Serial clock
EECS EQU 2  ; Chip select
EEDI EQU 4  ; Data in
EEDO EQU 8  ; Data out

;========================
;=======          =======
;       WRITE_EEDI
;=======          =======
;========================
write_eedi:
    ; DX = EEPROM Control Register
    ; CF=1 to write a 1, 0 otherwise
    salc
    and     al, EEDI
    or      al, EECS
    out     dx, al
    call    sleep_3us

    or      al, EESK
    out     dx, al
    jmp     sleep_3us

;==================================
;=======                    =======
;       READ_EEPROM_REGISTER
;=======                    =======
;==================================
read_eeprom_register:
    ; CX = register number (MSB of r# is MSB of CX)
    ; Returns BX = register contents
    ;         BP = # address bits
    mov     dx, word [csr_io_bar]
    add     dx, 0xe     ; EEPROM Control Register

    ; Chip select
    mov     al, EECS | EESK
    out     dx, al
    call    sleep_3us

    ; Write Read Opcode (110)
    stc
    call    write_eedi
    stc
    call    write_eedi
    clc
    call    write_eedi

    ; Write into the address field until we get a 0 in EEDO:
    xor     bp, bp
.write_addr:
    add     cx, cx
    call    write_eedi
    inc     bp
    in      al, dx
    test    al, EEDO
    jnz     .write_addr

    ; Read the register:
    mov     cx, 16      ; Loop once per bit
    xor     bx, bx      ; Result
.loop2:
    clc
    call    write_eedi  ; Do a clock cycle
    in      al, dx
    shr     al, 3
    and     al, 1       ; 1 if EEDO was set, 0 otherwise

    add     bx, bx
    add     bl, al
    loop    .loop2

    ; Chip deselect
    mov     al, 0
    out     dx, al
    jmp     sleep_3us

;=========================
;=======           =======
;       DUMP_EEPROM
;=======           =======
;=========================
dump_eeprom:
    ; Puts the data in eeprom_dump
    ; Returns AX = # 16-bit registers
    xor     cx, cx                  ; Read register 0
    call    read_eeprom_register    ; BP := # address bits
    xchg    ax, bp
    mov     dx, .nregisters_is
    call    print_str_word

    ; Starting reg = 1 << (16 - nregister_bits)
    ; Reg increment = Starting reg
    ; # registers = 1 << nregister_bits
    mov     dx, 1
    push    dx
    push    ax
    neg     al
    add     al, 16
    xchg    ax, cx
    shl     dx, cl  ; DX := Reg increment

    xor     cx, cx  ; CX := Current reg
    mov     di, eeprom_dump
.loop:
    push    cx
    push    dx
    call    read_eeprom_register

    xchg    ax, bx  ; AX := value
    stosw

    pop     dx      ; Reg increment
    pop     cx      ; Current reg
    add     cx, dx
    jnc     .loop

    pop     cx      ; # address bits
    pop     ax      ; 1
    shl     ax, cl
    jmp     crlf

.nregisters_is: db `Number of EEPROM register bits is $`

;=====================================
;=======                       =======
;       GET_MAC_ADDRESS_OR_QUIT
;=======                       =======
;=====================================
get_mac_address_or_quit:
    call    dump_eeprom ; Returns AX=# words

    call    check_eeprom_checksum
    mov     dx, .sum_is_str
    call    print_str_word

    cmp     ax, 0xbaba
    mov     dx, .bad_sum_str
    jne     print_message_and_quit

    mov     dx, .your_mac_is
    call    print_str
    mov     si, eeprom_dump
    mov     di, mac_address
    mov     cx, 6
    jmp     .lin

.lout:
    mov     ah, 2
    mov     dl, ':'
    int     0x21
.lin:
    lodsb
    stosb
    call    print_byte
    loop    .lout

    jmp     crlf

.sum_is_str: db `EEPROM checksum = $`
.bad_sum_str: db `.\r\n\r\nError! It should be BABA. Aborting.\r\n$`
.your_mac_is: db `\r\nMAC Address: $`

;===================================
;=======                     =======
;       CHECK_EEPROM_CHECKSUM
;=======                     =======
;===================================
check_eeprom_checksum:
    ; AX = # words in the EEPROM
    ; Returns the checksum (should be 0xBABA) in AX
    ;
    ; The checksum is the sum of all the words in the EEPROM.
    xchg    ax, cx
    mov     si, eeprom_dump
    xor     bx, bx
.loop:
    lodsw
    add     bx, ax
    loop    .loop

    xchg    ax, bx
    ret

