
/** file waver.c - prototyping code to generate WAV file programatically
  *
  * tinkering around to get this to work, might not be doing things right.
  * as a matter of fact you (i?) should be using libsoundfile or something!
  *
  * actually i failed to compile libsoundfile and would have to learn
  * their API too... :/
  *
  * it's pretty crude, with repeated kludges even. for instance
  * some chunk records are represented by arrays of integer,
  * but the file format uses int32_t's and int16_t's...
  *
  * therefore array offsets don't match file offsets!
  *
  * in an attempt to manage things more consistantly (did we at all??)
  * the first 8 file-bytes' worth of data commented out.
  *
  * that way you can see print("fmt "), maybe not worth it but it's working now.
  *
  * also spits bytes directly into stdout so you would
  * use as: $ ./waver > waver.wav ;
  *
  * hope your operating system outputs binary text via stdout!
  * --> if it ONLY makes the terminal look funny try the reclear.sh script
  *
  * uses static arrays, not dynamic memory, so space could run out.
  *
  * not sure if positive & negative should use same max amplitude.
  *
  * not sure how to add copyright/song title/etc metadata,
  * just add it to metatag i guess.
  *
  * just set char *metatag = ""; to leave that junk out of the file.
  *
  * adding some crude input (from stdin of course!) before calling it a night.
  *
  * update: input isn't actually changing default values...
  *         i'm tired let's do this tomorrow!
  * update: ~7:30am next day, number problem fixed, always use integer values.
  *
  * Makefiles are as hard to make as a program... Arghhh!
  *
  * intended usage $ ./waver < soundcmd.txt > mywav.wav ;
  *          each line of soundcmd.txt has two (whole) numbers,
  *          first the frequency of tone,
  *          second the duration in milliseconds (hope Hz is multiple of 1000)
  *
  *          that will have to do until better notation dreamed up,
  *          or experimenting with chords. whichever comes first.
  *
  * author: Wayne Colvin (waynecolvin @ hotmail . com)
  * date  : start - Wednesday, June 12th, 2013 (2013-06-13)
  *         current - Thursday, June 13th, 2013 (next day)
  *
  * Do what you like with it, makes use of reference to documentation
  * found on internet. some examples appended as base64 binary.
 **/



#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>


/************************************************************************

    format variant generated:

    mono=1 for one audio channel, width=2 for pcm_s16le, samples=duration*Hz

    ("RIFF" $ContainSize+4 "WAVE"
            ("fmt " int32:18 {note: basic PCM only}
                    WAVE_FORMAT_PCM=int16:1
                    mono=int16:1
                    Hz=int32:[8000 { or whatever }]
                    byterate=int32:[Hz*mono*width]
                    align=int16:[mono*width]
                    bitdepth=int16:[8*width] { fractional bytes round up } )
                    fmt_extra_len=int16:0
            ("LIST" $MetadataSize
                    { uncertain if '\0' and even-offset boundary required }
                    ("INFOISFT" int32:[strlen including '\0'] "blah blah..."))
            ("data" int32:[mono*width*samples] ... ))

    note: the "LIST" chunk may follow "data" instead but we put it before.
          maybe we'll switch that (update: did), in case it helps odd lengths.
          { my music player tolerates even/odd lengths for metatag however }

          now it's sad not seeing it at head of hexdump :(

          the "fmt " chunk must precede "data", not sure if MUST immediately
          follow "WAVE" however.

          update: my music player tolerated "fmt " not coming first!
          but rearrange it back just in case.

          the reference (url below) says that some software (naively) assumes
          a combined header of 44 bytes. that would be with the "fmt " chuck
          included as part of one big fat header so we better keep it first.

          music player thinks a duration of zero seconds is an error,
          but split-second files with at least one(!) sample okay.

reference: http://www-mmsp.ece.mcgill.ca/documents/AudioFormats/WAVE/WAVE.html
            
****************************************************************************/

//-----------------------------------------------------------

#define dummy (-1)
#define Hz (44000)

/** write little endian int of size len to stdout as binary data,
  * this works for negative two's compliment numbers.
  *
  * maybe not one's compliment numbers on odd hadware.
  *
  * two's compliment arithmatic could be simulated with
  * bit operators like (0xFFFFFFFFF... ^ (unsigned)datum)) + 1 ;
  * or whatever it would be, but don't trust me!
  *
  * why risk it? the world has enough bugs...
  *
  * datum should be within range of integer type emitted,
  * long integers will need to be at least 32 bit (that should be standard C).
**/

void emit(int len, long datum) {
    int b;
    while(len--) {
        b = datum % 256;
        datum = datum / 256;
        putc(b, stdout);
    }
    return;
}

//-----------------------------------------------------------

int ContentSize;

//------------------------

int FmtSize = 18;
unsigned int fmt[] = {
//  'f', 'm', 't', ' ',
//  18,
    1,
    1,
    Hz,
    2*Hz*1,
    2,
    16,
    0
};

/* for use with emit() */
int fmtlen[] = {
//  1,1,1,1,
//  4,
    2,
    2,
    4,
    4,
    2,
    2,
    2,
    dummy
};

//---------------------

char *metatag = "This audio WAV file was created by: waver (aka 'dumbcoder-0.01'), wicked!";

int MetaDataSize;
unsigned char MetaData[] = {

    /** "LIST" chunk can cantain various things but we just put encoder info,
      * just like FFmpeg does! That isn't necessary but cool ;)
      *
      * Media player tolerates even/odd length string
      * (if both "LIST" and "INFOISFT" sizes are consistant)
      * but pad with '\0' to an even file offset alignment anyways
      * just to be on the safe side.
      *
      * It's uncertain if "INFOISFT" must end with a null byte ('\0')
      * but other example files in Download/ folder did so we will too.
     **/

//  'L', 'I', 'S', 'T',
//  0, // <--

    'I','N','F','O',  'I','S','F','T',

    /* int32 length of string, including padding __align(2)__ */
    0 // <-- byte offset [16], but ARRAY offset [8 { formerly known as 13 }] !
};

int metadatalen[] = {
//  1,1,1,1,
//  4,
    1,1,1,1,  1,1,1,1,
    4,
    dummy
};

//------------------------------

int DataSize;

#define MAX_SAMPLES (Hz*60*5) // 5 minutes, change if need more
int16_t SampleData[MAX_SAMPLES];

int SampNum = 0;;

//---------------------------------------------------------------------

#define USE_THIS (32767) // not sure if same amplitude for positive & negative

/** duration might be read as a whole number of milliseconds,
  * but returns as seconds with possible fractional part.
**/

int readLine(double *freq, double *dur) {
    int r1, r2;
    int f, d;

    if( feof(stdin) ) return 1;

    /* scanf() refused to read double floating point without a decimal */

    r1 = scanf("%d", &f); *freq = (double)f;
    r2 = scanf("%d", &d); *dur = (double)d / 1000.0;

    if(r1 == EOF || r2 == EOF) return 1;
    if(r1 <= 0 || r2 <= 0) return 1;

    if(*freq < 0) {
        *freq = -1 * (*freq);
    }

    if(*freq >= (Hz / 2)) {
        fprintf(stderr,
            "warning: exceeding nyquist frequency, reverse tone may occur\n");
    }

    if(*dur <= 0) {
        fprintf(stderr, "warning: zero time duration for this tone\n");
        dur = 0;
    }

    return 0;
}


int writeLine(double freq, double dur, double vol) {
    int i, n, mult = 1; // program doesn't do overtones but multiple left in.
    double tmp;
    int16_t samp;

    /** the point of taking modulus (e.g. (i % Hz) / Hz) was
      * to keep the factor of 2*pi within the interval [0..1)
      * so as to minimize potential roundoff error of sin().
      *
      * a duration of a tone might be really long and
      * depending how sin() is calculated increased roundoff
      * error might have occurred further from 0.0
      *
      * argument dur is in seconds (with possible fractional part)
    **/

    n = dur * Hz;
    for(i = 0; i < n; i++) {
        tmp = (2.0 * M_PI * (mult*i % Hz) / Hz) * freq;
        samp = (int16_t)(vol * USE_THIS * sin(tmp));

        if(SampNum < MAX_SAMPLES-1) {
            SampleData[SampNum] = samp;
            SampNum++;
        }
        else {
            fprintf(stderr, "warning: this tone could not be completed, "
                            "no space\n");

            fprintf(stderr,
                    "increase limit by editing #define MAX_SAMPLES (whatever) "
                    "in waver.c and recompiling\n");

            return 1;
        }
    }

    return 0;
}


//---------------------------------------------

int main(int argc, char argv[]) {
    int i, done = 0;

    double freq = 440.0, // in Hertz
           vol  = 0.80,  // as percent of max, 0.80 = 80%
           dur  = 2.0;   // in seconds, may be a decimal


    if(Hz % 1000 != 0) {
        fprintf(stderr, "warning: recommend Hz be a multiple of 1000\n");
    }

    SampNum = 0;

    done = readLine(&freq, &dur);
    while( !done ) {
        if( writeLine(freq, dur, vol) ) break;
        if( readLine(&freq, &dur) ) break;
    }


//---------------------------- write output file ---------------------------

    FmtSize = 0;
    for(i = 0; fmtlen[i] != dummy; i++) { FmtSize += fmtlen[i]; }

    /* FmtSize should equal 18, not counting "fmt " nor run length */

//----

    DataSize = 2*SampNum;

//----

    MetaDataSize = 0;
    if(metatag != NULL) { MetaDataSize = strlen(metatag) + 1; }

    MetaData[8] = MetaDataSize;
    MetaDataSize += 12;

    if(metatag == NULL || strlen(metatag) == 0) { MetaDataSize = 0; }

//------------------------------

    /* The extra 4 comes from "WAVE", exclude MetaData if none */
    
    ContentSize = 4 + (8 + FmtSize) + (8 + DataSize) + (8 + MetaDataSize);
    if(MetaDataSize == 0) ContentSize -= 8;

//----

    printf("RIFF");
    emit(4, ContentSize);
    printf("WAVE");

//----

    printf("fmt ");
    emit(4, FmtSize);
    for(i = 0; fmtlen[i] != dummy; i++) emit(fmtlen[i], fmt[i]);

//----

    printf("data");
    emit(4, DataSize);
    for(i = 0; i < SampNum; i++) emit(2, SampleData[i]);

//----

    if(MetaDataSize != 0) {
        printf("LIST");
        emit(4, MetaDataSize);
        for(i = 0; metadatalen[i] != dummy; i++) emit(metadatalen[i], MetaData[i]);
        printf("%s", metatag);
        emit(1, 0x00);
    }

//--------------------------------

    return 0;
}

