/* qnxflash.c
 * Copyright (c) 2000 GFRN systems.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: qnxflash.c,v 1.3 2000/05/28 18:38:19 freebsdfan Exp $
 *
 * $Log: qnxflash.c,v $
 * Revision 1.3  2000/05/28 18:38:19  freebsdfan
 * Corrected failures to exit Device ID mode on V4 Iopeners with RISE CPU.
 * Changed the method used to exit Device mode from the one step method to
 * the three step method. Added code to check that Device ID mode was exited
 * successfully.  Corrected a bug in WaitFlashComplete() which prevented
 * timeout errors from being detected. Doubled the loop count in
 * WaitFlashComplete(). Added the -V switch to enable verbose error and status
 * messages. Added -c switch to allow Device ID checks to be disabled.
 *
 * Revision 1.2  2000/05/11 17:40:51  freebsdfan
 * Added support for FreeBSD 4.0. Added support for Atmel AT49F020 (not tested).
 * Added -l switch to list supported devices.
 *
 * Revision 1.1.1.1  2000/04/20 04:27:26  freebsdfan
 * initial public release
 *
 */

#if (defined(__unix__) || defined(unix)) && !defined(USG)
    #include <sys/param.h>
#endif

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/stat.h>

#ifdef  BSD
   #include <unistd.h>
   #include <machine/cpufunc.h>
   #define  OUTPD outl
   #define  INPD  inl
#else /* qnx */
   #include <conio.h>
   #include <i86.h>
   #define  OUTPD outpd
   #define  INPD  inpd
   int  getopt(int argc,char **argv,char *opts);
#endif

#define DEVICE_ID_EXIT_TRIES  10
#define WAIT_FLASH_COMPLETE_TO   0xfffff

#define FALSE        0
#define TRUE         !FALSE

#define SECTOR_SIZE        4096

#define BIOS_SIZE          0x40000  /* 256k */
#define SST_ID             0xBF     /* SST Manufacturer's ID code   */
#define SST_39SF020        0xB6     /* SST 39SF020 device code      */
#define ATMEL_ID           0x1F     /* Atmel Manufacturer's ID code */
#define AT49F020           0x0B     /* AT49F020 device code         */

struct {
   int  Algorithm;
   unsigned char Man_ID;
   unsigned char Dev_ID;
   char *Desc;
} Devices[] = {
   { 1,  SST_ID,     SST_39SF020,   "SST 39SF020" },
   { 1,  ATMEL_ID,   AT49F020,      "Atmel AT49F020" },
   { 0 } /* end of table marker */
};

#define PCI_CONFIG_ADR     0xcf8
#define PCI_CONFIG_DATA    0xcfc

#define INDEX_BUS_CONTROL  0x40
#define INDEX_ROM_CONTROL  0x43

void usage();
int  Openshm();
int  GetDeviceID(int Verbose);
int  ReadBIOS(char *readfile);
int  ProgramBIOS(char *readfile);
int  VerifyBIOS(char *verifyfile);

char *ChipBase;
int fd = -1;
FILE *fp = NULL;
long OldBusControl;
long OldPCI_ConfigAdr;
char *optarg;
int optind = 1;
int opterr = 1;
int optopt;
int opt_disable_verify = FALSE;
int report_all_verify_errors = FALSE;
int opt_verbose = FALSE;
int no_id_check = FALSE;

#ifdef  BSD
int iofd = -1;
#endif

char DiskBuf[512];

int mainline(int arg_ct, char **p_argv);

int main(int arg_ct, char **p_argv)
{
   int exit_value;

   exit_value = mainline(arg_ct,p_argv);

/* Clean up resources */
   if(ChipBase != (char *) -1) {
      munmap(ChipBase,BIOS_SIZE);
   }

   if(fd != -1) {
      close(fd);
   }

   if(fp != NULL) {
      fclose(fp);
   }
   return exit_value;
}

int mainline(int arg_ct, char **p_argv)
{
   int          i;
   char *ourargs = "r:w:v:idalVc";
   char *readfile = NULL;
   char *writefile = NULL;
   char *verifyfile = NULL;
   int opt_devid = FALSE;
   int got_opt = FALSE;
   int opt;
   int result = 0;

#ifdef  BSD
   printf("Freeflash version 1.02 for i-open'ed boxen.\n");
#else
   printf("Qnxflash version 1.02 for i-open'ed boxen.\n");
#endif

   printf("Copyleft 2000, all wrongs reserved by a FreeBSD fan.\n");
   printf("Check http://openflash.sourceforge.net for updates.\n\n");
   if(!Openshm()) {
      return 1;
   }

   while((opt = getopt(arg_ct,p_argv,ourargs)) != EOF){
      switch(opt){
         case 'r':
            readfile = optarg;
            got_opt = TRUE;
            break;

         case 'w':
            writefile = optarg;
            got_opt = TRUE;
            break;

         case 'v':
            verifyfile = optarg;
            got_opt = TRUE;
            break;

         case 'i':
            opt_devid = TRUE;
            got_opt = TRUE;
            break;

         case 'd':
            opt_disable_verify = TRUE;
            break;

         case 'a':
            report_all_verify_errors = TRUE;
            break;

         case 'V':
            opt_verbose = TRUE;
            break;

         case 'l':
            got_opt = TRUE;
            printf("The following devices are supported by this program:\n");
            for(i = 0; Devices[i].Algorithm != 0; i++) {
               printf("%s\n",Devices[i].Desc);
            }
            break;

         case 'c':
            no_id_check = TRUE;
            break;

         case '?':
         default:
            printf("Unknown argument\n");
            usage();
            return 2;
      }
   }

   if(!got_opt) {
   /* no options specified, just give usage summary */
      usage();
      return 2;
   }

   if(opt_devid || writefile) {
   /* Check the device ID */
      if(!GetDeviceID(opt_devid)) {
      // Get device ID failed.
         result = 3;
      }
   }

   if(readfile != NULL) {
   /* Read the BIOS image */
      if((result = ReadBIOS(readfile)) != 0) {
      // Error
         return result;
      }
   }

   if(writefile != NULL) {
   /* Write the BIOS image */
      result = ProgramBIOS(writefile);
   }
   else if(verifyfile != NULL) {
   /* Verify the BIOS image */
      result = VerifyBIOS(verifyfile);
   }
   return result;
}

#ifdef  BSD
int Openshm()
{
/* we need to do IO so get IO privilege */

   iofd = open("/dev/io",O_RDWR);
   if(iofd == -1) {
       fprintf(stderr,"Error: Couldn't open /dev/io, %s\n",strerror(errno));
       return 0;
   }

   /* Open physical memory */
   fd = open("/dev/mem", O_RDWR, 0777);
   if(fd == -1) {
      fprintf(stderr, "Error: Couldn't open /dev/mem, %s\n", strerror(errno));
      return 0;
   }

   /* Map BIOS ROM */
   ChipBase = mmap(0,BIOS_SIZE,PROT_READ | PROT_WRITE,MAP_SHARED,fd,0xfffc0000);
   if(ChipBase == (char *) -1) {
      fprintf(stderr, "Error: mmap failed, %s\n",strerror(errno));
      return 0;
   }

   return 1;
}

#else /* qnx version */

int Openshm()
{
   /* Open physical memory */
   fd = shm_open("Physical", O_RDWR, 0777);
   if(fd == -1) {
      fprintf(stderr, "Open failed:%s\n", strerror(errno));
      return 0;
   }

   /* Map BIOS ROM */
   ChipBase = mmap(0,BIOS_SIZE,PROT_READ | PROT_WRITE | PROT_NOCACHE,
                   MAP_SHARED,fd,0xfffc0000);
   if(ChipBase == (char *) -1) {
      fprintf(stderr, "mmap failed : %s\n",strerror(errno));
      return 0;
   }
   return 1;
}
#endif

/*
   From VT82c686 data sheet, page 54:
PCI configuration space accesses for functions 0-6 use PCI
configuration mechanism 1 (see PCI specification revision 2.2
for more details). The ports respond only to double-word
accesses. Byte or word accesses will be passed on unchanged.

Port 0xCFB - 0xCF8 - Configuration Address
31 Configuration Space Enable
0 Disable
1 Convert configuration data port writes to configuration cycles on the PCI bus

30-24 Reserved

23-16 PCI Bus Number
Used to choose a specific PCI bus in the system

15-11 Device Number
Used to choose a specific device in the system

10-8 Function Number
Used to choose a specific function if the selected device supports multiple
functions

7-2 Register Number
Used to select a specific DWORD in the devices configuration space

1-0 Fixed

Port 0xCFF - 0xCFC - Configuration Data
There are 7 functions implemented in the VT82C686A:
Function # Function
0 PCI to ISA Bridge
1 IDE Controller
2 USB Controller Ports 0-1
3 USB Controller Ports 2-3
4 Power Management, SMBus & Hardware Monitor
5 AC97 Audio Codec Controller
6 MC97 Modem Codec Controller


   From VT82c686 data sheet, page 55-56:

Offset 0x40 - ISA Bus Control
7 ISA Command Delay
0 Normal
1 Extra

6 Extended ISA Bus Ready
0 Disable
1 Enable

5 ISA Slave Wait States
0 4 Wait States
1 5 Wait States

4 Chipset I/O Wait States
0 2 Wait States
1 4 Wait States

3 I/O Recovery Time
0 Disable
1 Enable

2 Extend-ALE
0 Disable
1 Enable

1 ROM Wait States
0 1 Wait State
1 0 Wait States

0 ROM Write
0 Disable
1 Enable

Offset 0x43 - ROM Decode Control

Setting these bits enables the indicated address range to be included in the
ROMCS# decode:

7 FFFE0000h-FFFEFFFFh
6 FFF80000h-FFFDFFFFh
5 FFF00000h-FFF7FFFFh (CF/CG)
4 000E0000h-000EFFFFh (CF/CG)
5 000E8000h-000EFFFFh (CD/CE)
4 000E0000h-000E7FFFh (CD/CE)
3 000D8000h-000DFFFFh
2 000D0000h-000D7FFFh
1 000C8000h-000CFFFFh
0 000C0000h-000C7FFFh

*/

void MapBIOS()
{
   /* enable flash chip selects in high memory */
#ifndef BSD
   _disable();  /* no interrupts while messing with Flash ! */
#endif

   /* save current PCI configuration address setting (just in case it matters) */

   OldPCI_ConfigAdr = INPD(PCI_CONFIG_ADR);

   /* Select offset 0x40 (bus control) */
   OUTPD(PCI_CONFIG_ADR,0x80003840);

   /* save old setting */
   OldBusControl = INPD(PCI_CONFIG_DATA);

   /* enable BIOS ROM CS in 0xfff80000 -> 0xffeffff & enable ROM writes */
   OUTPD(PCI_CONFIG_DATA,OldBusControl | 0xc0000001);
}

void UnMapBIOS()
{
   /* Disable access to Super IO configuration registers */
   /* restore old control register settings */

   OUTPD(PCI_CONFIG_DATA,OldBusControl);
   OUTPD(PCI_CONFIG_ADR,OldPCI_ConfigAdr);

#ifndef BSD
   _enable();   /* Ok to interrupt again */
#endif
}

void usage()
{
#ifdef  BSD
   printf("Usage: freeflash [-r <file>] [-w <file>] [-v <file>] [-iadlcV]\n");
#else
   printf("Usage: qnxflash [-r <file>] [-w <file>] [-v <file>] [-iadlcV]\n");
#endif
   printf("       -r Read bios image from chip.\n");
   printf("       -w Write bios image to chip.\n");
   printf("       -v Verify chip data against image file.\n");
   printf("       -i display bios chip's device Id.\n");
   printf("       -a report All verify errors.\n");
   printf("       -d Disable verify while programming.\n");
   printf("       -l List supported devices.\n");
   printf("       -c disable device id Check.\n");
   printf("       -V Verbose messages.\n");
}

int GetDeviceID(int Verbose)
{
   unsigned char ManufactureID;
   unsigned char DeviceID;
   unsigned char Temp[32];
   int i;
   int pass;

   if(no_id_check) {
   // Debug aid, bypass device ID check.
      return TRUE;
   }

   MapBIOS();
// Save some of the original contents of the FLASH so we can detect possible
// failure to exit Device ID mode.

   for(i = 0; i < sizeof(Temp); i++) {
      Temp[i] = ChipBase[i];
   }

   ChipBase[0x5555] = 0xaa;
   ChipBase[0x2AAA] = 0x55;
   ChipBase[0x5555] = 0x90;

   ManufactureID = ChipBase[0];
   DeviceID = ChipBase[1];

   for(pass = 0; pass < DEVICE_ID_EXIT_TRIES; pass++) {
      // Exit Device ID
      ChipBase[0x5555] = 0xaa;
      ChipBase[0x2AAA] = 0x55;
      ChipBase[0x5555] = 0xF0;
      for(i = 0; i < sizeof(Temp); i++) {
         if(Temp[i] != ChipBase[i]) {
            if(opt_verbose) {
               printf("Failure to exit Device ID #%d\n",pass+1);
               printf("Offset %d = 0x%02x, should be 0x%02x.\n",
                      i,ChipBase[i],Temp[i]);
            }
            break;
         }
      }
      if(i == sizeof(Temp)) {
         break;
      }
   }
   UnMapBIOS();

   if(pass == DEVICE_ID_EXIT_TRIES) {
      printf("Return from Device ID to normal operation failed.\n");
      return FALSE;
   }

   if(Verbose) {
      printf("Manufacture code: 0x%02X, Device code: 0x%02X\n",
             ManufactureID,DeviceID);
   }

   for(i = 0; Devices[i].Algorithm != 0; i++) {
      if(ManufactureID == Devices[i].Man_ID && DeviceID == Devices[i].Dev_ID) {
         break;
      }
   }
   if(Devices[i].Algorithm != 0) {
      if(Verbose) {
         printf("Device is a %s.\n",Devices[i].Desc);
      }
      return TRUE;
   }
   else {
      if(Verbose) {
         printf("Unknown device.\n\n");
      }
      else {
         printf("Error: Device ID is unknown, aborting operation.\n\n");
      }
      printf("Manufacture's code = 0x%x, Device code = 0x%0x\n",
             ManufactureID,DeviceID);
      return FALSE;
   }
   return TRUE;
}

int ReadBIOS(char *readfile)
{
   int i;

   printf("Copying BIOS image to \"%s\".\n",readfile);
   if((fp = fopen(readfile,"w")) == NULL) {
      printf("Unable to open file \"%s:%s\n",readfile,strerror(errno));
      return 4;
   }

   for(i = 0; i < BIOS_SIZE; i+= sizeof(DiskBuf)) {
      MapBIOS();
      memcpy(DiskBuf,(char *) &ChipBase[i],sizeof(DiskBuf));
      UnMapBIOS();
      if(fwrite(DiskBuf,sizeof(DiskBuf),1,fp) != 1) {
         printf("Error writing file: %s\n",strerror(errno));
         return 5;
      }
      printf("\r0x%05x",i);
   }
   fclose(fp);
   fp = NULL;
   printf("\r0x%05x\n",i);
   printf("Complete.\n");
   return 0;
}

int WaitFlashComplete(volatile char *TestByte)
{
   char LastByte;
   char ThisByte;
   int Timeout = WAIT_FLASH_COMPLETE_TO + 1;

   LastByte = *TestByte;
   while(--Timeout) {
      ThisByte = *TestByte;
      if(((ThisByte ^ LastByte) & 0x40) == 0) {
      /* DB6 stopped toggling, flash is ready */
         break;
      }
      LastByte = ThisByte;
   }
   return Timeout;
}


int VerifyBIOS(char *verifyfile)
{
   int i,j;
   char is;
   int errors = 0;

   printf("Verifing BIOS image against \"%s\".\n",verifyfile);
   if((fp = fopen(verifyfile,"r")) == NULL) {
      printf("Unable to open file \"%s:%s\n",verifyfile,strerror(errno));
      return 6;
   }

   for(i = 0; i < BIOS_SIZE; i+= sizeof(DiskBuf)) {
      if(fread(DiskBuf,sizeof(DiskBuf),1,fp) != 1) {
         printf("Error reading file: %s\n",strerror(errno));
         return 7;
      }
      MapBIOS();
      for(j = 0; j < sizeof(DiskBuf); j++) {
         is = ChipBase[i+j];
         if(DiskBuf[j] != is) {
         // Shit, verify error !
            if(!report_all_verify_errors && errors++ >= 16) {
               printf("More than 16 verify errors, aborting. "
                      "(use -a to see all errors)\n");
               break;
            }
            printf("Verify error @ 0x%05X, Data = 0x%02X, should be 0x%02X.\n",
                   i+j,is,DiskBuf[j]);
         }
      }
      UnMapBIOS();
      if(j != sizeof(DiskBuf)) {
         return 8;
      }
      printf("\r0x%05x",i);
   }
   fclose(fp);
   fp = NULL;
   printf("\r0x%05x\n",i);
   printf("Verify complete, no errors found.\n");
   return 0;
}

int ProgramIt(FILE *fp)
{
   int i, j;
   int WaitOk = 1;
   char is;

   /* erase the flash */

   printf("Erasing FLASH...");
   fflush(stdout);

   MapBIOS();

/* Issue the erase chip command */
   ChipBase[0x5555] = 0xAA;
   ChipBase[0x2AAA] = 0x55;
   ChipBase[0x5555] = 0x80;
   ChipBase[0x5555] = 0xAA;
   ChipBase[0x2AAA] = 0x55;
   ChipBase[0x5555] = 0x10;
   WaitOk = WaitFlashComplete(ChipBase);
   UnMapBIOS();

   if(!WaitOk) {
      printf("\nErase failed, timeout waiting for DB6 to stop toggling.\n");
      return 10;
   }
   else if(opt_verbose) {
      printf("\nAfter erase WaitFlashComplete() looped %d times.\n",
             WAIT_FLASH_COMPLETE_TO - WaitOk);
   }

   printf("\nVerifing erasure...");
   fflush(stdout);

   MapBIOS();
   for(i = 0; i < BIOS_SIZE; i++) {
      is = ChipBase[i];
      if(is != (char) 0xff) {
      // Erasure failure
         break;
      }
   }
   UnMapBIOS();

   if(i != BIOS_SIZE) {
      printf("\nErase failed @ 0x%05x, data is 0x%02x, should be 0xff.\n",i,is);
      return 9;
   }
   else {
      printf("\nProgramming ...\n");
   }

   for(i = 0; i < BIOS_SIZE; i+= sizeof(DiskBuf)) {
      if(fread(DiskBuf,sizeof(DiskBuf),1,fp) != 1) {
         printf("Error reading file: %s\n",strerror(errno));
         return 7;
      }
      MapBIOS();
      for(j = 0; j < sizeof(DiskBuf); j++) {
         ChipBase[0x5555] = 0xAA;
         ChipBase[0x2AAA] = 0x55;
         ChipBase[0x5555] = 0xA0;
         ChipBase[i+j] = DiskBuf[j];
         if(!(WaitOk = WaitFlashComplete(&ChipBase[i+j]))) {
            break;
         }
         if(!opt_disable_verify) {
            if(ChipBase[i+j] != DiskBuf[j]){
            /* verify error */
               break;
            }
         }

      }
      UnMapBIOS();
      if(j != sizeof(DiskBuf)) {
      // Report verify error
         if(!WaitOk) {
            printf("\nProgram failed, timeout waiting for DB6 to stop toggling.\n");
            return 11;
         }
         else {
            printf("\nVerify error @ 0x%05x, Data = 0x%02x, should be 0x%02x.\n",
                   i+j,is,DiskBuf[j]);
            return 12;
         }
      }
      printf("\r0x%05x",i);
      fflush(stdout);
   }
   printf("\r0x%05x\n",i);
   return 0;
}

int ProgramBIOS(char *writefile)
{
   int result;
   struct stat StatBuf;
   char far_jmp[2];
   char c;

   if((fp = fopen(writefile,"r")) == NULL) {
      printf("Unable to open file \"%s\":%s\n",writefile,strerror(errno));
      return(6);
   }

/* a few quick sanity checks are called for ... check the filesize */

   if(stat(writefile,&StatBuf)) {
      printf("Unable to stat file \"%s\":%s\n",writefile,strerror(errno));
      return(13);
   }

   if(StatBuf.st_size != BIOS_SIZE) {
      printf("Error: \"%s\" is not the correct size for a BIOS image.\n",
             writefile);
      printf("The file size is %d, it should be %d.\n",
             (int) StatBuf.st_size,BIOS_SIZE);
      return(14);
   }

/*   make sure there's a far jump at 0xffff:0. */

   if(fseek(fp,0x3fff0,SEEK_SET)) {
      printf("Error seeking \"%s\": %s\n",writefile,strerror(errno));
      return(15);
   }

   if(fread(far_jmp,sizeof(far_jmp),1,fp) != 1) {
      printf("Error reading power on jump: %s\n",strerror(errno));
      return(16);
   }

   if(far_jmp[0] != (char) 0xea || far_jmp[1] != (char) 0x5b) {
      printf("Error: \"%s\" does not look like a valid BIOS image.\n",writefile);
      printf("There isn't a far jump @ 0xffff:0.\n");
      printf("(Expected 0xEA 0x5B, but found %02X %02X)\n",far_jmp[0],far_jmp[1]);
      return(17);
   }

   if(fseek(fp,0,SEEK_SET)) {
      printf("Error seeking \"%s\": %s\n",writefile,strerror(errno));
      return(18);
   }

   printf("About to reprogram your BIOS from \"%s\".\n",writefile);
   printf("Are you really, really, REALLY sure you want to do this ?");
   fflush(stdout);

   c = getchar();
   if(c == 'y' || c == 'Y') {
      printf("\nWhatever you do now DON'T turn off power until this program "
             "completes!\n");
      printf("\nOk, here we go...\n");
   }
   else {
      printf("\nI don't blame you!\n");
      return 0;
   }

   result = ProgramIt(fp);
   fclose(fp);
   fp = NULL;


   if(result == 0) {
      printf("Program and verify complete, no errors.\n");
   }
   else {
      printf("\nThe program operation failed, carefully consider what to do next.\n");
      printf("If you reset or power cycle at this point your i-opener will probably\n");
      printf("not boot and you will have to remove the BIOS chip and reprogram it\n");
      printf("by other means to recover.\n\n");
      printf("If I were you I think I'd try running this program again with my\n");
      printf("fingers crossed a bit tighter this time.\n\n");
      printf("Sorry !\n");
   }
   return result;
}

/*
 * getopt a wonderful little function that handles the command line.
 * available courtesy of AT&T.
 */

#ifndef  BSD

int
getopt(argc, argv, opts)
int   argc;
char  **argv, *opts;
{
   static int sp = 1;
   register int c;
   register char *cp;

   if(sp == 1)
      if(optind >= argc ||
         argv[optind][0] != '-' || argv[optind][1] == '\0')
         return(EOF);
      else if(strcmp(argv[optind], "--") == 0) {
         optind++;
         return(EOF);
      }
   optopt = c = argv[optind][sp];
   if(c == ':' || (cp=strchr(opts, c)) == NULL) {
      printf(": illegal option -- ", c);
      if(argv[optind][++sp] == '\0') {
         optind++;
         sp = 1;
      }
      return('?');
   }
   if(*++cp == ':') {
      if(argv[optind][sp+1] != '\0')
         optarg = &argv[optind++][sp+1];
      else if(++optind >= argc) {
         printf(": option requires an argument -- ", c);
         sp = 1;
         return('?');
      } else
         optarg = argv[optind++];
      sp = 1;
   } else {
      if(argv[optind][++sp] == '\0') {
         sp = 1;
         optind++;
      }
      optarg = NULL;
   }
   return(c);
}
#endif
