<?php
/**********************************************************************************
WIKINDX: Bibliographic Management system.
Copyright (C)

This program is free software; you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program;
if not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

The WIKINDX Team 2004
sirfragalot@users.sourceforge.net
**********************************************************************************/
/*
* IMPORTCOMMON class
*
* Provides methods common to the import scripts.
* @author Mark Grimshaw
* @version 1
* $Header: /cvsroot/wikindx/wikindx3/core/file/import/IMPORTCOMMON.php,v 1.17 2005/06/04 21:40:00 sirfragalot Exp $
*/
class IMPORTCOMMON
{
// Constructor
	function IMPORTCOMMON($db)
	{
		$this->db = $db;
		include_once("core/html/HTML.php");
		include_once("core/messages/LANGUAGE.php");
		$this->noSortArray = LANGUAGE::noSort();
		include_once("core/html/HTML.php");
		$this->html = new HTML();
		include_once("core/html/MISC.php");
		include_once("core/collection/COLLECTION.php");
		$this->collection = new COLLECTION($this->db);
		include_once("core/publisher/PUBLISHER.php");
		$this->publisher = new PUBLISHER($this->db);
		include_once("core/creator/CREATOR.php");
		$this->creator = new CREATOR($this->db);
		include_once("core/keyword/KEYWORD.php");
		$this->keyword = new KEYWORD($this->db);
		$this->index = 0;
	}
/*
* titleExists - check if a title already exists in the database.
* We check on title, resource type and, if it exists, primary creator surname.
* 
* @author Mark Grimshaw
* @param string $title - import title
* @param string $creatorSurname - primary creator's surname (may be FALSE)
* @param string $type - import resource type
* @return BOOLEAN
*/
	function titleExists($title, $creatorSurname, $type)
	{
		$title = trim($title);
		$join = $condition2 = FALSE;
		if($creatorSurname)
		{
			$recordset = $this->db->select(array('WKX_creator'), array('id'), 
				" WHERE " . $this->db->formatfield('surname') . "=" . 
				$this->db->tidyInput($creatorSurname));
			while($row = $this->db->loopRecordSet($recordset)) // May be more than one creator with same surname
				$ids[] = $row['id'];
			if(isset($ids))
			{
				$join = " LEFT JOIN " . $this->db->formatTable("WKX_resource_creator") . " ON " .
					$this->db->formatField("WKX_resource_creator.id") . "=" . 
					$this->db->formatField("WKX_resource.id");
				foreach($ids as $id)
					$condition[] = "(" . $this->db->tidyInput($id) . "," . 
					$this->db->formatField("creator1") . ")";
				$condition2 = " AND (FIND_IN_SET" . join(" OR FIND_IN_SET", $condition) . ")";
			}
			else
				return FALSE;
		}
		$condition1 = " WHERE " . $this->db->concat(
			array(
				"CASE WHEN " . $this->db->formatField('noSort') . " IS NULL" . 
				" THEN " . $this->db->tidyInputNoTrim('') . 
				" ELSE " . 
					$this->db->concat(array($this->db->formatField('noSort'), 
					$this->db->tidyInputNoTrim(' '))) . 
				" END", 
				" CASE WHEN " . $this->db->formatField('subtitle') . " IS NULL" . 
				" THEN " . $this->db->formatField('title') . 
				" ELSE " . 
					$this->db->concat(array($this->db->formatField('title'), 
					$this->db->tidyInputNoTrim(': '), $this->db->formatField('subtitle'))) . 
				" END"
			)) . "=" . $this->db->tidyInput($title) . 
			" AND " . $this->db->formatField('type') . "=" . $this->db->tidyInput($type);
		$fields = $condition2 ? array('title', 'creator1') : array('title');
		$recordset = $this->db->select(array('WKX_resource'), $fields, $join . $condition1 . $condition2);
		if($this->db->numRows($recordset))
			return TRUE;
		return FALSE;
	}
/* writeCreatorTable - write creators to WKX_creator table
*
* @author Mark Grimshaw
*
* @param $firstname
* @param $initials (space-delimited)
* @param $prefix (von, della, de etc.)
* @param $surname
* @return lastAutoId - suitable for inserting into WKX_resource_creator `creator1`, `creator2` etc. field
*/
	function writeCreatorTable($firstname, $initials, $prefix, $surname)
	{
		if($id = $this->creator->checkExists($surname, $firstname, $initials, $prefix))
			return $id;
		if($surname)
		{
			$fields[] = "surname";
			$values[] = $surname;
		}
		if($firstname)
		{
			$fields[] = "firstname";
			$values[] = $firstname;
		}
		if($initials)
		{
			$fields[] = "initials";
			$values[] = $initials;
		}
		if($prefix)
		{
			$fields[] = "prefix";
			$values[] = $prefix;
		}
		$this->db->insert('WKX_creator', $fields, $values);
		return $this->db->lastAutoId('WKX_creator');
	}
/*
* writeCollectionTable - write WKX_collection table if necessary.
* @author Mark Grimshaw
*
* @param $title - title of collection (e.g. journal title, newspaper, book title etc.)
* @param $short - abbreviated title of collection
* @param $wkType - the WIKINDX resource type
* @return collection ID suitable for inserting into WKX_resource_misc `collection` field
*/
	function writeCollectionTable($title, $short, $wkType)
	{
		$this->index++;
		$collection = array('type' => $wkType, 'collection' => $title, 'collectionShort' => $short);
		return $this->collection->writeCollectionTable($collection);
	}
/*
* writePublisherTable - write WKX_publisher table if necessary.
* @author Mark Grimshaw
*
* @param $name - name of publisher
* @param $location - location of publisher
* @param $wkType - the WIKINDX resource type
* @return ID suitable for inserting into WKX_resource_misc `publisher` field
*/
	function writePublisherTable($name, $location, $wkType)
	{
		if($wkType == 'proceedings_article')
		{
			$publisher = array('type' => $wkType, 'confPublisherName' => $name, 
				'confPublisherLocation' => $location);
			$id = $this->publisher->writePublisherTable($publisher, FALSE, TRUE);
		}
		else
		{
			$publisher = array('type' => $wkType, 'publisherName' => $name, 'publisherLocation' => $location);
			$id = $this->publisher->writePublisherTable($publisher);
		}
		return $id;
	}
/*
* writeResourceKeywordTable - write WKX_resource_keyword table and, if necessary, WKX_keyword table
* @author Mark Grimshaw
*
* @param $keywords - string of keywords from import file
* @param $resourceId - INT of this resource's id from WKX_resource table
* @param $importType - string for type of import
*/
	function writeResourceKeywordTable($keywords, $resourceId, $importType)
	{
// bibtex keywords field should be delimited by ';'
		if($importType == 'bibtex')
			$keywords = preg_split("/;/", trim($keywords));
		else if($importType == 'endnote')
			$keywords = preg_split("/;/", trim($keywords));
		foreach($keywords as $keyword)
		{
			$keyword = trim($keyword);
			if(!$keyword)
				continue;
			$tempK[] = $keyword;
		}
		if(!isset($tempK))
			return;
		$keywords = array_unique($tempK);
		foreach($keywords as $keyword)
		{
			if($id = $this->keyword->checkExists($keyword))
				$keywordIds[] = $id;
			else
				$keywordInserts[] = $keyword;
		}
		if(isset($keywordInserts))
		{
			foreach($keywordInserts as $insert)
			{
				$this->db->insert('WKX_keyword', array('keyword'), array($insert));
				$keywordIds[] = $this->db->lastAutoID('WKX_keyword');
			}
		}
		if(isset($keywordIds))
		{
			$string = join(",", $keywordIds);
			$this->db->insert('WKX_resource_keyword', array('id', 'keywords'), 
				array($resourceId, $string));
		}
	}
/* splitTitle - split title into WIKINDX component parts (noSort, title : subtitle)
*
* @author Mark Grimshaw
*
* @param $title
* @return $noSort
* @return $title
* @return $subtitle
*/
	function splitTitle($title)
	{
		$noSort = $subtitle = FALSE;
		$array = preg_split("/:\s+/", $title, 2);
		if(sizeof($array) > 1)
		{
			$title = trim(array_shift($array));
			$subtitle = trim(implode('', $array));
		}
		else
			$title = trim($array[0]);
		foreach($this->noSortArray as $pattern)
		{
			if(preg_match("/^($pattern)\s(.*)/i", $title, $matches))
			{
				$title = trim($this->html->removeNl($matches[2]));
				$noSort = trim($this->html->removeNl($matches[1]));
				break;
			}
		}
		return array($noSort, $title, $subtitle);
	}
/*
* createDir - create a directory (if it doesn't exist) based on session ID
* 
* @author Mark Grimshaw
*
* @param $dirName - directory name to make (usually "files/" . session_id() . "/")
* @return $dh directory handle or FALSE if unable to create directory
*/
	function createDir($dirName)
	{
		if(!$dh = @opendir($dirName))
		{
			if(!mkdir($dirName))
				return FALSE;
			else
				$dh = @opendir($dirName);
		}
		return $dh;
	}
/*
* printDuplicates - Print a list of discarded duplicates.
* 
* @author Mark Grimshaw
*
* @param INT $numDiscarded - num. duplicates discarded
* @param ARRAY $titlesDiscarded - authors + titles of discarded duplicates
* @return browser print string.
*/
	function printDuplicates($numDiscarded, $titlesDiscarded)
	{
		include_once("core/messages/MESSAGES.php");
		$messages = new MESSAGES();
		if(($numDiscarded <= 50) && !empty($titlesDiscarded))
		{
			$reject = $messages->text("import", "discarded", " " . $numDiscarded);
			foreach($titlesDiscarded as $title)
				$reject .= MISC::br() . $title;
			return MISC::p($reject);
		}
		return MISC::p($messages->text("import", "discarded", " " . $numDiscarded));
	}
}
?>
