<?php
/**********************************************************************************
WIKINDX: Bibliographic Management system.
Copyright (C)

This program is free software; you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program;
if not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

The WIKINDX Team 2004
sirfragalot@users.sourceforge.net
**********************************************************************************/
/*****
*	BIBTEXIMPORTSTAGE2: BibTeX STAGE2 import class
*
*	$Header: /cvsroot/wikindx/wikindx3/core/file/import/BIBTEXIMPORTSTAGE2.php,v 1.32 2005/06/25 02:59:15 sirfragalot Exp $
*****/
/*
* @author Mark Grimshaw
* @version 2
*/
class BIBTEXIMPORTSTAGE2
{
// Constructor
	function BIBTEXIMPORTSTAGE2($db, $vars)
	{
		$this->db = $db;
		$this->vars = $vars;
		include_once("core/session/SESSION.php");
		$this->session = new SESSION();
		include_once("core/messages/MESSAGES.php");
		$this->messages = new MESSAGES();
		include_once("core/messages/ERRORS.php");
		$this->errors = new ERRORS();
		include_once("core/messages/SUCCESS.php");
		$this->success = new SUCCESS();
// Load config arrays
		include_once("core/file/BIBTEXCONFIG.php");
		$this->config = new BIBTEXCONFIG();
		$this->config->bibtex();
		include_once("core/file/BIBTEXMAP.php");
		$this->map = new BIBTEXMAP();
// need to use English constants for BibTeX
		include_once("languages/en/CONSTANTS.php");
		$constants = new CONSTANTS_en();
		$constants->convertNumbers();
		$this->editionNumbers = array_flip($constants->cardinalToOrdinalWord());
		include_once("core/html/MISC.php");
		include_once("core/html/HTML.php");
		$this->html = new HTML();
		include_once("core/html/FORM.php");
		include_once("core/html/TABLE.php");
		include_once("core/file/TAG.php");
		$this->tag = new TAG($this->db);
		include_once("core/file/import/BIBTEXCREATORPARSE.php");
		$this->parseCreator = new BIBTEXCREATORPARSE();
		include_once("core/file/import/BIBTEXMONTHPARSE.php");
		$this->monthObj = new BIBTEXMONTHPARSE();
		include_once("core/file/import/BIBTEXPAGEPARSE.php");
		$this->pages = new BIBTEXPAGEPARSE();
		include_once("core/file/import/IMPORTCOMMON.php");
		$this->common = new IMPORTCOMMON($this->db);
		include_once("core/messages/UTF8.php");
		$this->utf8 = new UTF8();
		$this->resourceAdded = $this->resourceDiscarded = 0;
	}
/*
* init - start the process
* 
* @author Mark Grimshaw
* @param string $paste	If TRUE, this is an ordinary user cut 'n' pasting bibtex entries into a textarea box.
* @param string $message - optional error message
* @return string
*/
	function init($paste, $message = FALSE)
	{
// if session variable 'bibtexImportLock' is TRUE, user is simply reloading this form
		if($this->session->getVar('importLock'))
			$this->badInput($this->errors->text("done", "fileImport"));
		$this->paste = $paste;
		$this->fileName = $this->gatherStage1();
		include_once("core/file/import/BIBTEXPARSE.php");
		$parse = new BIBTEXPARSE();
		$parse->expandMacro = TRUE; // substitute @string values
		$this->rejectTitles = array();
		$parse->openBib($this->fileName);
		$parse->extractEntries();
		$parse->closeBib();
		list($null, $this->strings, $entries) = $parse->returnArrays(); // don't need preamble
		if(empty($entries))
		{
			$pString = MISC::p($this->messages->text("import", "empty"));
			$this->session->setVar('importLock', TRUE);
			return $pString;
		}
// NB - we need to write data to database as UTF-8 and parse all bibTeX values for laTeX code
		$this->entries = $this->convertEntries($entries);
		$this->writeDb();
		@unlink($this->fileName); // remove garbage - ignore errors
		$pString = MISC::p($this->success->text("bibtexImport"));
		$pString .= MISC::p($this->messages->text("import", "added", " " . $this->resourceAdded));
		$pString .= $this->common->printDuplicates($this->resourceDiscarded, $this->rejectTitles);
		$this->session->setVar('importLock', TRUE);
		return $pString;
	}
/*
* writeDb - write input to the database.
* 
* @author Mark Grimshaw
*/
	function writeDb()
	{
		$tagWritten = $stringWritten = $this->tagId = FALSE;
		$pasteLimit = $this->session->getVar("setup_superadmin") ? FALSE : TRUE;
		foreach($this->entries as $entry)
		{
			$authors = $editors = array();
// For a user cut 'n' pasting. Admin is unlimited.
			if($pasteLimit && ($this->resourceAdded >= $this->session->getVar("setup_maxPaste")))
				break;
			$this->keywords = $this->note = $this->abstract = $this->authorList = 
				$this->editorList = $this->url = $this->month = $this->day = $this->confPublisherId = 
				$this->thesisType = $this->publisherId = $this->collectionId = FALSE;
			$wkType = $this->getType($entry);
			if(($authorWkField = array_search('author', $this->map->{$wkType}['resource_creator'])) && 
				array_key_exists('author', $entry) && $entry['author'])
				$authors = $this->parseCreator->parse($entry['author']);
// For duplicate checking, we try to find the surname of the primary creator ($authors[0][2] is the surname)
			if(!empty($authors) && array_key_exists(0, $authors) && $authors[0][2])
				$creatorSurname = $authors[0][2];
			else
				$creatorSurname = FALSE;
			if(!$this->session->getVar('import_importDuplicates') && 
				$this->common->titleExists($entry['title'], trim($creatorSurname), $wkType))
			{
				$rejectTitle = $entry['title'] . ".";
				if(array_key_exists('author', $entry) && $entry['author'])
					$rejectTitle = trim($entry['author']) . " " . $rejectTitle;
				$this->rejectTitles[] = $rejectTitle;
				$this->resourceDiscarded++;
				continue;
			}
// bibTeX's 'article' type can be wikindx's journal_article, magazine_article or newspaper_article.  If there is no 'month' field, 
// we assume the first, if there's 'month' but no day part of that field, we assume the second and, if there's a day part, assume 
// the third. So, before we can write the resource table and its `type` field, we need to query any month field in the import.
			list($this->startMonth, $this->startDay, $this->endMonth, $this->endDay) = $this->grabMonth($entry);
// A bibtex type with a howpublished field containing a URL is mapped to wikindx's web_article type.
			list($this->url, $this->howPublished) = $this->grabHowPublished($entry);
			list($this->rejected, $entry) = $this->reject($entry, $wkType);
//print_r($entry); print "<P>";
//print_r($rejected); print "<P>";
			$this->resourceId = $this->writeResourceTable($entry, $wkType);
// add any import tag and get tag auto ID.  We write it here after the resource table in case we forbid duplicates and all 
// bibtex entries are duplicates - we don't want an empty tag in the WKX_tag table.
			if(!$tagWritten)
			{
				$this->tagId = $this->writeTagTable();
				$tagWritten = TRUE;
			}
			if(!$stringWritten)
			{
				$this->writeBibtexStringTable();
				$stringWritten = TRUE;
			}
			if(!empty($authors))
				$this->authorList = $this->writeCreatorTable($authors);
			if(($editorWkField = array_search('editor', $this->map->{$wkType}['resource_creator'])) && 
				array_key_exists('editor', $entry) && $entry['editor'])
			{
				$editors = $this->parseCreator->parse($entry['editor']);
				$this->editorList = $this->writeCreatorTable($editors);
			}
			if($authorWkField || $editorWkField)
				$this->writeResourceCreatorTable($authorWkField, $editorWkField);
			$this->writePublisherTable($entry, $wkType);
			$this->writeCollectionTable($entry, $wkType);
			$this->writeResourceMiscTable($entry, $wkType);
			$this->writeResourceYearTable($entry, $wkType);
			$this->writeResourcePageTable($entry);
			$this->writeResourceNoteTable($entry);
			$this->writeResourceAbstractTable($entry);
			$this->writeResourceKeywordTable($entry);
			$this->writeResourceCategoryTable();
			$this->writeResourceTimestampTable();
			$this->writeImportRawTable();
			$this->resourceAdded++;
		}
// update total no. resources in summary table
		$recordset = $this->db->select(array('WKX_database_summary'), array("totalResources"));
		$totalResources = $this->db->fetchOne($recordset) + $this->resourceAdded;
		$this->db->update('WKX_database_summary', array("totalResources" => $totalResources));
	}

/*
* reject -- gather rejected fields that wikindx does not recognise for that type and remove from $entry
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @return $wkType - the WIKINDX resource type for this bibtex entry
* @return $rejected - array of rejected field and their values (with bibTeX delimiters added back in)
* @return $newEntry - $entry with $rejected elements removed
*/
	function reject($entry, $wkType)
	{
		$rejectedEntry = FALSE;
		$index = 0;
		foreach($entry as $key => $value)
		{
			if(($key == 'bibtexEntryType') || 
			($key == 'howpublished') || ($key == 'abstract') || ($key == 'keywords'))
			{
				$newEntry[$key] = $value;
				continue;
			}
			if($key == 'bibtexCitation')
			{
				$rejected['citation'] = trim($value);
				continue;
			}
			if($key == 'note') // Use 'note' in preference to 'annote'
			{
				$newEntry[$key] = $value;
				continue;
			}
			if(($key == 'annote') && !array_key_exists('note', $entry)) // Use 'note' in preference to 'annote'
			{
				$newEntry[$key] = $value;
				continue;
			}
			if(($key == 'month') && $this->url && ($wkType == 'web_article'))
				continue;
			if(array_search($key, $this->map->{$wkType}['possible']) !== FALSE)
			{
				$newEntry[$key] = $value;
				continue;
			}
// If we get here, we have a bibtex field and value that are not recognised by wikindx. Need to to store this in case user 
// has requested that unused fields are also stored in the database.
// Return any @STRING substitution in $value back to original state
			$rejectedEntry = TRUE;
// Do @string substitutions
			if($this->strings && ($strKey = array_search($value, $this->strings)))
					$rejected[$key] = $strKey;
// No substitution so return quoted
			else
				$rejected[$key] = "\"" . $value . "\"";
		}
		if(!$rejectedEntry)
			return array(FALSE, $entry);
		return array($rejected, $newEntry);
	}
/*
* getType - figure out what wikindx type this entry is
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @return $wkType - the WIKINDX resource type for this bibtex entry
*/
	function getType($entry)
	{
		if($entry['bibtexEntryType'] == 'article')
		{
			if($this->day)
				$wkType = 'newspaper_article';
			else if($this->month)
				$wkType = 'magazine_article';
			else // no day or month
				$wkType = 'journal_article';
		}
		else if(($entry['bibtexEntryType'] == 'misc') && $this->url)
			$wkType = 'web_article';
		else if($entry['bibtexEntryType'] == 'misc')
			$wkType = 'miscellaneous';
		else if($entry['bibtexEntryType'] == 'mastersthesis')
		{
			$wkType = 'thesis';
			$this->thesisType = "master's";
		}
		else if($entry['bibtexEntryType'] == 'phdthesis')
		{
			$wkType = 'thesis';
			$this->thesisType = "PhD";
		}
		else if($entry['bibtexEntryType'] == 'conference') // same as `inproceedings`
			$wkType = 'proceedings_article';
		else if($entry['bibtexEntryType'] == 'incollection') // same as `inbook`
			$wkType = 'book_article';
		else if($entry['bibtexEntryType'] == 'collection') // same as `proceedings`
			$wkType = 'proceedings';
		else if(!$wkType = array_search($entry['bibtexEntryType'], $this->map->types))
			$wkType = 'miscellaneous'; // everything else
		return $wkType;
	}
/*
* writeResourceTable - write WKX_resource table and get lastAutoId
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @param $wkType - the WIKINDX resource type for this bibtex entry
* @return autoId
*/
	function writeResourceTable($entry, $wkType)
	{
// bibTeX has no way of saying whether a thesis is a thesis or a dissertation so here we force it to 'thesis'.
		if($wkType == 'thesis')
		{
			$fields[] = 'field2';
			$values[] = 'thesis';
		}
		$fields[] = 'type';
		$values[] = $wkType;
		list($noSort, $title, $subtitle) = $this->common->splitTitle($entry['title']);
		$fields[] = 'title';
		$values[] = $title;
		if($noSort)
		{
			$fields[] = 'noSort';
			$values[] = $noSort;
		}
		if($subtitle)
		{
			$fields[] = 'subtitle';
			$values[] = $subtitle;
		}
		if($this->thesisType)
		{
			$fields[] = 'field1';
			$values[] = $this->thesisType;
		}
		if(($wkType == 'miscellaneous') && $this->howPublished)
		{
			$fields[] = $this->map->miscellaneous['howpublished'];
			$values[] = $this->howPublished;
		}
		if($this->url) // from @misc 'howpublished' field
			$entry['url'] = $this->url;
		foreach($entry as $bibField => $bibValue)
		{
// ISBN, ISSN and URL are uppercase in BIBTEXMAP but everything in $entry is lowercase
			if(($bibField == 'url') || ($bibField == 'isbn') || ($bibField == 'issn'))
				$bibField = strtoupper($bibField);
			if($wkField = array_search($bibField, $this->map->{$wkType}['resource']))
			{
				$fields[] = $wkField;
				$values[] = $bibValue;
			}
		}
		$this->db->insert('WKX_resource', $fields, $values);
		return $this->db->lastAutoID('WKX_resource');
	}
/*
* writeCreatorTable - write WKX_creator table and get lastAutoId
* @author Mark Grimshaw
*
* @param $creators - assoc array of creators for one entry
* @return comma-separated list of creator IDs ready for insertion into WKX_resource_creator table.
*/
	function writeCreatorTable($creators)
	{
		if(!$creators)
			return FALSE;
		foreach($creators as $array)
		{
			$prefix = trim($array[3]);
			$surname = trim($array[2]);
			$firstname = trim($array[0]);
			$initials = trim($array[1]);
			$creatorIds[] = $this->common->writeCreatorTable($firstname, $initials, $prefix, $surname);
		}
		return implode(',', $creatorIds);
	}
/*
* writeResourceCreatorTable - write to WKX_resource_creator table
* @author Mark Grimshaw
*
* @param $authorWkField - wikindx equivalent field in WKX_resource_creator for a bibtex 'author' field.
* @param $editorWkField - wikindx equivalent field in WKX_resource_creator for a bibtex 'editor' field.
*/
	function writeResourceCreatorTable($authorWkField, $editorWkField)
	{
		if($authorWkField && $this->authorList)
		{
			$fields[] = $authorWkField;
			$values[] = $this->authorList;
		}
		if($editorWkField && $this->editorList)
		{
			$fields[] = $editorWkField;
			$values[] = $this->editorList;
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$this->db->insert('WKX_resource_creator', $fields, $values);
	}
/*
* writeCollectionTable - write WKX_collection table
* @author Mark Grimshaw
*
* The only input from bibtex that can be a wikindx 'collection' is the 'journal' field or, for @inbook, 
* the 'booktitle' field.
*
* @param $entry - assoc array of one entry for import.
* @param $wkType - the WIKINDX resource type for this bibtex entry
*/
	function writeCollectionTable($entry, $wkType)
	{
		$title = FALSE;
		if(($wkType == 'book_article') && !array_key_exists('booktitle', $entry))
			return;
		else if(array_key_exists('booktitle', $entry))
			$title = trim($entry['booktitle']);
		if(!$title && !array_key_exists('journal', $entry))
			return;
		else if(!$title && array_key_exists('journal', $entry))
			$title = trim($entry['journal']);
		if(!$title)
			return;
		if(is_array($this->strings))
			$short = array_search($title, $this->strings);
		else
			$short = FALSE;
		$this->collectionId = $this->common->writeCollectionTable($title, $short, $wkType);
	}
/*
* writePublisherTable - write WKX_publisher table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @param $wkType - the WIKINDX resource type for this bibtex entry
*/
	function writePublisherTable($entry, $wkType)
	{
		$organization = $publisherName = $publisherLocation = $conferenceLocation = FALSE;
		if(array_key_exists('publisher', $entry))
			$publisherName = trim($entry['publisher']);
		if(array_key_exists('organization', $entry) && ($wkType != 'proceedings_article'))
			$publisherName = trim($entry['organization']);
		else if(array_key_exists('organization', $entry))
			$organization = trim($entry['organization']);
		else if(array_key_exists('school', $entry))
			$publisherName = trim($entry['school']);
		else if(array_key_exists('institution', $entry))
			$publisherName = trim($entry['institution']);
		if(!$organization && !$publisherName)
			return;
		if(array_key_exists('address', $entry))
			$publisherLocation = trim($entry['address']);
		if(array_key_exists('location', $entry))
		{
			if($wkType == 'proceedings_article')
				$conferenceLocation = trim($entry['location']);
			else
				$publisherLocation = trim($entry['location']);
		}
		if($wkType == 'proceedings_article')
		{
			$this->publisherId = 
				$this->common->writePublisherTable($organization, $conferenceLocation, $wkType);
			$this->confPublisherId = 
				$this->common->writePublisherTable($publisherName, $publisherLocation, $wkType);
		}
		else
			$this->publisherId = 
				$this->common->writePublisherTable($publisherName, $publisherLocation, $wkType);
	}
/*
* writeResourceMiscTable - write WKX_resource_misc table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @param $wkType - the WIKINDX resource type for this bibtex entry
*/
	function writeResourceMiscTable($entry, $wkType)
	{
		foreach($entry as $bibField => $bibValue)
		{
			if($wkField = array_search($bibField, $this->map->{$wkType}['resource_misc']))
			{
				$fields[] = $wkField;
				$values[] = $bibValue;
			}
		}
		if($this->collectionId)
		{
			$fields[] = 'collection';
			$values[] = $this->collectionId;
		}
		if($this->publisherId)
		{
			$fields[] = 'publisher';
			$values[] = $this->publisherId;
		}
		if($this->confPublisherId)
		{
			$fields[] = "miscField1";
			$values[] = $this->confPublisherId;
		}
		if($this->tagId)
		{
			$fields[] = 'tag';
			$values[] = $this->tagId;
		}
		if(($wkType == 'newspaper_article') || ($wkType == 'magazine_article') || 
			($wkType == 'proceedings_article') || ($wkType == 'proceedings') || 
			($wkType == 'journal_article') || ($this->url && ($wkType == 'web_article')))
		{
			if($this->startMonth)
			{
				$fields[] = 'miscField3';
				$values[] = $this->startMonth;
			}
			if($this->startDay)
			{
				$fields[] = 'miscField2';
				$values[] = $this->startDay;
			}
		}
		if(($wkType == 'proceedings_article') || ($wkType == 'proceedings') || ($wkType == 'magazine_article'))
		{
			if($this->endMonth)
			{
				$fields[] = 'miscField6';
				$values[] = $this->endMonth;
			}
			if($this->endDay)
			{
				$fields[] = 'miscField5';
				$values[] = $this->endDay;
			}
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$fields[] = 'addUserIdResource';
		$values[] = $this->session->getVar("setup_userId");
		$this->db->insert('WKX_resource_misc', $fields, $values);
	}
/*
* writeResourceYearTable - write WKX_resource_year table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
* @param $wkType - the WIKINDX resource type for this bibtex entry
*/
	function writeResourceYearTable($entry, $wkType)
	{
		foreach($entry as $bibField => $bibValue)
		{
			if($wkField = array_search($bibField, $this->map->{$wkType}['resource_year']))
			{
				$fields[] = $wkField;
				$values[] = $bibValue;
			}
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$this->db->insert('WKX_resource_year', $fields, $values);
	}
/*
* writeResourcePageTable - write WKX_resource_page table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
*/
	function writeResourcePageTable($entry)
	{
		if(!array_key_exists('pages', $entry))
			return;
		list($pageStart, $pageEnd) = $this->pages->init($entry['pages']);
		if($pageStart)
		{
			$fields[] = 'pageStart';
			$values[] = $pageStart;
		}
		if($pageEnd)
		{
			$fields[] = 'pageEnd';
			$values[] = $pageEnd;
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$this->db->insert('WKX_resource_page', $fields, $values);
	}
/*
* writeResourceNoteTable - write WKX_resource_note table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
*/
	function writeResourceNoteTable($entry)
	{
		if(array_key_exists('note', $entry))
		{
			$fields[] = 'text';
			$values[] = $entry['note'];
		}
		else if(array_key_exists('annote', $entry))
		{
			$fields[] = 'text';
			$values[] = $entry['annote'];
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$fields[] = 'addUserIdNote';
		$values[] = $this->session->getVar("setup_userId");
		$this->db->insert('WKX_resource_note', $fields, $values);
	}
/*
* writeResourceAbstractTable - write WKX_resource_abstract table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
*/
	function writeResourceAbstractTable($entry)
	{
		if(array_key_exists('abstract', $entry))
		{
			$fields[] = 'abstract';
			$values[] = $entry['abstract'];
		}
		if(!isset($fields))
			return;
		$fields[] = 'id';
		$values[] = $this->resourceId;
		$fields[] = 'addUserIdAbstract';
		$values[] = $this->session->getVar("setup_userId");
		$this->db->insert('WKX_resource_abstract', $fields, $values);
	}
/*
* writeResourceKeywordTable - write WKX_resource_keyword table
* @author Mark Grimshaw
*
* @param $entry - assoc array of one entry for import.
*/
	function writeResourceKeywordTable($entry)
	{
		if(array_key_exists('keywords', $entry) && trim($entry['keywords']))
			$this->common->writeResourceKeywordTable($entry['keywords'], $this->resourceId, 'bibtex');
	}
/*
* writeResourceCategoryTable - write WKX_resource_category table
* @author Mark Grimshaw
* @param $entry - assoc array of one entry for import.
*/
	function writeResourceCategoryTable()
	{
		if(!$groups = $this->session->getVar('import_groups'))
			$groups = 1; // force to 'General' category
		$this->db->insert('WKX_resource_category', array('id', 'categories'), 
			array($this->resourceId, $groups));
	}
/*
* writeImportRawTable - write WKX_import_raw table
* @author Mark Grimshaw
*/
	function writeImportRawTable()
	{
		if(empty($this->rejected) || !$this->session->getVar("import_raw"))
			return;
		$rejected = '';
		foreach($this->rejected as $key => $value)
			$rejected .= "$key = $value\n";
		$fields[] = 'id';
		$values[] = $this->resourceId;
		if(isset($this->bibtexStringId))
		{
			$fields[] = 'stringId';
			$values[] = $this->bibtexStringId;
		}
		$fields[] = 'text';
		$values[] = base64_encode(serialize($rejected));
		$fields[] = 'importType';
		$values[] = 'bibtex';
		$this->db->insert('WKX_import_raw', $fields, $values);
	}
/*
* writeTagTable - write import tag to WKX_tag table
* 
* @author Mark Grimshaw
* @return lastAutoId
*/
	function writeTagTable()
	{
		if($tagId = $this->session->getVar('import_tagId'))
			return $tagId;
		if(!$tag = $this->session->getVar('import_tag'))
			return FALSE;
		$this->db->insert('WKX_tag', array('tag'), 
			array(utf8_encode($this->html->removeNl($tag))));
		return $this->db->lastAutoID('WKX_tag');
	}
/*
* writeBibtexStringTable - write $this->strings to WKX_bibtex_string table
* 
* @author Mark Grimshaw
*/
	function writeBibtexStringTable()
	{
		if(!empty($this->strings) && $this->session->getVar("import_raw"))
		{
			$fields[] = 'text';
			foreach($this->strings as $key => $value)
				$raw[] = '@STRING{' . $key . '=' . $value . '}';
			$values[] = base64_encode(serialize(implode("\n", $raw)));
			$this->db->insert('WKX_bibtex_string', $fields, $values);
			$this->bibtexStringId = $this->db->lastAutoId('WKX_bibtex_string');
		}
	}
/*
* writeResourceTimestampTable - write timestamp to WKX_resource_timestamp table
* 
* @author Mark Grimshaw
*/
	function writeResourceTimestampTable()
	{
		$this->db->insert('WKX_resource_timestamp', array('id', 'timestamp'), 
			array($this->resourceId, $this->db->formatTimestamp()));
	}
/*
* grabHowPublished - check for type of howpublished field in bibtex misc entry
* 
* @author Mark Grimshaw
* @param $entry - assoc array of one entry for import.
* @return array(URL, howPublished)
*/
	function grabHowPublished($entry)
	{
		$url = $howPublished = FALSE;
		if(($entry['bibtexEntryType'] == 'misc') && array_key_exists('howpublished', $entry))
		{
			if(preg_match("#^\\\url{(.*://.*)}#", $entry['howpublished'], $match))
				$url = $match[1];
			else
				$howPublished = $entry['howpublished'];
		}
		return array($url, $howPublished);
	}
/*
* grabMonth - check for any month field and split into component day/month fields
* 
* @author Mark Grimshaw
* @param $entry - assoc array of one entry for import.
* @return array(startMonth, startDay, endMonth, endDay)
*/
	function grabMonth($entry)
	{
		$startMonth = $startDay = $endMonth = $endDay = FALSE;
		if(array_key_exists('month', $entry))
			list($startMonth, $startDay, $endMonth, $endDay) = $this->monthObj->init($entry['month']);
		return array($startMonth, $startDay, $endMonth, $endDay);
	}
/*
* gatherStage1 - gather input from stage 1 and return a fullpath filename for parsing.
* If $this->paste is TRUE, this is a user cut 'n' pasting bibtex entries in a textarea box. We write the input to a 
* temporary file.
* 
* @author Mark Grimshaw
* @return string
*/
	function gatherStage1()
	{
// reset the session variable finalise flag
		$this->session->setVar('importLock', FALSE);
// bibtex_groups is a multiple select box so handle as array
		if(isset($this->vars['import_groups']) && $this->vars['import_groups'])
		{
			if(!$this->session->setVar('import_groups', trim(implode(',', $this->vars['import_groups']))))
				$this->badInput($this->errors->text("sessionError", "write"));
		}
		if(isset($this->vars['import_raw']) && $this->vars['import_raw'])
		{
			if(!$this->session->setVar('import_raw', 1))
				$this->badInput($this->errors->text("sessionError", "write"));
		}
		if(isset($this->vars['import_importDuplicates']) && $this->vars['import_importDuplicates'])
		{
			if(!$this->session->setVar('import_importDuplicates', 1))
				$this->badInput($this->errors->text("sessionError", "write"));
		}
// Force to 1 => 'General' category
		if(!$this->session->getVar('import_groups'))
		{
			if(!$this->session->setVar('import_groups', 1))
				$this->badInput($this->errors->text("sessionError", "write"));
		}
		$dirName = "files/" . session_id() . "/";
		if(!$dh = $this->common->createDir($dirName))
			$this->badInput($this->errors->text("file", "folder"));
		if(!$this->paste)
		{
//			global $_FILES;
			if(!isset($_FILES['import_file']))
			{
				if($file = $this->session->getVar('import_file'))
					return $dirName = "files/" . session_id() . "/" . $file;
				else
				{
					if($dh)
						closedir($dh);
					$this->badInput($this->errors->text("file", "upload"));
				}
			}
// Check for file input
			if(!move_uploaded_file($_FILES['import_file']['tmp_name'], 
				$dirName . $_FILES['import_file']['name']))
			{
				if($dh)
					closedir($dh);
				$this->badInput($this->errors->text("import_file", "upload"));
			}
			if($dh) closedir($dh);
			if(!$this->session->setVar('import_file', $_FILES['import_file']['name']))
				$this->badInput($this->errors->text("sessionError", "write"));
			if($this->vars['import_tag'])
			{
				if(!$tagId = $this->tag->checkExists($this->vars['import_tag']))
				{
					if(!$this->session->setVar('import_tag', $this->vars['import_tag']))
						$this->badInput($this->errors->text("sessionError", "write"));
				}
				else
				{
					if(!$this->session->setVar('import_tagId', $tagId))
						$this->badInput($this->errors->text("sessionError", "write"));
				}
			}
			else if(isset($this->vars['import_tagId']) && $this->vars['import_tagId'])
			{
				if(!$this->session->setVar('import_tagId', $this->vars['import_tagId']))
					$this->badInput($this->errors->text("sessionError", "write"));
			}
			return $dirName = "files/" . session_id() . "/" . $_FILES['import_file']['name'];
		}
		else
		{
			if(!trim($this->vars['import_paste']))
				$this->badInput($this->errors->text("inputError", "missing"));
			include_once("core/file/FILE.php");
			$file = new FILE($this->db);
			if(!$fileName = $file->createFileName('.bib'))
				$this->badInput($this->errors->text("file", "write", ": $fileName"));
			if(!$fp = fopen("$fileName", "w"))
				$this->badInput($this->errors->text("file", "write", ": $fileName"));
			if(!fputs($fp, stripslashes(trim($this->vars['import_paste']))))
				$this->badInput($this->errors->text("file", "write", ": $fileName"));
			fclose($fp);
			return $fileName;
		}
	}
/*
* convertEntries - convert any laTeX code and convert to UTF-8 ready for storing in the database
* 
* @author Mark Grimshaw
* @param string $entries - multidimensional array of entries
* @return multidimensional array of converted entries.
*/
	function convertEntries($entries)
	{
		foreach($this->config->bibtexSpCh as $key => $value)
		{
			$replaceBibtex[] = chr($key);
			$matchBibtex[] = preg_quote("/$value/");
		}
		foreach($this->config->bibtexSpChOld as $key => $value)
		{
			$replaceBibtex[] = chr($key);
			$matchBibtex[] = preg_quote("/$value/");
		}
		foreach($this->config->bibtexSpChOld2 as $key => $value)
		{
			$replaceBibtex[] = chr($key);
			$matchBibtex[] = preg_quote("/$value/");
		}
		foreach($this->config->bibtexSpChLatex as $key => $value)
		{
			$replaceBibtex[] = chr($key);
			$matchBibtex[] = preg_quote("/$value/");
		}
		$index = 0;
		foreach($entries as $array)
		{
			foreach($array as $key => $value)
			{
				$value = preg_replace($matchBibtex, $replaceBibtex, $value);
// Transform ISO-8859-1 strings to UTF-8 - web browser encoding is set to UTF-8.  This is fine for form input 
// and means that such input is stored in the db as UTF-8 but text file input (as here) requires an explicit conversion.
				$temp[$index][$key] = utf8_encode($value);
			}
			$index++;
		}
		return $temp;
	}
/*
* badInput - error handling - send user back to previous stage.
* 
* @author Mark Grimshaw
* @param string $error - any error message
*/
	function badInput($error)
	{
		include_once("core/template/TEMPLATE.php");
		$template = new TEMPLATE('content');
		include_once("core/file/import/BIBTEXIMPORTSTAGE1.php");
		$stage = new BIBTEXIMPORTSTAGE1($this->db, $this->vars);
		include_once("core/html/CLOSE.php");
		$template->setVar('heading', $this->messages->text("heading", "bibtexImport"));
// fix for user using browser back buttons - forces to the more innoccuous user paste.
		if(!isset($this->paste))
			$this->paste = TRUE;
		if(!$this->paste)
			$template->setVar('formStart', FORM::formMultiHeader("importBibtexStage2"));
		else
			$template->setVar('formStart', FORM::formHeader("resourcePasteBibtexStage2"));
		$template->setVar('body', $stage->init($this->paste, $error));
		$template->setVar('formEnd', FORM::formEnd());
		new CLOSE($this->db, $template->process());
	}
}
?>
