<?php
/**********************************************************************************
WIKINDX: Bibliographic Management system.
Copyright (C)

This program is free software; you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program;
if not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

The WIKINDX Team 2004
sirfragalot@users.sourceforge.net
**********************************************************************************/
/*
* EXPORT COMMON class
*
* Provides methods common to the export scripts.
* @author Mark Grimshaw
* @version 1
* $Header: /cvsroot/wikindx/wikindx3/core/file/export/EXPORTCOMMON.php,v 1.12 2005/06/11 01:46:32 sirfragalot Exp $
*/
class EXPORTCOMMON
{
// Constructor
	function EXPORTCOMMON($db)
	{
		$this->db = $db;
		include_once("core/cite/CITEFORMAT.php");
		$this->cite = new CITEFORMAT($this->db, FALSE);
		include_once("core/html/HTML.php");
		include_once("core/file/FILE.php");
		$this->file = new FILE($this->db);
		include_once("core/messages/MESSAGES.php");
		$this->messages = new MESSAGES();
		include_once("core/messages/ERRORS.php");
		$this->errors = new ERRORS();
		include_once("core/messages/UTF8.php");
		$this->utf8 = new UTF8();
		$this->roman = array('/I/', '/II/', '/III/', '/IV/', '/V/', '/VI/', '/VII/', '/VIII/', '/IX/', '/X/');
	}
/*
* listFiles - list all exported files in the user's session directory within files/
* 
* @author Mark Grimshaw
*/
	function listFiles()
	{
		include_once("core/html/MISC.php");
		list($dirName, $deletePeriod, $array) = $this->file->listFiles();
		if(!$dirName)
			$this->failure($this->errors->text("file", "read"));
		$pString = MISC::p($this->messages->text("file", "contents"));
		foreach($array as $key => $value)
			$pString .= MISC::a("link", $key, $dirName . '/' . $key, "_blank") . MISC::br();
		$minutes = $deletePeriod/60;
		$pString .= MISC::hr();
		$pString .= MISC::p($this->messages->text("file", "warning", " $minutes "));
		return $pString;
	}
/*
* titleFormat - format and return the resource title from the supplied SQL $row
* 
* @author Mark Grimshaw
* @param assocArray $row
* @param boolean $bibtex
* @return string
*/
	function titleFormat($row, $bibtex = FALSE)
	{
		$noSort = $row['noSort'] ? stripslashes($row['noSort']) . ' ' : FALSE;
		if($row['subtitle'])
			$string = $noSort . stripslashes($row['title']) . ": " . stripslashes($row['subtitle']);
		else
			$string = $noSort . stripslashes($row['title']);
		if($bibtex)
			return $string;
// If !bibtex, remove any braces that have been inserted to maintain case of characters - only required for resource title
		return preg_replace("/[{|}]/", "", $string);
	}
/*
* grabNote - grab note from WKX_resource_note and strip BBCode and (optionally), multiple whitespace
* 
* @author Mark Grimshaw
* @param assocArray $row
* @param boolean $exportType
* @return string
*/
	function grabNote($row, $exportType)
	{
		$recordset = $this->db->select(array('WKX_resource_note'), array('text'), 
			" WHERE " . $this->db->formatField('id') . "=" . $this->db->tidyInput($row['resourceId']));
		$note = $this->db->fetchRow($recordset);
		if($note['text'])
		{
			$text = $this->cite->citeParse(stripslashes($note['text']), FALSE);
			$text = HTML::stripBBCode($text);	
// replace all whitespace (TABS, CR, \n etc.) with single space.
			if(($exportType == 'ris') || ($exportType == 'endnote') || ($exportType == 'bibtex'))
				return preg_replace("/\s/", " ", $text);
// For bibtex, ensure first letter is capitalized
			if($exportType == 'bibtex')
				return $this->utf8->ucfirst($text);
			return $text;
		}
		return '';
	}
/*
* grabAbstract - abstract from WKX_resource_abstract and strip BBCode and (optionally), multiple whitespace
* 
* @author Mark Grimshaw
* @param assocArray $row
* @param boolean $exportType
* @return string
*/
	function grabAbstract($row, $exportType)
	{
		$recordset = $this->db->select(array('WKX_resource_abstract'), array('abstract'), 
			" WHERE " . $this->db->formatField('id') . "=" . $this->db->tidyInput($row['resourceId']));
		$abstract = $this->db->fetchRow($recordset);
		if($abstract['abstract'])
		{
			$text = $this->cite->citeParse(stripslashes($abstract['abstract']), FALSE);
			$text = HTML::stripBBCode($text);
// replace all whitespace (TABS, CR, \n etc.) with single space.
			if(($exportType == 'ris') || ($exportType == 'endnote') || ($exportType == 'bibtex'))
				return preg_replace("/\s/", " ", $text);
// For bibtex, ensure first letter is capitalized
			if($exportType == 'bibtex')
				return $this->utf8->ucfirst($text);
			return $text;
		}
		return '';
	}
/*
* pageFormat - return formatted pageStart and pageEnd with different delimiters
* 
* @author Mark Grimshaw
* @param assocArray $row
* @param string $exportType
* @return string
*/
	function pageFormat($row, $exportType)
	{
		$page = FALSE;
		if($row['pageStart'])
		{
			if($exportType == 'endnote')
				$page = stripslashes($row['pageStart']);
			else if($exportType == 'ris')
				$page = 'SP  - ' . stripslashes($row['pageStart']);
			if($exportType == 'bibtex')
				$page = stripslashes($row['pageStart']);
		}
		if($row['pageEnd'])
		{
			if($exportType == 'endnote')
				$page .= '-' . stripslashes($row['pageEnd']);
			else if($exportType == 'ris')
				$page .= "\r" . 'EP  - ' . stripslashes($row['pageEnd']);
			if($exportType == 'bibtex')
				$page .= '--' . stripslashes($row['pageEnd']);
		}
		if($page)
			return $page;
		return '';
	}
/*
* keywordFormat - return formatted keywords with different delimiters
* 
* @author Mark Grimshaw
* @param assocArray $row
* @param string $exportType
* @return string
*/
	function keywordFormat($row, $exportType)
	{
		$recordset = $this->db->select(array('WKX_keyword'), array("keyword"), 
			" LEFT JOIN " . $this->db->formatTable("WKX_resource_keyword") . 
			" ON FIND_IN_SET(" . $this->db->formatField("WKX_keyword.id") . ", " . 
			$this->db->formatField('keywords') . ")" . 
			" WHERE " . $this->db->formatField("WKX_resource_keyword.id") ." = " . 
			$this->db->tidyInput($row['resourceId']));
		if($this->db->numRows($recordset))
		{
			while($kw = $this->db->loopRecordSet($recordset))
				$k[] = stripslashes($kw['keyword']);
			if($exportType == 'endnote')
				return join(";", $k);
			else if($exportType == 'ris')
			{
// asterisk (character 42) is not allowed in the author, keywords, or periodical name fields - replace with '#'
				foreach($k as $key => $value)
					$k[$key] = 'KW  - ' . preg_replace("/\*/", "#", $value);
				return join("\r", $k);
			}
			else if($exportType == 'bibtex')
				return join(",", $k);
		}
		return '';
	}
/*
* formatName - format one name depending on the export type
* 
* @author Mark Grimshaw
* @param assocArray $creatorRow
* @param string $exportType
* @return string
*/
	function formatName($creatorRow, $exportType)
	{
		$surname = $firstname = $initials = '';
// WIKINDX stores Jr., IV etc. at end of surname...
		if($creatorRow['surname'])
		{
			if($creatorRow['prefix'])
				$surname = stripslashes($creatorRow['prefix']) . " " . 
				stripslashes($creatorRow['surname']);
			else
				$surname = stripslashes($creatorRow['surname']);
		}
		if($creatorRow['firstname'])
			$firstname = stripslashes($creatorRow['firstname']);
		if($creatorRow['initials'])
		{
			if($exportType == 'endnote')
				$initials = implode(' ', explode(' ', stripslashes($creatorRow['initials'])));
			else if($exportType == 'ris')
				$initials = implode('.', explode(' ', stripslashes($creatorRow['initials']))) . ".";
			else if($exportType == 'bibtex')
				$initials = implode('. ', explode(' ', stripslashes($creatorRow['initials']))) . ".";
		}
		if($exportType == 'ris')
		{
			if($firstname && $initials)
				return $surname . ',' . $firstname . ',' . $initials;
			else if($firstname)
				return $surname . ',' . $firstname;
			else if($initials)
				return $surname . ',' . $initials;
		}
		else if($exportType == 'endnote')
		{
			if($firstname && $initials)
				return $surname . ',' . $firstname . ' ' . $initials;
			else if($firstname)
				return $surname . ',' . $firstname;
			else if($initials)
				return $surname . ',' . $initials;
		}
		else if($exportType == 'bibtex')
		{
			if(preg_match("/(.*)(Sr\.|jr\.)/i", $surname, $matches))
				$surname = trim($matches[1]) . ", " . trim($matches[2]);
			if(preg_match("/(.*)\s(I|II|III|IV|V|VI|VII|VIII|IX|X)$/", $surname, $matches))
				$surname = trim($matches[1]) . ", " . trim($matches[2]);
			if($firstname && $initials)
				
				return $surname . ", " . $firstname . ' ' . $initials;
			else if($firstname)
				return $surname . ", " . $firstname;
			else if($initials)
				return $surname . ", " . $initials;
		}
		return $surname; // if all else fails
	}
}
?>
