<?php
/**********************************************************************************
WIKINDX: Bibliographic Management system.
Copyright (C)

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program; 
if not, write to the 
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

The WIKINDX Team 2004
sirfragalot@users.sourceforge.net
**********************************************************************************/
/*****
*	Endnote Map class.
*	Provides mappings for Endnote export/import
*
*	$Header: /cvsroot/wikindx/wikindx3/core/file/ENDNOTEMAP.php,v 1.12 2005/06/11 03:11:48 sirfragalot Exp $
*****/
class ENDNOTEMAP
{
/*
* Constructor
*
* @author Mark Grimshaw
* @version 1
*/
	function ENDNOTEMAP()
	{
		$this->loadMap();
	}
/*
* Create arrays based on database table and resource type.
* The arrays are named for WIKINDX resource types.
* The key of the resource_xxx arrays is the database field within the WKX_resource_xxx database table. For exporting 
* Endnote tabbed files this is not needed but will prove useful should there be an eventual Endnote import.
* The key of the resource_xxx array is the WIKINDX field in that table while the value is the Endnote 
* equivalent field (Generic).  For an export, these keys can be found as the fields in the SQL $row assoc. array.
*
* The database matrix can be found in docs/resourceDatabaseMatrix.html
*
* Some things must be sorted out on the fly in the import/export code:
* 1/ Endnote only has a field 'Pages' whereas WIKINDX uses pageStart and pageEnd fields.
* 2/ Endnote has a separate type for an 'Edited Book' - WIKINDX makes no such distinction but, if a WIKINDX book 
* has editors but no author, then it is an Endnote 'Edited Book'.
* 3/ WIKINDX book and book_article types can have a reprintYear in `year2`.  If a book has this in addition to a 
* publicationYear in `year1` then we need to switch the two fields so that `year1` is Endnote's 'Original Publication'
* field and `year2` is Endnote's 'Year'.  They are defined here as if there were no reprintYear.
* 4/ Most dates (newspaper, magazine, hearing etc.) have to be created on the fly from database fields `miscField2`
* (day) and `miscField3` (month).
* 5/ A WIKINDX 'title' is comprised of `subtitle`, `title` and `noSort` fields.
*
* @author Mark Grimshaw
* @version 1
*/
	function loadMap()
	{
// WIKINDX resource types (keys) and their Endnote 8.0 equivalent (values).
		$this->types = array(
				'book'			=>	'Book',
				'book_article'		=>	'Book Section',
				'journal_article'	=>	'Journal Article',
				'newspaper_article'	=>	'Newspaper Article',
				'magazine_article'	=>	'Magazine Article',
				'proceedings'		=>	'Generic',
				'conference_paper'	=>	'Conference Paper',
				'proceedings_article'	=>	'Conference Proceedings',
				'thesis'		=>	'Thesis',
				'web_article'		=>	'Electronic Source',
				'film'			=>	'Film',
				'broadcast'		=>	'Broadcast',
				'music_album'		=>	'Generic',
				'music_track'		=>	'Generic',
				'music_score'		=>	'Generic',
				'artwork'		=>	'Artwork',
				'software'		=>	'Computer Program',
				'audiovisual'		=>	'Audiovisual Material',
				'database'		=>	'Online Database',
				'government_report'	=>	'Government Report/Document',
				'report'		=>	'Report',
				'hearing'		=>	'Hearing',
				'statute'		=>	'Statute',
				'legal_ruling'		=>	'Legal Rule/Regulation',
				'case'			=>	'Case',
				'bill'			=>	'Bill',
				'patent'		=>	'Patent',
				'personal'		=>	'Personal Communication',
				'unpublished'		=>	'Unpublished Work',
				'classical'		=>	'Classical Work',
				'manuscript'		=>	'Manuscript',
				'map'			=>	'Map',
				'chart'			=>	'Chart or Table',
				'miscellaneous'		=>	'Generic',
			);
// Endnote versions earlier than 8.0 have a _very_ different XML format and type naming to version 8.0 *&^$*&^!
// XML import mappings for Endnote v8.0 and above?
		$this->importTypes8 = array_unique($this->types);
		unset($this->importTypes8[array_search('Generic', $this->importTypes8)]);
		$this->importTypes8['miscellaneous'] = 'Generic';
// Pre-Endnote v8 resource types
		$this->importTypesPre8 = array(
				'book'			=>	1,
				'book_article'		=>	7,
				'journal_article'	=>	0,
				'newspaper_article'	=>	5,
				'magazine_article'	=>	8,
				'proceedings_article'	=>	3,
				'thesis'		=>	2,
				'web_article'		=>	16,
				'film'			=>	21,
				'artwork'		=>	13,
				'software'		=>	6,
				'audiovisual'		=>	12,
				'report'		=>	10,
				'hearing'		=>	19,
				'case'			=>	18,
				'bill'			=>	17,
				'statute'		=>	22,
				'patent'		=>	15,
				'personal'		=>	4,
				'manuscript'		=>	20,
				'map'			=>	11,
				'chart'			=>	26,
				'miscellaneous'		=>	31,
			);
// Export uses the '*Generic' type so we must list all the possible Endnote generic fields that WIKINDX can use.
		$this->generic = array(
				'Author',
				'Year',
				'Title',
				'Secondary Author',
				'Secondary Title',
				'Short Title',
				'Place Published',
				'Publisher',
				'Volume',
				'Number of Volumes',
				'Number',
				'Pages',
				'Section',
				'Tertiary Author',
				'Tertiary Title',
				'Edition',
				'Date',
				'Type of Work',
				'Subsidiary Author',
				'ISBN/ISSN',
				'Original Publication',
				'Keywords',
				'Abstract',
				'Notes',
				'URL',
			);
// Map Endnote XML field names to Endnote tabbed file equivalents (why are they different? -- &@^#*&@!).  Keys 
// are XML fields, values are tabbed fields and should match values in the specific WIKINDX type arrays below.
// Some fields are dealt with on the fly in the code so are not listed here.
		$this->endnoteXmlFields8 = array(
				'secondary-title'	=>	'Secondary Title', // Collection or Series Title
				'tertiary-title'	=>	'Tertiary Title',
				'alt-title'		=>	'Short Title',
				'volume'		=>	'Volume',
				'number'		=>	'Number',
				'url'			=>	'URL',
				'num-vols'		=>	'Number of Volumes',
				'publisher'		=>	'Publisher',
				'pub-location'		=>	'Place Published',
				'isbn'			=>	'ISBN/ISSN',
				'orig-pub'		=>	'Original Publication',
				'edition'		=>	'Edition',
				'authors'		=>	'Author',
				'secondary-authors'	=>	'Secondary Author',
				'tertiary-authors'	=>	'Tertiary Author',
				'subsidiary-authors'	=>	'Subsidiary Author',
				'year'			=>	'Year',
				'abstract'		=>	'Abstract',
				'notes'			=>	'Notes',
				'section'		=>	'Section',
				'work-type'		=>	'Type of Work',
				'pages'			=>	'Pages',
			);
		$this->endnoteXmlFieldsPre8 = array(
				'SECONDARY_TITLE'	=>	'Secondary Title', // Collection or Series Title
				'TERTIARY_TITLE'	=>	'Tertiary Title',
				'SHORT_TITLE'		=>	'Short Title',
				'VOLUME'		=>	'Volume',
				'NUMBER'		=>	'Number',
				'URL'			=>	'URL',
				'NUMBER_OF_VOLUMES'	=>	'Number of Volumes',
				'PUBLISHER'		=>	'Publisher',
				'PLACE_PUBLISHED'	=>	'Place Published',
				'ISBN'			=>	'ISBN/ISSN',
				'ORIGINAL_PUB'		=>	'Original Publication',
				'EDITION'		=>	'Edition',
				'AUTHOR'		=>	'Author',
				'SECONDARY_AUTHOR'	=>	'Secondary Author',
				'TERTIARY_AUTHOR'	=>	'Tertiary Author',
				'SUBSIDIARY_AUTHOR'	=>	'Subsidiary Author',
				'YEAR'			=>	'Year',
				'ABSTRACT'		=>	'Abstract',
				'NOTES'			=>	'Notes',
				'SECTION'		=>	'Section',
				'TYPE_OF_WORK'		=>	'Type of Work',
				'PAGES'			=>	'Pages',
			);
// For exporting, these are fields that WIKINDX does not store but that the user may request to be exported.
// The mapping is Endnote XML export fields => Endnote tabbed file import fields.
// Commented out items are just not accepted by Endnote when importing a tabbed file but _should_ be the correct field names ... buggy commercial software! ;)
		$this->exportRawFields = array(
				"accession-num"		=>		"Accession Number",
				"custom1"			=>		"Custom 1",
				"custom2"			=>		"Custom 2",
				"custom3"			=>		"Custom 3",
				"custom4"			=>		"Custom 4",
				"custom5"			=>		"Custom 5",
				"custom6"			=>		"Custom 6",
				"custom7"			=>		"Custom 7",
				"auth-address"		=>		"Author Address",
				"reprint-edition"	=>		"Reprint Edition",
				"call-num"			=>		"Call Number",
				"label"				=>		"Label",
				"caption"			=>		"Caption",
				"reviewed-item"		=>		"Reviewed Item",
//				"electronic-resource-num"	=>	"Electronic Resource Number",
//				"remote-database-name"	=>	"Name of Database",
//				"remote-database-provider"	=>	"Database Provider",
//				"research-notes"	=>		"Research Notes",
//				"language"			=>		"Language",
//				"modified-date"		=>		"Last Modified Date",
				"alt-title"			=>		"Alternate Title",
//				"translated-title"	=>		"Translated Title"
			);
// Book resource type
		$this->book = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Title', // Series Title
					'field2'	=>	'Edition',
					'field4'	=>	'Volume',
				),
				'resource_misc'	=>	array(
					'miscField4'	=>	'Number of Volumes',
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
					'year2'		=>	'Original Publication',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Secondary Author', // Editor
					'creator3'	=>	'Subsidiary Author', // Translator
					'creator5'	=>	'Tertiary Author', // Series Editor
				),
			);
		$this->book['resource_publisher']['publisherName'] = 'Publisher';
		$this->book['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->book['resource_collection'] = array(); // need empty array
// book article/chapter resource type
		$this->book_article = $this->book;
		$this->book_article['resource']['field1'] = 'Tertiary Title'; // overwrite
		$this->book_article['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->book_article['resource_collection']['collectionTitleShort'] = 'Short Title';
// Journal article resource type
		$this->journal_article = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Volume',
					'field2'	=>	'Number',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->journal_article['resource_publisher'] = array(); // need empty array
		$this->journal_article['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->journal_article['resource_collection']['collectionTitleShort'] = 'Short Title';
// Newspaper article type
		$this->newspaper_article = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Section',
					'field2'	=>	'Place Published',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->newspaper_article['resource_publisher'] = array(); // need empty array
		$this->newspaper_article['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->newspaper_article['resource_collection']['collectionTitleShort'] = 'Short Title';
// proceedings article
		$this->proceedings_article = array(
				'resource'	=>	array(
					'field1'	=>	'Tertiary Title', // Series Title
					'field3'	=>	'Volume',
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->proceedings_article['resource_publisher']['publisherName'] = 'Publisher';
		$this->proceedings_article['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->proceedings_article['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->proceedings_article['resource_collection']['collectionTitleShort'] = 'Short Title';
// Web article resource type
		$this->web_article = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->web_article['resource_publisher'] = array(); // need empty array
		$this->web_article['resource_collection'] = array(); // need empty array
// Thesis resource type
		$this->thesis = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Volume', // thesis or dissertation?
					'field2'	=>	'Type of Work',
					'field5'	=>	'Secondary Title', // Department
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->thesis['resource_publisher']['publisherName'] = 'Publisher';
		$this->thesis['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->thesis['resource_collection'] = array(); // need empty array
// Film resource type
		$this->film = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Place Published',
					),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Subsidiary Author',
				),
			);
		$this->film['resource_publisher']['publisherName'] = 'Publisher';
		$this->film['resource_collection'] = array(); // need empty array
// Broadcast resource type
		$this->broadcast = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Subsidiary Author',
				),
			);
		$this->broadcast['resource_publisher']['publisherName'] = 'Publisher';
		$this->broadcast['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->broadcast['resource_collection'] = array(); // need empty array
// Software resource type
		$this->software = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
					'field4'	=>	'Edition', // Version
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Programmer
				),
			);
		$this->software['resource_publisher']['publisherName'] = 'Publisher';
		$this->software['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->software['resource_collection'] = array(); // need empty array
// Artwork resource type
		$this->artwork = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->artwork['resource_publisher']['publisherName'] = 'Publisher';
		$this->artwork['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->artwork['resource_collection'] = array(); // need empty array
// Audiovisual resource type
		$this->audiovisual = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Title',
					'field2'	=>	'Type of Work',
					'field3'	=>	'Edition',
					'field5'	=>	'Number',
				),
				'resource_misc'	=>	array(
					'miscField4'	=>	'Number of Volumes',
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Subsidiary Author', // Performers
					'creator5'	=>	'Secondary Author', // Series Editor
				),
			);
		$this->audiovisual['resource_publisher']['publisherName'] = 'Publisher';
		$this->audiovisual['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->audiovisual['resource_collection'] = array(); // need empty array
// case resource type
		$this->case = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Title', // Reporter
					'field4'	=>	'Volume', // Reporter Volume
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator3'	=>	'Subsidiary Author', // Counsel
				),
			);
		$this->case['resource_publisher']['publisherName'] = 'Publisher';
		$this->case['resource_collection'] = array(); // need empty array
// bill resource type
		$this->bill = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Section', // Code Section
					'field2'	=>	'Secondary Title', // Code
					'field3'	=>	'Volume', // Code Volume
					'field4'	=>	'Edition', // Session
					'field5'	=>	'Number', // Bill Number
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array( // we require an empty array to still be present
				),
			);
		$this->bill['resource_publisher'] = array(); // need empty array
		$this->bill['resource_collection'] = array(); // need empty array
// classical resource type
		$this->classical = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field4'	=>	'Volume',
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Attribution
				),
			);
		$this->classical['resource_publisher'] = array(); // need empty array
		$this->classical['resource_collection'] = array(); // need empty array
// Conference Paper resource type
		$this->conference_paper = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->conference_paper['resource_publisher']['publisherName'] = 'Publisher';
		$this->conference_paper['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->conference_paper['resource_collection'] = array(); // need empty array
// Legal Rule/Regulation resource type
		$this->legal_ruling = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Section', // Section Number
					'field2'	=>	'Type of Work',
					'field3'	=>	'Edition',
					'field4'	=>	'Volume', // Rule Number
					),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->legal_ruling['resource_publisher']['publisherName'] = 'Publisher';
		$this->legal_ruling['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->legal_ruling['resource_collection'] = array(); // need empty array
// Government Report/Documentation resource type
		$this->government_report = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Section',
					'field2'	=>	'Secondary Author', // Department
					'field3'	=>	'Edition',
					'field4'	=>	'Volume',
					'field5'	=>	'Number', // Issue number
					),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
					),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					),
			);
		$this->government_report['resource_publisher']['publisherName'] = 'Publisher';
		$this->government_report['resource_collection'] = array(); // need empty array
// Report/Documentation resource type
		$this->report = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'field1'	=>	'Secondary Title', // Series Title
					'field2'	=>	'Type of Work',
					'field5'	=>	'ISBN/ISSN', // Report number
					),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
					),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					),
			);
		$this->report['resource_publisher']['publisherName'] = 'Publisher';
		$this->report['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->report['resource_collection'] = array(); // need empty array
// Hearing resource type
		$this->hearing = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Title', // Committee
					'field2'	=>	'Tertiary Title', // Legislative Body
					'field3'	=>	'Edition', // Session
					'field4'	=>	'Number', // Document Number
					),
				'resource_misc'	=>	array(
					'miscField4'	=>	'Number of Volumes',
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
					),
				'resource_creator' =>	array(  // empty array required
					),
			);
		$this->hearing['resource_publisher']['publisherName'] = 'Publisher';
		$this->hearing['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->hearing['resource_collection'] = array(); // need empty array
// Online database resource type
		$this->database = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->database['resource_publisher']['publisherName'] = 'Publisher';
		$this->database['resource_collection'] = array(); // need empty array
// Magazine article type
		$this->magazine_article = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
					'field4'	=>	'Volume',
					'field3'	=>	'Number',
					'field1'	=>	'Edition',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->magazine_article['resource_publisher'] = array(); // need empty array
		$this->magazine_article['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->magazine_article['resource_collection']['collectionTitleShort'] = 'Short Title';
// Manuscript type
		$this->manuscript = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
					'field3'	=>	'Number',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->manuscript['resource_publisher'] = array(); // need empty array
		$this->manuscript['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->manuscript['resource_collection']['collectionTitleShort'] = 'Short Title';
// Map type
		$this->map = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Title', // Series Title
					'field2'	=>	'Type of Work',
					'field3'	=>	'Edition',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Cartographer
					'creator5'	=>	'Secondary Author', // Series Editor
				),
			);
		$this->map['resource_publisher']['publisherName'] = 'Publisher';
		$this->map['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->map['resource_collection'] = array(); // need empty array
// Chart type
		$this->chart = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Secondary Author', // Name of File
					'field2'	=>	'Secondary Title', // Image Source Program
					'field3'	=>	'Volume', // Image Size
					'field4'	=>	'Type of Work',
					'field5'	=>	'Edition', // Version
					'field6'	=>	'Number',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Created By
				),
			);
		$this->chart['resource_publisher']['publisherName'] = 'Publisher';
		$this->chart['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->chart['resource_collection'] = array(); // need empty array
// Statute type
		$this->statute = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field1'	=>	'Number', // Public Law Number
					'field2'	=>	'Secondary Title', // Code
					'field3'	=>	'Edition', // Session
					'field4'	=>	'Section',
					'field5'	=>	'Volume', // Code Number
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(  // empty array required
				),
			);
		$this->statute['resource_publisher'] = array(); // need empty array
		$this->statute['resource_collection'] = array(); // need empty array
// Patent type
		$this->patent = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'field1'	=>	'Secondary Title', // Published Source
					'field2'	=>	'Volume', // Patent Version Number
					'field3'	=>	'Number', // Application Number
					'field4'	=>	'Type of Work', // Patent Type
					'field5'	=>	'Section', // International Patent Number
					'field6'	=>	'Tertiary Author', // International Title (careful!)
					'field7'	=>	'Edition', // International Patent Classification
					'field8'	=>	'ISBN/ISSN', // Patent Number
					'field9'	=>	'Custom 6', // Legal Status
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Inventor
					'creator2'	=>	'Secondary Author', // Issuing Organisation
					'creator3'	=>	'Custom 4', // Attorney/Agent
					'creator4'	=>	'Tertiary Title', // International Author (careful!)
				),
			);
		$this->patent['resource_publisher']['publisherName'] = 'Publisher';
		$this->patent['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->patent['resource_collection'] = array(); // need empty array
// Personal Communication type
		$this->personal = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Secondary Author', // Recipient
				),
			);
		$this->personal['resource_publisher'] = array(); // need empty array
		$this->personal['resource_collection'] = array(); // need empty array
// Unpublished type
		$this->unpublished = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->unpublished['resource_publisher']['publisherName'] = 'Publisher'; // Institution
		$this->unpublished['resource_publisher']['publisherLocation'] = 'Place Published'; // Institution Location
		$this->unpublished['resource_collection'] = array(); // need empty array
// proceedings (complete set)
		$this->proceedings = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array( // we require an empty array to still be present
				),
			);
		$this->proceedings['resource_publisher']['publisherName'] = 'Publisher';
		$this->proceedings['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->proceedings['resource_collection'] = array(); // need empty array
// Music Album type
		$this->music_album = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Performer
					'creator2'	=>	'Subsidiary Author', // Composer
					'creator3'	=>	'Tertiary Author', // Conductor
				),
			);
		$this->music_album['resource_publisher']['publisherName'] = 'Publisher';
		$this->music_album['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->music_album['resource_collection'] = array(); // need empty array
// Music Track type
		$this->music_track = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
					'creator2'	=>	'Subsidiary Author',
					'creator3'	=>	'Tertiary Author',
				),
			);
		$this->music_track['resource_publisher']['publisherName'] = 'Publisher';
		$this->music_track['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->music_track['resource_collection']['collectionTitle'] = 'Secondary Title';
		$this->music_track['resource_collection']['collectionTitleShort'] = 'Short Title';
// Music Score type
		$this->music_score = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field3'	=>	'Edition',
				),
				'resource_misc'	=>	array(  // empty array required
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author', // Composer
					'creator2'	=>	'Secondary Author', // Editor
				),
			);
		$this->music_score['resource_publisher']['publisherName'] = 'Publisher';
		$this->music_score['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->music_score['resource_collection'] = array(); // need empty array
// Miscellaneous resource type
		$this->miscellaneous = array(
				'resource'	=>	array(
					'url'		=>	'URL',
					'isbn'		=>	'ISBN/ISSN',
					'field2'	=>	'Type of Work',
				),
				'resource_misc'	=>	array( // we require an empty array to still be present
					),
				'resource_year'	=>	array(
					'year1'		=>	'Year',
				),
				'resource_creator' =>	array(
					'creator1'	=>	'Author',
				),
			);
		$this->miscellaneous['resource_publisher']['publisherName'] = 'Publisher';
		$this->miscellaneous['resource_publisher']['publisherLocation'] = 'Place Published';
		$this->miscellaneous['resource_collection'] = array(); // need empty array
	}
}
?>
