#!/usr/bin/perl -w
###############################################################################
# Name: FetchYahoo
# Purpose: retrieves messages from Yahoo! Mail, saving them to a local spool
# Description:  FetchYahoo is a Perl script that downloads mail from a Yahoo!
#               webmail account to a local mail spool. It is meant to replace
#               fetchmail for people using Yahoo! mail since Yahoo!'s POP
#               service is no longer free. It downloads messages to a local
#               mail spool, including all parts and attachments . It then
#               deletes messages unless requested not to. It can also forward
#               messages to another e-mail address.
# Author:  Ravi Ramkissoon
# Author's E-mail: ravi_ramkissoon@yahoo.com
# License: Gnu Public License
# Homepage: http://fetchyahoo.twizzler.org
# Created: 04.12.02
# Modified: 04.02.03
my $version = "2.5" . "3";
#
# Installation instructions are in the INSTALL file.

# for the latest version and changes check, in order:
# http://fetchyahoo.twizzler.org
# http://freshmeat.net/projects/fetchyahoo
###############################################################################
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
###############################################################################

use strict;

use Getopt::Long ();
use HTTP::Request::Common qw(GET POST);
use HTTP::Cookies ();
use LWP::UserAgent ();
use MIME::Head ();
sub GetRedirectUrl($);
sub GetEmailAddress($);
sub ParseConfigFile();
sub Localize($);
sub EmptyTrash();
sub EmptyBulk();
sub Logout();
sub Delete(@);
sub MarkUnread(@);
sub MyDie($);
sub MyGet;
sub checkExternal();

# Configure these or use the interactive input
my $username = 'yahoo-user-name';
my $password = 'yahoo-password';

# mail spool, mbox file and procmail configs
my $useSpool  = 1;              # set this to 0 to disable outputting to a file
my $spoolName = '/var/spool/mail/local-user-name';
my $spoolMode = 'append';       # either 'append', 'pipe' or 'overwrite'
                                # use 'pipe' for procmail or other filter
# proxy configs
my $useProxy = 0;               # set this to 1 to enable use of a web proxy
my $proxyHost = 'proxy.hostname.com';
my $proxyPort = 80;
my $proxyUser = 'proxyAuthenicationUserName';
my $proxyPass = 'proxyAuthenicationPassword';

my $useHTTPS = 1;               # set this to 0 to turn off HTTPS and transfer
                                # all information in plaintext (INSECURE)
                                # using HTTPS requires Crypt::SSLeay or
                                # IO::Socket::SSL

# mail forwarding configs
my $useForward = 0;             # set this to 1 to enable mail forwarding
my $mailHost = 'outgoing.mail.com'; # set this to yr smtp outgoing mail server
my $sendToAddress = 'me@myhost.com';  # the e-mail address to be forwarded to
my $sendFromAddress = 'me@myhost.com'; 
# the e-mail address to use as the from address. This is used only if the
# message being forwarded has no From header

# daemon mode config. If this is 0, the program runs only once and terminates.
# Otherwise this is the number of minutes between successive mail checks.
my $repeatInterval = 0;

# the below defaults can be overridden from the commandline
my $newOnly = 0;        # download all (0) or just new (1) messages
my $noDelete = 0;       # to not delete messages set this to 1
my $quiet = 0;          # to suppress regular (non-error) output set this to 1
my $noerrors = 0;  # to suppress error output, set this to 1
my $noDownload = 0;     # to delete msgs and not download them, set this to 1
my $emptyBulk = 0;      # to empty bulk folder (always happens before fetch)
my $emptyTrashAfter = 0;  # to empty trash after downloading msgs set this to 1
my $emptyTrashBefore = 0; # to empty trash b4 downloading msgs, set this to 1 
my $logout = 0;          # to have fetchyahoo logout at the end, set this to 1
my $leaveUnread=0;    # to leave messages as unread on the server,set this to 1
	              # only makes sense to use this with noDelete
my $noFromLine=0;     # if you use a program/filter which doesn't expect a 
	                    # From_ line appended to the message, set this to 1
my $statusOnly=0;     # if you want only the number of messages, set this to 1
my $box = 'Inbox';    # to download from a different folder, set this
                      # eg 'Bulk' will get messages from the Bulk folder
my $getExternal=0;  # if set to 1, messages from external mailboxes
                           # configured on Yahoo will also be retrieved

# use LWP::Debug qw(+);        # uncomment this for lots of debugging messages

#### Nothing below here is intended for user configuration

# I may need to edit these in future
# my $userAgent= 'FetchYahoo/' . $version;    switched to Galeon user string
my $userAgent = 
     "Mozilla/5.0 Galeon/1.2.5 (X11; Linux i686; U;) Gecko/20020911";

my $loginURL = 'http://login.yahoo.com/config/login';
my $HTTPSloginURL = 'https://login.yahoo.com/config/login';
my $mailURL = "http://mail.yahoo.com";
my $versionString = "FetchYahoo Version " . $version . "\n" .
  "Homepage: http://fetchyahoo.twizzler.org\n";
my $maxMidsPerURL = 75;

# other variables used
my $overwriteFlag = 0;
my $spool;
my $proxyURL;
my $smtp;
my $altConfigFile;

# flag for help and version
my $helpFlag = 0;
my $versionFlag = 0;

my %map  = ();     # hash for extension->MIMEtype mappings

my $help = <<EOF
$0 [options] 

Retrieves messages from the inbox of a Yahoo user using the web interface
and stores them in the specified local spool/mbox file. Deletes the messages
downloaded unless requested not to.

Options specified on the commandline take precedence over options specified
in the configuration file, which in turn take precedence over options hardcoded
within the fetchyahoo program file.

--version                      print the version and exit
--help                         display help showing the program options
--quiet                        suppress regular (non-error) messages
--noerrors                     suppress error messages

--configfile=config-file       use config-file as the configuration file
--username=yahoousername       use yahoousername as the login
--password=pass                use pass as the password
--spoolfile=spool-file         use spool-file as the file to spool messages to
--folder=folder-name           download messages from folder-name instead

--nohttps                      use an insecure plaintext login instead of HTTPS
--newonly                      only download new (i.e. unread) messages
--nodelete                     do not delete messages after downloading them
--nodownload                   do not download messages 
                               (but still delete them/empty trash if requested)

--append                       append messages to spool-file (default)
--overwrite                    overwrite spool-file (instead of appending)
--pipe                         pipe messages to a program instead of
                               a spool-file

--emptytrashbefore             empty trash before downloading messages
--emptytrashafter              empty trash after downloading messages
--emptybulk                    empty bulk mail folder (always before fetch)
--leaveunread                  leave messages unread on the server
--repeatinterval=N             check for mail every N minutes
--statusonly                   only get number of messages
--getexternal                  retrieve messages from external mailboxes also

--nofromline                   leave out the leading From_ line
--logout                       log out of Yahoo! when done

--proxyhost=proxy.host.org     hostname for proxy [ DEFAULT off ]
--proxyport=N                  port for proxy [DEFAULT 80 ]
--proxyuser=proxy-user         username for proxy authentication[ DEFAULT none]
--proxypass=proxy-pass         password for proxy authentication[ DEFAULT none]

--noquiet                      opposite of --quiet [DEFAULT]
--errors                       opposite of --noerrors [DEFAULT]
--delete                       opposite of --nodelete [DEFAULT]
--nologout                     opposite of --logout [DEFAULT]
--noempty                      do not empty trash or bulk [DEFAULT]
--allmsgs                      get all msgs (not only new ones) [DEFAULT]
--download                     opposite of --nodownload [DEFAULT]
--includefromline              opposite of --nofromline [ DEFAULT ]
--markread                     opposite of --leaveunread [ DEFAULT ]
--nostatusonly                 opposite of --statusonly [ DEFAULT ]
--nogetexternal                opposite of --getexternal [ DEFAULT ]
--https                        use a secure login via HTTPS [ DEFAULT ]

At least username and password must be specified somewhere (commandline, 
config-file or in fetchyahoo.pl)

EOF
;

# S T A R T   M A I N   P R O G R A M

# parse input options for an alternate config file
Getopt::Long::Configure('pass_through');
Getopt::Long::GetOptions ('configfile=s'  => \$altConfigFile);

# config file options take precedence over hardcoded (within-file)  options
ParseConfigFile();

# get other command-line input options. These take precedence over all others
Getopt::Long::Configure('no_pass_through');
Getopt::Long::GetOptions
(
 'newonly'           => \$newOnly,
 'help'              => \$helpFlag,
 'version'           => \$versionFlag,
 'noDelete'          => \$noDelete,
 'username=s'        => \$username,
 'password=s'        => \$password,
 'spoolfile=s'       => \$spoolName,
 'quiet!'            => \$quiet,
 'noerrors'          => \$noerrors,
 'nodownload'        => \$noDownload,
 'emptybulk'         => \$emptyBulk,
 'emptytrashafter'   => \$emptyTrashAfter,
 'emptytrashbefore'  => \$emptyTrashBefore,
 'logout!'           => \$logout,
 'statusonly!'       => \$statusOnly,
 'repeatinterval=i'  => \$repeatInterval,
 'noempty'           => sub {$emptyTrashAfter=0;$emptyTrashBefore=0
                               ;$emptyBulk=0},
 'download'          => sub { $noDownload=0; },
 'allmsgs'           => sub { $newOnly=0; }, 
 'delete'            => sub { $noDelete=0; },
 'leaveunread'       => \$leaveUnread,
 'nofromline'        => \$noFromLine,
 'markread'          => sub { $leaveUnread=0; },
 'includefromline'   => sub { $noFromLine=0; },
 'errors'            => sub { $noerrors=0; },
 'pipe'              => sub { $spoolMode='pipe'; },
 'append'            => sub { $spoolMode='append'; },
 'overwrite'         => sub { $spoolMode='overwrite'; },
 'folder=s'          => \$box ,
 'getexternal!'      => \$getExternal,
 'proxyhost=s'       => sub { $proxyHost= $_[1] ; $useProxy=1;},
 'proxyport=s'       => \$proxyPort,
 'proxyuser=s'       => \$proxyUser,
 'proxypass=s'       => \$proxyPass,
 'https!'            => \$useHTTPS );

# set some required variables

if ($box eq 'bulk' || $box eq 'Bulk') { $box = "%40B%40Bulk"; }
if ($box eq 'sent') { $box = "Sent"; }
if ($box eq 'draft') { $box = "Draft"; }
if ($box eq 'trash') { $box = "Trash"; }
if ($box eq 'inbox') { $box = "Inbox"; }

my $homesuff = '/ym/ShowFolder?box='.$box;
my $msgsuff = $homesuff.'&PRINT=1&Nhead=f&toc=1&MsgId=';
$box =~ s/%/%%/g ;                  # fix  for template which is used in printf
my $bodyPartUrlTemplate = "/ym/ShowLetter?box=".$box."&MsgId=%s&bodyPart=%s";

# For a proxy with authentication, create a URL like
# http://user:pass@host:port/
unless ($proxyPass eq 'proxyAuthenicationPassword') {
    $proxyHost = $proxyPass . '@' . $proxyHost; }
unless ($proxyUser eq 'proxyAuthenicationUserName') {
    $proxyHost = $proxyUser . ':' . $proxyHost; }

$proxyURL = 'http://' . $proxyHost . ':' . $proxyPort;
$proxyURL = $proxyHost . ':' . $proxyPort if ($useHTTPS) ;

if ( $useProxy && $proxyHost eq "proxy.hostname.com" &&
     exists ($ENV{'HTTP_PROXY'}) ) {
    $proxyURL = $ENV{'HTTP_PROXY'}; 
}

if ( $useProxy && $proxyHost eq "proxy.hostname.com" &&
     exists ($ENV{'http_proxy'}) ) {
    $proxyURL = $ENV{'http_proxy'}; 
}

$loginURL = $HTTPSloginURL if ($useHTTPS) ;

# unbuffer STDOUT
select((select(STDOUT), $| = 1)[0]);

# check if help or version was requested
if ($helpFlag) { print $versionString . "\n" . $help; exit; }
if ($versionFlag) { print $versionString; exit; }

# check for common errors (forgot to edit variables)
if ($username eq 'yahoo-user-name')  {
    print "No username specified.\nPlease enter your Yahoo! username: ";
    $username = <STDIN> ;
    chomp($username);
    $password = 'yahoo-password';
}

if ($password eq 'yahoo-password')  {

  my $useReadKey=0;

  # check for Term::ReadKey
  eval ("use Term::ReadKey");
  if ($@) {
	print "* WARNING * Term::ReadKey is not installed. ".
    "Your password will be displayed on the screen.\n\n".
    "Either Ctrl-C and install Term::ReadKey or ".
    "make sure noone is looking at your screen.\n\n";
  }
  else { $useReadKey = 1; }

  print "Please enter your Yahoo! password: ";
  if ($useReadKey) {
    ReadMode('noecho');          #hide output
    $password = ReadLine(0);     #get input
    ReadMode('normal');          #back to normal mode
  }
  else { $password = <STDIN>; }
  chomp($password);
}

if ( $useSpool && $spoolName eq "/var/spool/mail/local-user-name") {
    print "No mailbox or mailspool specified.\n";
    print "Please enter the path to and name of your mail spool or mailbox ".
      "(eg /var/spool/mail/username): ";
    $spoolName = <STDIN>;
    chomp($spoolName);
}

if ($spoolMode eq 'append') {
    $spool = '>>' . $spoolName ; }
elsif ($spoolMode eq 'pipe') {
    $spool = '|' . $spoolName ; }
elsif ($spoolMode eq 'overwrite') {
    $spool = '>' . $spoolName ; }
else { $spool = '>>' . $spoolName ; }     # the default is to append

if ( $useProxy && $proxyHost eq "proxy.hostname.com") {
    print "If you are using a web proxy (use-proxy=1), you must " .
	"specify the proxy hostname.\n\n";
    print $versionString . "\n" . $help; exit; 
}

if (!$quiet) {
    if ($useHTTPS) { print "Logging in securely via SSL as $username " }
    else { print "Logging in insecurely via plaintext as $username " }
    print "on " . (scalar localtime) . "\n";

    if ($useProxy) { print "Using $proxyURL as a webproxy.\n" }
    if ($repeatInterval>0) { 
      print "Running in daemon mode. Will check every $repeatInterval" . 
	    " minutes.\n"; }
}

# if daemon mode is chosen, fork into the background
if ($repeatInterval>0) {
    print "Forking into the background.\n" unless $quiet ;
    $SIG{CHLD} = 'IGNORE';
    my $pid = fork;
    exit if $pid;
    die "Couldn't fork into background: $!" unless defined ($pid) ;
}

startfetch:

if ($useForward) {  # check that everything is setup for mailforwarding

    if ( $sendToAddress eq 'me@myhost.com') {
	print "If you are forwarding the messages (use-forward=1), you must " .
	    "specify the e-mail address to forward to.\n\n";
	print $versionString . "\n" . $help; exit; 
    }

    if ( $mailHost eq 'outgoing.mail.com') {
	print "If you are forwarding the messages (use-forward=1), you must " .
	    "specify an smtp server" . 
	    " (localhost if you have one installed locally).\n\n";
	print $versionString . "\n" . $help; exit; 
    }

    eval ("use Net::SMTP");     # make sure Net::SMTP is installed for mail-forwarding
    if ($@) {
	die "Net::SMTP is not installed. It must be installed to use ". 
	    "mail-forwarding\n";
    }

    # we only setup the smpt connection if we find messages
}

# grab login cookies
my $ua = LWP::UserAgent->new;
my $cookie_jar = HTTP::Cookies->new();
my $url = "";
my $request;

$ua->cookie_jar($cookie_jar);
$ua->agent($userAgent);
if ($useProxy) {
    if ($useHTTPS) {
	$ENV{HTTPS_PROXY} = $proxyURL;
    } else {
    	$ua->proxy('http', $proxyURL);
    }
}

$request = POST $loginURL,
    [
     '.tries' => '1',
     #'.done'  => 'URL to go to later',
     '.src'   => 'ym',
     '.intl'  => 'us',
     'login'  => $username,
     'passwd' => $password,
     ];

$request->content_type('application/x-www-form-urlencoded');
$request->header('Accept' => '*/*');
$request->header('Allowed' => 'GET HEAD PUT');

my $content = MyGet($request, 'log in', 1);

if ( $content =~ /Invalid Password/ ) {
    MyDie("Failed: Wrong password entered for $username"); 
}

if ( $content =~ /ID does not exist/ ) {
    MyDie("Failed: Yahoo user $username does not exist\n");
}

# print "url is $url\n\n";

$url=~/https?:\/\/(.*?)\./;

# set localization strings
my %strings;
my $language;

if (defined $1) {
  $language = $1;
  %strings = Localize($1);
}
else {
  $language = 'Undetected';
  %strings = Localize('us');
}

if ($box eq '%%40B%%40Bulk') { $box = 'Bulk'; }
if (!$quiet) {
    print "Successfully logged in as $username.\n"; 
    print "Country code : $language\tFolder: $box\n";
}

# setup URLs
$url =~ /(http:\/\/.*?)\// ;
my $baseurl = $1;
my $homeurl = $baseurl . $homesuff ;
my $msgurl = $baseurl . $msgsuff ; 
my $logouturl = $baseurl . "/ym/Logout";
my $numurls = 0;
my @delurls;
my @unreadurls;
my $emptyurl;

$delurls[0] = $homeurl . "\&DEL=Delete";
$unreadurls[0] = $homeurl . "\&UNR=1";

# get all message IDs
my $msgcount = 0;
my $pagecount = 0;
my $numMsgs ;
my $startMsg ;
my $endMsg ;
my @msgids ;
my $crumb;

# if requested, print summaries for external mailboxes
if ($getExternal) { checkExternal(); }

if ( $newOnly) {
    print "Only retrieving new messages\n" unless $quiet;
    $homeurl = $homeurl . "\&Nview=u";
}

if ( $leaveUnread) {
    print "Leaving messages unread on the server\n" unless $quiet;
}

# empty trash before downloading messages, if requested
# parsing the empty trash URL from a parsed inbox summary page does NOT work
# because it changes message IDs so deleting messages would fail
if ($emptyTrashBefore) { 
    my $tmpurl = $baseurl . "/ym/Folders" ;
    $request = GET  $tmpurl ;
    $content = MyGet($request, 'get Folders listing for Empty', 0);
    if ($content eq "FAILED") {
      goto startDownload ;
    }
    $content =~ /href=\"((.*?)\?ET=1(.*?))\"/;
    if (defined $1) { $emptyurl = $baseurl . $1; }
    else { 
      print "Warning: Couldn't get empty trash URL\n" unless $noerrors; 
      goto startDownload ;
    }
    EmptyTrash();
}

if ($emptyBulk) { 
    my $tmpurl = $baseurl . "/ym/Folders" ;
    $request = GET  $tmpurl ;
    $content = MyGet($request, 'get Folders listing for Empty', 0);
    if ($content eq "FAILED") {
      goto startDownload ;
    }
    $content =~ /href=\"((.*?)\?EB=1(.*?))\"/;
    if (defined $1) { $emptyurl = $baseurl . $1; }
    else { 
      print "Warning: Couldn't get empty bulk URL\n" unless $noerrors; 
      goto startDownload ;
    }
    EmptyBulk();
}

# loop over all inbox summary pages
startDownload:
my $mainPage;
do {

  # get summary page
  my $tmpurl = $homeurl . "\&Npos=$pagecount&order=down&sort=date" ; 
  $request = GET  $tmpurl ;
  $mainPage = MyGet($request, 'get Folder $box listing', 1);

  #parse for number of messages
  if ($mainPage =~ /$strings{'msg_range'}/) { 
    $startMsg = $1 ;
    $endMsg = $2 ;
    $numMsgs = $3;  }
  elsif ($mainPage =~ /$strings{'new_msg_range'}/) { 
    $startMsg = $1 ;
    $endMsg = $2 ;
    $numMsgs = $3;  }
  elsif ($mainPage =~ /$strings{'no_msgs'}/) {
    if (!$quiet) { print "There are no messages in the $box folder.\n"; }

    if ($emptyTrashAfter) {     # if requested, empty trash
	    $mainPage =~ /href=\"((.*?)\?ET=1(.*?))\"/;
	    if (defined $1) { $emptyurl = $baseurl . $1; }
	    else { 
        print "Warning: Couldn't get empty trash URL\n" unless $noerrors; 
	    }
	    EmptyTrash();
    }

    if ($logout) { Logout; }                    # if requested, logout

    # if repeat interval is non-zero, loop after repeatInterval minutes
    if ($repeatInterval > 0) {
	    sleep (60*$repeatInterval);
	    goto startfetch ;
    }
    exit;
  }
  elsif ($mainPage =~ /$strings{'new_no_msgs'}/) {
    print "There are no messages in the $box folder.\n" unless $quiet;

    if ($emptyTrashAfter) {     # if requested, empty trash
	    $mainPage =~ /href=\"((.*?)\?ET=1(.*?))\"/;
	    if (defined $1) { $emptyurl = $baseurl . $1; }
	    else {
        print "Warning: Couldn't get empty trash URL\n" unless $noerrors; 
	    }
	    EmptyTrash();
    }

    if ($logout) { Logout; }                    # if requested, logout	
    # if repeat interval is non-zero, loop after repeatInterval minutes
    if ($repeatInterval > 0) {
	    sleep (60*$repeatInterval);
	    goto startfetch ;
    }
    exit;
  }
  elsif ($mainPage =~ /There was a problem accessing your mailbox/ &&
         $mainPage =~ /This is most likely a temporary problem/) {
    MyDie "Could not find folder $box. Remember that folder names are " .
	    "case-sensitive.\n" unless $noerrors;
  }
  else {
    print $mainPage . "\n\n";;
    print '<<< Please e-mail this output to ravi_ramkissoon@yahoo.com >>>' ."\n";
    MyDie("\n\nFailed: Can't parse summary page.\n"); }

  if ($statusOnly) {
    print "$numMsgs message".(($numMsgs>1)?"s":"") . " found in the $box folder.\n";

    if ($emptyTrashAfter) {     # if requested, empty trash
	    $mainPage =~ /href=\"((.*?)\?ET=1(.*?))\"/;
	    if (defined $1) { $emptyurl = $baseurl . $1; }
	    else { 
        print "Warning: Couldn't get empty trash URL\n" unless $noerrors; 
	    }
	    EmptyTrash();
    }
	
    if ($logout) { Logout; }                    # if requested, logout

    # if repeat interval is non-zero, loop after repeatInterval minutes
    if ($repeatInterval > 0) {
	    sleep (60*$repeatInterval);
	    goto startfetch ;
    }
    exit;
  }

  print "Getting Message ID(s) for message(s) $startMsg - $endMsg.\n" unless $quiet;

  # parse summary page for message IDs
  foreach my $word (split ' ', $mainPage) {
    if ($word =~ /ShowLetter\?MsgId=([0-9_-]+)/ ) {
	    unless (grep ( /$1/ , @msgids )) {
        $msgcount = $msgcount + 1;
        $msgids[$msgcount-1] = $1 ;
	    }
    }
    elsif ($word =~ /DMid=([0-9_]+)/) {
	    unless (grep ( /$1/ , @msgids )) {
        $msgcount = $msgcount + 1;
        $msgids[$msgcount-1] = $1 ;
	    }
    }
  }
  $pagecount = $pagecount+1 ;            # next summary page
} until $numMsgs == $endMsg ;

$mainPage =~ /name=\".crumb\" value=\"(.*?)\"/ ;
if (defined($1)) { $crumb = $1; } 
elsif (!$noDelete)  { 
  print "Warning: Can't get crumb.\n" unless $noerrors; }

if ($emptyTrashAfter) {
  $mainPage =~ /\"((.*?)\?ET=1(.*?))\"/;
  if (defined $1) { $emptyurl = $baseurl . $1; }
  else {
    print "Warning: Couldn't get empty trash URL.\n" unless $noerrors; }
}

if (!$quiet) { print "Got $msgcount Message IDs\n"; }
my $delCount = 0;
my $downloadCount = 0;
my $unreadCount = 0;

if ($noDownload) {
  if (!$quiet) { print "Not downloading messages\n"; }
  foreach my $msgid (@msgids) { 
    # add message to deletion list
    $delurls[$numurls] = $delurls[$numurls] . "\&Mid=$msgid";
    $delCount = $delCount+1;
    if ($delCount%$maxMidsPerURL == 0) {
      $numurls=$numurls+1;
      $delurls[$numurls] = $homeurl . "\&DEL=Delete";
      $delurls[$numurls-1] = $delurls[$numurls-1] . "\&.crumb=$crumb";
    }
  }
  goto startDelete;
}

# initiate smtp connection if we are forwarding mail
if ($useForward) {
  $smtp = Net::SMTP->new($mailHost);
  if (!$smtp) {
    MyDie("Failed: Unable to connect to server $mailHost to forward ".
          "mail. \n");
  }
}

@msgids = reverse(@msgids);            # download msg IDs in correct order

# loop over all Message IDs
foreach my $msgid (@msgids) { 

  # add message to unread list
  $unreadurls[$numurls] = $unreadurls[$numurls] . "\&Mid=$msgid";
  $unreadCount = $unreadCount+1;

  my $tmpurl = $baseurl . sprintf($bodyPartUrlTemplate, $msgid, "HEADER") ; 
  my $header;
  $request = GET $tmpurl;
  $header = MyGet($request, "get header of message $msgid. It will be "
                  . "skipped and not deleted", 0);

  if ($header eq "FAILED") { next; }

  my @foo = split /\n/, $header;
  my $fromLine = shift @foo;     # save From_ line for later
  my $fromName; my $fromRest;

  # Yahoo!'s From_ line is broken, fix it
  if (!defined ($fromLine)) {
    # can't parse From_ line, make a new one
    $fromName = '-';
    $fromRest = scalar localtime ;
  }
  else {
    $fromLine =~ /From (.*?)\s*((Mon|Tue|Wed|Thu|Fri|Sat|Sun).*?)$/ ;

    if ( !defined ($1) || !defined($2) ) {  # can't parse From_ line, make a new one
	    $fromName = '-';
	    $fromRest = scalar localtime ;
    }
    else {
	    $fromName = $1 ;
	    $fromRest = $2 ;
	    $fromName =~ s/ /_/g;
    }
  }

  $fromLine = "From " . $fromName . " " . $fromRest . "\n" ;

  my $mimeHead = new MIME::Head(\@foo);

  # if we can't parse at least To or From or Date assume this has failed
  unless ($mimeHead->get('From') || $mimeHead->get('To')
          || $mimeHead->get('Date') || $mimeHead->get('Return-Path')
          || $mimeHead->get('X-Apparently-To'))  {
    print $mimeHead->as_string;
    print "\n\nCan't find message $msgid. It will be skipped and not deleted.\n\n" .
	    '<<< Please e-mail this output to ravi_ramkissoon@yahoo.com >>>' . "\n"; 
    next;
  }

  $mimeHead->delete("Content-Length");
  # Remove Yahoo! Mail's internal headers
  $mimeHead->delete("X-RocketMail");
  $mimeHead->delete("X-RocketUID");
  $mimeHead->delete("X-RocketMIF");
  $mimeHead->delete("X-RocketRCL");
  $mimeHead->delete("X-Track");
  $mimeHead->delete("X-Rocket-Server");
  $mimeHead->delete("X-Rocket-Track");

  # Add our own header
  $mimeHead->add('X-FetchYahoo',"version ".$version." MsgId ".$msgid);

  # This fixes a bug affecting non-English latin characters
  # is it ok if we always change CTE from quoted-printable->8bit ?
  if ($mimeHead->get('Content-Transfer-Encoding') &&
      $mimeHead->get('Content-Transfer-Encoding') =~
      /^(quoted-printable|base64)$/i) {
    $mimeHead->add("X-FetchYahoo-Content-Transfer-Encoding-Autoconverted",
                   "from ".$mimeHead->get('Content-Transfer-Encoding').
                   " to 8bit");
    $mimeHead->replace("Content-Transfer-Encoding", "8bit");
  }

  my $message = $mimeHead->as_string."\n"; # message we are constructing

  $request = GET $baseurl . sprintf($bodyPartUrlTemplate, $msgid, "TEXT");

  my $rawPart = MyGet($request, "get body of message $msgid.\n"
                      . "Message will be skipped and not deleted.\n", 0);
	
  if ($rawPart eq "FAILED" ) { next; }

  $message .= $rawPart . "\n\n";
  $message =~ s/\n>From /\n>>From /g ; # slightly extended RFC 822
  $message =~ s/\nFrom /\n>From /g ; 

  # add message to deletion list
  $delurls[$numurls] = $delurls[$numurls] . "\&Mid=$msgid";
  $delCount = $delCount+1;
  if ($delCount%$maxMidsPerURL == 0) {
    $numurls=$numurls+1;
    $delurls[$numurls] = $homeurl . "\&DEL=Delete";
    $delurls[$numurls-1] = $delurls[$numurls-1] . "\&.crumb=$crumb";
    $unreadurls[$numurls] = $homeurl . "\&UNR=1";
    $unreadurls[$numurls-1] = $unreadurls[$numurls-1] . "\&.crumb=$crumb";
  }


  if ($useSpool) {
    # send From_line and message to the specified spool/file
    open SPOOL, "$spool" or
	    MyDie("Failed: Couldn't open output: $spool");
	
    flock SPOOL, 2 unless ($spoolMode eq 'pipe');  # lock the mailbox file

    if (!$noFromLine) {
	    print SPOOL $fromLine or 
        MyDie("Failed: Couldn't write to output: $spool");
    }
    print SPOOL "$message" or 
	    MyDie("Failed: Couldn't write to output: $spool");
	
    flock SPOOL, 8 unless ($spoolMode eq 'pipe'); # unlock the mailbox file
    close SPOOL;
  }

  # if overwrite mode is chosen, we only need to overwrite the first time
  if (!$overwriteFlag and $spoolMode eq "overwrite") {
    $overwriteFlag = 1;
    $spool = '>>' . $spoolName
  }

  # mail fowarding stuff goes here
  if ($useForward) {
    $smtp->mail(GetEmailAddress($mimeHead->get('From')) || $sendFromAddress);
    $smtp->to($sendToAddress);	
    $smtp->data();
    $smtp->datasend($message);
    $smtp->dataend();	
  }

  $downloadCount = $downloadCount +1 ;
  # Progress indicator
  if (!$quiet) {
    if ($downloadCount%5) {
	    print ".";
    } elsif ($downloadCount%10) {
	    print "5";
    } else {
	    printf("[%d]\n", $downloadCount);
    }
  }

}

if ($useForward) { $smtp->quit; }

if (!$quiet) { print "\nFinished downloading $downloadCount messages.\n"; }

if ($leaveUnread) {
  $unreadurls[$numurls] = $unreadurls[$numurls] . "\&.crumb=$crumb";
  MarkUnread(@unreadurls);
}

startDelete:
if ( ! $noDelete) {
  $delurls[$numurls] = $delurls[$numurls] . "\&.crumb=$crumb";
  Delete(@delurls);
}
else {
  print "Messages have not been deleted.\n" unless $quiet;
}

if ($emptyTrashAfter) { EmptyTrash(); }
if ($logout) { Logout(); }

# if repeat interval is non-zero, loop after repeatInterval minutes
if ($repeatInterval > 0) {
  sleep (60*$repeatInterval);
  goto startfetch ;
}

###############################################################################
# Subroutines
###############################################################################

# return the URL we're redirected to
sub GetRedirectUrl($) {
  my $response = $_[0];
  my $url = $response->header('Location');

  if ($url =  $response->header('Location')) {
    # the Location URL is sometimes non-absolute which is not allowed, fix it
    local $URI::ABS_ALLOW_RELATIVE_SCHEME = 1;
    my $base = $response->base;
    $url = $HTTP::URI_CLASS->new($url, $base)->abs($base);
  }
  elsif ($response->content =~
         /^<html>\s*<head>\s*<script language=\"/ &&
         $response->content =~
         /<meta http-equiv="Refresh" content="0; url=(http:\/\/.*?)">/) {
    $url = $1;
  }
  else {
    return undef;
  }

  return $url;
}

sub GetEmailAddress($) {
  my $addr = $_[0];

  # match A @ B
   $addr =~ /([.0-9a-zA-Z-_]+)@([.0-9a-zA-Z-_]+)/;
   if (defined $1 && defined $2) {
     return $1 . "@" . $2;
   }
   else {
     return undef;
   }
}

# parameters (request, action_being_attempted, die_if_failed)
# if die_if_failed is 0 failure will be denoted by returning "FAILED"
sub MyGet {
  my $request = $_[0];
  my $response;
  my $tries;

  for($tries=0;$tries<3;$tries++) {
    if ( $useProxy
         && !($proxyUser eq "proxyAuthenicationUserName")
         && !($proxyPass eq "proxyAuthenicationPassword") ) {
        $request->proxy_authorization_basic($proxyUser, $proxyPass);
    }
    $response = $ua->simple_request($request);

    while ( $response->is_redirect ||
            ( $response->content =~
              /^<html>\s*<head>\s*<script language=\"/ &&
              $response->content =~
              /<meta http-equiv="Refresh" content="0; url=(http:\/\/.*)">/)) {
      $cookie_jar->extract_cookies($response);
      $url = GetRedirectUrl($response);
      $request = GET $url;
      if ( $useProxy
           && !($proxyUser eq "proxyAuthenicationUserName")
           && !($proxyPass eq "proxyAuthenicationPassword") ) {
        $request->proxy_authorization_basic($proxyUser, $proxyPass);
      }
      $response = $ua->simple_request($request);
    }
    my $content =  $response->content;

    # check for broken pages
    if ( $content =~ /^<html><head><title>Yahoo! -\n404/ ||
         ($content=~/^<!--web/ && 
          $content=~/There was a problem accessing your account/) ||
         $content =~ /^<html><head><title>Document Error: Data follows/ ||
         ($content=~/^<!--web/ && 
          $content=~
          /he request your browser sent was missing some needed information/)||
         ($content=~/^<html>/ && 
          $content=~
          /<title>Sorry, this page is not currently available<\/title>/ ) ||
         ($content=~/^<!--web/ && 
          $content=~
          /Your login session has expired/ )  ||
         ($content=~/^<!--web/ && 
          $content=~
          /Invalid Mailbox State/ ) ||
         ($content=~/^<html>/ &&
          $content=~ /A JavaScript implementation of the RSA Data Security/ &&
          $content=~ /These functions implement the basic operation for each round of the/) )
      { print "Failed $tries\n"; next; }

    if ( $response->is_success ) { return $content; }
  }

  if ( $response->is_error
       && $request->url->scheme eq "https"
       && $response->message =~ /LWP::Protocol::https/ )
    {
      die "HTTPS secure login is now turned on by default.\n\n".
        "Unable to login securely with HTTPS. You may need to install ".
        "the Crypt::SSLeay or IO::Socket::SSL perl module. Please ".
        "check the webpage or INSTALL file for information on how to ".
        "install perl modules.\n\n".
        "Alternatively you can turn off HTTPS secure login to use an".
        "insecure plaintext login (-nohttps or edit the config file).\n";
    }

  if ($_[2]) {
    print $request->uri()."\n" . $response->status_line . "\n"
      unless $noerrors;
    MyDie("Failed: Couldn't " . $_[1] . ".\n");
  }
  else {
    print $request->uri()."\n" . $response->status_line . "\n"
      unless $noerrors;
    print"Warning: Couldn't " . $_[1] . ".\n" unless $noerrors;
    return "FAILED";
  }

}


sub MyDie($) {
  if ($repeatInterval > 0) {
    sleep (60*$repeatInterval);
    goto startfetch ;
  }

  die shift unless $noerrors ;
  die ;
}

sub Delete(@) {
  my $cnt = $numurls;
  my @lst = @_;
  while ($cnt>=0) {
    # print "url - $cnt " . $lst[$cnt] . "\n";
    $request = GET $lst[$cnt] ;
    $content = MyGet($request, 'delete messages', 1);
    if ( $content =~ /<head><title>Yahoo\!\s*-\s*404 Not Found<\/title>/s)
      {
        $content = MyGet($request, 'delete messages', 1);
        if ($content =~ /<head><title>Yahoo\!\s*-\s*404 Not Found<\/title>/s ){
          MyDie("Failed: Couldn't delete messages\n");
        }
      }
    $cnt = $cnt-1;
  }
  print $delCount . " message(s) have been deleted.\n" unless $quiet ;
}

sub MarkUnread(@) {
  my $cnt = $numurls;
  my @lst = @_;
  while ($cnt>=0) {
    $request = GET $lst[$cnt] ;
    $content = MyGet($request, 'mark messages unread', 1);
    if ( $content =~ /<head><title>Yahoo\!\s*-\s*404 Not Found<\/title>/s)
      {
        $content = MyGet($request, 'mark messages unread', 1);
        if ($content =~ /<head><title>Yahoo\!\s*-\s*404 Not Found<\/title>/s ){
          MyDie("Failed: Couldn't mark messages unread\n");
        }
      }
    $cnt = $cnt-1;
  }
  print $unreadCount . " message(s) have been ". "left unread.\n" unless $quiet;
}

sub EmptyTrash() {
    $request = GET  $emptyurl ;
    $content = MyGet($request, 'empty trash',0);
    print "Trash emptied.\n" unless ($content eq "FAILED" || $quiet);
}

sub EmptyBulk() {
    $request = GET  $emptyurl ;
    $content = MyGet($request, 'empty bulk',0);
    print "Bulk emptied.\n" unless ($content eq "FAILED" || $quiet);
}

sub Logout() {
    $request = GET  $logouturl ;
    $content = MyGet($request, 'logout',1);
    print "Logged out.\n" unless $quiet;
}


sub ParseConfigFile() {

  if ($altConfigFile) {
    open(CONFIGFILE,$altConfigFile) || 
	    die "Can`t open config file $altConfigFile\n" ; 
  }  else {
    open(CONFIGFILE, $ENV{"HOME"} . "/.fetchyahoorc") ||
	    open(CONFIGFILE,"/etc/fetchyahoorc") || return;
  }

  while (<CONFIGFILE>) {
    next if (/^\s*\#/);              # ignore lines with starting with a #

    if (/^\s*username\s*=\s*(.*?)\s*$/i) {
 	    $username = $1;
    } elsif (/^\s*password\s*=\s*(.*?)\s*$/i) {
 	    $password = $1;
    } elsif (/^\s*use-https\s*=\s*(.*?)\s*$/i) {
 	    $useHTTPS = $1;

    } elsif (/^\s*use-spool\s*=\s*(.*?)\s*$/i) {
 	    $useSpool = $1;
    } elsif (/^\s*spool\s*=\s*(.*?)\s*$/i) {
 	    $spoolName = $1;
    } elsif (/^\s*spool-mode\s*=\s*(.*?)\s*$/i) {
 	    $spoolMode = $1;

    } elsif (/^\s*use-proxy\s*=\s*(.*?)\s*$/i) {
 	    $useProxy = $1;
    } elsif (/^\s*proxy-host\s*=\s*(.*?)\s*$/i) {
 	    $proxyHost = $1;
    } elsif (/^\s*proxy-port\s*=\s*(.*?)\s*$/i) {
 	    $proxyPort = $1;
    } elsif (/^\s*proxy-username\s*=\s*(.*?)\s*$/i) {
 	    $proxyUser = $1;
    } elsif (/^\s*proxy-password\s*=\s*(.*?)\s*$/i) {
 	    $proxyPass = $1;

    } elsif (/^\s*use-forward\s*=\s*(.*?)\s*$/i) {
 	    $useForward = $1;
    } elsif (/^\s*mail-host\s*=\s*(.*?)\s*$/i) {
 	    $mailHost = $1;
    } elsif (/^\s*send-to\s*=\s*(.*?)\s*$/i) {
 	    $sendToAddress = $1;
    } elsif (/^\s*send-from\s*=\s*(.*?)\s*$/i) {
 	    $sendFromAddress = $1;

    } elsif (/^\s*new-messages-only\s*=\s*(.*?)\s*$/i) {
 	    $newOnly = $1;
    } elsif (/^\s*no-delete\s*=\s*(.*?)\s*$/i) {
 	    $noDelete = $1;
    } elsif (/^\s*use-https\s*=\s*(.*?)\s*$/i) {
 	    $useHTTPS = $1;
    } elsif (/^\s*quiet\s*=\s*(.*?)\s*$/i) {
 	    $quiet = $1;
    } elsif (/^\s*get-external\s*=\s*(.*?)\s*$/i) {
 	    $getExternal = $1;

    } elsif (/^\s*empty-bulk\s*=\s*(.*?)\s*$/i) {
 	    $emptyBulk = $1;
    } elsif (/^\s*empty-trash-after\s*=\s*(.*?)\s*$/i) {
 	    $emptyTrashAfter = $1;
    } elsif (/^\s*empty-trash-before\s*=\s*(.*?)\s*$/i) {
 	    $emptyTrashBefore = $1;
    } elsif (/^\s*logout\s*=\s*(.*?)\s*$/i) {
 	    $logout = $1;
    } elsif (/^\s*repeat-interval\s*=\s*(.*?)\s*$/i) {
 	    $repeatInterval = $1;
    } elsif (/^\s*no-errors\s*=\s*(.*?)\s*$/i) {
 	    $noerrors = $1;

    } elsif (/^\s*leave-unread\s*=\s*(.*?)\s*$/i) {
 	    $leaveUnread = $1;
    } elsif (/^\s*no-from-line\s*=\s*(.*?)\s*$/i) {
 	    $noFromLine = $1;
    } elsif (/^\s*status-only\s*=\s*(.*?)\s*$/i) {
 	    $statusOnly = $1;
    } elsif (/^\s*folder\s*=\s*(.*?)\s*$/i) {
 	    $box = $1;
    }

  }
  close(CONFIGFILE);
}

sub Localize($) {
  my ($cc) = @_;
  if (not grep /$cc/, ('us','fr','es','e1','de','it','br','ca','uk','cf')) {
    print "Country Code '$cc' not found. We will try the translation for 'us'.\n"
      unless $quiet;
    $cc='us';
  }
  my $strings;
  my %localized_strings = 
    ('us' => { 'msg_range' => 'showing (\d+)-(\d+) of (\d+)',
               'new_msg_range' => 'Messages (\d+)-(\d+) of (\d+)',
               'no_msgs'   => 'Folder\s*[\sa-zA-Z]*\s*has\s*no\s+',
               'new_no_msgs' => 'This\s*folder\s*has\s*no[\sunread]*messages',
               'p_view'    => 'Printable\&nbsp;View' },
     'fr' => { 'msg_range' => '(\d+)-(\d+) sur (\d+)',
               'new_msg_range' => 'Messages (\d+)-(\d+) sur (\d+)',
               'no_msgs'   =>
               'Dossier\s*Bote\s*de\s*rception\s*sans\s*messages',
               'new_no_msgs'   =>
               'Ce\s*dossier\s*ne\s*contient\s*pas\s*de\s*messages',
               'p_view'    => 'Version\&nbsp;imprimable' },
     'es' => { 'msg_range' => 'Mostrando (\d+)-(\d+) de (\d+)',
               'new_msg_range' => 'Mensajes (\d+)-(\d+) de (\d+)',
               'no_msgs'   => 
               'La\s*carpeta\s*Bandeja\s*de\s*entrada\s*est\s*vaca',
               'new_no_msgs' =>
               'No\s*hay\s*ningn\s*mensaje\s*sin\s*leer\s*en\s*la\s*carpeta',
               'p_view'    => 'Vista para imprimir' },
     'e1' => { 'msg_range' => 'Mostrando (\d+)-(\d+) de (\d+)',
               'new_msg_range' => 'Mensajes (\d+)-(\d+) de (\d+)',
               'no_msgs'   => 'La\s*carpeta\s*Bandeja\s*de\s*entrada\s*est\s*vaca',
               'new_no_msgs' =>'Esta\s*carpeta\s*no\s*tiene\s*mensajes\s*mensajes\s*no\s*ledos',
               'p_view'    => 'Vista para imprimir' },
     'de' => { 'msg_range' => 'Nachrichten (\d+)-(\d+) von (\d+)',
               'no_msgs'   => 'Dieser\s*Ordner\s*hat\s*keine\s*E-Mails',
               'p_view'    => 'Druckform' },
     'it' => { 'msg_range' => 'mostra (\d+)-(\d+) di (\d+)',
               'no_msgs'   => 'La\s*cartella\s*In\s*arrivo\s*non\s*contiene\s*messaggi',
               'p_view'    => 'Anteprima&nbsp;di&nbsp;stampa' },
     'br' => { 'msg_range' => 'exibindo (\d+)-(\d+) de (\d+)',
               'new_msg_range' => 'Mensagens (\d+)-(\d+) de (\d+)',
               'no_msgs'   => 'A\s*pasta\s*Caixa\s*de\s*entrada\s*no',
               'new_no_msgs'  => 'Esta\s*pasta\s*no\s*tem\s*mensagens',
               'p_view'    => 'Visualizar&nbsp;impresso' }
    );

  $cc = 'us' if $cc eq 'uk';
  $cc = 'us' if $cc eq 'ca';
  $cc = 'fr' if $cc eq 'cf';

  if ($strings = $localized_strings{$cc}) { return %$strings; } 
  else { return 0; }

}

sub checkExternal() {
  my @content;
  my @extboxen;
  my $tmpurl;
  print "Checking external boxen..\n" unless $quiet;
  $request = GET $baseurl . "/ym/External";
  $content = MyGet ($request, 'get external mailboxes listing', 0);
  #$content =~ s/Exhref=\"(.*)\"/$1/g;
  @content = split(/\n/, $content);
  @extboxen = grep s/.*href=\"(.*?External\?GET[^"]*)\".*/$1/g,@content;  #" heh

  # just loading the URL for an external mailbox loads the messages from it
  foreach $tmpurl (@extboxen) {
    $_ = $tmpurl;
    s/.*Srvr=(.*?)\&.*/$1/;
    print $_ . "\n" unless $quiet ;
    $request = GET $baseurl . $tmpurl;
    $content = MyGet ($request, 'get external mailbox messages', 0);
  }
}

