; Sketchy Example Program
; Copyright (C) 2005 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---purpose---
; Process trees.

; ---example---
; (define tree
;   '(((() (1 1) (() (2 2) ())) (3 3) (() (4 4) ())) (5 5) ()))
; (tree-assoc 1 tree <)

;c lib/list.l
;c lib/equalp.l equal?
;c lib/caar.l cadr cddr
;c lib/caaar.l caddr cadar
;c lib/caaaar.l caddar
;c lib/reverse.l

; ---code---
; Retrieve a binding from an association tree.
; An association tree (atree) is an ordered binary
; tree containing (symbol.value) pairs. Each node
; of the tree has the format
; -
; (left (symbol.value) right)
; -
; where left and right are either nodes or '().

(define tree-assoc (lambda (x a p)
  (letrec
    ((tassoc (lambda (a)
      (cond ((null? a) '())
        ((equal? (caadr a) x) (cadr a))
        ((p x (caadr a)) (tassoc (car a)))
        (#t (tassoc (caddr a)))))))
    (tassoc a))))

; Insert a node into an ordered tree.

; (define tree-insert (lambda (x tree p)
;   (cond ((null? tree) (list '() x '()))
;     ((p x (cadr tree))
;       (list (tree-insert x (car tree) p)
;             (cadr tree) (caddr tree)))
;     (#t (list (car tree) (cadr tree)
;               (tree-insert x (caddr tree) p))))))

(define tree-insert (lambda (x tree p)
  (letrec
    ((t-ins (lambda (tree)
      (cond ((null? tree) (list '() x '()))
        ((p x (cadr tree))
          (list (t-ins (car tree))
                (cadr tree) (caddr tree)))
        (#t (list (car tree) (cadr tree)
                  (t-ins (caddr tree))))))))
    (t-ins tree))))

; Sort a list using tree sort. A predicate is used
; to define the desired order of members. It should
; return #T for each two subsequent members in the
; desired order.

(define tree-sort (lambda (x p)
  (letrec
    ((sort-tree (lambda (x tree)
      (cond ((null? x) (tree-flatten tree))
        (#t (sort-tree (cdr x)
          (tree-insert (car x) tree p)))))))
    (sort-tree x '()))))

; Flatten a binary tree. Convert a binary tree to
; a flat list using depth first traversal.

(define tree-flatten (lambda (x)
  (letrec
    ((flatten (lambda (x r s)
      (cond ((null? x)
          (cond ((null? s) (reverse r))
            (#t (flatten (caddar s) (cons (cadar s) r)
                  (cdr s)))))
        (#t (flatten (car x) r (cons x s)))))))
    (flatten x '() '()))))

; Convert an ordered list into a balanced tree.

(define tree-balance (lambda (x p)
  ; Traverse the list using divide and conquer:
  ; first insert the middle element, then the
  ; left sub-list and finally the right sub-list.
  (letrec
    ((divide-list (lambda (x rx tree)
      (cond ((null? x) tree)
        ((eq? (car x) (car rx))
          (divide-list
            (cdr x) (reverse (cdr x))
            (divide-list
              (reverse (cdr rx)) (cdr rx)
              (tree-insert (car x) tree p))))
        ((eq? (car x) (cadr rx))
          (divide-list
            (cddr x) (reverse (cddr x))
            (divide-list
              (reverse (cdr rx)) (cdr rx)
              (tree-insert (car rx) tree p))))
        (#t (divide-list (cdr x) (cdr rx) tree))))))
    (divide-list x (reverse x) '()))))

