; Sketchy Example Program
; Copyright (C) 2005,2006 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---purpose---
; Re-name variables of lambda expressions.
; Lambda variables are re-name in such a way
; that no conflicts can arise between names
; of nested lambda expressions. After
; LVRENAME-ing an expression, full beta
; reduction can be done using SUBST.
; --
; For instance,
; -
; ((LVRENAME '(LAMBDA (X) (LAMBDA (X) X)))
; -
; => (LAMBDA (X:) (LAMBDA (X:I) X:I))

; ---args---
; X - S-expression

; ---keywords---
; LVRENAME function, lambda abstraction, beta reduction
; lambda variables, lambda variable renaming, example

; ---example---
; (lvrename '(lambda (x) (lambda (x) x))) => (lambda (x:) (lambda (x:i) x:i))

:require explode.l
:require implode.l
;c lib/list.l
;c lib/assoc.l
;c lib/reverse.l
;c lib/append.l
;c lib/neqp.l neq?
;c lib/not.l
;c lib/listp.l list?
;c lib/plus.l +
;c lib/minus.l -
;c lib/zerop.l zero?
;c lib/caar.l cadr cddr
;c lib/caaar.l caddr

; ---code---
(define lvrename (lambda (x)
  (letrec

  ; Re-name X to first match of NLIST
  ((rename (lambda (x nlist)
    (let ((nx (assoc x nlist)))
      (cond ((null? nx) x) (#t (cdr nx))))))

  ; Re-name each symbol in L
  (rename-list (lambda (l nlist r)
    (cond ((null? l) (reverse r))
      (#t (rename-list (cdr l) nlist
            (cons (rename (car l) nlist) r))))))

  ; Create a binding of the form
  ; (NAME . UNIQUE-NAME)
  ; Unique names are created by appending :i...
  (make-assoc (lambda (name level)
    (letrec
      ((append-suffix (lambda (name suffix)
        (implode (append (explode name) '(:) suffix))))
      (mk-assoc (lambda (n res)
        (cond ((zero? n)
            (cons name (append-suffix name res)))
          (#t
            (mk-assoc (- n 1) (cons 'i res)))))))
      (mk-assoc level '()))))

  ; Create a set of local (unique) names
  ; OLD   - NLIST so far
  ; NEW   - list of symbols to add to NLIST
  ; LEVEL - level of nested LAMBDA
  (localnames (lambda (old new level)
    (cond ((null? new) old)
      (#t (cons (make-assoc (car new) level)
            (localnames old (cdr new) level))))))

  ; Check if X is a valid lambda expression
  (lambda-p (lambda (x)
    (cond ((neq? (car x) 'lambda) #f)
      ((null? (cdr x)) #f)
      ((not (list? (cdr x))) #f)
      ((not (list? (cadr x))) #f)
      (#t (not (null? (cddr x)))))))

  ; Re-name all lambda variables in X.
  ; NLIST holds the bindings used to re-name symbols.
  ; LEVEL is the current nesting level of LAMBDA.
  (lvr (lambda (x nlist level)
    (cond ((null? x) '())
      ((symbol? x) (rename x nlist))
      ((lambda-p x)
        (let ((new-nlist (localnames nlist (cadr x) level)))
          (list 'lambda
                (rename-list (cadr x) new-nlist '())
                (lvr (caddr x) new-nlist (+ 1 level)))))
      (#t (cons (lvr (car x) nlist level)
                (lvr (cdr x) nlist level)))))))

  (lvr x '() 0))))

