; Sketchy Example Program
; Copyright (C) 2005,2006 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---purpose---
; Convert arithmetic expressions from prefix to
; infix notation. Prefix expressions may contain
; variables [symbols like X], numbers [quoted
; integer numbers like -57], and these functions:
; +, -, *, QUOTIENT, EXPT. Functions are translated
; to these operators: + - * / ^.
; --
; For instance,
; -
; (INFIX '(+ -15 17))
; -
; gives
; -
; '(#-15 + #17).
; --
; INFIX will insert parentheses ([ and ]) where
; required.

; ---keywords---
; INFIX function, prefix to infix, RPN, conversion

; ---args---
; X - arithmetic expression

; ---example---
; (infix '(+ 12 (* 3 (expt 4 (+ 5 6))))) => (12 + 3 * 4 ^ [ 5 + 6 ])

;c lib/digits.l
;c lib/list.l
;c lib/caar.l cadr
;c lib/caaar.l caddr
;c lib/append.l
;c lib/member.l
;c lib/assoc.l
;c lib/plus.l +
;c lib/minus.l -
;c lib/times.l *
;c lib/quotient.l
;c lib/expt.l
;c lib/normalize.l

; ---code---
(define infix (lambda (x)
  (letrec

  ; Operator symbols
  ((+ '+) (- '-) (* '*) (/ '/) (^ '^) ([ '[) (] '])

  ; Function->operator translation
  (ops '((+ . +) (- . -) (* . *) (quotient . /) (expt . ^)))

  ; Precedence list (descending order)
  (precedence '((high) (expt) (* quotient) (+ -) (low)))

  (function? (lambda (x)
    (member x '(+ - * quotient expt))))

  ; Has the function associated with the operator X
  ; a higher precedence than Y?
  (higherprec? (lambda (x y)
    (letrec
      ((hp (lambda (x y l)
        (cond ((null? l) #f)
          ((member x (car l))
            (cond ((member y (car l)) #f) (#t #t)))
          ((member y (car l)) #f)
          (#t (hp x y (cdr l)))))))
    (hp x y precedence))))

  (parenthesize (lambda (x) (append '([) x '(]))))

  (inf (lambda (x op)
    (cond ((number? x)
        ; Number: remove positive sign
        (list (normalize x)))
      ((symbol? x) (list x))
      ((function? (car x))
        ; Convert prefix to infix
        (let ((sub (append (inf (cadr x) (car x))
                     (list (cdr (assoc (car x) ops)))
                     (inf (caddr x) (car x)))))
          ; If the surrounding operation has a higher
          ; precedence, parenthesize this expression
          (cond ((higherprec? op (car x))
              (parenthesize sub))
            (#t sub))))
      (#t (bottom (list 'syntax: x)))))))

  (inf x 'low))))

