; Sketchy Example Program
; Copyright (C) 2005,2006 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---purpose---
; Evaluate arithmetic expressions.
; Evaluate all operations with numeric arguments
; (or subexpressions evaluating to numeric arguments)
; to their results. Reduce expressions with one variable
; argument using rules such as X+0 -> X and X^0 -> 1.
; Return the evaluated expression.
; --
; The following functions are recognized:
; +, -, *, QUOTIENT, EXPT
; --
; Variables are represented by single letters.

; ---keywords---
; EVALUATE function, evaluation, partial, math expressions
; arithmetic expressions, arithmetics

; ---args---
; X - expression

; ---example---
; (evaluate '(* x (expt 2 (+ 3 4)))) => (* x 128)

;c lib/list.l
;c lib/caar.l cadr.l
;c lib/caaar.l caddr.l
;c lib/zerop.l zero?
;c lib/plus.l +
;c lib/minus.l -
;c lib/times.l *
;c lib/quotient.l
;c lib/expt.l
;c lib/member.l

; ---code---
(define evaluate (lambda (x)
  (letrec

  ; Evaluate (fn left-num right-num)
  ((compute (lambda (op left right)
    (cond ((eq? op '+) (+ left right))
          ((eq? op '-) (- left right))
          ((eq? op '*) (* left right))
          ((eq? op 'quotient) (quotient left right))
          ((eq? op 'expt) (expt left right))
          (#t (bottom 'compute)))))

  (zero-p (lambda (x)
    (cond ((symbol? x) #f)
      (#t (zero? x)))))

  (one-p (lambda (x)
    (cond ((symbol? x) #f)
      (#t (= 1 x)))))

  (_evaluate (lambda (x)
    (cond ((symbol? x) x) ; Pass thru symbols and numbers
      ((number? x) x)
      ; Evaluate or reduce binary functions
      ((member (car x) '(+ - * quotient expt))
        ; First evaluate the arguments
        (let ((left (_evaluate (cadr x)))
             (right (_evaluate (caddr x))))
          (cond
            ; Both arguments constant,
            ; compute the result
            ((and (number? left) (number? right))
              (compute (car x) left right))
            ((eq? (car x) '+)
              (cond ((zero-p right) left)   ; X+0 => X
                ((zero-p left) right)       ; 0+X => X
                (#t (list (car x) left right))))
            ((eq? (car x) '-)
              (cond ((zero-p right) left)   ; X-0 => X
                (#t (list (car x) left right))))
            ((eq? (car x) '*)
              (cond ((zero-p right) '0)   ; X*0 => 0
                ((zero-p left) '0)        ; 0*X => 0
                ((one-p left) right)      ; X*1 => X
                ((one-p right) left)      ; 1*X => X
                (#t (list (car x) left right))))
            ((eq? (car x) 'quotient)
              (cond ((zero-p right)     ; X/0 => _|_
                  (bottom 'divide-by-zero))
                ((zero-p left) '0)      ; 0/X => 0
                ((one-p right) left)    ; X/1 => X
                (#t (list (car x) left right))))
            ((eq? (car x) 'expt)
              (cond ((zero-p right) '1)   ; X^0 => 1
                ((zero-p left) '0)        ; 0^X => 0
                ((one-p left) '1)         ; 1^X => 1
                ((one-p right) left)      ; X^1 => X
                (#t (list (car x) left right))))
            ; Unknown operator?
            (#t (bottom 'this-cannot-happen)))))
      (#t (bottom (list 'unknown-operator: x)))))))

  (_evaluate x))))

