; Sketchy Example Program
; Copyright (C) 2005 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---purpose---
; Date computations.

; ---example---
; (date-difference '(1969 7 21) '(2000 1 1)) => 11121

;c lib/caar.l cadr
;c lib/caaar.l caaar caddr
;c lib/logand.l
;c lib/zerop.l zero?
;c lib/greater.l >
;c lib/remainder.l
;c lib/minus.l -
;c lib/plus.l +
;c lib/times.l *
;c lib/quotient.l
;c lib/list-ref.l
;c lib/less.l <

; ---code---
; Definition of the 'date' data structure.
; Dates are lists of numbers. Order of year,
; month, day is in ISO format. For instance,
; July 23rd 1968 becomes '(1968 7 23).

(define year-of (lambda (date) (car date)))

(define month-of (lambda (date) (cadr date)))

(define day-of (lambda (date) (caddr date)))

; Convert a given date to the number of days
; since 01-01-01. This function will deliver
; wrong results for dates before 1582-10-15.

(define date-to-days (lambda (date)
  (letrec

    ((days-per-month
      '(0 31 59 90 120 151 181 212 243 273 304 334))

    (leap-year-p (lambda (year)
      (cond ((zero? (remainder year 400)) #t)
        ((zero? (remainder year 100)) #f)
        (#t (zero? (remainder year 4))))))

    (year-to-days (lambda (year)
      (- (+ (+ (* year 365)
               (quotient year 4))
            (quotient year 400))
         (quotient year 100))))

    (nth (lambda (a n)
      (list-ref a (- n 1))))

    (month-to-days (lambda (year month)
      (cond ((and (leap-year-p year) (> month 2))
          (+ 1 (nth days-per-month month)))
        (#t (nth days-per-month month))))))

    (+ (year-to-days (- (year-of date) 1))
       (+ (month-to-days (year-of date) (month-of date))
          (day-of date))))))

; Compute the - between two dates
; in days. The range covered by the dates given
; may not include the 15th of October in 1582,
; or the result will be wrong.

(define date-difference (lambda (date1 date2)
  (cond ((date-before-p date1 date2)
      (- (date-to-days date2)
         (date-to-days date1)))
    (#t (- (date-to-days date1)
           (date-to-days date2))))))

; Test if two dates are in chronological order.
; Return #T, if the first date is 'before' the
; second one and otherwise #F.

(define date-before-p (lambda (date1 date2)
  (cond ((< (year-of date1) (year-of date2)) #t)
        ((> (year-of date1) (year-of date2)) #f)
        ((< (month-of date1) (month-of date2)) #t)
        ((> (month-of date1) (month-of date2)) #f)
     (#t (< (day-of date1) (day-of date2))))))

; Test if two dates are in reverse chronological
; order. Return #T, if the first date is 'after'
; the second one and otherwise #F.

(define date-after-p (lambda (date1 date2)
  (date-before-p date2 date1)))

