/*
 * Sketchy -- An interpreter for purely applicative Scheme
 * Copyright (C) 2005,2006 Nils M Holm <nmh@t3x.org>
 * Derived from ArrowLISP, Copyright (C) 1998-2005 Nils M Holm.
 * See the file LICENSE for conditions of use.
 *
 * How to compile:
 *
 * cc -o sketchy -O sketchy.c
 * Flags:
 *    -DSMALL compiles a small memory version.
 *    -DSIGNAL redirects SIGINT to the error handler.
 */

#include <stdlib.h>
#ifdef __TURBOC__
 #include <io.h>
#else
 #include <unistd.h>
 #ifndef __MINGW32__
  #define setmode(fd, mode)
 #endif
#endif
#include <stdio.h>
#ifdef SIGNAL
 #include <signal.h>
#endif
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <time.h>

#define MAJORVERSION	12
#define RELEASE		"2006-02-25"

#define MAXPATHL	256

#define NBASEREG	10	/* Number of base registers */
#define MAXWRAPCOL	255	/* Max. wrap column */

/* Maximum length for symbols and imploded lists */
#ifdef SMALL
#define	TEXTLEN		128
#else
#define	TEXTLEN		1024
#endif

/*
 * Number of nodes.
 * (Recommended: 16-bit: 12280, >=32-bit: 131072)
 */
#ifdef SMALL
#define	POOLSIZE	12280
#else
#define	POOLSIZE	131072
#endif

#define VPOOLSIZE	POOLSIZE

/* Tag Masks */
#define	AFLAG	0x01	/* Atom flag, Car = char, CDR = next */
#define	MFLAG	0x02	/* Mark flag of garbage collector */
#define SFLAG	0x04	/* State flag of garbage collector */
#define XFLAG	0x08	/* Extended type flag */

#define	EOT	-1	/* EOT indicator */
#define	DOT	-2	/* Internal: dot character */
#define	RPAREN	-3	/* Internal: right parenthesis */

/* Evaluator states */
#define	MATOM	'0' 	/* Processing Atom */
#define	MLIST	'1' 	/* Processing List */
#define	MBETA	'2' 	/* Beta-reducing */
#define	MBIND	'3' 	/* Processing bindings of LET */
#define	MBINR	'4' 	/* Processing bindings of LETREC */
#define	MLETR	'5' 	/* Finish LET or LETREC */
#define	MCOND	'6' 	/* Processing predicates of COND */
#define	MCONJ	'7' 	/* Processing arguments of AND */
#define	MDISJ	'8' 	/* Processing arguments of OR */
#define	MBEGN	'9' 	/* Processing BEGIN */

struct counter {
	int	nred, nr1k, nr1m, nr1g;
};

int	NIL;			/* Not In List (or Pool) */
int	Car[POOLSIZE],		/* Vector holding CAR fields */
	Cdr[POOLSIZE];		/* Vector holding CDR fields */
char	Tag[POOLSIZE];		/* Vector holding TAG fields */
int	Vpool[VPOOLSIZE];	/* Vector pool */
int	Vptr;			/* Free space pointer of Vpool */
char	*Infile;		/* Input file name */
char	DirName[MAXPATHL];	/* Source directory */
char	ExpPath[MAXPATHL];	/* Expanded path of input file */
char	Path[MAXPATHL];		/* Path to input file */
FILE	*Input;			/* Current input stream */
int	Rejected;		/* Unread character */
FILE	*Output;		/* Current output stream */
int	Mstack, Lstack;		/* Mode stack, List stack */
int	Bstack;			/* Binding stack, used by LET/LETREC */
int	Estack;			/* Environment stack, for fixing closures */
int	Parent;			/* Parent pointer used in D/S/W GC */
int	Free;			/* Freelist */
int	Symbols;		/* Symbol table */
int	Packages;		/* Package list */
int	Digits[10];		/* Digit symbols */
int	SafeSymbols;		/* Safe copy of symbols */
int	Stack, Stack0;		/* Global stack, bottom of Stack */
int	Frame;			/* Current call frame */
int	Tmp, Tmp2;		/* Safe locations */
int	*Base[NBASEREG];	/* Base registers (roots for GC) */
int	Level;			/* Nesting level during input */
int	LoadLev;		/* Nesting level of LOAD meta cmds */
int	EvLev;			/* Number of nested EVALs */
int	ErrFlag, FatalFlag;	/* Error flags */
int	Function;		/* Name of current lambda function */
int	Trace;			/* Name of traced function */
char	PrtBuf[MAXWRAPCOL+1];	/* Print buffer, */
int	Pp;			/* Pointer */
int	Quoted;			/* Quote flag of PRINT */
int	DisplayMode;		/* Display Mode flag */
int	MaxAtoms, MaxCells;	/* Memory use gauge */
int	Line;			/* Input line number */
struct counter
	Reductions,		/* Reduction counter */
	Allocations;		/* Allocation counter */
int	StatFlag;		/* Statistics flag */
int	Ntrace;			/* Max functions to print in call trace */
int	ClPrLev;		/* Closure Print level */
int	ArrowMode;		/* Enable arrow comments */
int	StrictApply;		/* Strict (R5RS) apply flag */
int	LexEnv;			/* Environment for creating closures */
int	Bound;			/* Variables bound in a closure */
int	Closure_type;		/* Type of c. to fix in fixAllClosures() */
char	Image[TEXTLEN];		/* Name of initial load image */

/* Builtin symbol pointers (for fast lookup) */
int	S_bottom, S_char, S_closure, S_continuation,
	S_eof, S_false, S_lambda, S_number, S_primitive, S_quote,
	S_special, S_special_cbv, S_string, S_t, S_true, S_void,
	S_last, S_0, S_1,
	S_2, S_3, S_4, S_5, S_6, S_7, S_8, S_9;

/* Command line flags */
int	O_batch;	/* Batch (non-interactive) mode */

/* Primitive function opcodes */
enum {	P_BOTTOM, P_CAR, P_CDR, P_CHAR_TO_INTEGER, P_CHAR_CI_EQP,
	P_CHAR_CI_LTP, P_CHAR_EQP, P_CHAR_LTP, P_CHARP, P_CONS,
	P_DELETE_FILE, P_DISPLAY, P_EOF_OBJECTP, P_EQP,
	P_INTEGER_TO_CHAR, P_INTEGER_TO_LIST, P_LIST_TO_INTEGER,
	P_LIST_TO_STRING, P_NLESS, P_NMINUS, P_NPLUS, P_NULLP,
	P_NUMBERP, P_PACKAGE, P_PAIRP, P_PROCEDUREP, P_READ,
	P_READ_CHAR, P_RECURSIVE_BIND, P_STRING_TO_LIST,
	P_STRING_TO_SYMBOL, P_STRING_APPEND, P_STRING_LENGTH,
	P_STRING_REF, P_STRINGP, P_SUBSTRING, P_SYMBOL_TO_STRING,
	P_SYMBOLP, P_VOID, P_WRITE, N_PRIMITIVES };

/* Primitive function pointers. */
int	(*Primitives[N_PRIMITIVES])(int);

/* Special form opcodes */
enum {	SF_AND, SF_APPLY, SF_BEGIN, SF_CALLCC, SF_COND, SF_DEFINE,
	SF_LAMBDA, SF_LET, SF_LETREC, SF_OR, SF_QUOTE,
	SF_WITH_INPUT_FROM_FILE, SF_WITH_OUTPUT_TO_FILE, N_SPECIALS };

/* Special form handler pointers */
int	(*Specials[N_SPECIALS])(int, int *, int *, int *);

/* unused arg in special form handler */
#define USE(arg)	arg = NULL

/*
 * Prototypes
 */
void	_print(int n);
void	_prnum(int n, int w, char *spaces);
int	_rdch(void);
int	addPackage(int sym);
int	addPrim(char *name, int opcode);
int	addSpecial(char *name, int opcode, int cbv);
int	addSym(char *s, int v);
int	alloc(int pcar, int pcdr);
int	alloc3(int pcar, int pcdr, int ptag);
int	allocv(int type, int size);
int	assoc(int x, int n);
int	atomic(int n);
int	badArgLst(int n);
int	bindArgs(int n, int name);
void	bindLet(int env);
void	bsave(int n);
int	bunsave(int k);
void	bye(char *p);
int	capture(void);
#ifdef SIGNAL
 void	catchIntr(int);
#endif
int	character(void);
int	ckCont(int n);
void	clearStats(void);
int	closure(int n);
void	collect(int n);
int	copy2(int n);
int	copyBindings(void);
void	count(struct counter *c, int k);
int	digitToValue(int n);
void	display(int n);
int	doAnd(int n, int *pcf, int *pmode, int *pcbn);
int	doApply(int n, int *pcf, int *pmode, int *pcbn);
int	doBegin(int n, int *pcf, int *pmode, int *pcbn);
int	doBottom(int n);
int	doCallCC(int n, int *pcf, int *pmode, int *pcbn);
int	doCar(int n);
int	doCdr(int n);
int	doCharCiEqP(int n);
int	doCharCiLtP(int n);
int	doCharEqP(int n);
int	doCharLtP(int n);
int	doCharP(int n);
int	doCharToInteger(int n);
int	doCond(int n, int *pcf, int *pmode, int *pcbn);
int	doCons(int n);
int	doDefine(int n, int *pcf, int *pmode, int *pcbn);
int	doDeleteFile(int n);
int	doDisplay(int n);
int	doEofObjectP(int n);
int	doEqP(int n);
int	doIntegerToChar(int n);
int	doIntegerToList(int n);
int	doLambda(int n, int *pcf, int *pmode, int *pcbn);
int	doLet(int n, int *pcf, int *pmode, int *pcbn);
int	doLetrec(int n, int *pcf, int *pmode, int *pcbn);
int	doListToInteger(int n);
int	doListToString(int n);
int	doNLess(int n);
int	doNMinus(int n);
int	doNPlus(int n);
int	doNullP(int n);
int	doNumberP(int n);
int	doOr(int n, int *pcf, int *pmode, int *pcbn);
int	doPackage(int n);
int	doPairP(int n);
int	doProcedureP(int n);
int	doQuote(int n, int *pcf, int *pmode, int *pcbn);
int	doRead(int n);
int	doReadChar(int n);
int	doRecursiveBind(int n);
int	doStringAppend(int n);
int	doStringLength(int n);
int	doStringP(int n);
int	doStringRef(int n);
int	doStringToList(int n);
int	doStringToSymbol(int n);
int	doSubstring(int n);
int	doSymbolP(int n);
int	doSymbolToString(int n);
int	doVoid(int n);
int	doWithInputFromFile(int n, int *pcf, int *pmode, int *pcbn);
int	doWithOutputToFile(int n, int *pcf, int *pmode, int *pcbn);
int	doWrite(int n);
void	dump(char *p);
void	dumpState(char *s, int m);
void	dumpSymbols(char *p);
int	error(char *m, int n);
int	eval(int n);
int	evalClause(int n);
int	evalLet(void);
char	*expandPath(char *s);
int	explodeNum(char *s);
void	fatal(char *m);
int	findPackage(int sym);
int	findPsym(char *s, int y);
int	findSym(char *s);
int	finishLet(int rec);
void	fixAllClosures(int b, int type);
int	fixCachedClosures(void);
void	fixClosuresOf(int n, int bindings);
int	flatCopy(int n, int *lastp);
int	gc(void);
void	gcv(void);
void	gc_stats(char *p);
void	getCharArgs(int n, int *pc1, int *pc2, char *msg);
void	getDirName(char *path, char *pfx);
int	getFactors(char *msg, int n, int *p1, int *p2);
int	getName(char *p);
int	getPred(void);
void	getOpts(int argc, char **argv);
void	help(char *p);
void	help_input(void);
int	implodeStr(char *s);
void	init1(void);
void	init2(void);
void	mark(int n);
int	isAlist(int n);
int	isBound(int n);
int	isSymList(int m);
int	lazyAtom(int n);
int	length(int n);
void	license(char *p);
void	load(char *p);
void	loadImage(char *p);
int	localize(int n, int *exprp);
char	*locase(char *s);
void	lsave(int n);
int	lunsave(int k);
int	main(int argc, char **argv);
void	markVec(int n);
void	meta(void);
int	mkChar(int x);
int	mkLexEnv(int term, int locals);
void	msave(int v);
int	munsave(void);
int	newDefine(int n);
void	newLine(void);
int	nextLet(int n);
void	nl(void);
int	nreverse(int n);
int	numericStr(char *s);
void	pl(char *s);
void	pr(char *s);
void	prDepth(int n);
int	primitive(int *np);
void	print(int n);
int	printChar(int n);
int	printClosure(int n);
int	printCont(int n);
int	printNum(int n);
int	printPrim(int n);
int	printQuote(int n);
int	printSpecial(int n);
void	printStats(void);
int	printString(int n);
void	printTrace(int n);
void	printValue(struct counter *c, char *s);
void	prnum(int n, int w);
void	prznum(int n, int w);
int	quote(int n);
int	readList(void);
int	rdch(void); int c;
void	repl(void);
void	require(char *p);
void	reset(struct counter *c);
void	resetState(void);
void	restoreBindings(int values);
int	resume(int n);
int	reverse(int n);
void	save(int n);
void	setFlag(char *name, int *intp, int val);
int	setupCond(int n);
int	setupLet(int n);
int	setupLogOp(int n);
int	skip(void);
int	special(int *np, int *pcf, int *pmode, int *pcbn);
int	stringLiteral(void);
void	subst(int old, int new, int *p);
int	symOrNum(int c);
int	tagged(int n);
void	tailCall(void);
void	trace(char *p);
void	traceCall(void);
void	unbindArgs(void);
void	unmarkVecs(void);
int	unreadable(void);
int	unsave(int k);
void	updatePackages(int old, int new);
int	usage(void);
int	valueOf(char *src, int n);
int	valueToDigit(int n);
void	version(char *p);
int	wrongArgs(int n);
int	xread(void);
int	yesno(char *p, int *pf);

/* Emit a newline sequence */
void newLine(void) {
	putc('\n', Output);
	if (Output == stdout) fflush(Output);
}

/* Print the string S thru a buffered interface. */
void pr(char *s) {
	fputs(s, Output);
}

/* Print a padded number with leading characters. */
void _prnum(int n, int w, char *spaces) {
	char	b[20];
	int	k;

	sprintf(b, "%d", n);
	k = strlen(b);
	if (k < w) pr(&spaces[k]);
	pr(b);
}

/* Print number with leading spaces. */
void prnum(int n, int w) {
	_prnum(n, w, "      ");
}

/* Print number with leading zeroes. */
void prznum(int n, int w) {
	_prnum(n, w, "000");
}

/* Flush the output buffer and emit a newline sequence. */
void nl(void) {
	newLine();
}

/* Convert string to lower case. */
char *locase(char *s) {
	int	k, i;

	k = strlen(s)+1;
	for (i=0; i<k; i++)
		if ('A' <= s[i] && s[i] <= 'Z')
			s[i] = s[i] + ('a' - 'A');
	return s;
}

/* Print function names on call stack */
void traceCall(void) {
	int	s, n;

	s = Frame;
	n = Ntrace;
	while (s != NIL) {
		if (!n || Cdr[s] == NIL || Car[Cdr[s]] == NIL) break;
		if (n == Ntrace) pr("* Trace:");
		n = n-1;
		pr(" ");
		Quoted = 1;
		_print(Car[Cdr[s]]);
		s = Car[s];
	}
	if (n != Ntrace) nl();
}

/*
 * Print error message M and set ErrFlag.
 * If N != -1 print an additional line containing '* N'.
 */
int error(char *m, int n) {
	if (ErrFlag) return NIL;
	pr("* ");
	if (Infile) {
		pr(locase(Infile));
		pr(": ");
	}
	prnum(Line, 0);
	pr(": ");
	if (Function != NIL) {
		Quoted = 1;
		_print(Function);
	}
	else {
		pr("REPL");
	}
	pr(": ");
	pr(m);
	if (n != -1) {
		if (m[0]) pr(": ");
		Quoted = 1;
		_print(n);
	}
	nl();
	if (!FatalFlag && Frame != NIL) traceCall();
	if (O_batch) exit(1);
	ErrFlag = -1;
	Function = NIL;
	return NIL;
}

/* Print message M and halt the interpreter. */
void fatal(char *m) {
	ErrFlag = 0;
	FatalFlag = 1;
	error(m, -1);
	pr("* Fatal error, aborting");
	nl();
	exit(1);
}

/* Reset counter. */
void reset(struct counter *c) {
	c->nred = 0;
	c->nr1k = 0;
	c->nr1m = 0;
	c->nr1g = 0;
}

/* Increment counter. */
void count(struct counter *c, int k) {
	c->nred = c->nred+k;
	if (c->nred >= 1000) {
		c->nred = c->nred - 1000;
		c->nr1k = c->nr1k + 1;
		if (c->nr1k >= 1000) {
			c->nr1k = 0;
			c->nr1m = c->nr1m+1;
			if (c->nr1m >= 1000) {
				c->nr1m = 0;
				c->nr1g = c->nr1g+1;
				if (c->nr1g >= 1000)
					fatal("statistics: counter overflow");
			}
		}
	}
}

/* Print counter value. */
void printValue(struct counter *c, char *s) {
	if (c->nr1g) {
		prznum(c->nr1g, 0); pr(",");
	}
	if (c->nr1m || c->nr1g) {
		prznum(c->nr1m, c->nr1g?3:0); pr(",");
	}
	if (c->nr1k || c->nr1m || c->nr1g) {
		prznum(c->nr1k, (c->nr1m||c->nr1g)?3:0); pr(",");
	}
	prznum(c->nred, (c->nr1k||c->nr1m||c->nr1g)?3:0);
	pr(" ");
	pr(s);
	nl();
}

/* Mark the object with the offset N in the Vpool */
void markVec(int n) {
	int	*p;

	p = &Vpool[Car[Cdr[n]] - 2];
	*p = n;
}

/*
 * Mark nodes which can be accessed through N.
 * This routine uses the Deutsch/Schorr/Waite algorithm
 * (aka pointer reversal algorithm) which marks the
 * nodes of a pool in constant space.
 * It uses the MFLAG and SFLAG to keep track of the
 * state of the current node.
 * Each visited node goes through these states:
 * M==0 S==0 unvisited, process CAR
 * M==1 S==1 CAR visited, process CDR
 * M==1 S==0 completely visited, return to parent
 */
void mark(int n) {
	int	p;

	Parent = NIL;	/* Initially, there is no parent node */
	while (1) {
		/* Reached a leaf? */
		if (n == NIL || Tag[n] & MFLAG) {
			/* When the current node is a leaf and there is */
			/* no parent, the entire tree is marked. */
			if (Parent == NIL) break;
			if (Tag[Parent] & SFLAG) {
				/* State 2: the CDR of the parent has */
				/* not yet been marked (S of Parent set). */
				/* Swap CAR and CDR pointers and */
				/* proceed with CDR. Set State==3. */
				p = Cdr[Parent];
				Cdr[Parent] = Car[Parent];
				Car[Parent] = n;
				Tag[Parent] &= ~SFLAG;	/* S=0 */
				Tag[Parent] |=  MFLAG;	/* M=1 */
				n = p;
			}
			else {
				/* State 3: CAR and CDR of parent done. */
				/* Return to the parent and restore */
				/* parent of parent */
				p = Parent;
				Parent = Cdr[p];
				Cdr[p] = n;
				n = p;
			}
		}
		else {
			/* State 1: The current node has not yet been */
			/* visited. */
			if (Tag[n] & AFLAG) {
				/* If the node is an atom, go directly */
				/* to state 3: Save the parent in CDR, */
				/* make the current node the new parent */
				/* and move to its CDR. */
				p = Cdr[n];
				Cdr[n] = Parent;
				/*Tag[n] &= ~SFLAG;*/	/* S=0 */
				Parent = n;
				n = p;
				Tag[Parent] |= MFLAG;	/* M=1 */
			}
			else {
				/* If this node is a vector, mark it */
				if (Tag[n] & XFLAG) markVec(n);
				/* Go to state 2: like above, but save */
				/* the parent in CAR and proceed to CAR. */
				p = Car[n];
				Car[n] = Parent;
				Tag[n] |= MFLAG;	/* M=1 */
				Parent = n;
				n = p;
				Tag[Parent] |= SFLAG;	/* S=1 */
			}
		}
	}
}

/* Mark vector of the Vpool unused */
void unmarkVecs(void) {
	int	p, k, link;

	p = 0;
	while (p < Vptr) {
		link = p;
		k = Vpool[p+1];
		p += (k + sizeof(int)-1) / sizeof(int) + 2;
		Vpool[link] = NIL;
	}
}

/*
 * Mark and Sweep Garbage Collection.
 * First, mark all nodes which can be accessed through
 * Base Registers (Base[]) and then reclaim untagged
 * nodes.
 */
int gc(void) {
	int	i, k;

	k = 0;
#ifdef DEBUG
	pr("GC called");
	nl();
#endif
	for (i=0; i<NBASEREG; i++) mark(Base[i][0]);
	Free = NIL;
	for (i=0; i<POOLSIZE; i++) {
		if (!(Tag[i] & MFLAG)) {
			Cdr[i] = Free;
			Free = i;
			k = k+1;
		}
		else {
			Tag[i] &= ~MFLAG;
		}
	}
	if (MaxAtoms < POOLSIZE-k) MaxAtoms = POOLSIZE-k;
#ifdef DEBUG
	prnum(k, 0);
	pr(" nodes reclaimed");
	nl();
#endif
	return k;
}

/* Allocate a fresh node and initialize with PCAR,PCDR,PTAG. */
int alloc3(int pcar, int pcdr, int ptag) {
	int	n;

	if (StatFlag) count(&Allocations, 1);
	if (Free == NIL) {
		gc();
		if (Free == NIL) fatal("out of nodes");
	}
	n = Free;
	Free = Cdr[Free];
	Car[n] = pcar;
	Cdr[n] = pcdr;
	Tag[n] = ptag;
	return n;
}

/* Allocate a fresh node and initialize with PCAR,PCDR. */
int alloc(int pcar, int pcdr) {
	return alloc3(pcar, pcdr, 0);
}

/* In situ vector pool garbage collection and compaction */
void gcv(void) {
	int	k, to, from;

	unmarkVecs();
	gc();		/* re-mark life vecs */
	to = from = 0;
	while (from < Vptr) {
		k = Vpool[from+1];
		k = (k + sizeof(int) - 1) / sizeof(int) + 2;
		if (Vpool[from] != NIL) {
			if (to != from) {
				memmove(&Vpool[to], &Vpool[from],
					k * sizeof(int));
				Car[Cdr[Vpool[to]]] = to + 2;
			}
			to += k;
		}
		from += k;
	}
	Vptr = to;
	if (Vptr > MaxCells) MaxCells = Vptr;
}

/* Allocate vector from vpool */
int allocv(int type, int size) {
	int	v, n, wsize;

	wsize = (size + sizeof(int) - 1) / sizeof(int) + 2;
	if (Vptr + wsize >= VPOOLSIZE) {
		gcv();
		if (Vptr + wsize >= VPOOLSIZE)
			fatal("out of vector space");
	}
	v = Vptr;
	Vptr += wsize;
	Tmp2 = alloc3(v+2, NIL, AFLAG);
	n = alloc3(type, Tmp2, XFLAG);
	Tmp2 = NIL;
	Vpool[v] = n;
	Vpool[v+1] = size;
	return n;
}

/* Save node N on the Stack. */
void save(int n) {
	Tmp = n;	/* Otherwise, alloc() might recycle this node. */
	Stack = alloc(n, Stack);
	Tmp = NIL;
}

/*
 * Pop K nodes off the Stack and return
 * the most recently popped one.
 */
int unsave(int k) {
	int	n = NIL; /*LINT*/

	while (k) {
		if (Stack == NIL) fatal("stack underflow");
		n = Car[Stack];
		Stack = Cdr[Stack];
		k = k-1;
	}
	return n;
}

/* Save value V on the M-Stack. */
void msave(int v) {
	/* Since the Mstack holds integer values rather than */
	/* nodes, the values are packaged in the character */
	/* fields of atoms. */
	Car[Mstack] = alloc3(v, Car[Mstack], AFLAG);
}

/* Pop a value off the M-Stack and return i */
int munsave(void) {
	int	v;

	if (Car[Mstack] == NIL) fatal("m-stack underflow");
	v = Car[Car[Mstack]];		/* See msave() */
	Car[Mstack] = Cdr[Car[Mstack]];
	return v;
}

/* Save node N on the L-Stack. */
void lsave(int n) {
	Tmp = n;	/* Otherwise, alloc() might recycle this node. */
	Lstack = alloc(n, Lstack);
	Tmp = NIL;
}

/*
 * Pop K nodes off the L-Stack and return
 * the most recently popped one.
 */
int lunsave(int k) {
	int	n = NIL; /*LINT*/

	while (k) {
		if (Lstack == NIL) fatal("l-stack underflow");
		n = Car[Lstack];
		Lstack = Cdr[Lstack];
		k = k-1;
	}
	return n;
}

/* Save node N on the B-Stack. */
void bsave(int n) {
	Tmp = n;	/* Otherwise, alloc() might recycle this node. */
	Bstack = alloc(n, Bstack);
	Tmp = NIL;
}

/*
 * Pop K nodes off the B-Stack and return
 * the most recently popped one.
 */
int bunsave(int k) {
	int	n = NIL; /*LINT*/

	while (k) {
		if (Bstack == NIL) fatal("b-stack underflow");
		n = Car[Bstack];
		Bstack = Cdr[Bstack];
		k = k-1;
	}
	return n;
}

/*
 * Read a single character from the input stream
 * and return it. Rdch()==EOT indicates that the
 * input is exhausted.
 * Characters will be stripped to seven bits.
 */
int _rdch(void) {
	int	c;

	if (Rejected != EOT) {
		c = Rejected;
		Rejected = EOT;
		return c;
	}
	c = getc(Input);
	if (feof(Input)) return EOT;
	if (c == '\n') Line = Line+1;
	return c;
}

/* Read a character and convert it to lower case. */
int rdch(void) {
	return tolower(_rdch());
}

/*
 * Find a symbol named S in the symbol table Y.
 * Each symbol is represented by a (NAME.VALUE) pair
 * where NAME is a list of character nodes and value
 * may be any valid S-expression.
 * The symbol table is a list containing symbol
 * pairs ((N1.V1) ...).
 * When a symbol named S is found, return its
 * pair (S.V) and otherwise return NIL.
 */
int findPsym(char *s, int y) {
	int	n, i;

	while (y != NIL) {
		n = Car[Car[y]];
		i = 0;
		while (n != NIL && s[i]) {
			if (s[i] != (Car[n] & 255)) break;
			n = Cdr[n];
			i = i+1;
		}
		if (n == NIL && !s[i]) return Car[y];
		y = Cdr[y];
	}
	return NIL;
}

/*
 * Find the symbol S in the symbol table of any
 * package in the package list.
 */
int findSym(char *s) {
	int	p, y;

	/* First search the current package */
	y = findPsym(s, Symbols);
	if (y != NIL) return y;
	/* No match, search other packages. */
	p = Packages;
	while (p != NIL) {
		y = findPsym(s, Cdr[Car[p]]);
		if (y != NIL) return y;
		p = Cdr[p];
	}
	return NIL;
}

/* Implode a string. */
int implodeStr(char *s) {
	int	i, n, m, a;

	i = 0;
	if (s[i] == 0) return NIL;
	a = n = NIL;
	while (s[i]) {
		m = alloc3(s[i], NIL, AFLAG);
		if (n == NIL) {		/* Protect the first character */
			n = m;
			save(n);
		}
		else {			/* Just append the rest */
			Cdr[a] = m;
		}
		a = m;
		i = i+1;
	}
	unsave(1);
	return n;
}

/* Update symbol table pointer in package lis */
void updatePackages(int old, int new) {
	int	p;

	p = Packages;
	while (p != NIL) {
		if (Cdr[Car[p]] == old) {
			Cdr[Car[p]] = new;
			return;
		}
		p = Cdr[p];
	}
	if (Packages != NIL)
		fatal("symbol table not in package list? *BOOM*");
}

/*
 * Add the symbol S to the symbol table if it
 * does not already exist. If it does exist,
 * return the existing symbol.
 * When adding a new symbol, initialize the
 * VALUE field with V. If V==0, make the symbol
 * a constant (bind it to itself).
 * Return the pair representing the symbol S.
 */
int addSym(char *s, int v) {
	int	n, m, osym;

	n = findSym(s);
	if (n != NIL) return n;
	n = implodeStr(s);
	save(n);
	m = alloc(n, v? v: n);
	save(m);
	osym = Symbols;
	Symbols = alloc(m, Symbols);
	unsave(2);
	updatePackages(osym, Symbols);
	return m;
}

/* Add primitive procedure. */
int addPrim(char *name, int opcode) {
	int	y;

	y = addSym(name, 0);
	Cdr[y] = alloc(S_primitive, NIL);
	Cdr[Cdr[y]] = alloc3(opcode, NIL, AFLAG);
	Cdr[Cdr[Cdr[y]]] = y;
	return y;
}

/* Add special form handler. */
int addSpecial(char *name, int opcode, int cbv) {
	int	y;

	y = addSym(name, 0);
	Cdr[y] = alloc(cbv? S_special_cbv: S_special, NIL);
	Cdr[Cdr[y]] = alloc3(opcode, NIL, AFLAG);
	Cdr[Cdr[Cdr[y]]] = y;
	return y;
}

/* Find a package. */
int findPackage(int sym) {
	int	p;

	p = Packages;
	while (p != NIL) {
		if (Car[Car[p]] == sym) return Car[p];
		p = Cdr[p];
	}
	return NIL;
}

/* Add a package. */
int addPackage(int sym) {
	int	y, p;

	y = findPackage(sym);
	if (y != NIL) return Cdr[y];
	p = alloc(sym, NIL);
	save(p);
	Packages = alloc(p, Packages);
	unsave(1);
	return Cdr[p];
}

/*
 * Read a list (S0 ... SN) and return (a pointer to) it.
 * This routine also recognizes pairs of the form (S0.S1).
 * For empty lists, it returns NIL.
 */
int readList(void) {
	int	n,	/* Node read */
		m,	/* Ptr to list */
		a,	/* Used to append nodes to m */
		c;	/* Member counter */
	char	*badpair;

	badpair = "bad pair";
	Level = Level+1;
	m = alloc(NIL, NIL);	/* Root node */
	save(m);
	a = NIL;
	c = 0;
	while (1) {
		if (ErrFlag) return NIL;
		n = xread();
		if (n == S_eof)  {
			if (LoadLev) return S_eof;
			fatal("EOF in list");
		}
		if (n == DOT) {
			if (c < 1) {
				error(badpair, -1);
				continue;
			}
			n = xread();
			Cdr[a] = n;
			if (n == RPAREN || xread() != RPAREN) {
				error(badpair, -1);
				continue;
			}
			unsave(1);
			Level = Level-1;
			return m;
		}
		if (n == RPAREN) break;
		if (a == NIL) 
			a = m;		/* First member: insert at root */
		else
			a = Cdr[a];	/* Following members: append */
		Car[a] = n;
		Cdr[a] = alloc(NIL, NIL); /* Alloc space for next member */
		c = c+1;
	}
	Level = Level-1;
	if (a != NIL) Cdr[a] = NIL;	/* Remove trailing empty node */
	unsave(1);
	return c? m: NIL;
}

/*
 * Read characters from the input device until a non-space
 * character [^\t ] is found. Return that character.
 */
int skip(void) {
	int	c;

	c = rdch();
	while (c == ' ' || c == '\t') c = rdch();
	return c;
}

/* Variables to dump to image file */
int *ImageVars[] = {
	&StatFlag, &ClPrLev, &ArrowMode, &StrictApply,
	&Symbols, &Packages, &Free, &Vptr,
	&S_char, &S_closure, &S_continuation, &S_false,
	&S_lambda, &S_number, &S_primitive, &S_quote, &S_special,
	&S_special_cbv, &S_string, &S_t, &S_true, &S_void, &S_last,
	&S_0, &S_1, &S_2, &S_3, &S_4, &S_5, &S_6, &S_7, &S_8, &S_9,
NULL };

/* Dump node pool image to file named in P. */
void dump(char *p) {
	int	fd, n, i;
	int	**v;
	char	magic[17];

	fd = open(p, O_CREAT | O_WRONLY, 0644);
	setmode(fd, O_BINARY);
	if (fd < 0) {
		error("cannot create file", -1);
		pr("* "); pr(p); nl();
		return;
	}
	strcpy(magic, "SKETCHY_________");
	magic[7] = sizeof(int);
	magic[8] = MAJORVERSION;
	n = 0x12345678;
	memcpy(&magic[10], &n, sizeof(int));
	write(fd, magic, 16);
	n = POOLSIZE;
	write(fd, &n, sizeof(int));
	v = ImageVars;
	i = 0;
	while (v[i]) {
		write(fd, v[i], sizeof(int));
		i = i+1;
	}
	if (	write(fd, Car, POOLSIZE*sizeof(int)) != POOLSIZE*sizeof(int) ||
		write(fd, Cdr, POOLSIZE*sizeof(int)) != POOLSIZE*sizeof(int) ||
		write(fd, Tag, POOLSIZE) != POOLSIZE ||
		write(fd, Vpool, VPOOLSIZE*sizeof(int)) !=
			VPOOLSIZE*sizeof(int)
	) {
		close(fd);
		pr("dump failed"); nl();
		return;
	}
	close(fd);
	if (!O_batch) {
		pr("dumped"); nl();
	}
}

/*
 * Extract directory name of PATH into PFX.
 */
void getDirName(char *path, char *pfx) {
	char	*p;

	if (strlen(path) > 256)
		fatal("path too long in ':l'");
	strcpy(pfx, path);
	p = strrchr(pfx, '/');
	if (p == NULL)
		strcpy(pfx, ".");
	else
		*p = 0;
}

/* Expand leading '~/' and '=' in path names */
char *expandPath(char *s) {
	char	*var, *r, *v;

	if (!strncmp(s, "~/", 2)) {
		var = "HOME";
		r = &s[2];
	}
	else if (!strncmp(s, "=", 1)) {
		var = "SKETCHYSRC";
		r = &s[1];
	}
	else
		return s;
	if ((v = getenv(var)) == NULL) return s;
	if (strlen(v) + strlen(r) + 2 >= MAXPATHL) {
		error("path too long in ':l' or ':c'", -1);
		return s;
	}
	sprintf(ExpPath, "%s/%s", v, r);
	return ExpPath;
}

/*
 * Load S-expressions from the external file or device
 * named in the string P. Return nothing.
 */
void load(char *p) {
	FILE	*ofile, *nfile;
	int	r;
	char	*oname;
	int	oline;

	if (LoadLev > 0) {
		if (strlen(p) + strlen(DirName) >= MAXPATHL) {
			error("path too long in ':l' or ':c'", -1);
			return;
		}
		if (*p != '.' && *p != '/')
			sprintf(Path, "%s/%s", DirName, p);
		else
			strcpy(Path, p);
		p = Path;
	}
	else {
		p = expandPath(p);
		getDirName(p, DirName);
	}
	nfile = fopen(p, "r");
	if (nfile == NULL) {
		error("cannot open file", -1);
		pr("* "); pr(p); nl();
		return;
	}
	LoadLev = LoadLev + 1;
	/* Save old I/O state */
	r = Rejected;
	/* Run the toplevel loop with redirected I/O */
	ofile = Input;
	Input = nfile;
	oline = Line;
	Line = 1;
	oname = Infile;
	Infile = p;
	repl();
	Infile = oname;
	Line = oline;
	/* Restore previous I/O state */
	Rejected = r;
	Input = ofile;
	LoadLev = LoadLev - 1;
	fclose(nfile);
	if (Level) error("unbalanced parentheses in loaded file", -1);
	if (!LoadLev && !O_batch) {
		pr("loaded"); nl();
	}
}

/*
 * Read all characters up to the next delimiter (white space or EOT)
 * from the input device and store them in the vector P.
 * Convert each character read to lower case.
 * Return the terminating character.
 * This routine is used to accept parameters to meta commands.
 */
int getName(char *p) {
	int	c, i;

	c = skip();
	i = 0;
	while (c != ' ' && c != '\t' && c != '\n' && c != '\r' && c != EOT) {
		p[i] = c;
		if (i >= TEXTLEN-2) {
			error("name too long", -1);
			return 0;
		}
		c = rdch();
		i = i+1;
	}
	p[i] = 0;
	return c;
}

/*
 * Extract a YES/NO, ON/OFF, 0/1 argument used to
 * enable and disable options.
 */
int yesno(char *p, int *pf) {
	int	c, f;

	c = getName(p);
	f = 2;
	if (p[0] == 'y' || p[0] == '1') f = -1;
	if (p[0] == 'n' || p[0] == '0') f = 0;
	if (p[0] == 'o') {
		if (p[1] == 'n') f = -1;
		if (p[1] == 'f') f = 0;
	}
	if (f == 2) error("Usage: command on/off", -1);
	pf[0] = f;
	return c;
}

/* Print a line followed by a newline sequence. */
void pl(char *s) {
	pr(s); nl();
}


/* Print the conditions of use. */
void license(char *p) {
	int	w;

	p = NULL;
pl(
"Sketchy -- An Interpreter for Purely Applicative Scheme");
pl(
"Copyright (C) 2005,2006 Nils M Holm.  All rights reserved.");
pl("");
pl(
"Redistribution and use in source and binary forms, with or without");
pl(
"modification, are permitted provided that the following conditions");
pl(
"are met:");
pl(
"1. Redistributions of source code must retain the above copyright");
pl(
"   notice, this list of conditions and the following disclaimer.");
pl(
"2. Redistributions in binary form must reproduce the above copyright");
pl(
"   notice, this list of conditions and the following disclaimer in the");
pl(
"   documentation and/or other materials provided with the distribution.");
pl("");
pl(
"THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND");
pl(
"ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE");
pl(
"IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE");
pl(
"ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE");
pl(
"FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL");
pl(
"DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS");
pl(
"OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)");
pl(
"HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT");
pl(
"LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY");
pl(
"OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF");
pl(
"SUCH DAMAGE.");
pl("");
}

/* Normal termination */
void bye(char *p) {
	p = NULL;
	if (!O_batch) {
		pr("Bye"); nl();
	}
	exit(0);
}

/* Dump symbol table */
void dumpSymbols(char *p) {
	int	pk, y;

	y = findSym(p);
	Quoted = 1;
	pk = Packages;
	pr("Packages:");
	while (pk != NIL) {
		pr(" ");
		_print(Car[Car[pk]]);
		if (Cdr[Car[pk]] == Symbols) pr("[open]");
		pk = Cdr[pk];
	}
	nl();
	y = findPackage(y);
	_print(Car[y]);
	pr(": ");
	_print(Cdr[y]);
	nl();
}

/* Set and acknowledge state of flag */
void setFlag(char *name, int *intp, int val) {
	if (val != 2) {
		intp[0] = val;
		if (!LoadLev && !O_batch) {
			pr(name);
			pl(intp[0]? " on": " off");
		}
	}
}

void help_input(void) {
	pr("Meta commands may be abbreviated as long as an"); nl();
	pr("abbreviation is unambiguous:"); nl();
	pr(":a on  is the same as  :arrow-comments on"); nl();
	pr("but  :dump x  does not work, because is may expand"); nl();
	pr("to both  :dump-image x  and  :dump-symbols x."); nl();
	nl();
	pr("Commands containing a dash may be abbreviated by"); nl();
	pr("using the first character of the command plus the"); nl();
	pr("character after the dash:"); nl();
	pr(":ds  is the same as  :dump-symbols"); nl();
	nl();
	pr("Flags turn switches on or off. The values 'yes',"); nl();
	pr("'on', and '1' activate a switch, and the values"); nl();
	pr("'no', 'off', and '0' deactivate it."); nl();
}

void help(char *p) {
	if (p[0] && !strncmp(p, "input", strlen(p))) {
		help_input();
		return;
	}
	pr(":arrow-comm flag  - turn => comments on/off"); nl();
	pr(":closure-form 0..2 - set closure print level"); nl();
	pr(":dump-image file   - dump image to file"); nl();
	pr(":dump-sym [sym]    - dump symbol table"); nl();
	pr(":gc                - collect garbage"); nl();
	pr(":help [input]      - print help text (also ?)"); nl();
	pr(":load file         - load file (~/file = $HOME/file"); nl();
	pr("                     and =file = $SKETCHYSRC/file)"); nl();
	pr(":quit              - quit"); nl();
	pr(":r5rs-apply flag   - restricted (R5RS) APPLY on/off"); nl();
	pr(":require file      - load conditionally (see :load)"); nl();
	pr(":show-license      - print conditions of use"); nl();
	pr(":show-version      - print version"); nl();
	pr(":statistics flag   - statistics on/off"); nl();
	pr(":trace [name]      - trace function (:trace = off)"); nl();
}

void require(char *p) {
	char	*q1, *q2;
	int	y;

	q1 = strchr(p, '/');
	q2 = strchr(q1? q1: p, '.');
	if (q2 != NULL) *q2 = 0;
	y = findSym(locase(q1? &q1[1]: p));
	if (q2 != NULL) *q2 = '.';
	if (y == NIL || Cdr[y] == S_void) {
		load(locase(p));
	}
	else if (!LoadLev && !O_batch) {
		pr("already present"); nl();
	}
}

void gc_stats(char *p) {
	int	k;

	p = NULL;
	k = gc();
	gcv();
	prnum(MaxAtoms, 6);
	pr(" nodes in use"); nl();
	prnum(k, 6);
	pr(" nodes in freelist"); nl();
	prnum(MaxCells, 6);
	pr(" vector cells in use"); nl();
	prnum(VPOOLSIZE-Vptr, 6);
	pr(" free vector cells"); nl();
	MaxAtoms = 0;
	MaxCells = 0;
}

void trace(char *p) {
	Trace = findSym(p);
	if (Trace != NIL) {
		if (!O_batch) {
			pr("tracing "); pr(p); nl();
		}
	}
	else if (p[0]) {
		error("no such symbol", -1);
		pr("* "); pr(p); nl();
	}
	else {
		if (!O_batch) {
			pr("trace off"); nl();
		}
	}
}

void version(char *p) {
	pr("Sketchy (C) 2006 Nils M Holm"); nl();
	pr("Latest modification: ");
	pr(RELEASE); nl();
	prnum(POOLSIZE, 6);
	pr(" nodes, ");
	prnum(VPOOLSIZE, 6);
	pr(" vector cells"); nl();
}

#define MT_FLAG	0
#define MT_INT	1
#define MT_PROC	2

struct MetaCmd {
	char	*cmd;
	int	type;
	int	lo, hi;
	int	*pval;
	void	(*fn)(char *s);
} MetaCmds[] = {
	{	"arrow-comments", MT_FLAG, 0,0,	&ArrowMode,	NULL	},
	{	"closure-form",	MT_INT, 0,2,	&ClPrLev,	NULL	},
	{	"dump-image",	MT_PROC, 1,0,	NULL,		dump	},
	{	"dump-symbols",	MT_PROC, 0,0,	NULL,		dumpSymbols},
	{	"gc",		MT_PROC, 0,0,	NULL,		gc_stats},
	{	"help",		MT_PROC, 0,0,	NULL,		help	},
	{	"?",		MT_PROC, 0,0,	NULL,		help	},
	{	"load",		MT_PROC, 1,0,	NULL,		load	},
	{	"quit",		MT_PROC, 0,0,	NULL,		bye	},
	{	"r5rs-apply",	MT_FLAG, 0,0,	&StrictApply,	NULL	},
	{	"require",	MT_PROC, 1,0,	NULL,		require	},
	{	"show-license",	MT_PROC, 0,0,	NULL,		license	},
	{	"show-version",	MT_PROC, 0,0,	NULL,		version	},
	{	"statistics",	MT_FLAG, 0,0,	&StatFlag,	NULL	},
	{	"trace",	MT_PROC, 0,0,	NULL,		trace	},
	{	NULL,		0, 0,0,		NULL		},
};

/* Read and process meta (colon) commands. */
void meta(void) {
	int	i, k, err;
	char	s[TEXTLEN], *p, *q;
	int	v;
	
	err = 0;
	c = getName(s);
	k = strlen(s);
	for (i=0; MetaCmds[i].cmd; i++) {
		if (!strncmp(MetaCmds[i].cmd, s, k)) {
			if (	MetaCmds[i+1].cmd &&
				!strncmp(MetaCmds[i+1].cmd, s, k)
			) {
				error("ambiguous meta command (:h = help)",
					-1);
				err = -1;
			}
			break;
		}
	}
	if (!err && !MetaCmds[i].cmd && strlen(s) == 2) {
		/* Try abbreviations */
		for (i=0; MetaCmds[i].cmd; i++) {
			p = MetaCmds[i].cmd;
			q = strchr(p, '-');
			if (q != NULL && p[0] == s[0] && q[1] == s[1]) {
				err = 0;
				break;
			}

		}
	}
	if (!MetaCmds[i].cmd) {
		error("unknown meta command (:h = help)", -1);
		err = -1;
	}
	if (err) {
		/* invalid command */
	}
	else if (c == '\n' || c == EOT) {
		s[0] = 0;
		err = -1;
	}
	else if (MetaCmds[i].type == MT_FLAG) {
		c = yesno(s, &v);
	}
	else if (MetaCmds[i].type == MT_INT) {
		c = getName(s);
		v = atoi(s);
		if (v < MetaCmds[i].lo || v > MetaCmds[i].hi)
			error("value out of range", -1);
	}
	else {
		c = getName(s);
	}
	while (c != '\n' && c != EOT) c = rdch();
	if (err && MetaCmds[i].type != MT_PROC)
		error("missing parameter", -1);
	if (ErrFlag) return;
	if (MetaCmds[i].type == MT_FLAG || MetaCmds[i].type == MT_INT) {
		*(MetaCmds[i].pval) = v;
		if (!LoadLev && !O_batch) {
			pr(MetaCmds[i].cmd);
			pr(" = ");
			if (MetaCmds[i].type == MT_FLAG)
				pr(v? "on": "off");
			else
				prnum(v, 0);
			nl();
		}
	}
	else {
		if (MetaCmds[i].lo && !s[0])
			error("missing parameter", -1);
		else
			(*MetaCmds[i].fn)(s);
	}
}

/* Is N a 'real' (non-NIL) Atom? */
int atomic(int n) {
	return n != NIL && Car[n] != NIL && (Tag[Car[n]] & AFLAG);
}

/* Is N a tagged list (an internal type)? */
int tagged(int n) {
	if (	n == NIL || Car[n] == NIL ||
		Car[n] == S_true || Car[n] == S_false
	)
		return 0;
	n = Car[Car[n]];
	return (Tag[n] & AFLAG) && Car[n] == '#';
}

/* Is N a lazy atom (atomic or tagged list)? */
int lazyAtom(int n) {
	return atomic(n) || tagged(n);
}

/* Quote an expression */
int quote(int n) {
	int	q;

	save(n);
	q = alloc(S_quote, NIL);
	save(q);
	Cdr[q] = alloc(n, NIL);
	unsave(2);
	return q;
}

/*
 * Check whether a string represents a number.
 * Numbers are defined as [+-]?[0-9]+.
 */
int numericStr(char *s) {
	int	i;

	i = 0;
	if (s[0] == '+' || s[0] == '-')
		i = 1;
	if (!s[i]) return 0;
	while (s[i]) {
		if (!isdigit(s[i])) return 0;
		i = i+1;
	}
	return 1;
}

/* Explode a string into a list of one-character symbols. */
int explodeNum(char *s) {
	int	i, l, x, y;
	char	name[3];

	i = 0;
	l = alloc(S_number, NIL);
	x = l;
	save(l);
	strcpy(name, "0d");
	while (s[i]) {
		name[0] = s[i];
		name[1] = isdigit(s[i])? 'd': 0;
		y = addSym(name, NIL);
		Cdr[x] = alloc(y, NIL);
		x = Cdr[x];
		i = i+1;
	}
	unsave(1);
	return l;
}

/* Report unreadable object */
int unreadable(void) {
	int	c;
	char	buf[2];

	error("unreadable object", -1);
	pr("* #<");
	buf[1] = 0;
	while (1) {
		c = rdch();
		if (c == '>' || c == '\n') break;
		buf[0] = c;
		pr(buf);
	}
	pr(">"); nl();
	return NIL;
}

/* Create a character literal. */
int mkChar(int x) {
	int	n;

	n = Tmp2 = alloc(S_char, NIL);
	Cdr[Tmp2] = alloc3(x, NIL, AFLAG);
	Tmp2 = NIL;
	return n;
}

/* Read a character literal. */
int character(void) {
	char	buf[10];
	int	i, c;

	for (i=0; i<9; i++) {
		c = _rdch();
		if (i > 0 && !isalpha(c)) break;
		buf[i] = c;
	}
	Rejected = c;
	buf[i] = 0;
	if (i == 0) c = ' ';
	else if (i == 1) c = buf[0];
	else if (!strcmp(buf, "space")) c = ' ';
	else if (!strcmp(buf, "newline")) c = '\n';
	else {
		error("bad character name", -1);
		c = 0;
	}
	return mkChar(c);
}

/* Read a string literal. */
int stringLiteral(void) {
	char	s[TEXTLEN+1];
	int	c, i, n, q;
	int	inv;

	i = 0;
	q = 0;
	c = _rdch();
	inv = 0;
	while (q || c != '"') {
		if (i >= TEXTLEN-2) {
			error("symbol too long", -1);
			i = i-1;
		}
		if (q && c != '"' && c != '\\') {
			s[i++] = '\\';
			inv = 1;
		}
		s[i] = c;
		q = !q && c == '\\';
		if (!q) i = i+1;
		c = _rdch();
	}
	s[i] = 0;
	n = allocv(S_string, i+1);
	strcpy((char *) &Vpool[Car[Cdr[n]]], s);
	if (inv) error("invalid escape sequence in string", n);
	return n;
}

/*
 * Read a symbol or a numeric literal. When reading a
 * symbol, add it to the global symbol table.
 */
int symOrNum(int c) {
	char	s[TEXTLEN];
	int	i;

	i = 0;
	while (	c != ' ' && c != '\t' && c != '\n' &&
		c != '\r' && c != '(' && c != ')' &&
		c != ';' && c != '.' && c != '#' &&
		c != '\''
	) {
		if (i >= TEXTLEN-2) {
			error("symbol too long", -1);
			i = i-1;
		}
		s[i] = c;
		i = i+1;
		c = rdch();
	}
	s[i] = 0;
	Rejected = c;
	if (numericStr(s)) return explodeNum(s);
	return addSym(s, S_void);
}

/*
 * Read an expression from the current input stream
 * and return (a pointer to) it.
 * Each valid expression is either an Atom or a List.
 */
int xread(void) {
	int	c;

	c = rdch();
	while (1) {	/* Skip spaces and comments, run meta commands */
		while (c == ' ' || c == '\t' || c == '\n' || c == '\r')
			c = rdch();
		if (c == ':' && !Level) {
			meta();
			c = '\n';
			ErrFlag = 0;
			continue;
		}
		if (ArrowMode && c == '=') {
			c = rdch();
			if (c != '>') {
				Rejected = c;
				c = '=';
				break;
			}
		}
		else if (c != ';')
			break;
		while (c != '\n') c = rdch();
	}
	if (c == EOT) return S_eof;
	if (c == '(') {
		return readList();
	}
	else if (c == '\'') {
		return quote(xread());
	}
	else if (c == '#') {
		c = rdch();
		if (c == 'f') return S_false;
		if (c == 't') return S_true;
		if (c == '\\') return character();
		if (c == '<') return unreadable();
		return error("bad # syntax", -1);
	}
	else if (c == '"') {
		return stringLiteral();
	}
	else if (c == ')') {
		if (!Level) return error("unexpected ')'", -1);
		return RPAREN;
	}
	else if (c == '.') {
		if (!Level) return error("unexpected '.'", -1);
		return DOT;
	}
	else {
		return symOrNum(c);
	}
}

/* Misc. error reporting and handling routines... */

int wrongArgs(int n) {
	return error("wrong argument count", n);
}

int badArgLst(int n) {
	return error("bad argument list", n);
}

/* Evaluate N=(CONS M M2) */
int doCons(int n) {
	int	m, m2;

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	m2 = Car[Cdr[m]];
	m = alloc(Car[m], m2);
	return m;
}

/* Evaluate N=(CAR M) */
int doCar(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	if (atomic(Car[m]) || Car[m] == NIL || tagged(Car[m]))
		return error("non-pair in 'car'", Car[m]);
	return Car[Car[m]];
}

/* Evaluate N=(CDR M) */
int doCdr(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	if (atomic(Car[m]) || Car[m] == NIL || tagged(Car[m])) {
		return error("non-pair in 'cdr'", Car[m]);
	}
	m = Cdr[Car[m]];
	return m;
}

/* Evaluate N=(CHAR->INTEGER M) */
int doCharToInteger(int n) {
	int	m, i, c;
	char	b[4];

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_char)
		return error("non-character in 'char->integer'", m);
	c = Car[Cdr[m]];
	i = 3;
	b[i] = 0;
	while (c || i == 4) {
		i = i-1;
		b[i] = c % 10 + '0';
		c = c / 10;
	}
	return explodeNum(&b[i]);
}

/* Get arguments of CHAR... primitives with type checking. */
void getCharArgs(int n, int *pc1, int *pc2, char *msg) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL) {
		wrongArgs(n);
		return;
	}
	if (atomic(Car[m]) || Car[Car[m]] != S_char) {
		error(msg, Car[m]);
		return;
	}
	pc1[0] = Car[Cdr[Car[m]]];
	m = Cdr[m];
	if (atomic(Car[m]) || Car[Car[m]] != S_char)
		error(msg, Car[m]);
	pc2[0] = Car[Cdr[Car[m]]];
}

/* Evaluate N=(CHAR-CI<? M1 M2) */
int doCharCiLtP(int n) {
	int	c1, c2;

	getCharArgs(n, &c1, &c2, "non-char in char-ci<?");
	return tolower(c1) < tolower(c2)? S_true: S_false;
}

/* Evaluate N=(CHAR-CI==? M1 M2) */
int doCharCiEqP(int n) {
	int	c1, c2;

	getCharArgs(n, &c1, &c2, "non-char in char-ci=?");
	return tolower(c1) == tolower(c2)? S_true: S_false;
}

/* Evaluate N=(CHAR<? M1 M2) */
int doCharLtP(int n) {
	int	c1, c2;

	getCharArgs(n, &c1, &c2, "non-char in char<?");
	return c1 < c2? S_true: S_false;
}

/* Evaluate N=(CHAR==? M1 M2) */
int doCharEqP(int n) {
	int	c1, c2;

	getCharArgs(n, &c1, &c2, "non-char in char=?");
	return c1 == c2? S_true: S_false;
}

/* Evaluate N=(CHAR? M) */
int doCharP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	return !atomic(m) && Car[m] == S_char ? S_true: S_false;
}

/* Return the value of a digit; */
int digitToValue(int n) {
	int	y;

	y = Car[n];
	if (	!isdigit(Car[y]) ||
		(Car[Cdr[y]]) != 'd' ||
		Cdr[Cdr[y]] != NIL
	) {
		error("non-digit in natural arithmetic operation", n);
		return -1;
	}
	return Car[y] - '0';
}

/* Convert a value to a digit. */
int valueToDigit(int n) {
	return Digits[n];
}

/* Reverse a list */
int reverse(int n) {
	int	m;

	m = NIL;
	save(m);
	while (n != NIL) {
		m = alloc(Car[n], m);
		Car[Stack] = m;
		n = Cdr[n];
	}
	unsave(1);
	return m;
}

/* Compute the length of a list */
int length(int n) {
	int	k;

	k = 0;
	while (n != NIL) {
		n = Cdr[n];
		k = k+1;
	}
	return k;
}

/* Retrieve factors of a numeric function */
int getFactors(char *msg, int n, int *p1, int *p2) {
	int	m, m2;

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_number)
		return error(msg, m);
	m2 = Car[Cdr[Cdr[n]]];
	if (atomic(m2) || m2 == NIL || Car[m2] != S_number)
		return error(msg, m2);
	*p1 = Cdr[m];
	*p2 = Cdr[m2];
	return ErrFlag;
}

/* Evaluate N=(N< M1 M2) */
int doNLess(int n) {
	int	f1, f2;
	int	k1, k2;
	int	v1, v2;

	getFactors("non-integer in 'n<'", n, &f1, &f2);
	save(f1);
	save(f2);
	k1 = length(f1);
	k2 = length(f2);
	if (k1 != k2) {
		unsave(2);
		if (k1 < k2) return S_true;
		return S_false;
	}
	while (f1 != NIL) {
		v1 = digitToValue(Car[f1]);
		v2 = digitToValue(Car[f2]);
		if (v1 != v2) {
			unsave(2);
			if (v1 < v2) return S_true;
			return S_false;
		}
		f1 = Cdr[f1];
		f2 = Cdr[f2];
	}
	unsave(2);
	return S_false;
}

/* Evaluate N=(N- M1 M2) */
int doNMinus(int n) {
	int	f1, f2, res;
	int	r, borrow;

	getFactors("non-integer in 'n-'", n, &f1, &f2);
	f1 = reverse(f1);
	save(f1);
	f2 = reverse(f2);
	save(f2);
	res = NIL;
	save(res);
	borrow = 0;
	while (f1 != NIL) {
		r = digitToValue(Car[f1])
			- (f2 == NIL? 0: digitToValue(Car[f2]))
			- borrow;
		if (r < 0) {
			r += 10;
			borrow = 1;
		}
		else {
			borrow = 0;
		}
		res = alloc(valueToDigit(r), res);
		Car[Stack] = res;
		if (f1 != NIL) f1 = Cdr[f1];
		if (f2 != NIL) f2 = Cdr[f2];
	}
	if (f2 != NIL || borrow)
		error("negative difference in 'n-'", n);
	while (Car[res] == S_0 && Cdr[res] != NIL)
		res = Cdr[res];
	res = alloc(S_number, res);
	unsave(3);
	return res;
}

/* Evaluate N=(N+ M1 M2) */
int doNPlus(int n) {
	int	f1, f2, res;
	int	r, carry;

	getFactors("non-integer in 'n+'", n, &f1, &f2);
	f1 = reverse(f1);
	save(f1);
	f2 = reverse(f2);
	save(f2);
	res = NIL;
	save(res);
	carry = 0;
	while (f1 != NIL || f2 != NIL || carry) {
		r = (f1 == NIL? 0: digitToValue(Car[f1]))
			+ (f2 == NIL? 0: digitToValue(Car[f2]))
			+ carry;
		if (r > 9) {
			r -= 10;
			carry = 1;
		}
		else {
			carry = 0;
		}
		res = alloc(valueToDigit(r), res);
		Car[Stack] = res;
		if (f1 != NIL) f1 = Cdr[f1];
		if (f2 != NIL) f2 = Cdr[f2];
	}
	res = alloc(S_number, res);
	unsave(3);
	return res;
}

/* Evaluate N=(INTEGER->CHAR M) */
int doIntegerToChar(int n) {
	int		m, p;
	unsigned char	c;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_number)
		return error("non-integer in 'integer->char'", m);
	p = Cdr[m];
	c = 0;
	while (p != NIL) {
		c = c * 10 + Car[Car[Car[p]]] - '0';
		p = Cdr[p];
	}
	if (c > 127)
		return error("value out of range in 'integer->char'", m);
	return mkChar(c);
}

/* Evaluate N=(INTEGER->LIST M) */
int doIntegerToList(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_number)
		return error("non-integer in 'integer->list'", m);
	return Cdr[m];
}

/* Evaluate N=(LIST->INTEGER M) */
int doListToInteger(int n) {
	int	m, check, p, d;

	m = Cdr[n];
	if (m == NIL || (Cdr[m] != NIL && Cdr[Cdr[m]] != NIL))
		return wrongArgs(n);
	check = Cdr[Cdr[m]] != NIL;
	m = Car[m];
	if (atomic(m) || tagged(m))
		return error("non-list in 'list->integer'", m);
	if (m == NIL)
		return error("empty list in 'list->integer'", m);
	if (check) {
		p = m;
		d = Car[p];
		if (	atomic(d) &&
			(Car[Car[d]] == '+' ||
			 Car[Car[d]] == '-') &&
			Cdr[Car[d]] == NIL
		)
			p = Cdr[p];
		while (p != NIL) {
			d = Car[p];
			if (	d != S_0 && d != S_1 && d != S_2 &&
				d != S_3 && d != S_4 && d != S_5 &&
				d != S_6 && d != S_7 && d != S_8 &&
				d != S_9
			)
				error(
				"non-digit in argument to 'list->integer'",
					d);

			p = Cdr[p];
		}
	}
	return alloc(S_number, m);
}

/* Evaluate N=(NULL M) */
int doNullP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	return Car[m] == NIL? S_true: S_false;
}

/* Evaluate N=(EOF-OBJECT? M M2) */
int doEofObjectP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL)
		return wrongArgs(n);
	return Car[m] == S_eof? S_true: S_false;
}

/* Evaluate N=(EQ? M M2) */
int doEqP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	return Car[m] == Car[Cdr[m]]? S_true: S_false;
}

/* Evaluate N=(LIST->STRING M)  */
int doListToString(int n) {
	int	m, p, i, k, ch;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (m == NIL) {
		p = allocv(S_string, 1);
		s = (char *) &Vpool[Car[Cdr[p]]];
		s[0] = 0;
		return p;
	}
	if (lazyAtom(m)) return error("non-list in 'list->string'", m);
	k = 0;
	for (p = m; p != NIL; p = Cdr[p]) {
		if (lazyAtom(p))
			return error("improper list in 'list->string'",
				Car[Cdr[n]]);
		k++;
	}
	p = allocv(S_string, k+1);
	i = 0;
	s = (char *) &Vpool[Car[Cdr[p]]];
	while (m != NIL) {
		ch = Car[m];
		if (atomic(ch) || Car[ch] != S_char)
			return error("non-char in argument to 'list->string'",
				ch);
		s[i++] = Car[Cdr[ch]];
		m = Cdr[m];
	}
	s[i] = 0;
	return p;
}

/* Evaluate N=(NUMBER? M) */
int doNumberP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	return atomic(m) || m == NIL? S_false:
		Car[m] == S_number? S_true: S_false;
}

/* Evaluate N=(PACKAGE [N1 [N2]]) */
int doPackage(int n) {
	int	m;

	m = Cdr[n];
	if (m != NIL && Cdr[m] != NIL)
		return wrongArgs(n);
	m = m == NIL? NIL: Car[m];
	Symbols = addPackage(m);
	return m;
}

/* Evaluate N=(PAIR? M) */
int doPairP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	return atomic(m) || tagged(m) || m == NIL? S_false:
		S_true;
}

/* Evaluate N=(PROCEDURE? M) */
int doProcedureP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL)
		return S_false;
	if (	Car[m] == S_closure ||
		Car[m] == S_continuation ||
		Car[m] == S_primitive ||
		Car[m] == S_special_cbv
	)
		return S_true;
	return S_false;
}

/* Evaluate N=(STRING->LIST M)  */
int doStringToList(int n) {
	int	m, a, lst, k, i;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_string)
		return error("non-string in 'string->list'", m);
	s = (char *) &Vpool[Car[Cdr[m]]];
	k = Vpool[Car[Cdr[m]] - 1] - 1;
	if (*s == 0) return NIL;
	lst = alloc(NIL, NIL);
	save(lst);
	a = lst;
	i = 0;
	while (i < k) {
		Car[a] = mkChar(s[i++]);
		if (i < k) {
			Cdr[a] = alloc(NIL, NIL);
			a = Cdr[a];
		}
	}
	unsave(1);
	return lst;
}

/* Evaluate N=(STRING->SYMBOL M)  */
int doStringToSymbol(int n) {
	int	m;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_string)
		return error("non-string in 'string->symbol'", m);
	s = (char *) &Vpool[Car[Cdr[m]]];
	if (s[0] == 0)
		return error("empty string in 'string->symbol'", m);
	return addSym(s, S_void);
}

/* Evaluate N=(STRING-APPEND M ...) */
int doStringAppend(int n) {
	int	m, p;
	int	k, len, o;
	int	new;
	char	*s, *q;

	m = Cdr[n];
	k = 0;
	while (m != NIL) {
		p = Car[m];
		if (atomic(p) || p == NIL || Car[p] != S_string)
			return error("non-string in 'string-append'", p);
		s = (char *) &Vpool[Car[Cdr[p]]];
		o = k;
		len = strlen(s);
		k = k + len;
		if (k < 0 || k - len != o)
			return error("string too long in 'string-append'", -1);
		m = Cdr[m];
	}
	new = allocv(S_string, k+1);
	q = (char *) &Vpool[Car[Cdr[new]]];
	q[0] = 0;
	m = Cdr[n];
	while (m != NIL) {
		p = Car[m];
		s = (char *) &Vpool[Car[Cdr[p]]];
		strcpy(q, s);
		q = &q[strlen(q)];
		m = Cdr[m];
	}
	return new;
}

/* Evaluate N=(STRING-LENGTH M) */
int doStringLength(int n) {
	int	m;
	char	*s;
	char	buf[20];

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_string)
		return error("non-string in 'string-length'", m);
	s = (char *) &Vpool[Car[Cdr[m]]];
	sprintf(buf, "%d", strlen(s));
	return explodeNum(buf);
}

/* Convert Sketchy integer to C integer */
int valueOf(char *src, int n) {
	int	org, v, o, d, neg;
	char	buf[100];

	v = 0;
	org = n;
	n = Cdr[n];
	neg = 0;
	if (Car[Car[Car[n]]] == '+') {
		n = Cdr[n];
	}
	else if (Car[Car[Car[n]]] == '-') {
		n = Cdr[n];
		neg = 1;
	}
	while (n != NIL) {
		o = v;
		d = digitToValue(Car[n]);
		v = v * 10 + d;
		if (v < 0 || (v - d) / 10 != o) {
			sprintf(buf, "value too big in '%s'", src);
			error(buf, org);
		}
		n = Cdr[n];
	}
	return neg? -v: v;
}

/* Evaluate N=(STRING-REF M1 M2) */
int doStringRef(int n) {
	int	m, m2, pos;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	m2 = Cdr[m];
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_string)
		return error("non-string in argument 1 of 'string-ref'", m);
	m2 = Car[m2];
	if (atomic(m2) || m2 == NIL || Car[m2] != S_number)
		return error("non-number in argument 2 of 'string-ref'", m2);
	s = (char *) &Vpool[Car[Cdr[m]]];
	pos = valueOf("string-ref", m2);
	if (pos < 0 || pos >= strlen(s))
		return error("offset out of range in 'string-ref'", m2);
	return mkChar(s[pos]);
}

/* Evaluate N=(STRING? M) */
int doStringP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	return !atomic(m) && Car[m] == S_string ? S_true: S_false;
}

/* Evaluate N=(SUBSTRING M1 M2 M3) */
int doSubstring(int n) {
	int	m, m2, m3, start, end, k;
	char	*s, *q;
	int	new;

	m = Cdr[n];
	if (	m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] == NIL ||
		Cdr[Cdr[Cdr[m]]] != NIL
	)
		return wrongArgs(n);
	m2 = Cdr[m];
	m = Car[m];
	if (atomic(m) || m == NIL || Car[m] != S_string)
		return error("non-string in argument 1 of 'substring'", m);
	m3 = Cdr[m2];
	m2 = Car[m2];
	if (atomic(m2) || m2 == NIL || Car[m2] != S_number)
		return error("non-number in argument 2 of 'substring'", m2);
	m3 = Car[m3];
	if (atomic(m3) || m3 == NIL || Car[m3] != S_number)
		return error("non-number in argument 3 of 'substring'", m2);
	s = (char *) &Vpool[Car[Cdr[m]]];
	start = valueOf("substring", m2);
	end = valueOf("substring", m3);
	if (start < 0 || start > strlen(s))
		return error("offset out of range in 'substring'", m2);
	if (end < start || end > strlen(s))
		return error("bad range in 'substring'", -1);
	k = end - start;
	new = allocv(S_string, k+1);
	q = (char *) &Vpool[Car[Cdr[new]]];
	memcpy(q, &s[start], k);
	q[k] = 0;
	return new;
}

/* Evaluate N=(SYMBOL->STRING M)  */
int doSymbolToString(int n) {
	int	m, p, k, q, i;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	if (!atomic(m))
		return error("non-symbol in 'symbol->string'", m);
	k = 1;
	for (p = Car[m]; p != NIL; p = Cdr[p])
		k++;
	q = allocv(S_string, k);
	s = (char *) &Vpool[Car[Cdr[q]]];
	p = Car[m];
	for (i = 0; i<k; i++) {
		s[i] = Car[p];
		p = Cdr[p];
	}
	s[i-1] = 0;
	return q;
}


/* Evaluate N=(SYMBOL? M)  */
int doSymbolP(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = Car[m];
	return atomic(m) && m != S_true && m != S_false ?
		S_true: S_false;
}

/* Extract clause from argument list of COND. */
/* Check the syntax of the clause. */
/* Return the predicate of the clause for */
/* evaluation in EVAL. */
int getPred(void) {
	int	e;

	e = Car[Car[Bstack]];
	if (	atomic(e) || e == NIL ||
		Cdr[e] == NIL || Cdr[Cdr[e]] != NIL
	)
		return error("bad clause in 'cond'", e);
	return Car[e];	/* predicate */
}

/*
 * Setup context for evaluation of (COND (P1 E1) ... (Pn En)).
 * The context consits of a list of clauses.
 * return the predicate of the first clause.
 */
int setupCond(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL) return wrongArgs(n);
	bsave(m);
	return getPred();
}

/*
 * Evaluate next clause of COND.
 * N is the value of the current predicate.
 * If N=*T*, return the expression of the predicate.
 * If N=*F*, return the predicate of the next clause.
 * When returning the expression of a predicate (N=*T*),
 * set the context on the Bstack to NIL to signal that
 * a true clause was found.
 * Clauses and predicates are returned to EVAL for
 * evaluation.
 */
int evalClause(int n) {
	int	e;

	e = Car[Bstack];
	if (n == S_false) {
		Car[Bstack] = Cdr[e];
		if (Car[Bstack] == NIL)
			return error("no default in 'cond'", -1);
		return getPred();
	}
	else {
		e = Car[Cdr[Car[e]]];
		Car[Bstack] = NIL;
		return e;
	}
}

/*
 * Setup context for evaluation of (AND ...) and (OR ...)
 * Return the first expression of the form.
 */
int setupLogOp(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL) return wrongArgs(n);
	bsave(m);
	return Car[m];
}

/*
 * Unbind the arguments of LAMBDA, LET and LETREC.
 * See also BINDARGS().
 */
void unbindArgs(void) {
	int	v;

	Frame = unsave(1);
	Function = unsave(1);
	v = bunsave(1);		/* Caller's namelist */
	while (v != NIL) {
		Cdr[Car[v]] = unsave(1);
		v = Cdr[v];
	}
}

/*
 * Check if the symbol N is bound in the current lexical
 * environment.
 */
int isBound(int n) {
	int	b;

	b = Bound;
	while (b != NIL) {
		if (atomic(b)) {
			if (n == b) return 1;
			break;
		}
		if (n == Car[b]) return 1;
		b = Cdr[b];
	}
	b = Car[LexEnv];
	while (b != NIL) {
		if (Car[Car[b]] == n) return 1;
		b = Cdr[b];
	}
	return 0;
}

/*
 * Recursively collect free variables and add their symbols
 * and values to the current lexical environment.
 */
void collect(int n) {
	if (n == NIL || (Tag[n] & AFLAG) || tagged(n)) return;
	if (atomic(n)) {
		if (isBound(n)) return;
		Car[LexEnv] = alloc(NIL, Car[LexEnv]);
		Car[Car[LexEnv]] = alloc(n, Car[n] == Cdr[n]? n: Cdr[n]);
		return;
	}
	collect(Car[n]);
	collect(Cdr[n]);
}

/* Create lexical environmen */
int mkLexEnv(int term, int locals) {
	LexEnv = alloc(NIL, NIL);
	save(LexEnv);
	Bound = locals;
	collect(term);
	unsave(1);
	return Car[LexEnv];
}

/* Create a closure from a lambda expression. */
int closure(int n) {
	int	cl, env, args, term;

	if (ErrFlag) return NIL;
	cl = alloc(S_closure, NIL);
	save(cl);
	args = Car[Cdr[n]];
	Cdr[cl] = alloc(args, NIL);
	term = Car[Cdr[Cdr[n]]];
	Cdr[Cdr[cl]] = alloc(term, NIL);
	if (Cdr[Cdr[Cdr[n]]] == NIL) {
		env = mkLexEnv(term, args);
		save(env);
		if (env != NIL) {
			Cdr[Cdr[Cdr[cl]]] = alloc(env, NIL);
			Estack = alloc(env, Estack);
		}
		unsave(1);
	}
	else {
		Cdr[Cdr[Cdr[cl]]] = alloc(Car[Cdr[Cdr[Cdr[n]]]], NIL);
	}
	unsave(1);
	return cl;
}

/*
 * Fix cached recursive bindings in closures.
 * Return 1, if applications of CALL/CC were
 * found in the cache and otherwise 0.
 */
int fixCachedClosures(void) {
	int	a, ee, e;
	int	cont;

	cont = 0;
	if (ErrFlag || Estack == NIL) return 0;
	a = Car[Bstack];
	while (a != NIL) {
		ee = Estack;
		while (ee != NIL) {
			e = Car[ee];
			if (Car[e] == S_continuation) {
				cont = 1;
			}
			else {
				while (e != NIL) {
					if (Car[a] == Car[Car[e]]) {
						Cdr[Car[e]] = Cdr[Car[a]];
						break;
					}
					e = Cdr[e];
				}
			}
			ee = Cdr[ee];
		}
		a = Cdr[a];
	}
	return cont;
}

/*
 * Fix references to symbols of B
 * in all closures of n.
 */
void fixClosuresOf(int n, int bindings) {
	int	ee, e;
	int	bb, b;

	if (n == NIL || atomic(n)) return;
	if (Car[n] == Closure_type) {
		fixClosuresOf(Car[Cdr[Cdr[n]]], bindings);
		ee = Cdr[Cdr[Cdr[n]]];
		if (ee == NIL) return;
		ee = Car[ee];
		while (ee != NIL) {
			e = Car[ee];
			bb = bindings;
			while (bb != NIL) {
				b = Car[bb];
				if (Car[b] == Car[e])
					Cdr[e] = Cdr[b];
				bb = Cdr[bb];
			}
			ee = Cdr[ee];
		}
		return;
	}
	if (tagged(n)) return;
	fixClosuresOf(Car[n], bindings);
	fixClosuresOf(Cdr[n], bindings);
}

/*
 * Fix recursive bindings in closures.
 * This is a slower, but more accurate version of
 * FixCachedClosures. It is only needed when
 * CALL/CC occurs inside of LETREC.
 */
void fixAllClosures(int b, int type) {
	int	p;

	Closure_type = type;
	p = b;
	while (p != NIL) {
		fixClosuresOf(Cdr[Car[p]], b);
		p = Cdr[p];
	}
}

/* Check if N is an alist. */
int isAlist(int n) {
	if (atomic(n)) return 0;
	while (n != NIL) {
		if (lazyAtom(Car[n]) || !atomic(Car[Car[n]]))
			return 0;
		n = Cdr[n];
	}
	return -1;
}

/* Check if M is a list of symbols. */
int isSymList(int m) {   
	int	a;

	a = m;
	while (m != NIL) {
		if (!atomic(Car[m])) return 0;
		if (atomic(Cdr[m])) break;
		m = Cdr[m];
	}
	return 1;
}

/* Evaluate to (ASSOC X N) */
int assoc(int x, int n) {
	while (n != NIL) {
		if (Car[Car[n]] == x)
			return Car[n];
		n = Cdr[n];
	}
	return NIL;
}

/* Evaluate N=(RECURSIVE-BIND M) */
int doRecursiveBind(int n) {
	int	m, env;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	env = Car[m];
	if (!isAlist(env))
		return error("non-alist in recursive-bind", env);
	fixAllClosures(env, S_lambda);
	return env;
}

/*
 * Set up a context for processing N=(LET ((MA1 eval[MX2]) ...) MN)
 * and N=(LETREC ((MA1 eval[MX2]) ...) MN).
 * Save
 * - the complete LET/LETREC expression on the Bstack
 * - the environment on the Bstack
 * - a list of new bindings on the Bstack (initially empty)
 * - two copies of the saved name list on the Stack
 */
int setupLet(int n) {
	int	m;	/* Argument pointer */

	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	m = Car[m];
	if (atomic(m))
		return error("bad environment in 'let' or 'letrec'", m);
	bsave(n);	/* save entire LET/LETREC */
	bsave(m);	/* save environment */
	bsave(NIL);	/* list of bindings */
	bsave(NIL);	/* save empty name list */
	save(Estack);
	Estack = NIL;
	return m;
}

/*
 * Process one binding of LET/LETREC:
 * bind value to name, advance to next binding.
 * Return:
 * non-NIL - more bindings in environment
 * NIL     - last binding done
 */
int nextLet(int n) {
	int	m, p;

	m = Car[Cdr[Cdr[Bstack]]];	/* rest of environment */
	if (m == NIL) return NIL;
	p = Car[m];
	Tmp2 = n;
	Car[Cdr[Bstack]] = alloc(NIL, Car[Cdr[Bstack]]);
	Car[Car[Cdr[Bstack]]] = alloc(Car[p], n);
	Tmp2 = NIL;
	Car[Cdr[Cdr[Bstack]]] = Cdr[m];
	return Cdr[m];
}

/*
 * Evaluate value to bind inside a LET/LETREC:
 * - check syntax
 * - save name to bind to
 * - save original binding of name
 * - return value for evaluation in EVAL
 */
int evalLet(void) {
	int	m, p, v;

	m = Car[Cdr[Cdr[Bstack]]];
	p = Car[m];
	/* Each binding must have the form (atom expr) */
	if (	atomic(p) || Cdr[p] == NIL || atomic(Cdr[p]) ||
		Cdr[Cdr[p]] != NIL || !atomic(Car[p])
	) {
		/* In case of an error, get rid of the */
		/* partial environment. */
		v = bunsave(1);
		bunsave(3);
		bsave(v);
		Estack = unsave(1);
		save(Function);
		save(Frame);
		unbindArgs();
		return error("bad binding in 'let' or 'letrec'", p);
	}
	Car[Bstack] = alloc(Car[p], Car[Bstack]);	/* Save name */
	/* Evaluate the new value of the current symbol */
	return Car[Cdr[p]];
}

/* Reverse a list in situ */
int nreverse(int n) {
	int	this, next, x;

	if (n == NIL) return NIL;
	this = n;
	next = Cdr[n];
	Cdr[this] = NIL;
	while (next != NIL) {
		x = Cdr[next];
		Cdr[next] = this;
		this = next;
		next = x;
	}
	return this;
}

/* Establish the bindings of LET/LETREC. */
void bindLet(int env) {
	int	b;

	while (env != NIL) {
		b = Car[env];
		save(Cdr[Car[b]]);	/* Save old value */
		Cdr[Car[b]] = Cdr[b];	/* Bind new value */
		env = Cdr[env];
	}
}

/*
 * Finish processing bindings of LET/LETREC:
 * finish context and return term.
 */
int finishLet(int rec) {
	int	m, v, b, e;

	Tmp2 = alloc(NIL, NIL);	/* Create safe storage */
	Cdr[Tmp2] = alloc(NIL, NIL);
	Cdr[Cdr[Tmp2]] = alloc(NIL, NIL);
	Cdr[Cdr[Cdr[Tmp2]]] = alloc(NIL, NIL);
	v = bunsave(1);
	b = bunsave(1);	/* get bindings */
	m = bunsave(2);	/* drop environment, get full LET/LETREC */
	b = nreverse(b);	/* needed for UNBINDARGS() */
	e = unsave(1);	/* outer Estack */
	Car[Tmp2] = b;		/* protect b, m, v */
	Car[Cdr[Tmp2]] = m;
	Car[Cdr[Cdr[Tmp2]]] = v;
	Cdr[Cdr[Cdr[Tmp2]]] = e;
	bindLet(b);
	bsave(v);
	if (rec) {
		if (fixCachedClosures())
			/* If CALL/CC was involved */
			fixAllClosures(b, S_closure);
	}
	Estack = e;
	save(Function);			/* required by UNBINDARGS() */
	save(Frame);
	Tmp2 = NIL;
	return Car[Cdr[Cdr[m]]];	/* return term of LET/LETREC */
}

/*
 * Substitute each OLD in *P with NEW.
 */
void subst(int old, int new, int *p) {
	if (*p == NIL) return;
	if (lazyAtom(*p)) {
		if (*p == old) *p = new;
		return;
	}
	subst(old, new, &Car[*p]);
	subst(old, new, &Cdr[*p]);
}

/* Make symbol N local to the current package. */
/* Also fix recursive references to N in EXPR. */
int localize(int n, int *exprp) {
	int	y, osym;

	y = Symbols;
	while (y != NIL) {
		if (n == Car[y]) return n;
		y = Cdr[y];
	}
	osym = Symbols;
	Symbols = alloc(NIL, Symbols);
	Car[Symbols] = alloc(Car[n], S_void);
	updatePackages(osym, Symbols);
	subst(n, Car[Symbols], exprp);
	return Car[Symbols];
}

/* Evaluate N=(BOTTOM ...) */
int doBottom(int n) {
	save(n);
	n = alloc(S_bottom, Cdr[n]);
	unsave(1);
	return error("", n);
}

/*
 * Create a flat copy of a list.
 * Store a reference to the last member
 * of the copy in lastp.
 */
int flatCopy(int n, int *lastp) {
	int	a, m, last;

	if (n == NIL) {
		lastp[0] = NIL;
		return NIL;
	}
	m = alloc(NIL, NIL);
	save(m);
	a = m;
	last = m;
	while (n != NIL) {
		Car[a] = Car[n];
		last = a;
		n = Cdr[n];
		if (n != NIL) {
			Cdr[a] = alloc(NIL, NIL);
			a = Cdr[a];
		}
	}
	unsave(1);
	lastp[0] = last;
	return m;
}

/*
 * Create a 2-level copy of the Lstack
 * and fix cyclic references. N = Lstack.
 */
int copy2(int n) {
	int	m, a, last, unused;

	if (n == NIL) return NIL;
	m = alloc(NIL, NIL);
	a = m;
	save(m);
	while (n != NIL) {
		if (Cdr[n] == NIL || Cdr[Cdr[n]] == NIL)
			fatal("internal: unexpected NIL in COPY");
		Car[a] = flatCopy(Car[n], &last);
		Cdr[a] = alloc(NIL, NIL);
		Car[Cdr[a]] = flatCopy(Car[Cdr[n]], &unused);
		Cdr[Cdr[a]] = alloc(last, NIL);
		n = Cdr[Cdr[Cdr[n]]];
		if (n != NIL) {
			Cdr[Cdr[Cdr[a]]] = alloc(NIL, NIL);
			a = Cdr[Cdr[Cdr[a]]];
		}
	}
	unsave(1);
	return m;
}

/*
 * Copy names and values of the symbol table
 * into an alist.
 */
int copyBindings(void) {
	int	y, p, ny, pk, q;

	pk = Packages;
	p = alloc(NIL, NIL);
	ny = p;
	q = NIL;
	save(p);
	while (pk != NIL) {
		y = Cdr[Car[pk]];
		while (y != NIL) {
			Car[p] = alloc(Car[y], Cdr[Car[y]]);
			y = Cdr[y];
			Cdr[p] = alloc(NIL, NIL);
			q = p;
			p = Cdr[p];
		}
		pk = Cdr[pk];
	}
	if (q != NIL) Cdr[q] = NIL;
	unsave(1);
	return Car[ny] == NIL? NIL: ny;
}

/* Restore values of the symbol table. */
void restoreBindings(int values) {
	int	b;

	while (values != NIL) {
		b = Car[values];
		Cdr[Car[b]] = Cdr[b];
		values = Cdr[values];
	}
}

/* Capture interpreter state, creating a continuation. */
int capture(void) {
	int	cont, x;

	cont = alloc(S_continuation, NIL);
	save(cont);
	Estack = alloc(cont, Estack);
	Cdr[cont] = alloc(NIL, NIL);
	Car[Cdr[cont]] = alloc(Car[Mstack], NIL);
	Cdr[Cdr[cont]] = alloc(NIL, NIL);
	Car[Cdr[Cdr[cont]]] = copy2(Lstack);
	Cdr[Cdr[Cdr[cont]]] = alloc(Cdr[Cdr[Stack]], NIL);
	Cdr[Cdr[Cdr[Cdr[cont]]]] = alloc(NIL, NIL);
	Car[Cdr[Cdr[Cdr[Cdr[cont]]]]] = flatCopy(Bstack, &x);
	Cdr[Cdr[Cdr[Cdr[Cdr[cont]]]]] = alloc(Estack, NIL);
	Cdr[Cdr[Cdr[Cdr[Cdr[Cdr[cont]]]]]] = alloc(NIL, NIL);
	Car[Cdr[Cdr[Cdr[Cdr[Cdr[Cdr[cont]]]]]]] = copyBindings();
	unsave(1);
	return cont;
}

/* Check the arguments of a continuation. */
int ckCont(int n) {
	if (	Cdr[n] == NIL || Cdr[Cdr[n]] == NIL ||
		Cdr[Cdr[Cdr[n]]] == NIL || Cdr[Cdr[Cdr[Cdr[n]]]] == NIL ||
		Cdr[Cdr[Cdr[Cdr[Cdr[n]]]]] == NIL ||
		Cdr[Cdr[Cdr[Cdr[Cdr[Cdr[n]]]]]] == NIL ||
		Cdr[Cdr[Cdr[Cdr[Cdr[Cdr[Cdr[n]]]]]]] != NIL
	)
		error("invalid continuation", -1);
	return n;
}

/* Re-activate a captured continuation N. */
int resume(int n) {
	int	m, state, x;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	ckCont(Car[n]);
	if (ErrFlag) return NIL;
	state = Cdr[Car[n]];
	Car[Mstack] = Car[Car[state]];
	Lstack = Car[Cdr[state]];
	Stack = Car[Cdr[Cdr[state]]];
	Bstack = flatCopy(Car[Cdr[Cdr[Cdr[state]]]], &x);
	Estack = Car[Cdr[Cdr[Cdr[Cdr[state]]]]];
	restoreBindings(Car[Cdr[Cdr[Cdr[Cdr[Cdr[state]]]]]]);
	return Car[m];
}

/* Evaluate N=(VOID) */
int doVoid(int n) {
	if (Cdr[n] != NIL) return wrongArgs(n);
	return S_void;
}

/* Evaluate N=(READ) */
int doRead(int n) {
	if (Cdr[n] != NIL) return wrongArgs(n);
	return xread();
}

/* Evaluate N=(READ-CHAR) */
int doReadChar(int n) {
	int	c;

	if (Cdr[n] != NIL) return wrongArgs(n);
	c = _rdch();
	return c == EOT? S_eof: mkChar(c);
}

/* Evaluate N=(WRITE M) */
int doWrite(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	Quoted = 1;
	_print(Car[m]);
	return S_void;
}

/* Evaluate N=(DISPLAY M) */
int doDisplay(int n) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	display(Car[m]);
	return S_void;
}

int doDeleteFile(int n) {
	int	m, f;
	char	*s;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	f = Car[m];
	if (atomic(f) || f == NIL || Car[f] != S_string)
		return error("non-string in 'delete-file'", f);
	s = (char *) &Vpool[Car[Cdr[f]]];	/* file name */
	return unlink(s)? S_false: S_true;
}

/*
 * Check whether (CAR NP[0]) is a builtin procedure.
 * If it is one, run the appropriate routine, save
 * its result in NP[0], and return -1.
 * Return 0 if (CAR NP[0]) is not a builtin procedure.
 */
int primitive(int *np) {
	int	n, y;
	int	(*op)(int);

	n = np[0];
	y = Car[n];
	if (Car[y] != S_primitive || ErrFlag) return 0;
	op = Primitives[Car[Cdr[y]]];
	n = (*op)(n);
	np[0] = n;
	return -1;
}

/* Evaluate N=(AND ...) */
int doAnd(int n, int *pcf, int *pmode, int *pcbn) {
	USE(pcbn);
	if (Cdr[n] == NIL) {
		return S_true;
	}
	else if (Cdr[Cdr[n]] == NIL) {
		*pcf = 1;
		return Car[Cdr[n]];
	}
	else {
		*pcf = 2;
		*pmode = MCONJ;
		return setupLogOp(n);
	}
}

/* Evaluate N=(APPLY M) */
int doApply(int n, int *pcf, int *pmode, int *pcbn) {
	int	m, p;

	*pcf = 1;
	USE(pmode);
	*pcbn = 1;
	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	if (Car[m] == NIL || atomic(Car[m]))
		return error("non-procedure in 'apply'", Car[m]);
	p = Car[Car[m]];
	if (!StrictApply && p == S_special)
		;	/* OK */
	else if (p != S_primitive && p != S_special_cbv && p != S_closure)
		return error("non-procedure in 'apply'", Car[m]);
	p = Car[Cdr[m]];
	while (p != NIL) {
		if (atomic(p) || tagged(p)) return
			error("improper list in 'apply'", Car[Cdr[m]]);
		p = Cdr[p];
	}
	return alloc(Car[m], Car[Cdr[m]]);
}

/* Evaluate N=(BEGIN ...) */
int doBegin(int n, int *pcf, int *pmode, int *pcbn) {
	USE(pcbn);
	if (Cdr[n] == NIL) {
		return S_void;
	}
	else if (Cdr[Cdr[n]] == NIL) {
		*pcf = 1;
		return Car[Cdr[n]];
	}
	else {
		*pcf = 2;
		*pmode = MBEGN;
		return setupLogOp(n);
	}
}

/* Evaluate N=(CALL/CC M) */
int doCallCC(int n, int *pcf, int *pmode, int *pcbn) {
	int	m;

	USE(pmode);
	USE(pcbn);
	*pcf = 1;
	if (EvLev > 1) {
		error("'call/cc' cannot be used here; this is a known bug",
		-1);
		return NIL;
	}
	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	m = alloc(Car[m], NIL);
	save(m);
	Cdr[m] = alloc(capture(), NIL);
	unsave(1);
	return m;
}

/* Evaluate N=(COND M1 ...) */
int doCond(int n, int *pcf, int *pmode, int *pcbn) {
	*pcf = 2;
	*pmode = MCOND;
	USE(pcbn);
	return setupCond(n);
}

/*
 * Evaluate N=(DEFINE (M ...) M2)
 */
int newDefine(int n) {
	int	m, y;

	m = Cdr[n];
	if (Car[m] == NIL)
		return error("missing function name in 'define'",
			Car[m]);
	if (!isSymList(Car[m])) return badArgLst(Car[m]);
	y = Car[Car[m]];
	save(Car[Cdr[m]]);
	Tmp2 = alloc(S_lambda, NIL);
	Cdr[Tmp2] = alloc(Cdr[Car[m]], NIL);
	Cdr[Cdr[Tmp2]] = alloc(Car[Cdr[m]], NIL);
	Cdr[Cdr[Cdr[Tmp2]]] = alloc(NIL, NIL);
	y = localize(y, &Car[Cdr[m]]);
	Cdr[y] = eval(Tmp2);
	Tmp2 = NIL;
	unsave(1);
	return y;
}

/*
 * Evaluate N=(DEFINE M eval[M2])
 * The name M already has been added to the
 * symbol table by READ().
 */
int doDefine(int n, int *pcf, int *pmode, int *pcbn) {
	int	m, v, y;

	if (EvLev > 1) {
		error("'define' is limited to the top level", -1);
		return NIL;
	}
	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	y = Car[m];
	if (!atomic(y)) return newDefine(n);
	/* Protect the unevaluated expression */
	v = Car[Cdr[m]];
	save(v);
	/* If we are binding to a lambda expression, */
	/* add a null environment */
	if (!atomic(v) && Car[v] == S_lambda) {
		if (	Cdr[v] != NIL && Cdr[Cdr[v]] != NIL &&
			Cdr[Cdr[Cdr[v]]] == NIL
		) {
			Cdr[Cdr[Cdr[v]]] = alloc(NIL, NIL);
		}
	}
	y = localize(y, &Car[Cdr[m]]);
	/* Evaluate and bind second argument */
	Cdr[y] = eval(Car[Cdr[m]]);
	unsave(1);
	return Car[m];
}

/* Check LAMBDA syntax and create closure from lambda expression. */
int doLambda(int n, int *pcf, int *pmode, int *pcbn) {
	int	m;

	m = Cdr[n];
	if (	m == NIL || Cdr[m] == NIL ||
		(Cdr[Cdr[m]] != NIL && Cdr[Cdr[Cdr[m]]] != NIL)
	)
		return wrongArgs(n);
	if (Cdr[Cdr[m]] != NIL && !isAlist(Car[Cdr[Cdr[m]]]))
		return error("bad environment in 'lambda'",
			Car[Cdr[Cdr[m]]]);
	if (!atomic(Car[m])) {
		if (tagged(Car[m])) return badArgLst(Car[m]);
		if (!isSymList(Car[m])) return badArgLst(Car[m]);
	}
	return Car[n] == S_closure? n: closure(n);
}

/* Evaluate N=(LET ENV EXPR) */
int doLet(int n, int *pcf, int *pmode, int *pcbn) {
	*pcf = 2;
	*pmode = MBIND;
	USE(pcbn);
	if (setupLet(n) != NIL)
		return evalLet();
	else
		return NIL;
}

/* Evaluate N=(LETREC ENV EXPR) */
int doLetrec(int n, int *pcf, int *pmode, int *pcbn) {
	*pcf = 2;
	*pmode = MBINR;
	USE(pcbn);
	if (setupLet(n) != NIL)
		return evalLet();
	else
		return NIL;
}

/* Evaluate N=(OR ...) */
int doOr(int n, int *pcf, int *pmode, int *pcbn) {
	USE(pcbn);
	if (Cdr[n] == NIL) {
		return S_false;
	}
	else if (Cdr[Cdr[n]] == NIL) {
		*pcf = 1;
		return Car[Cdr[n]];
	}
	else {
		*pcf = 2;
		*pmode = MDISJ;
		return setupLogOp(n);
	}
}

/* Evaluate N=(QUOTE M) */
int doQuote(int n, int *pcf, int *pmode, int *pcbn) {
	int	m;

	m = Cdr[n];
	if (m == NIL || Cdr[m] != NIL) return wrongArgs(n);
	return (Car[m]);
}

/* Evaluate N=(WITH-INPUT-FROM-FILE M1 M2) */
int doWithInputFromFile(int n, int *pcf, int *pmode, int *pcbn) {
	int	m, f;
	int	r, p;
	FILE	*ofile, *nfile;
	char	*s;

	USE(pcf);
	USE(pmode);
	USE(pcbn);
	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	f = Car[m];
	if (atomic(f) || f == NIL || Car[f] != S_string)
		return error("non-string in 'string->symbol'", f);
	s = (char *) &Vpool[Car[Cdr[f]]];	/* file name */
	if ((nfile = fopen(s, "r")) == NULL)
		return error("cannot open input file", f);
	/* Save old input state, create new */
	ofile = Input;
	Input = nfile;
	r = Rejected;
	Rejected = EOT;
	p = alloc(Car[Cdr[m]], NIL);	/* thunk => (thunk) */
	save(p);
	n = eval(p);
	unsave(1);
	/* Restore input state */
	Input = ofile;
	Rejected = r;
	fclose(nfile);
	return n;
}

/* Evaluate N=(WITH-OUTPUT-TO-FILE M1 M2) */
int doWithOutputToFile(int n, int *pcf, int *pmode, int *pcbn) {
	int	m, f;
	int	p;
	FILE	*ofile, *nfile;
	char	*s;

	USE(pcf);
	USE(pmode);
	USE(pcbn);
	m = Cdr[n];
	if (m == NIL || Cdr[m] == NIL || Cdr[Cdr[m]] != NIL)
		return wrongArgs(n);
	f = Car[m];
	if (atomic(f) || f == NIL || Car[f] != S_string)
		return error("non-string in 'string->symbol'", f);
	s = (char *) &Vpool[Car[Cdr[f]]];	/* file name */
	if ((nfile = fopen(s, "w")) == NULL)
		return error("cannot create output file", f);
	/* Save old output state, create new */
	fflush(Output);
	ofile = Output;
	Output = nfile;
	p = alloc(Car[Cdr[m]], NIL);	/* thunk => (thunk) */
	save(p);
	n = eval(p);
	unsave(1);
	/* Restore output state */
	Output = ofile;
	fclose(nfile);
	return n;
}

/*
 * Check whether (CAR NP[0]) is a special form handler.
 * If it is one, run the appropriate routine, save
 * its result in NP[0], and return -1.
 * Return 0 if (CAR NP[0]) is not a SF handler .
 */
int special(int *np, int *pcf, int *pmode, int *pcbn) {
	int	n, y;
	int	(*op)(int, int *, int *, int *);

	n = np[0];
	y = Car[n];
	if (ErrFlag) return 0;
	if (Car[y] == S_special || Car[y] == S_special_cbv) 
		op = Specials[Car[Cdr[y]]];
	else if (atomic(y) &&
		(Car[Cdr[y]] == S_special ||
		 Car[Cdr[y]] == S_special_cbv)
	)
		op = Specials[Car[Cdr[Cdr[y]]]];
	else
		return 0;
	np[0] = (*op)(n, pcf, pmode, pcbn);
	return -1;
}

/*
 * Bind the arguments of a LAMBDA function.
 * For a lambda application N=((LAMBDA (X1 ... Xn) S [ENV]) Y1 ... Yn)
 * this includes the following steps for j in {1,...,n}:
 *	1) save Xj in V
 *	2) save the value of Xj
 *	3) bind Xj to Yj
 * This routine results in  S' == S[X1/Y1] ... [Xn/Yn].
 * S->S' is performed by creating a new context. BINDARGS()
 * has no function result. It is used by EVAL() to perform
 * BETA reduction steps.
 */
int bindArgs(int n, int name) {
	int	fa,	/* Formal arg list */
		aa,	/* Actual arg list */
		e;	/* S, as above */
	int	env;	/* Optional lexical environment */
	int	p;
	int	at;	/* Atomic argument list flag */

	if (ErrFlag) return NIL;
	fa = Car[Cdr[Car[n]]];
	at = atomic(fa);
	aa = Cdr[n];
	p = Cdr[Cdr[Car[n]]];
	e = Car[p];
	env = Cdr[p] != NIL ? Car[Cdr[p]]: NIL;
	bsave(NIL);
	while ((fa != NIL && aa != NIL) || at) {
		if (!at) {
			/* Save name */
			Car[Bstack] = alloc(Car[fa], Car[Bstack]);
			save(Cdr[Car[fa]]);		/* Save value */
			Cdr[Car[fa]] = Car[aa];		/* Bind arg */
			fa = Cdr[fa];
			aa = Cdr[aa];
		}
		if (atomic(fa)) {	/* improper argument list */
			Car[Bstack] = alloc(fa, Car[Bstack]);	/* Save name */
			save(Cdr[fa]);	/* Save value */
			Cdr[fa] = aa;	/* Bind remaining arg list */
			fa = NIL;
			aa = NIL;
			break;
		}
	}
	while (env != NIL) {
		p = Car[env];
		Car[Bstack] = alloc(Car[p], Car[Bstack]);/* Save name */
		save(Cdr[Car[p]]);		/* Save value */
		Cdr[Car[p]] = Cdr[p];		/* Bind lex val */
		env = Cdr[env];
	}
	if (fa != NIL || aa != NIL) {
		wrongArgs(n);
		n = NIL;
	}
	else {
		n = e;
	}
	save(Function);
	Function = name;
	save(Frame);
	Frame = Stack;
	return n;
}

/*
 * Print application of traced function N in the form
 *	* (NAME A1 ... An)
 * PRINT() cannot be used because it would print NAME in
 * its expanded (LAMBDA...) form which is not desirable.
 */
void printTrace(int n) {
	pr("+ ");
	pr("(");
	Quoted = 1;
	_print(Trace);
	while (1) {
		n = Cdr[n];
		if (n == NIL) break;
		pr(" ");
		_print(Car[n]);
	}
	pr(")"); nl();
}

/* Print depth of stack N. */
void prDepth(int n) {
	int	k;

	k = 0;
	while (n != NIL) {
		n = Cdr[n];
		k = k+1;
	}
	prnum(k, 6); pr(" ");
}

/* Debugging: dump interpreter state */
void dumpState(char *s, int m) {
	pr(s); nl();
	pr("       ");
	pr("Mode        = "); prnum(m-'0',0); nl();
	prDepth(Lstack);
	pr("Lstack      = "); print(Lstack); nl();
	pr("       ");
	pr("Mstack      = "); print(Mstack); nl();
	prDepth(Stack);       pr("Car[Stack]  == "); print(Car[Stack]); nl();
	pr("       ");
	pr("Stack0      = "); prnum(Stack0,0); nl();
	prDepth(Bstack);
	pr("Bstack      = "); print(Bstack); nl();
}

/* Do tail call optimization. */
void tailCall(void) {
	int	m, y;

	m = Car[Mstack];
	/* Skip over callee's LET/LETREC frames, if any */
	while (m != NIL && Car[m] == MLETR) {
		m = Cdr[m];
	}
	/* Parent not beta-reducing? Give up. */
	if (m == NIL || Car[m] != MBETA)
		return;
	/* Yes, this is a tail call: */
	/* - remove callee's LET/LETREC frames. */
	/* - remove caller's call frame. */
	while (1) {
		Tmp2 = unsave(1); /* M */
		unbindArgs();
		unsave(1);
		y = munsave();
		save(Tmp2);
		Tmp2 = NIL;
		if (y == MBETA) break;
	}
}

/*
 * Evaluate the term N and return its normal form.
 * This is the heart of the interpreter:
 * An iterative EVAL function with tail-call optimization and call/cc.
 * For details on the reduction rules, refer to the manual.
 */
int eval(int n) {
	int	m,	/* Result node */
		m2,	/* Root of result lists */
		a,	/* Used to append to result */
		cbn;	/* Use call-by-name/quotation in next iteration */
	int	mode,	/* Current state */
		cf;	/* Continue flag */
	int	nm;	/* name of function to apply */

	EvLev = EvLev + 1;
	save(Lstack);
	save(Bstack);
	save(Car[Mstack]);
	save(Stack0);
	Stack0 = Stack;
	mode = MATOM;
	cf = 0;
	cbn = 0;
	while (!ErrFlag) {
		if (StatFlag) count(&Reductions, 1);
		if (n == NIL) {			/* () -> () */
			m = NIL;
		}
		else if (atomic(n)) {		/* Symbol -> Value */
			if (cbn) {
				m = n;
				cbn = 0;
			}
			else {
				m = Cdr[n] == Car[n]? n: Cdr[n];
				if (m == S_void) {
					error("symbol not bound", n);
					break;
				}
			}
		}
		else if (Car[n] == S_closure ||
			Car[n] == S_primitive ||
			Car[n] == S_special ||
			Car[n] == S_special_cbv ||
			Car[n] == S_continuation ||
			Car[n] == S_number ||
			Car[n] == S_char ||
			Car[n] == S_string ||
			cbn == 2
		) {
			m = n;
			cbn = 0;
		}
		else {				/* List (...) and Pair (X.Y) */
			/*
			 * This block is used to DESCEND into lists.
			 * The following nodes/variables will be saved:
			 *	1) the original list (on Stack)
			 *	2) the current state (on Mstack)
			 *	3) the root of the result list (on Lstack)
			 *	4) a ptr to the next free node
			 *		in the result list (on Lstack)
			 *	5) a ptr to the next member of
			 *		the original list (on Lstack)
			 */
			m = Car[n];
			if (atomic(Cdr[n])) {
				error("improper list in application", n);
				n = NIL;
			}
			save(n);	/* Save original list */
			msave(mode);
			/* Check call-by-name built-ins and flag */
			if ((atomic(m) && Car[Cdr[m]] == S_special) || cbn) {
				cbn = 0;
				lsave(NIL);
				lsave(NIL);
				lsave(n);	/* Root of result list */
				n = NIL;
			}
			else {
				a = alloc(NIL, NIL);
				lsave(a);
				lsave(Cdr[n]);
				lsave(a);	/* Root of result list */
				n = Car[n];
			}
			mode = MLIST;
			continue;
		}
		/*
		 * The following loop is used to ASCEND back to the
		 * root of a list, thereby performing BETA reduction
		 * and creating result lists.
		 */
		while (1) if (mode == MBETA || mode == MLETR) {
			/* Finish BETA reduction */
			unbindArgs();
			unsave(1);	/* Original list */
			mode = munsave();
		}
		else if (mode == MLIST) {	/* Append to list, reduce */
			n = Car[Cdr[Lstack]];	/* Next member */
			a = Car[Cdr[Cdr[Lstack]]];	/* Place to append to */
			m2 = lunsave(1);	/* Root of result list */
			if (a != NIL)		/* Append new member */
				Car[a] = m;
			if (n == NIL) {		/* End of list */
				m = m2;
				lunsave(2);	/* Drop N,A */
 				/* Drop original list, remember first element */
				nm = Car[unsave(1)];
				save(m);	/* Save result */
				if (Trace == nm) printTrace(m);
				if (primitive(&m))
					;	/* primitive fn */
				else if (special(&m, &cf, &mode, &cbn))
					n = m;	/* special form */
				else if (!atomic(Car[m]) &&
					Car[m] != NIL &&
					Car[Car[m]] == S_closure
				) {
					/* Application: */
					/*   reduce ((lambda...)...) */
					nm = atomic(nm)? nm: NIL;
					/* If the caller is also */
					/* BETA-reducing, */
					/* this is a TAIL application. */
					tailCall();
					bindArgs(m, nm);
					/* N=S of ((LAMBDA (...) S) ...) */
					n = Car[Cdr[Cdr[Car[m]]]];
					cf = 2;
					mode = MBETA;
				}
				else if (!atomic(Car[m]) &&
					Car[m] != NIL &&
					Car[Car[m]] == S_continuation
				) {
					n = resume(m);
					cbn = 2;
					cf = 1;
				}
				else {
					error("application of non-function",
						nm);
					n = NIL;
				}
				if (cf != 2) {
					unsave(1);	/* Drop result */
					mode = munsave();
				}
				/* Continue this evaluation. */
				/* Leave the ASCENDING loop and descend */
				/* once more into N. */
				if (cf) break;
			}
			else {			/* Append to list */
				lsave(m2);
				/* Create space for next argument */
				Cdr[a] = alloc(NIL, NIL);
				Car[Cdr[Cdr[Lstack]]] = Cdr[a];
				Car[Cdr[Lstack]] = Cdr[n];
				n = Car[n];	/* Evaluate next member */
				break;
			}
		}
		else if (mode == MCOND) {
			n = evalClause(m);
			if (Car[Bstack] == NIL) {
				unsave(1);	/* Original list */
				bunsave(1);
				mode = munsave();
			}
			cf = 1;
			break;
		}
		else if (mode == MCONJ || mode == MDISJ) {
			Car[Bstack] = Cdr[Car[Bstack]];
			if (	(m == S_false && mode == MCONJ) || 
				(m != S_false && mode == MDISJ) ||
				Car[Bstack] == NIL
			) {
				unsave(1);	/* Original list */
				bunsave(1);
				mode = munsave();
				n = m;
				cbn = 1;
			}
			else if (Cdr[Car[Bstack]] == NIL) {
				n = Car[Car[Bstack]];
				unsave(1);	/* Original list */
				bunsave(1);
				mode = munsave();
			}
			else {
				n = Car[Car[Bstack]];
			}
			cf = 1;
			break;
		}
		else if (mode == MBEGN) {
			Car[Bstack] = Cdr[Car[Bstack]];
			if (Cdr[Car[Bstack]] == NIL) {
				n = Car[Car[Bstack]];
				unsave(1);	/* Original list */
				bunsave(1);
				mode = munsave();
			}
			else {
				n = Car[Car[Bstack]];
			}
			cf = 1;
			break;
		}
		else if (mode == MBIND || mode == MBINR) {
			if (nextLet(m) == NIL) {
				n = finishLet(mode == MBINR);
				mode = MLETR;
			}
			else {
				n = evalLet();
			}
			cf = 1;
			break;
		}
		else {	/* Atom */
			break;
		}
		if (cf) {	/* Continue evaluation if requested */
			cf = 0;
			continue;
		}
		if (Stack == Stack0) break;
	}
	while (Stack != Stack0) unsave(1);
	Stack0 = unsave(1);
	Car[Mstack] = unsave(1);
	Bstack = unsave(1);
	Lstack = unsave(1);
	EvLev = EvLev - 1;
	return m;		/* Return the evaluated expr */
}

/* Print lists of digits in condensed format. */
int printNum(int n) {
	char	s[2];

	if (Car[n] != S_number) return 0;
	s[1] = 0;
	n = Cdr[n];
	while (1) {
		if (n == NIL) break;
		s[0] = Car[Car[Car[n]]] & 255;
		pr(s);
		n = Cdr[n];
	}
	return -1;
}

/* Print expressions of the form (QUOTE X) as 'X. */
int printQuote(int n) {
	if (	Car[n] == S_quote &&
		Cdr[n] != NIL &&
		Cdr[Cdr[n]] == NIL
	) {
		n = Car[Cdr[n]];
		if (n != S_true && n != S_false) pr("'");
		_print(n);
		return 1;
	}
	return 0;
}

/* Print a closure. */
int printClosure(int n) {
	if (	Car[n] == S_closure &&
		Cdr[n] != NIL && !atomic(Cdr[n]) &&
		Cdr[Cdr[n]] != NIL && !atomic(Cdr[Cdr[n]])
	) {
		Quoted = 1;
		pr("#<closure ");
		_print(Car[Cdr[n]]);
		if (ClPrLev > 0) {
			pr(" ");
			_print(Car[Cdr[Cdr[n]]]);
			if (ClPrLev > 1 && Cdr[Cdr[Cdr[n]]] != NIL) {
				pr(" ");
				_print(Car[Cdr[Cdr[Cdr[n]]]]);
			}
		}
		pr(">");
		return -1;
	}
	return 0;
}

/* Print a continuation. */
int printCont(int n) {
	if (Car[n] == S_continuation) {
		pr("#<continuation>");
		return -1;
	}
	else {
		return 0;
	}
}

/* Print a character. */
int printChar(int n) {
	char	b[2];
	int	c;

	if (Car[n] != S_char) return 0;
	if (!DisplayMode) pr("#\\");
	c = Car[Cdr[n]];
	if (!DisplayMode && c == ' ') {
		pr("space");
	}
	else if (!DisplayMode && c == '\n') {
		pr("newline");
	}
	else {
		b[1] = 0;
		b[0] = c;
		pr(b);
	}
	return -1;
}

/* Print a string. */
int printString(int n) {
	char	b[2];
	int	k;
	char	*s;

	if (Car[n] != S_string) return 0;
	if (!DisplayMode) pr("\"");
	s = (char *) &Vpool[Car[Cdr[n]]];
	k = Vpool[Car[Cdr[n]] - 1];
	b[1] = 0;
	while (k) {
		b[0] = *s++;
		if (!DisplayMode)
			if (b[0] == '"' || b[0] == '\\')
				pr("\\");
		pr(b);
		k = k-1;
	}
	if (!DisplayMode) pr("\"");
	return -1;
}

/* Print a primitive function. */
int printPrim(int n) {
	if (Car[n] != S_primitive) return 0;
	pr("#<primitive ");
	Quoted = 1;
	_print(Cdr[Cdr[n]]);
	pr(">");
	return -1;
}

/* Print a special form handler. */
int printSpecial(int n) {
	if (Car[n] != S_special && Car[n] != S_special_cbv)
		return 0;
	pr(Car[n] == S_special? "#<special ": "#<special/cbv ");
	Quoted = 1;
	_print(Cdr[Cdr[n]]);
	pr(">");
	return -1;
}

/* Recursively print the term N. */
void _print(int n) {
	char	s[TEXTLEN+1];
	int	i;

	if (n == NIL) {
		if (!Quoted) {
			pr("'");
			Quoted = 1;
		}
		pr("()");
	}
	else if (Tag[n] & AFLAG) {
		/* Characters are limited to the symbol table */
		pr("#<unprintable object>");
	}
	else if (atomic(n)) {
		if (!Quoted) {
			if (	n != S_true &&
				n != S_false &&
				n != S_number &&
				n != S_eof &&
				n != S_void
			)
				pr("'");
			Quoted = 1;
		}
		i = 0;		/* Symbol */
		n = Car[n];
		while (n != NIL) {
			s[i] = Car[n];
			if (i < TEXTLEN-2) i = i+1;
			n = Cdr[n];
		}
		s[i] = 0;
		pr(s);
	}
	else {	/* List */
		if (printNum(n)) return;
		if (printChar(n)) return;
		if (printString(n)) return;
		if (printCont(n)) return;
		if (printClosure(n)) return;
		if (printPrim(n)) return;
		if (printSpecial(n)) return;
		if (!Quoted) {
			pr("'");
			Quoted = 1;
		}
		if (printQuote(n)) return;
		pr("(");
		while (n != NIL) {
			_print(Car[n]);
			n = Cdr[n];
			if (	n != NIL &&
				(atomic(n) || tagged(n))
			) {
				pr(" . ");
				_print(n);
				n = NIL;
			}
			if (n != NIL) pr(" ");
		}
		pr(")");
	}
}

/* Print quoted expression. */
void print(int n) {
	Quoted = 1;
	DisplayMode = 0;
	_print(n);
}

/* Pretty-Print expression. */
void display(int n) {
	Quoted = 1;
	DisplayMode = 1;
	_print(n);
	DisplayMode = 0;
}

/* Reset interpreter state. */
void resetState(void) {
	Stack = NIL;
	Lstack = NIL;
	Bstack = NIL;
	Estack = NIL;
	Frame = NIL;
	Function = NIL;
	EvLev = 0;
	Level = 0;
}

#ifdef SIGNAL
void catchIntr(int sig) {
	error("interrupted", -1);
	ErrFlag = 1;
	signal(SIGINT, catchIntr);
}
#endif

/* Initialize interpreter variables. */
void init1(void) {
	/* Misc. variables */
	NIL = POOLSIZE;
	Level = 0;
	resetState();
	ErrFlag = 0;
	FatalFlag = 0;
	Symbols = NIL;
	Packages = NIL;
	SafeSymbols = NIL;
	Tmp = NIL;
	Tmp2 = NIL;
	LoadLev = 0;
	Trace = NIL;
	Pp = 0;
	MaxAtoms = 0;
	MaxCells = 0;
	StatFlag = 0;
	Ntrace = 10;
	ClPrLev = 0;
	ArrowMode = 0;
	StrictApply = 0;
	Image[0] = 0;
	Line = 1;
	/* Initialize Freelist */
	Free = NIL;
	Vptr = 0;
	/* Clear input buffer */
	Infile = NULL;
	DirName[0] = 0;
	Input = stdin;
	Output = stdout;
	Rejected = EOT;
#ifdef SIGNAL
	signal(SIGINT, catchIntr);
#endif
}

/*
 * Second stage of initialization:
 * protect registers from GC,
 * build the free list,
 * create builtin symbols.
 */
void init2(void) {
	int	core;

	/* Protect base registers */
	Base[0] = &Symbols;
	Base[1] = &Stack;
	Base[2] = &Mstack;
	Base[3] = &Lstack;
	Base[4] = &Bstack;
	Base[5] = &Estack;
	Base[6] = &Tmp;
	Base[7] = &Tmp2;
	Base[8] = &SafeSymbols;
	Base[9] = &Packages;
	/* Create builtin symbols */
	S_0 = addSym("0d", 0); /* First GC will be triggered HERE */
	S_1 = addSym("1d", 0);
	S_2 = addSym("2d", 0);
	S_3 = addSym("3d", 0);
	S_4 = addSym("4d", 0);
	S_5 = addSym("5d", 0);
	S_6 = addSym("6d", 0);
	S_7 = addSym("7d", 0);
	S_8 = addSym("8d", 0);
	S_9 = addSym("9d", 0);
	Digits[0] = S_0;
	Digits[1] = S_1;
	Digits[2] = S_2;
	Digits[3] = S_3;
	Digits[4] = S_4;
	Digits[5] = S_5;
	Digits[6] = S_6;
	Digits[7] = S_7;
	Digits[8] = S_8;
	Digits[9] = S_9;
	/* Tags (especially #<primitive> and #<special*>) */
	/* must be defined before any primitives. */
	S_special = addSym("#<special>", 0);
	S_special_cbv = addSym("#<special/cbv>", 0);
	S_primitive = addSym("#<primitive>", 0);
	S_char = addSym("#<char>", 0);
	S_closure = addSym("#<closure>", 0);
	S_continuation = addSym("#<continuation>", 0);
	S_eof = addSym("#<eof>", 0);
	S_number = addSym("#<number>", 0);
	S_string = addSym("#<string>", 0);
	S_void = addSym("#<void>", 0);
	addSpecial("and", SF_AND, 0);
	addSpecial("apply", SF_APPLY, 1);
	addSpecial("begin", SF_BEGIN, 0);
	S_bottom = addPrim("bottom", P_BOTTOM);
	addSpecial("call-with-current-continuation", SF_CALLCC, 1);
	addSpecial("call/cc", SF_CALLCC, 1);
	addPrim("car", P_CAR);
	addPrim("cdr", P_CDR);
	addPrim("char->integer", P_CHAR_TO_INTEGER);
	addPrim("char-ci=?", P_CHAR_CI_EQP);
	addPrim("char-ci<?", P_CHAR_CI_LTP);
	addPrim("char=?", P_CHAR_EQP);
	addPrim("char<?", P_CHAR_LTP);
	addPrim("char?", P_CHARP);
	addSpecial("cond", SF_COND, 0);
	addPrim("cons", P_CONS);
	addSpecial("define", SF_DEFINE, 0);
	addPrim("delete-file", P_DELETE_FILE);
	addPrim("display", P_DISPLAY);
	addPrim("eof-object?", P_EOF_OBJECTP);
	addPrim("eq?", P_EQP);
	S_false = addSym("#f", 0);
	addPrim("integer->char", P_INTEGER_TO_CHAR);
	addPrim("integer->list", P_INTEGER_TO_LIST);
	S_lambda = addSpecial("lambda", SF_LAMBDA, 0);
	addSpecial("let", SF_LET, 0);
	addSpecial("letrec", SF_LETREC, 0);
	addPrim("list->integer", P_LIST_TO_INTEGER);
	addPrim("list->string", P_LIST_TO_STRING);
	addPrim("n+", P_NPLUS);
	addPrim("n-", P_NMINUS);
	addPrim("n<", P_NLESS);
	addPrim("null?", P_NULLP);
	addPrim("number?", P_NUMBERP);
	addSpecial("or", SF_OR, 0);
	addPrim("package", P_PACKAGE);
	addPrim("pair?", P_PAIRP);
	addPrim("procedure?", P_PROCEDUREP);
	S_quote = addSpecial("quote", SF_QUOTE, 0);
	addSpecial("with-input-from-file", SF_WITH_INPUT_FROM_FILE, 1);
	addSpecial("with-output-to-file", SF_WITH_OUTPUT_TO_FILE, 1);
	addPrim("procedure?", P_PROCEDUREP);
	addPrim("read", P_READ);
	addPrim("read-char", P_READ_CHAR);
	addPrim("recursive-bind", P_RECURSIVE_BIND);
	addPrim("string->symbol", P_STRING_TO_SYMBOL);
	addPrim("string->list", P_STRING_TO_LIST);
	addPrim("string-append", P_STRING_APPEND);
	addPrim("string-length", P_STRING_LENGTH);
	addPrim("string-ref", P_STRING_REF);
	addPrim("string?", P_STRINGP);
	addPrim("substring", P_SUBSTRING);
	addPrim("symbol->string", P_SYMBOL_TO_STRING);
	addPrim("symbol?", P_SYMBOLP);
	addPrim("void", P_VOID);
	addPrim("write", P_WRITE);
	S_true = addSym("#t", 0);
	S_last = addSym("**", 0);
	Mstack = alloc(NIL, NIL);
	Primitives[P_BOTTOM] = &doBottom;
	Primitives[P_CAR] = &doCar;
	Primitives[P_CDR] = &doCdr;
	Primitives[P_CHAR_TO_INTEGER] = &doCharToInteger;
	Primitives[P_CHAR_CI_EQP] = &doCharCiEqP;
	Primitives[P_CHAR_CI_LTP] = &doCharCiLtP;
	Primitives[P_CHAR_EQP] = &doCharEqP;
	Primitives[P_CHAR_LTP] = &doCharLtP;
	Primitives[P_CHARP] = &doCharP;
	Primitives[P_CONS] = &doCons;
	Primitives[P_DELETE_FILE] = &doDeleteFile;
	Primitives[P_DISPLAY] = &doDisplay;
	Primitives[P_EOF_OBJECTP] = &doEofObjectP;
	Primitives[P_EQP] = &doEqP;
	Primitives[P_INTEGER_TO_CHAR] = &doIntegerToChar;
	Primitives[P_INTEGER_TO_LIST] = &doIntegerToList;
	Primitives[P_LIST_TO_INTEGER] = &doListToInteger;
	Primitives[P_LIST_TO_STRING] = &doListToString;
	Primitives[P_NPLUS] = &doNPlus;
	Primitives[P_NMINUS] = &doNMinus;
	Primitives[P_NLESS] = &doNLess;
	Primitives[P_NULLP] = &doNullP;
	Primitives[P_NUMBERP] = &doNumberP;
	Primitives[P_PACKAGE] = &doPackage;
	Primitives[P_PAIRP] = &doPairP;
	Primitives[P_PROCEDUREP] = &doProcedureP;
	Primitives[P_READ] = &doRead;
	Primitives[P_READ_CHAR] = &doReadChar;
	Primitives[P_RECURSIVE_BIND] = &doRecursiveBind;
	Primitives[P_STRING_TO_LIST] = &doStringToList;
	Primitives[P_STRING_TO_SYMBOL] = &doStringToSymbol;
	Primitives[P_STRING_APPEND] = &doStringAppend;
	Primitives[P_STRING_LENGTH] = &doStringLength;
	Primitives[P_STRING_REF] = &doStringRef;
	Primitives[P_STRINGP] = &doStringP;
	Primitives[P_SUBSTRING] = &doSubstring;
	Primitives[P_SYMBOL_TO_STRING] = &doSymbolToString;
	Primitives[P_SYMBOLP] = &doSymbolP;
	Primitives[P_VOID] = &doVoid;
	Primitives[P_WRITE] = &doWrite;
	Specials[SF_AND] = &doAnd;
	Specials[SF_APPLY] = &doApply;
	Specials[SF_BEGIN] = &doBegin;
	Specials[SF_CALLCC] = &doCallCC;
	Specials[SF_COND] = &doCond;
	Specials[SF_DEFINE] = &doDefine;
	Specials[SF_LAMBDA] = &doLambda;
	Specials[SF_LET] = &doLet;
	Specials[SF_LETREC] = &doLetrec;
	Specials[SF_OR] = &doOr;
	Specials[SF_QUOTE] = &doQuote;
	Specials[SF_WITH_INPUT_FROM_FILE] = &doWithInputFromFile;
	Specials[SF_WITH_OUTPUT_TO_FILE] = &doWithOutputToFile;
	core = addSym("core", 0);
	Packages = alloc(core, Symbols);
	Packages = alloc(Packages, NIL);
	Symbols = addPackage(NIL);
}

/* Reset the reduction counter */
void clearStats(void) {
	reset(&Reductions);
	reset(&Allocations);
}

/* Print the number of reductions done in last EVAL */
void printStats(void) {
	printValue(&Reductions, "reduction steps");
	printValue(&Allocations, "nodes allocated");
}

/* This is the top-level Read-Eval-Print Loop. */
void repl(void) {
	int	n;

	while (1) {
		ErrFlag = 0;
		n = xread();
		if (n == S_eof) break;
		save(n);
		SafeSymbols = copyBindings();
		if (StatFlag) clearStats();
		n = eval(Car[Stack]);
		unsave(1);
		if (!ErrFlag) {
			Cdr[S_last] = n;
			if (!LoadLev) {
				if (!O_batch) pr("=> ");
				print(n); nl();
			}
			if (Stack != NIL)
				fatal("internal: unbalanced stack");
			if (StatFlag && !LoadLev) printStats();
		}
		else {
			restoreBindings(SafeSymbols);
		}
		resetState();
		while (Car[Mstack] != NIL) munsave();
	}
}

/* Load initial node pool image */
void loadImage(char *p) {
	int	fd, n, i;
	char	buf[17];
	int	**v;
	int	bad = 0;

	fd = open(p, O_RDONLY);
	setmode(fd, O_BINARY);
	if (fd < 0) {
		error("cannot open image", -1);
		pr("* "); pr(p); nl();
		exit(1);
	}
	read(fd, buf, 16);
	read(fd, &n, sizeof(int));
	if (memcmp(buf, "SKETCHY", 7)) {
		error("bad image (magic match failed)", -1);
		bad = 1;
	}
	if (buf[7] != sizeof(int)) {
		error("bad image (wrong cell size)", -1);
		bad = 1;
	}
	if (buf[8] != MAJORVERSION) {
		error("bad image (wrong version)", -1);
		bad = 1;
	}
	if (n != POOLSIZE) {
		error("bad image (wrong pool size)", -1);
		bad = 1;
	}
	memcpy(&n, &buf[10], sizeof(int));
	if (n != 0x12345678) {
		error("bad image (wrong architecture)", -1);
		bad = 1;
	}
	v = ImageVars;
	i = 0;
	while (v[i]) {
		read(fd, v[i], sizeof(int));
		i = i+1;
	}
	if (	read(fd, Car, POOLSIZE*sizeof(int)) != POOLSIZE*sizeof(int) ||
		read(fd, Cdr, POOLSIZE*sizeof(int)) != POOLSIZE*sizeof(int) ||
		read(fd, Tag, POOLSIZE) != POOLSIZE ||
		read(fd, Vpool, VPOOLSIZE*sizeof(int)) !=
			VPOOLSIZE*sizeof(int)
	) {
		error("bad image (bad file size)", -1);
		bad = 1;
	}
	close(fd);
	if (bad) {
		pr("* "); pr(p); nl();
		exit(1);
	}
}

/* Print usage and fail. */
int usage(void) {
	error("Usage: sketchy [-b] [image]", -1);
	exit(1);
}

/* Evaluate the command line options */
void getOpts(int argc, char **argv) {
	char	*a;
	int	i, j, k;

	O_batch = 0;
	i = 1;
	while (i < argc) {
		k = strlen(argv[i]);
		a = argv[i];
		if (a[0] != '-') break;
		for (j=1; j<k; j++)
			if (a[j] == 'b' || a[j] == 'B') O_batch = 1;
			else usage();
		i = i+1;
	}
	if (i < argc) strcpy(Image, a);
}

/*
 * Here the fun begins.
 */
int main(int argc, char **argv) {
	init1();
	getOpts(argc, argv);
	init2();
	if (!O_batch) {
		pr("Sketchy ");
		pr(RELEASE);
		pr(" (C) 2006 Nils M Holm"); nl();
	}
	if (Image[0]) {
		loadImage(Image);
	}
	else if (!O_batch) {
		pr("Warning: no image loaded"); nl();
	}
	repl();
	bye(NULL);
	return 0;
}

