; Sketchy Library
; Copyright (C) 2005 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---name---
; string->number

; ---conformance---
; R5RS
; Takes only integer arguments.

; ---purpose---
; Convert a string to a number. Radix must be in the
; range 2..16. If no radix is given, 10 is assumed.
; If the given string does not hold the representation
; of a valid integer, #F is returned.

; ---args---
; STR - string
; RADIX - optional radix

; ---keywords---
; STRING->NUMBER function, string, integer, number, conversion

; ---see-also---
; number->string

; ---example---
; (string->number "-7777" 8) => -4095

(define strtonum #t)

:require plus.l +
:require times.l *
:require less.l <
:require c-downcase.l

; ---code---
(define (string->number str . radix)
  (letrec

    ((digits (string->list "0123456789abcdef"))

    ; Find the value of a digit.
    (value (lambda (x)
      (letrec
        ((v (lambda (x d n)
          (cond ((null? d) 17)
            ((char=? (car d) x) n)
            (#t (v x (cdr d) (+ n 1)))))))
        (v (char-downcase x) digits 0))))

    ; Convert an unsigned number
    (conv (lambda (lst res rdx)
      (cond ((null? lst) res)
        (#t (let ((dval (value (car lst))))
              (cond ((< dval rdx)
                  (conv (cdr lst)
                    (+ (value (car lst))
                      (* res rdx))
                    rdx))
                (#t #f)))))))

    ; Convert a signed number
    (sconv (lambda (lst res rdx)
      (cond ((null? lst) #f)
        ((char=? (car lst) #\+)
          (conv (cdr lst) res rdx))
        ((char=? (car lst) #\-)
          (- (conv (cdr lst) res rdx)))
        (#t (conv lst res rdx)))))

    ; Get the radix
    (get-radix (lambda ()
      (cond ((null? radix) 10)
        ((< 1 (car radix) 17) (car radix))
        (#t (bottom
              '(bad radix in string->number)))))))

    (sconv (string->list str) 0 (get-radix))))

