; Sketchy Library
; Copyright (C) 2005 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---name---
; sqrt

; ---conformance---
; R5RS
; Result is an integer.

; ---purpose---
; Compute the square root of a natural number.
; (In fact, this function computes the greatest
; integer that is no greater than the square
; root of the given argument.)

; ---args---
; X - square of root to extract

; ---keywords---
; SQRT function, root, square, square root, arithmetics

; ---see-also---
; digits, expt, gcd

; ---example---
; (sqrt 144) => 12

(define sqrt #t)

:require list.l
:require equal.l =
:require negativep.l negative?
:require ntimes.l n*
:require nquotient.l
:require ngreater.l n>
:require nlteq.l n<=

; ---model---
; (define (sqrt square)
;   (letrec
;     ((sqr (lambda (x last)
;         (cond ((and (n<= (n* x x) square)
;               (n> (n* (n+ x 1) (n+ x 1)) square))
;             x)
;           (#t (sqr (nquotient
;                      (n+ x (nquotient square x))
;                      2)
;                    x))))))
;       (sqr square 0)))

; ---code---
(define (sqrt square)
  (letrec
    ((sqr (lambda (x last)
      (cond ((= last x) x)
        ((= last (n+ x 1))
          (cond ((n> (n* x x) square) (- x 1))
            (#t x)))
        (#t (sqr (nquotient
                   (n+ x (nquotient square x))
                   2)
                 x))))))
    (cond ((negative? square)
        (bottom (list 'sqrt square)))
      (#t (sqr square 0)))))

