; Sketchy Library
; Copyright (C) 2005 Nils M Holm. All rights reserved.
; See the file LICENSE of the Sketchy distribution
; for conditions of use.

; ---name---
; assq

; ---conformance---
; R5RS

; ---purpose---
; Retrieve a binding from an association list.
; An association list is a list of pairs where
; the CAR part of each pair holds a key and
; the CDR part of the pair holds the value
; associated with that key:
; -
; ((key1.value1) ... (keyN.valueN))
; -
; Unlike ASSOC, ASSQ uses EQ? to identify keys.
; Therefore, keys are limited to symbols when
; using ASSQ.

; ---args---
; X - key of value to be found
; A - association list

; ---keywords---
; ASSQ function, association lists, alists, associate

; ---see-also---
; assoc, memq

; ---example---
; (assq 'c '((a.i) (b.ii) (c.iii) (d.iv))) => (c . iii)

(define assq #t)

:require caar.l cdar

; ---model---
; (define (assq x a)
;   (cond ((null? a) #f)
;     ((eq? (caar a) x) (car a))
;     (#t (assq x (cdr a)))))

; ---code---
(define (assq x a)
  (letrec
    ((_assq (lambda (a)
      (cond ((null? a) #f)
        ((eq? (caar a) x) (car a))
        (#t (_assq (cdr a)))))))
    (_assq a)))

