#! /usr/bin/env python
# (c) 1999 - AccesSoft - See license.txt for details
# $Id: qsplit.py,v 1.1 2004/02/12 20:20:18 daveron Exp $

"""qsplit module for splitting quotes in MOOP

Exports:
qsplit(string) : splits a string into whitespace separated elements into a list of strings, without
                    breaking apart any single or double quoted elements that contain whitespace themselves
qsplit_list(list) : splits a string containing a comma-separated list into a list of strings, without
                    breaking apart any single or double quoted elements that contain commas themselves

>>> from qsplit import qsplit, qsplit_list
>>> s = 'This is "a quoted" string '
>>> print qsplit(s)
['This', 'is', '"a quoted"', 'string']
>>> s = 'This, is, "a, comma", separated string'
>>> print qsplit_list(s)
['This', 'is', '"a, comma"', 'separated string']
    
"""


#import regex
import re

r = re.compile("""\s*               # Skip leading whitespace
               (                    # Three alternative token styles
                [^\s"\'][^\s]*       # A non-quotemark or WS
                                    # character, up to the next
                                    # whitespace
                 |                  #   or
                "[^"]*"             # A double quote, followed
                                    # by non-quotes, ending with
                                    # another double quote
                  |                 #   or
                '[^']*'             # A single quote, followed
                                    # by non-quotes, ending with
                                    # another single quote
                )""",               # End of the alternatives
               re.VERBOSE)

rl = re.compile("""\s*               # Skip leading whitespace
               (                    # Three alternative token styles
                [^\s"\'][^,]*       # A non-quotemark or WS
                                    # character, up to the next
                                    # comma
                 |                  #   or
                "[^"]*"             # A double quote, followed
                                    # by non-quotes, ending with
                                    # another double quote
                  |                 #   or
                '[^']*'             # A single quote, followed
                                    # by non-quotes, ending with
                                    # another single quote
                )""",               # End of the alternatives
               re.VERBOSE)
               

def qsplit(s):
    lst=[]
    pos=0
    while (s != ""):
        res = r.match(s, pos)
        if res == None: break
        lst.append(res.group(1))
        pos=res.regs[0][1]
    return lst

def qsplit_list(s):
    lst=[] ; pos=0
    while (s != "" and pos < len(s)):
        res = rl.match(s, pos)
        if not res:
            break
        lst.append(res.group(1))
        pos = res.regs[0][1]+1
    return lst

def _test(verbose=0):
    import doctest
    import qsplit
    return doctest.testmod(qsplit, verbose=verbose)

if __name__ == '__main__':
    _test(0)
