#!/bin/sh
#
# Common Items for test scripts t-xxxx.sh
#
# Copyright (c) 2022 ... 2025 2026
#     John McCue <jmccue@sdf.org>
#
# Permission to use, copy, modify, and distribute this software
# for any purpose with or without fee is hereby granted, provided
# that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
# WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
# CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

if test "$SDF" != "YES"
then
    set -o pipefail
fi

OS=`uname -s`
HOST=`uname -n | awk -F '.' '{print $1}'`
ARCH=`uname -m`
export OS HOST MACHINE
if test "$USER" = ""
then
    USER=`id -un`
    export USER
fi

#--- Find Source Directory, based upon test script name
TEST_SCRIPT="$0"
TEST_HDIR="`dirname $TEST_SCRIPT`"
cd $TEST_HDIR
TEST_DIR_FULL="`pwd`"
TEST_DIR_NAME="`basename $TEST_DIR_FULL`"
cd ..
TEST_DIR_PARENT="`pwd`"
cd $TEST_HDIR
TEST_DIR_SRC=$TEST_DIR_FULL
export TEST_SCRIPT TEST_HDIR TEST_DIR_NAME TEST_DIR_PARENT TEST_DIR_SRC

cat << EOF
INFO========================================================================
PWD              `pwd`
TEST_SCRIPT      $TEST_SCRIPT

TEST_DIR_FULL    $TEST_DIR_FULL
TEST_DIR_NAME    $TEST_DIR_NAME
TEST_DIR_PARENT  $TEST_DIR_PARENT
TEST_DIR_SRC     $TEST_DIR_SRC
INFO========================================================================
EOF

#--- Set results Dir
if test -d /mnt/mfs/$USER
then
    TMPDIR=/mnt/mfs/$USER
    export TMPDIR
else
    if test "$TMPDIR" = ""
    then
	if test -d /mnt/tmpfs/$USER
	then
	    TMPDIR=/mnt/tmpfs/$USER
	else
	    TMPDIR=/tmp/$USER
	fi
	export TMPDIR
    fi
fi

DIR_RESULTS=$TMPDIR/$HOST/$TEST_DIR_NAME
export DIR_RESULTS

#
# f_msg() - print a message
#
f_msg()
{
    l_lm_msg="$1"
    l_lm_tmsg=`echo "$l_lm_msg" | cut -c 1-1`
    l_lm_now="`date '+%H:%M:%S'`"

    case "$l_lm_tmsg" in
	"W")
	    echo "$l_lm_now - WARN  $l_lm_msg"
	    ;;
	"S")
	    echo "$l_lm_now - SUCCESS $l_lm_msg"
	    ;;
	"I")
	    echo "$l_lm_now - INFO  $l_lm_msg"
	    ;;
	"A")
	    echo '######################################################################'
	    echo "START $l_lm_now - INFO  $l_lm_msg"
	    echo '======================================================================'
	    ;;
	"B")
	    echo '======================================================================'
	    echo "DONE  $l_lm_now - INFO  $l_lm_msg"
	    echo '######################################################################'
	    ;;
	"E")
	    echo "$l_lm_now - ERROR $l_lm_msg"
	    exit 2
	    ;;
	*)
	    echo "$l_lm_now - ABORT $l_lm_msg"
	    exit 2
	    ;;
    esac

} # END: f_msg()

f_help_setup()
{
    l_help_show="Y"
    
#--- Object specific help
    case "$TEST_DIR_NAME" in
	"jfile")
cat << EOF

Test Options for jfile.

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -a      Show all actual Character Details
    -n      Show '*' for non ASCII characters

EOF
	    ;;
	"jsplit")
cat << EOF

Test Options for jsplit.

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -b      Split the file using Binary Mode
    -t      Split the file using Text Mode

EOF
	    ;;
	"jcat")
cat << EOF

Optional Test Option for jcat
Default is to use 2 small files.

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)

    -s      Test using a two small files
            This is the default if no arguments are used
            This test will execute for a very long time

    -L      Test using a very large third file.
            This test will execute for a very long time

EOF
	    ;;
	"jadd")
#NOT YET SUPPORTED#-dc     Use Delimited  Input with Decimal Point of a comma
#NOT YET SUPPORTED#-fc     Use Flat Fixed Input with Decimal Point of a comma
cat << EOF

Test Options for jadd

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -dp     Use Delimited  Input with Decimal Point of a period
    -fp     Use Flat Fixed Input with Decimal Point of a period

EOF
	    ;;
	"jmerge")
cat << EOF

Test Options for jmerge

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -s      Process Sorted Input Files
    -r      Process Random Ordered Input Files

EOF
	    ;;
	"jr")
cat << EOF

Test Options for jr

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)

    -d      Change a Delimited File into a Flat Fixed Length File
            and reformat Dates from MM/DD/YYYY to YYYYMMDD

    -L      Change a Large Delimited File into a Flat 
            Fixed Length File

    -s      Change a Small Delimited File with embedded spaces 
            into a Flat Fixed Length File

    -MM     Change a Flat Fixed Length File into a Delimited File 
            and reformat from YYYYMMDD Dates to MM/DD/YYYY

    -DD     Change a Flat Fixed Length File into a Delimited File
            and reformat from YYYYMMDD Dates to DD-MM-YYYY

EOF
	    ;;
	"jfilt")
cat << EOF

Test Options for jfilt

    Option  Meaning
    ------  ----------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -7      Write only 7-bit ASCII
    -8      Write UTF-8

EOF
	    ;;
	"jcsv")
cat << EOF

Test Options for jcsv, convert to/from CSV (Comma Separated Value File)

    Option  Meaning
    ------  ---------------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)
    -c      Input: Text to CSV, verbose -vvvv, Cols 2 & 3 forced to Text
    t       Input: Text to CSV, verbose -v
    t0      Input: Text to CSV, verbose -vvvv, Col 1 forced to Text
    t2      Input: Text to CSV, verbose -vvv
    tm      Input: Text to CSV, verbose -v,   Max Col Length 15
    tm2     Input: Text to CSV, verbose -vvv, Max Col Length 15
    t1      Input: CSV to Text, verbose -vvv
    tm1     Input: CSV to Text, verbose -vvv, Max Col Length 15
    regular Input: Airport File, verbose -vvvv, remove non-7bit ASCII
    ta      Input: Airport File, verbose -vvv,  remove non-7bit ASCII
    tma     Input: Airport File, verbose -vvv,  remove non-7bit ASCII
                   Max Field Length 40
    tmaa    Input: Airport File, verbose -vvv,  remove non-7bit ASCII
                   Max Field Length 30

    Option '-c' is the one I use the most

EOF
	    ;;
	"jgrep")
cat << EOF

Test Options for jgrep

    Option  Meaning
    ------  ---------------------------------------------------------------
    -h      Show Test Help and exit (also --help -H --HELP H HELP)

    T01     Search for 1 specific entry
    T02     Inverse Search for 1 specific entry
    T03     Search for entries from a file
    T04     Reverse Search for entries from a file

    T11     Search for 1 specific entry            in Delimited Column 2
    T12     Search for entries from a file         in Delimited Column 2
    T13     Reverse Search for 1 specific entry    in Delimited Column 2
    T14     Reverse Search for entries from a file in Delimited Column 2

    T21     Search for 1 specific entry         at char # 4
    T22     Search for entries from a file      at char # 4
    T23     Reverse Search for 1 specific entry at char # 4
    T24     Search for entries from a file      at char # 4

EOF
    esac

    if test "$g_is_error" = "Y"
    then
	return
    fi
cat << EOF
---------------------------------------------------------------------

You can use these optional arguments to generate a Makefile
prior to testing:

   YES_J    : Link to    j_lib2             (default)
   NO_J     : do NOT use j_lib2
   YES_MEM  : Test for Memory Issues
   NO_MEM   : Do NOT Test for Memory Issues (default)
   SKIP     : skip compile step, if possible

EOF
#--- leave
    exit 2

} # END: f_help_setup()

f_ckfile()
{
    l_ckf_mode="$1"
    l_ckf_file="$2"

    if test ! -f "$l_ckf_file"
    then
	f_msg "Es10: missing File $l_ckf_file"
    fi
    if test ! -r "$l_ckf_file"
    then
	f_msg "Es11: no read File $l_ckf_file"
    fi
    if test "$l_ckf_mode" = "w"
    then
	if test ! -w "$l_ckf_file"
	then
	    f_msg "Es12: no write File $l_ckf_file"
	fi
    fi
} # END: f_ckfile()
f_ckdir()
{
    l_ckd_mode="$1"
    l_ckd_dir="$2"

    if test "$l_ckd_mode" = "x"
    then
	l_ckd_mode="w"
	if test ! -d "$l_ckd_dir"
	then
	    mkdir -p "$l_ckd_dir" && chmod 700 "$l_ckd_dir"
	fi
    fi

    if test ! -d "$l_ckd_dir"
    then
	f_msg "Es13: missing Dir $l_ckd_dir"
    fi
    if test ! -r "$l_ckd_dir"
    then
	f_msg "Es14: no read Dir $l_ckd_dir"
    fi
    if test ! -x "$l_ckd_dir"
    then
	f_msg "Es15: no exec Dir $l_ckd_dir"
    fi
    if test "$l_ckd_mode" = "w"
    then
	if test ! -w "$l_ckd_dir"
	then
	    f_msg "Es16: no write Dir $l_ckd_dir"
	fi
    fi
} # END: f_ckdir()
f_ckprog()
{
    l_ckp_prog="$1"

    type "$l_ckp_prog" > /dev/null 2>&1
    if test "$?" -ne "0"
    then
	f_msg "Es17: missing prog $l_ckp_prog"
    fi
} # END: f_ckprog()

f_rmfile()
{
    l_rmf_file="$1"

    if test -f "$l_rmf_file"
    then
	if test -w "$l_rmf_file"
	then
	    rm "$l_rmf_file"
	    if test -f "$l_rmf_file"
	    then
		f_msg "Es18: FAILED to delete $l_rmf_file"
	    fi
	else
	    f_msg "Es19: no access to delete $l_rmf_file"
	fi
    fi

} # END: f_rmfile()

#
# main
#

FILE_MAKE_RESULTS="$DIR_RESULTS/make_results.txt"
FILE_VALGRIND_LOG="$DIR_RESULTS/valgrind.txt"
FILE_BUILD_RESULTS="$DIR_RESULTS/build_results.txt"
FILE_VERSION_RESULTS="$DIR_RESULTS/version_results.txt"
ARG=""
TEST_ARG=""
COMPILE="Y"
export FILE_MAKE_RESULTS FILE_VALGRIND_LOG FILE_BUILD_RESULTS
export FILE_VERSION_RESULTS ARG TEST_ARG COMPILE

#--- set source and work directories
if test -d "$TEST_DIR_FULL"
then
    cd $TEST_DIR_FULL
else
    f_msg "Ws00: $TEST_SCRIPT - cannot find Source Dir:"
    f_msg "Es01:    $TEST_DIR_FULL"
    exit 2
fi

#--- checks
f_ckdir  "x" "$TMPDIR"
f_ckdir  "x" "$DIR_RESULTS"

type valgrind > /dev/null 2>&1
if test "$?" -eq "0"
then
    HAVE_VALGRIND="Y"
else
    HAVE_VALGRIND="N"
fi
PROCESS_VALGRIND="N"
export HAVE_VALGRIND PROCESS_VALGRIND

if test "$SKIP_MESSAGES" != "Y"
then
    f_msg "Is02: Src Dir:        $TEST_DIR_FULL"
    f_msg "Is03: Work Dir:       $DIR_RESULTS"
    f_msg "Is04: Found valgrind: $HAVE_VALGRIND"
fi

#--- generate Makefile
for ARG in $@
do
    case $ARG in
	"SKIP")
	    if test -f Makefile
	    then
		COMPILE="N"
		export COMPILE
	    else
		f_msg "Ws11: Ignoring Option 'SKIP', Makefile does not exist"
	    fi
	    ;;
	"YES_J")
	    BYPASS_JLIB2="NO"
	    export BYPASS_JLIB2
	    ;;
	"NO_J")
	    BYPASS_JLIB2="YES"
	    export BYPASS_JLIB2
	    ;;
	"YES_MEM")
	    MEMORY_TEST="YES"
	    export MEMORY_TEST
	    ;;
	"NO_MEM")
	    MEMORY_TEST="NO"
	    export MEMORY_TEST
	    ;;
	"-h"|"h"|"-H"|"H"|"--help"|"help"|"HELP")
	    f_help_setup
	    ;;
	*)
	    TEST_ARG="$ARG"
	    export TEST_ARG
	    ;;
    esac
done

#--- find a shuffle program, needed to create some test data
type jshuffle > /dev/null 2>&1
if test "$?" -eq "0"
then
    PROG_SHUFFLE="jshuffle"
else
    case "$OS" in
	"Linux")
	    type shuf > /dev/null 2>&1
	    if test "$?" -eq "0"
	    then
		PROG_SHUFFLE="shuf"
	    else
		PROG_SHUFFLE="sort -R"
	    fi
	    ;;
	"OpenBSD")
	    PROG_SHUFFLE="sort -r"
	    ;;
	"NetBSD")
	    type shuffle > /dev/null 2>&1
	    if test "$?" -eq "0"
	    then
		PROG_SHUFFLE="shuffle -"
	    else
		PROG_SHUFFLE="sort"
	    fi
	    ;;
	*)
	    f_msg "Es20: $OS not 100% sopported, random sort disabled"
	    if test "$?" -eq "0"
	    then
		PROG_SHUFFLE="shuf"
	    else
		type shuffle > /dev/null 2>&1
		if test "$?" -eq "0"
		then
		    PROG_SHUFFLE="shuffle -"
		else
		    PROG_SHUFFLE="sort"
		fi
	    fi
	    ;;
    esac
fi
export PROG_SHUFFLE

#--- generate Makefile
if test "$COMPILE" = "Y"
then
    f_msg "Is11: Creating Makefile"
    if test -f Makefile
    then
	make clean > /dev/null 2>&1
    fi
    ./build.sh > $FILE_BUILD_RESULTS 2>&1
    if test "$?" -ne "0"
    then
	f_msg "Es12: build failed, see $FILE_BUILD_RESULTS"
    fi
    rm $FILE_BUILD_RESULTS > /dev/null 2>&1
fi

#--- Linux Memory Test Settings
if test "$OS" = "Linux"
then
    if test "$HAVE_VALGRIND" = "Y"
    then
	grep '^CFLAGS=' < Makefile | grep ' -g ' > /dev/null 2>&1
	if test "$?" -eq "0"
	then
	    PROCESS_VALGRIND="Y"
	    MSG_MEMORY="-- valgrind ENABLED"
	else
	    PROCESS_VALGRIND="N"
	    MSG_MEMORY="-- valgrind DISABLED"
	fi
	export PROCESS_VALGRIND MSG_MEMORY
    fi
fi

#--- OpenBSD Memory Test Settings
if test "$OS" = "OpenBSD"
then
    grep '^CFLAGS=' < Makefile | grep ' -g ' > /dev/null 2>&1
    if test "$?" -eq "0"
    then
	MSG_MEMORY="-- testing memory"
	OPENBSD_KTRACE="ktrace -tu"
	if test "$ARCH" = "i386"
	then
	    MALLOC_OPTIONS="D"
	else
	    MALLOC_OPTIONS="4"  # was D, see ~/Docs/System/OpenBSD/HOWTOs/mem_trace.txt
	fi
	export MALLOC_OPTIONS OPENBSD_KTRACE MSG_MEMORY
	f_msg "Is21: using ktrace malloc option $MALLOC_OPTIONS"
    else
	unset MALLOC_OPTIONS OPENBSD_KTRACE MSG_MEMORY
    fi
fi

#--- generate binary
if test "$COMPILE" = "Y"
then
    rm "$FILE_MAKE_RESULTS" > /dev/null 2>&1
    f_msg "Is31: Compiling"
    make -f Makefile >> $FILE_MAKE_RESULTS 2>&1
    if test "$?" -ne "0"
    then
	f_msg "Es32: FAILED: make, see $FILE_MAKE_RESULTS"
    fi
    if test "$TERM" != "dumb"
    then
	rm TAGS > /dev/null 2>&1
    fi
fi

### DONE
