/*
 * Copyright (c) 2021 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _WIN32
#ifndef _MSDOS
#include <sys/param.h>
#endif
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "rtave.h"

#define SCKARG 80
#define MSG_HELP_11 "Print Average Time Difference"

/*
 * verify_path() -- make sure dir/file name is OK
 */
int verify_path(char *fname)
{
  char *s;

  if (fname == (char *) NULL)
    return(PATH_IS_NULL);
  if (strlen(fname) < 1)
    return(PATH_IS_EMPTY);
  if (strlen(fname) > (PATH_MAX - 2))
    return(PATH_IS_BIG);
  if ((strlen(fname) > 1) && (*fname == '-'))
    return(PATH_INVALID);

  for (s = fname; (*s) != JLIB2_CHAR_NULL; s++)
    {
      if (isspace((int) (*s)))
	return(PATH_HAS_SPACE);
    }

  return(PATH_IS_VALID);

} /* verify_path() */

/*
 * check_path_file() -- Check and Save a Dir Name
 */
void check_path_file(FILE *efp, char *fname, char arg_switch)
{

  char sw = SWITCH_CHAR;
  char ag = arg_switch;

  if (arg_switch == JLIB2_CHAR_NULL)
    {
      sw = 'i';
      ag = 'n';
    }

  switch (verify_path(fname))
    {
      case PATH_IS_VALID:
	break;
      case PATH_IS_NULL:
	fprintf(efp, MSG_ERR_E004S, LIT_NULL,  sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_EMPTY:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_BIG:
	fprintf(efp, MSG_ERR_E004SB, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,   PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_HAS_SPACE:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_INVALID:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      default:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
    }

} /* check_path_file() */

/*
 * init_file() -- clear s_file
 */
void init_file(struct s_file_info *f)

{
  memset(f->fname, JLIB2_CHAR_NULL, ((PATH_MAX + 1)));
  f->fp          = (FILE *) NULL;
  f->ok_to_close = FALSE;
}

/*
 * open_in() -- open in file
 */
int open_in(struct s_file_info *err, struct s_file_info *ifile, char *fname)

{

  if (fname == (char *) NULL)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strlen(fname) < 1)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strncmp(fname, FILE_NAME_STD, PATH_MAX) == 0)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (ifile->ok_to_close == TRUE) /* file in use */
    {
      fprintf(err->fp, MSG_ERR_E112, fname);
      fprintf(err->fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  ifile->fp = fopen(fname, "r");

  if (ifile->fp == (FILE *) NULL)
    {
      fprintf(err->fp, MSG_WARN_W002, fname, strerror(errno));
      return(FALSE);
    }
 
  strncpy(ifile->fname, fname, PATH_MAX);
  ifile->ok_to_close = TRUE;
  return(TRUE);

} /* open_in() */

/*
 * open_out() -- save the file name and check status
 */
void open_out(struct s_file_info *f, char *fname, int force)

{

  if (fname == (char *) NULL)
    return;
  if (strlen(fname) < 1)
    return;
  if (strcmp(fname, FILE_NAME_STD) == 0)
    return;
  if (f->ok_to_close == TRUE) /* file in use */
    {
      fprintf(stderr, MSG_ERR_E112, fname);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  if (force == FALSE)
    {
      if ( j2_f_exist(fname) )
	{
	  fprintf(stderr, MSG_ERR_E025, fname);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  f->fp = fopen(fname, "w");
  if (f->fp == (FILE *) NULL)
    {
      fprintf(stderr, MSG_ERR_E002, fname);
      fprintf(stderr, "\t%s\n", strerror(errno));
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  strncpy(f->fname, fname, PATH_MAX);
  f->ok_to_close = TRUE;
  return;

} /* open_out() */

/*
 * close_file()
 */
void close_file(struct s_file_info *f)
{
  if (f->ok_to_close == TRUE)
    fclose(f->fp);

  init_file(f);
  
} /* close_file() */

/*
 * init_counters() -- initialize work area
 */
void init_counters(struct s_counters *c)

{

  c->seconds_ave         = (time_t) 0;
  c->seconds_max         = (time_t) 0;
  c->seconds_min         = (time_t) LONG_MAX;
  c->seconds_total_diff  = (COUNTER) 0;
  c->selected_records    = (COUNTER) 0;
  c->total_records       = (COUNTER) 0;
  memset(c->item_name, JLIB2_CHAR_NULL, (MAX_SIZE_ITEM + 1));

} /*  init_counters() */

/*
 * init_w() -- initialize work area
 */
void init_w(struct s_work *w)

{

  w->writes      = (COUNTER) 0;
  w->verbose     = 0;
  w->num_files   = 0;
  w->all         = FALSE;
  w->out_csv     = FALSE;
  w->show_title  = FALSE;
  w->force       = FALSE;

  init_file(&(w->out));
  init_file(&(w->err));

  w->out.fp = stdout;
  w->err.fp = stderr;

}  /* init_w() */

/*
 * show_rev()
 */
void show_rev(FILE *fp)
{
  fprintf(fp,"%s: %s\n", PROG_NAME, MSG_HELP_11);

#ifdef J_LIB2_H
  fprintf(fp, "\t%s\n", j2_get_build());
#endif

  fprintf(fp, "\t%s %s: %s - ", LIT_REV, PROG_NAME, RTAVE_H);
  fprintf(fp, LIT_INFO_04, __DATE__, __TIME__);

#ifdef OSTYPE
  fprintf(fp,"\t%s\n",OSTYPE);
#endif  /* OSTYPE  */
#ifdef PRETTY_NAME
  fprintf(fp,"\t%s\n",PRETTY_NAME);
#endif  /* OSTYPE  */

  exit(EXIT_FAILURE);

} /* show_rev() */

/*
 * show_brief_help();
 */
void show_brief_help(FILE *fp)
{
  fprintf(fp, USG_MSG_USAGE, PROG_NAME);
  fprintf(fp, "\t%s\n", MSG_HELP_11);
  fprintf(fp, USG_MSG_OPTIONS);
  fprintf(fp, USG_MSG_ARG_ALL_1,          SWITCH_CHAR, ARG_ALL);
  fprintf(fp, USG_MSG_ARG_FMT_CSV,        SWITCH_CHAR, ARG_FMT_CSV);
  fprintf(fp, USG_MSG_ARG_ERR,            SWITCH_CHAR, ARG_ERR);
  fprintf(fp, USG_MSG_ARG_FORCE,          SWITCH_CHAR, ARG_FORCE);
  fprintf(fp, USG_MSG_ARG_SHOW_HEADING,   SWITCH_CHAR, ARG_SHOW_HEADING);
  fprintf(fp, USG_MSG_ARG_HELP,           SWITCH_CHAR, ARG_HELP);
  fprintf(fp, USG_MSG_ARG_OUT,            SWITCH_CHAR, ARG_OUT);
  fprintf(fp, USG_MSG_ARG_VERSION,        SWITCH_CHAR, ARG_VERSION);
  fprintf(fp, USG_MSG_ARG_VERBOSE_8,      SWITCH_CHAR, ARG_VERBOSE);

  exit(EXIT_FAILURE);

} /* show_brief_help() */

/*
 * process_arg() -- process arguments
 */
void process_arg(int argc, char **argv, struct s_work *w)

{
  char ckarg[SCKARG];
  int opt         = 0;
  int i           = 0;
  int count_std   = 0;
  char *efile     = (char *) NULL;
  char *ofile     = (char *) NULL;

  snprintf(ckarg, SCKARG, "%c%c%c%c%c%c%c%c:%c:", 
	   ARG_ALL, ARG_FMT_CSV, ARG_HELP, ARG_SHOW_HEADING,
           ARG_VERBOSE, ARG_VERSION, ARG_FORCE,
           ARG_OUT, ARG_ERR);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	case ARG_ERR:
	  if (efile == (char *) NULL)
	    {
	      check_path_file(stderr, optarg, ARG_ERR);
	      efile = optarg;
	    }
	  else
	    {
	      fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_OUT:
	  if (ofile == (char *) NULL)
	    {
	      check_path_file(stderr, optarg, ARG_OUT);
	      ofile = optarg;
	    }
	  else
	    {
	      fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
	      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	      exit(EXIT_FAILURE);
	    }
	  break;
	case ARG_FORCE:
	  w->force = TRUE;
	  break;
	case ARG_ALL:
	  w->all = TRUE;
	  break;
	case ARG_FMT_CSV:
	  w->out_csv = TRUE;
	  break;
	case ARG_HELP:
	  show_brief_help(stderr);
	  break;
	case ARG_SHOW_HEADING:
	  w->show_title = TRUE;
	  break;
	case ARG_VERBOSE:
	  w->verbose++;
	  break;
	case ARG_VERSION:
	  show_rev(stderr);
	  break;
	default:
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	  break;
	}
    }

  /*** open 'out' files ***/
  open_out(&(w->err), efile, w->force);
  open_out(&(w->out), ofile, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* END process_arg() */

/*
 * init() -- initialize
 */
void init(int argc, char **argv, struct s_work *w)

{

  init_w(w);

  process_arg(argc, argv, w);

  if (w->verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      fprintf(w->err.fp, MSG_INFO_I087, (w->force      == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I224, (w->all        == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I084, (w->out_csv    == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I219, (w->show_title == TRUE ? LIT_YES : LIT_NO));
    }

}  /* init() */
