/*
 * Copyright (c) 2003 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <libgen.h>

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "jwc.h"

#define WCI_AMODE_NORMAL 0
#define WCI_AMODE_TAB    1
#define WCI_MAX_ENV_ARG  100
#define SCKARG           80

/*
 * verify_path() -- make sure dir/file name is OK
 */
int verify_path(char *fname)
{
  char *s;

  if (fname == (char *) NULL)
    return(PATH_IS_NULL);
  if (strlen(fname) < 1)
    return(PATH_IS_EMPTY);
  if (strlen(fname) > (PATH_MAX - 2))
    return(PATH_IS_BIG);
  if ((strlen(fname) > 1) && (*fname == '-'))
    return(PATH_INVALID);

  for (s = fname; (*s) != JLIB2_CHAR_NULL; s++)
    {
      if (isspace((int) (*s)))
	return(PATH_HAS_SPACE);
    }

  return(PATH_IS_VALID);

} /* verify_path() */

/*
 * check_path_file() -- Check and Save a Dir Name
 */
void check_path_file(FILE *efp, char *fname, char arg_switch)
{

  char sw = SWITCH_CHAR;
  char ag = arg_switch;

  if (arg_switch == JLIB2_CHAR_NULL)
    {
      sw = 'i';
      ag = 'n';
    }

  switch (verify_path(fname))
    {
      case PATH_IS_VALID:
	break;
      case PATH_IS_NULL:
	fprintf(efp, MSG_ERR_E004S, LIT_NULL,  sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_EMPTY:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_IS_BIG:
	fprintf(efp, MSG_ERR_E004SB, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,   PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_HAS_SPACE:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      case PATH_INVALID:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
      default:
	fprintf(efp, MSG_ERR_E004S, fname,     sw, ag);
	fprintf(efp, MSG_ERR_E000,  PROG_NAME, SWITCH_CHAR, ARG_HELP);
	exit(EXIT_FAILURE);
    }

} /* check_path_file() */

/*
 * init_line_count()
 */
void init_line_count(struct s_line_count *l)
{

  l->file_count    = (COUNT_NUM) 0;
  l->lines         = (COUNT_NUM) 0;
  l->words         = (COUNT_NUM) 0;
  l->bytes         = (COUNT_NUM) 0;
  l->min_line_size = (COUNT_NUM) 0;
  l->min_line_num  = (COUNT_NUM) 0;
  l->max_line_size = (COUNT_NUM) 0;
  l->max_line_num  = (COUNT_NUM) 0;

} /* init_line_count() */

/*
 * init_finfo() -- initialize out file structure
 */
void init_finfo(struct s_file_info *f)

{

  f->fp          = (FILE *) NULL;
  f->ok_to_close = FALSE;
  memset(f->fname, 0, (PATH_MAX + 1));

} /* init_finfo() */

/*
 * close_file() -- close an opened file
 */
void close_file(struct s_file_info *f)

{
  if (f->ok_to_close == TRUE)
    fclose(f->fp);

  init_finfo(f);

} /* close_file() */

/*
 * open_out() -- save the file name and check status
 */
void open_out(struct s_file_info *f, char *fname, int force)

{

  if (fname == (char *) NULL)
    return;
  if (strlen(fname) < 1)
    return;
  if (strcmp(fname, FILE_NAME_STD) == 0)
    return;
  if (f->ok_to_close == TRUE) /* file in use */
    {
      fprintf(stderr, MSG_ERR_E112, fname);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  if (force == FALSE)
    {
      if ( j2_f_exist(fname) )
	{
	  fprintf(stderr, MSG_ERR_E025, fname);
	  fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	  exit(EXIT_FAILURE);
	}
    }

  f->fp = fopen(fname, "w");
  if (f->fp == (FILE *) NULL)
    {
      fprintf(stderr, MSG_ERR_E002, fname);
      fprintf(stderr, "\t%s\n", strerror(errno));
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  strncpy(f->fname, fname, PATH_MAX);
  f->ok_to_close = TRUE;
  return;

} /* open_out() */

/*
 * open_in() -- open in file
 */
int open_in(struct s_file_info *err, struct s_file_info *ifile, char *fname)

{

  if (fname == (char *) NULL)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strlen(fname) < 1)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (strncmp(fname, FILE_NAME_STD, PATH_MAX) == 0)
    {
      ifile->fp = stdin;
      return(TRUE);
    }
  if (ifile->ok_to_close == TRUE) /* file in use */
    {
      fprintf(err->fp, MSG_ERR_E112, fname);
      fprintf(err->fp, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  ifile->fp = fopen(fname, "r");

  if (ifile->fp == (FILE *) NULL)
    {
      fprintf(err->fp, MSG_WARN_W002, fname, strerror(errno));
      return(FALSE);
    }
 
  strncpy(ifile->fname, fname, PATH_MAX);
  ifile->ok_to_close = TRUE;
  return(TRUE);

} /* open_in() */

/*
 * parse_arg() -- Set an argument value
 */
void parse_arg(work_area *w, int argc, char **argv)
{

  char ckarg[SCKARG];
  int  opt       = 0;
  int  i         = 0;
  int  count_std = 0;
  char *fout     = (char *) NULL; 
  char *ferr     = (char *) NULL; 

  snprintf(ckarg, SCKARG, "%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c:%c:",
          ARG_ALL, ARG_BYTE, ARG_EXPAND, ARG_FORCE, ARG_HELP, 
          ARG_INCLUDE_NL, ARG_LONG_LEN, ARG_SHORT_LINES,
          ARG_SHORT_LEN, ARG_VERBOSE, ARG_VERSION, ARG_WORD_COUNT,
          ARG_SHOW_HEADING, ARG_LONG_LINES,
          ARG_ERR, ARG_OUT);


  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	  case ARG_ERR:
	    if (ferr == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_ERR);
		ferr = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_FORCE:
	    w->force = TRUE;
	    break;
	  case ARG_HELP:
	    show_brief_help();
	    break;
	  case ARG_OUT:
	    if (fout == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_OUT);
		fout = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_SHOW_HEADING:
	    w->headings  = TRUE;
	    break;
	  case ARG_VERBOSE:
	    w->verbose++;
	    break;
	  case ARG_VERSION:
	    show_rev();
	    break;
	  case ARG_ALL:
	    w->headings      = TRUE;
	    w->show_bytes    = TRUE;
	    w->show_lines    = TRUE;
	    w->show_words    = TRUE;
	    w->show_min_line = TRUE;
	    w->show_min_num  = TRUE;
	    w->show_max_line = TRUE;
	    w->show_max_num  = TRUE;
	    break;
	  case ARG_BYTE:
	    w->show_bytes = TRUE;
	    break;
	  case ARG_EXPAND:
	    w->expand_tabs = TRUE;
	    break;
	  case ARG_LONG_LEN:
	    w->show_max_line = TRUE;
	    break;
	  case ARG_INCLUDE_NL:
	    w->show_lines = TRUE;
	    break;
	  case ARG_SHORT_LINES:
	    w->show_min_num = TRUE;
	    break;
	  case ARG_SHORT_LEN:
	    w->show_min_line = TRUE;
	    break;
	  case ARG_WORD_COUNT:
	    w->show_words = TRUE;
	    break;
	  case ARG_LONG_LINES:
	    w->show_max_num = TRUE;
	    break;
	  default:
	    fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(EXIT_FAILURE);
	    break;
	}
    }

  /*** if necessary - save stdout/err files and open ***/
  open_out(&(w->err), ferr, w->force);
  open_out(&(w->out), fout, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

} /* parse_arg() */

/*
 * init_work() -- clear work area
 */
void init_work(work_area *w)
{

  w->arg_switch        = SWITCH_CHAR;
  w->expand_tabs       = FALSE;
  w->force             = FALSE;
  w->headings          = FALSE;
  w->show_lines        = FALSE;
  w->show_words        = FALSE;
  w->show_bytes        = FALSE;
  w->show_min_line     = FALSE;
  w->show_min_num      = FALSE;
  w->show_max_line     = FALSE;
  w->show_max_num      = FALSE;
  w->show_arg_found    = FALSE;

  w->verbose           = 0;
  w->size_expanded     = 0;
  w->num_files         = 0;
  w->expanded          = (char *) NULL;
  w->bad_arg           = (char *) NULL;

  init_line_count(&(w->totals));
  init_finfo(&(w->err));
  init_finfo(&(w->out));

  w->out.fp = stdout;
  w->err.fp = stderr;

} /* init_work() */

/*
 * init() -- init
 */
void init(work_area *w, int argc, char **argv)
{

  init_work(w);

  parse_arg(w, argc, argv);

  if (w->verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      if (w->expand_tabs == TRUE)
	fprintf(w->err.fp, MSG_INFO_I167, EXPAND_TAB_DEFAULT);
      fprintf(w->err.fp, MSG_INFO_I087, (w->force         == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I212, (w->show_bytes    == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I214, (w->show_lines    == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I217, (w->show_words    == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I215, (w->show_min_num  == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I216, (w->show_min_line == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I218, (w->show_max_num  == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I213, (w->show_max_line == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I219, (w->headings      == TRUE ? LIT_YES : LIT_NO));
    }

} /* init() */
