/*
 * Copyright (c) 1999 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define _FILE_OFFSET_BITS 64
#define __USE_LARGEFILE64
#define _TIME_BITS 64

#ifndef _MSDOS
#include <sys/param.h>
#endif
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#ifdef OpenBSD
#include <err.h>
#endif
#ifdef __FreeBSD_version
#include <err.h>
#endif
#ifdef __NetBSD_Version__
#include <err.h>
#endif

#ifdef HAVE_JLIB
#include <j_lib2.h>
#include <j_lib2m.h>
#endif

#include "justleft.h"

#define SARG 80

/*
 * show_file_heading() -- Show run stats
 */
long int show_file_heading(FILE *fp, char *fname)

{
  long int bwrite = 0;

  bwrite += fprintf(fp, "%s\n", LIT_C80);

  if (fname == (char *) NULL)
    bwrite += fprintf(fp, "%s\n", LIT_STDIN);
  else
    {
      if (strcmp(fname, FILE_NAME_STD) == 0)
	bwrite += fprintf(fp, "%s\n", LIT_STDIN);
      else
	bwrite += fprintf(fp, "%s\n", fname);
    }

  bwrite += fprintf(fp, "%s\n", LIT_C80);

  return(bwrite);

} /* show_file_heading() */

/*
 * get_args() -- load arguments
 */
void get_args(int argc, char **argv, struct s_work *w)

{

  char ckarg[SARG];
  int opt        = 0;
  int i          = 0;
  int count_std  = 0;
  char *ofile    = (char *) NULL;
  char *efile    = (char *) NULL;

  init_finfo(&(w->err));
  init_finfo(&(w->out));
  w->out.fp     = stdout;
  w->err.fp     = stderr;
  w->num_files  = 0;
  w->verbose    = 0;
  w->force      = FALSE;
  w->rtw        = FALSE;

  snprintf(ckarg, SARG, "%c:%c:%c%c%c%c%c", 
          ARG_ERR,  ARG_OUT, 
          ARG_HELP, ARG_RTW, ARG_FORCE, ARG_VERSION, ARG_VERBOSE);

  while ((opt = getopt(argc, argv, ckarg)) != -1)
    {
      switch (opt)
	{
	  case ARG_HELP:
	    show_help();
	    break;
	  case ARG_VERSION:
	    show_rev();
	    break;
	  case ARG_FORCE:
	    w->force = (int) TRUE;
	    break;
	  case ARG_VERBOSE:
	    (w->verbose)++;
	    break;
	  case ARG_RTW:
	    w->rtw = (int) TRUE;
	    break;
	  case ARG_OUT:
	    if (ofile == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_OUT);
		ofile = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  case ARG_ERR:
	    if (efile == (char *) NULL)
	      {
		check_path_file(stderr, optarg, ARG_ERR);
		efile = optarg;
	      }
	    else
	      {
		fprintf(stderr, MSG_ERR_E074, SWITCH_CHAR, ARG_ERR);
		fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
		exit(EXIT_FAILURE);
	      }
	    break;
	  default:
	    fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
	    exit(EXIT_FAILURE);
	    break;
	}
    }

  /*** if necessary - save stdout/err files and open ***/
  open_out(&(w->err), efile, w->force);
  open_out(&(w->out), ofile, w->force);

  /*** Count number of files to process and check file name size */
  for (i = optind; i < argc; i++)
    {
      check_path_file(w->err.fp, argv[i], JLIB2_CHAR_NULL);
      (w->num_files)++;
      if (strncmp(argv[i], FILE_NAME_STD, PATH_MAX) == 0)
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
	  count_std++;
	}
      else
	{
	  if (w->verbose > 1)
	    fprintf(w->err.fp, MSG_INFO_I043L, argv[i]);
	}
    }
  if (w->num_files == 0)
    {
      (w->num_files)++;  /* stdin when no files */
      if (w->verbose > 1)
	fprintf(w->err.fp, MSG_INFO_I043L, LIT_STDIN);
    }
  if (count_std > 1)
    {
      fprintf(stderr, MSG_ERR_E113);
      fprintf(stderr, MSG_ERR_E000, PROG_NAME, SWITCH_CHAR, ARG_HELP);
      exit(EXIT_FAILURE);
    }

  /*** show arguments ***/
  if (w->verbose > 1)
    {
      fprintf(w->err.fp, MSG_INFO_I086,
              (strlen(w->err.fname) == 0 ? LIT_STDERR : w->err.fname));
      fprintf(w->err.fp, MSG_INFO_I090,
              (strlen(w->out.fname) == 0 ? LIT_STDOUT : w->out.fname));
      fprintf(w->err.fp, MSG_INFO_I081, w->num_files);
      fprintf(w->err.fp, MSG_INFO_I092, w->verbose);
      fprintf(w->err.fp, MSG_INFO_I186, (w->rtw   == TRUE ? LIT_YES : LIT_NO));
      fprintf(w->err.fp, MSG_INFO_I087, (w->force == TRUE ? LIT_YES : LIT_NO));
    }

} /* get_args() */

/*
 * process_a_file() -- reads in file, creating out file
 */
void process_a_file(struct s_work *w, char *fname, char **buf, size_t *bsize,
		    struct s_counts *t)

{
  ssize_t cbytes = (ssize_t) 0;
  struct s_counts lines;
  struct s_file_info ifile;

  init_counts(&lines);
  init_finfo(&ifile);
  
  if (w->verbose > 2)
    {
      lines.bytes_writes += show_file_heading(w->out.fp, fname);
      lines.writes += 3; /* this function writes 3 lines */
    }

  if ( open_in(&(w->err), &ifile, fname) == FALSE)
    return;

  /*** process data ***/
  while ((cbytes = j2_getline(buf, bsize, ifile.fp)) > (ssize_t) -1)
    {
      lines.read++;
      lines.bytes_read += cbytes;
      j2_bye_nl((*buf));
      j2_justleft((*buf));
      if (w->rtw == (int) TRUE)
	j2_rtw((*buf));
      if (strlen((*buf)) > 0)
	lines.bytes_writes += fprintf(w->out.fp,"%s\n",(*buf));
      else
	lines.bytes_writes += fprintf(w->out.fp,"\n");
      lines.writes++;
    }
  
  /*** complete ***/
  if (w->verbose > 0)
    {
      t->read += lines.read;
      t->writes += lines.writes;
      t->bytes_read += lines.bytes_read;
      t->bytes_writes += lines.bytes_writes;
      show_file_heading(w->err.fp, fname);
      fprintf(w->err.fp, MSG_INFO_I023, lines.read);
      fprintf(w->err.fp, MSG_INFO_I035, lines.writes);
      fprintf(w->err.fp, MSG_INFO_I024, lines.bytes_read);
      fprintf(w->err.fp, MSG_INFO_I041, lines.bytes_writes);
    }
  
  close_file(&ifile);

} /* process_a_file() */

/*
 * process_all() -- Process all files
 */
void process_all(int argc, char **argv, struct s_work *w)

{
  int i;
  char *buf = (char *) NULL;
  size_t bsiz = (size_t) 200;
  struct s_counts totl;
  
  init_counts(&totl);

  /* allocate initial read buffer memory (optional) */
  buf = (char *) calloc(bsiz, sizeof(char));
  if (buf == (char *) NULL)
    {
      fprintf(w->err.fp, MSG_ERR_E080, strerror(errno));
      return;
    }

  /* process files */
  for (i = optind; i < argc; i++)
    process_a_file(w, argv[i], &buf, &bsiz, &totl);

  if (i == optind)
    process_a_file(w, FILE_NAME_STD, &buf, &bsiz, &totl);

  if (buf != (char *) NULL)
    free(buf);

  if (w->verbose > 0)
    {
      show_file_heading(w->err.fp, LIT_TALLF);
      fprintf(w->err.fp, MSG_INFO_I023, totl.read);
      fprintf(w->err.fp, MSG_INFO_I035, totl.writes);
      fprintf(w->err.fp, MSG_INFO_I024, totl.bytes_read);
      fprintf(w->err.fp, MSG_INFO_I041, totl.bytes_writes);
    }

}  /* process_all() */

/*
 * main()
 */
int main(int argc, char **argv)

{

  time_t tstart = time(&tstart), tend;
  struct s_work w;

#ifdef OpenBSD
  if(pledge("stdio rpath wpath cpath",NULL) == -1)
    err(1,"pledge\n");
#endif

  get_args(argc, argv, &w);
  process_all(argc, argv, &w);

  if (w.verbose > 0)
    {
      fprintf(w.err.fp, "\n");
      fprintf(w.err.fp, MSG_INFO_I152S, (long long int) (time(&tend) - tstart));
    }

  close_file(&(w.out));
  close_file(&(w.err));
  exit(EXIT_SUCCESS);

}  /* main() */

/* END: justleft.c */
