#!/bin/sh
#
# jsplit - test with valgrind or ktrace
#
# Copyright (c) 2022 ... 2025 2026
#     John McCue <jmccue@sdf.org>
#
# Permission to use, copy, modify, and distribute this software
# for any purpose with or without fee is hereby granted, provided
# that the above copyright notice and this permission notice
# appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
# WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
# CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

g_dir_bhome=`dirname $0`
cd $g_dir_bhome
g_dir_bhome=`pwd`
. $g_dir_bhome/../test/tsetup.sh

g_file_in=""
g_file_binary="$DIR_RESULTS/in_binary.dd"
g_file_text="$DIR_RESULTS/in_text.txt"
g_file_err=$DIR_RESULTS/val_err.txt
g_file_combined=$DIR_RESULTS/val_combined
g_arg_mode=""

cd $TEST_DIR_SRC
f_ckprog cmp
f_ckprog jcomb
f_ckprog $TEST_DIR_SRC/jsplit

#--- validate arguments
case "$TEST_ARG" in
    "-b")
	g_arg_mode=""
	g_file_in="$g_file_binary"
	;;
    "-t")
	g_arg_mode="-t"
	g_file_in="$g_file_text"
	;;
    *)
	echo ""
	g_is_error="Y"
	f_help_setup
	f_msg "W010: Required test option missing or invalid, exiting"
	exit 2
	;;
esac

#--- create files
if test ! -f "$g_file_binary"
then
    f_msg "I100: creating $g_file_binary"
    dd if=/dev/urandom of="$g_file_binary" bs=1000 count=15
fi
if test ! -f "$g_file_text"
then
    f_msg "I101: creating $g_file_text"
    echo '    leading and trailing spaces         ' > "$g_file_text"
    tr -cd '[:upper:]' < /dev/urandom | fold -w  40 \
       | sed 14999q | sort >> "$g_file_text"
fi

#--- print version info
$TEST_DIR_SRC/jsplit -V > "$FILE_VERSION_RESULTS" 2>&1
cat << EOF >> $FILE_MAKE_RESULTS 2>&1
=====================================================================
Source  Dir: $TEST_DIR_SRC
Results Dir: $DIR_RESULTS

Program Rev:
---------------------------------------------------------------------
`cat $FILE_VERSION_RESULTS`
=====================================================================
EOF

#--- cleanup
f_msg "I100: Cleaning $DIR_RESULTS"
find $DIR_RESULTS -type f -name 'v0*.x' -exec rm '{}' \;
rm "$g_file_err" "$g_file_combined"         > /dev/null 2>&1
rm $FILE_VERSION_RESULTS $FILE_VALGRIND_LOG > /dev/null 2>&1

cd "$DIR_RESULTS"

#--- Test object
if test "$PROCESS_VALGRIND" = "Y"
then
    f_msg "A210: START jsplit $MSG_MEMORY" > "$FILE_VALGRIND_LOG"
    f_msg "I211: START jsplit $MSG_MEMORY"
    valgrind -s --tool=memcheck --leak-check=yes               \
                --leak-check=full --show-leak-kinds=all        \
                --track-origins=yes                            \
                --keep-stacktraces=alloc-then-free             \
	        $TEST_DIR_SRC/jsplit $g_arg_mode -M 5000              \
                                     -f -vv -e $g_file_err $g_file_in \
                >> "$FILE_VALGRIND_LOG" 2>&1
    if test "$TERM" = "dumb"
    then
	cat $FILE_VALGRIND_LOG
    fi
    f_msg "I281: Created: $FILE_VALGRIND_LOG"
    f_msg "I282:          $g_file_err"
    f_msg "I290: END   jsplit $MSG_MEMORY"
    f_msg "B291: END   jsplit $MSG_MEMORY"   >> "$FILE_VALGRIND_LOG"
else
    f_msg "I300: START ./jsplit $MSG_MEMORY"
    $OPENBSD_KTRACE $TEST_DIR_SRC/jsplit $g_arg_mode -M 5000 \
                     -f -vv -e $g_file_err $g_file_in
    f_msg "I301: Created: $g_file_err"
    if test "$OPENBSD_KTRACE" != ""
    then
	sleep 1
	kdump -u malloc
	sleep 1
	f_msg "I313: ***** Execute this:"
	f_msg "I314:       addr2line -e ./jsplit 0x????"
	f_msg "I315: ***** using the last hex val on the line for ./jsplit"
	f_msg "I316: ***** to see where the leak is (if any)"
    fi
    f_msg "I320: END   ./jsplit $MSG_MEMORY"
fi

#--- validate results
cd "$DIR_RESULTS"
f_msg "I401: checking results"
jcomb -f "$g_file_combined"
cmp "$g_file_combined" "$g_file_in" > /dev/null 2>&1
if test "$?" -eq "0"
then
    f_msg "I410: SUCCESS ./jsplit"
else
    f_msg "W411: FAILED  ./jsplit"
    f_msg "W412: See Files in Dir:"
    f_msg "E413    $DIR_RESULTS"
fi

exit 0
