/*
 * Copyright (c) 2012 ... 2025 2026
 *     John McCue <jmccue@sdf.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef JSPLIT_H

#define JSPLIT_H   "2.15 2025/07/30"
#define PROG_NAME  "jsplit"

#define JS_MAX_BYTES 1000

#ifndef JLIB2_CHAR_NULL
#define NO_JLIB 1
#define JLIB2_CHAR_NULL  ((char) '\0')
#endif

#ifndef NULL
#define NULL '\0'
#endif
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#ifndef SSIZE_T
#define SSIZE_T ssize_t
#endif

#ifdef MAXPATHLEN
#ifndef PATH_MAX
#define PATH_MAX MAXPATHLEN
#endif
#else
#ifdef PATH_MAX
#define MAXPATHLEN PATH_MAX
#endif
#endif

#ifdef NO_JLIB
typedef unsigned long jm_counter;  /* allow 16 bit systems */
#define JS_MAX_SPLIT      ((jm_counter) 9999999)
#define JS_MAX_SIZE       12         /* format v######.x   # < JS_MAX_SPLIT */
#define JS_FNAME          "v%07lu.x" /* see jm_counter above if changed     */
#endif

/*** structures ***/
struct s_file_info
  {
    FILE *fp;
    char fname[(PATH_MAX + 1)];
    int ok_to_close;
    jm_counter io;
  } ;

struct s_work
  {
    struct s_file_info err;     /* default stderr              */
    jm_counter min_allowed;     /* Minimum recs allowed        */
    jm_counter max_allowed;     /* split into # of bytes/lines */
    jm_counter max_items;       /* split into # of bytes/lines */
    jm_counter limit;           /* limit # of files created    */
    int num_files;              /* # of files to process       */
    int verbose;                /* TRUE or FALSE               */
    int force;                  /* TRUE or FALSE               */
    int split_binary;           /* TRUE or FALSE               */
    int random_split;           /* TRUE or FALSE               */
  } ;

#ifdef NO_JLIB
#define ARG_ERR           'e'  /* Output Error File                  */
#define ARG_FORCE         'f'  /* force create files                 */
#define ARG_HELP          'h'  /* Show Help                          */
#define ARG_LIMIT_FILES   'l'  /* Limit the number of files created  */
#define ARG_MAX_SIZE      'M'  /* Maximum Size                       */
#define ARG_MIN_SIZE      'L'  /* Low value for File Split           */
#define ARG_RANDOM_SPLIT  'r'  /* Random size on split               */
#define ARG_TEXT          't'  /* Process text mode                  */
#define ARG_VERBOSE       'v'  /* Verbose                            */
#define ARG_VERSION       'V'  /* Show Version Information           */
#define FILE_NAME_STD     "-"
#define LIT_INFO_04       "Build: %s %s\n"
#define LIT_LBINARY       "binary"
#define LIT_LBYTES        "bytes"
#define LIT_LLINES        "lines"
#define LIT_LTEXT         "text"
#define LIT_NO            "No"
#define LIT_READ          "read"
#define LIT_REV           "Revision"
#define LIT_STDERR        "(standard error)"
#define LIT_STDIN         "(standard input)"
#define LIT_STDOUT        "(standard output)"
#define LIT_TOTAL         "Total"
#define LIT_WRITTEN       "written"
#define LIT_YES           "Yes"
#define MSG_ERR_E000      "Try '%s %c%c' for more information\n"
#define MSG_ERR_E002      "ERROR E002: Cannot open '%s' for write, processing aborted\n"
#define MSG_ERR_E004LU    "ERROR E004: value %lu invalid value for %c%c\n"
#define MSG_ERR_E008      "ERROR E008: '%s' is an invalid value for %c%c, must be numeric\n"
#define MSG_ERR_E025      "ERROR E025: File %s cannot be created, already exists\n"
#define MSG_ERR_E028      "ERROR E028: too many files specified\n"
#define MSG_ERR_E055      "ERROR E055: Write Error on File '%s' : %s\n"
#define MSG_ERR_E056      "ERROR E056: Open Error on File '%s' : %s\n"
#define MSG_ERR_E084      "ERROR E084: Value for %c%c (%ld) less than the Value for %c%c (%ld)\n"
#define MSG_ERR_E085      "ERROR E085: Value for %c%c greater than 0\n"
#define MSG_ERR_E112      "ERROR E112: cannot open %s, already opened\n"
#define MSG_INFO_I043L    "I043: Input  File                  : %s\n"
#define MSG_INFO_I086     "I086: Errors written to File       : %s\n"
#define MSG_INFO_I087     "I087: Force Create Output          ? %s\n"
#define MSG_INFO_I092     "I092: Verbose Level                  %d\n"
#define MSG_INFO_I152S    "I152: Run Time: %lld second(s)\n"
#define MSG_INFO_I170     "I170: Split using Random Sizes\n"
#define MSG_INFO_I171     "I171: Minimum Split Size           : %-llu\n"
#define MSG_INFO_I172     "I172: Maximum Split Size           : %-llu\n"
#define MSG_INFO_I173     "I173: Split using a Fixed Size\n"
#define MSG_INFO_I174     "I174: Fixed Split Size             : %-llu\n"
#define MSG_INFO_I175     "I175: Max # Split Files to create  : %-llu\n"
#define MSG_INFO_I176     "I176: Split using Binary Mode\n"
#define MSG_INFO_I177     "I176: Split using Text Mode\n"
#define MSG_RPT_R12       "Splitting %s file %s\n"
#define MSG_RPT_R13       "          into multiple files with a maximum\n"
#define MSG_RPT_R14       "          of %lu %s per file,\n"
#define MSG_RPT_R15       "          limiting files created to %lu\n\n"
#define MSG_RPT_R16       "%s  in/out counts:\n"
#define MSG_RPT_R17       "    I/O type       I/O %s  File Name\n"
#define MSG_RPT_R18       "    --------- --------------- ------------------\n"
#define MSG_RPT_R19       "    %9s %15lu %s\n"
#define MSG_RPT_R20       "    ========= =============== ==================\n"
#define MSG_RPT_R21       "          Output Size Randomized using range of %lu to %lu\n"
#define SWITCH_CHAR       '-'
#define USG_MSG_ARG_ERR           "\t%c%c file\t\t: Write errors to file 'file', default stderr\n"
#define USG_MSG_ARG_FORCE         "\t%c%c\t\t: force create of files when found\n"
#define USG_MSG_ARG_HELP          "\t%c%c\t\t: Show brief help and exit\n"
#define USG_MSG_ARG_LIMIT_FILES   "\t%c%c num\t\t: Limit the number of files created\n"
#define USG_MSG_ARG_MAX_2         "\t%c%c max\t\t: Split file into 'max' bytes/lines per output file\n"
#define USG_MSG_ARG_MIN_2         "\t%c%c min\t\t: Split file into 'min' bytes/lines per output file\n"
#define USG_MSG_ARG_RANDOM_SPLIT  "\t%c%c\t\t: Split files using a Random Max Value\n"
#define USG_MSG_ARG_TEXT_1        "\t%c%c\t\t: Split in text mode\n"
#define USG_MSG_ARG_VERBOSE_5     "\t%c%c\t\t: show run stats on stderr\n"
#define USG_MSG_ARG_VERSION       "\t%c%c\t\t: Show revision information and exit\n"
#define USG_MSG_OPTIONS           "Options\n"
#define USG_MSG_USAGE_5           "usage:\t%s [OPTIONS] [FILE]\n"
#endif /* NO_JLIB */

/*** prototypes ***/
void init(int, char **, struct s_work *);
void init_finfo(struct s_file_info *);
void show_brief_help();
void show_rev();
void process_arg(int, char **, struct s_work *);
void open_out(struct s_file_info *, char *, char *, int);
void close_out(struct s_file_info *);
int  open_in(FILE **, char *, char *, FILE *);
void close_in(FILE **, char *);
void split_oout(struct s_file_info *, char *, char *, jm_counter *);
void splitbin(struct s_work *, char *);
void splittxt(struct s_work *, char *);
void split_log(int, struct s_work *, struct s_file_info *, char *, char *, char *, char *);

#ifdef NO_JLIB
SSIZE_T j2_getline(char **buf, size_t *n, FILE *fp);
int j2_f_exist(char *file_name);
int j2_is_numr(char *s);
long int j2_clr_str(char *s, char c, int size);
unsigned int j2_getseed(void);
unsigned int j2_randrange(unsigned int low, unsigned int max);
#endif /* NO_JLIB */

#endif /*  JSPLIT_H  */
